// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

#ifndef vtkWebGPUComputeOcclusionCuller_h
#define vtkWebGPUComputeOcclusionCuller_h

#include "vtkCallbackCommand.h" // for the bounds recomputed callback
#include "vtkCuller.h"
#include "vtkNew.h"                   // for new macro
#include "vtkRenderingWebGPUModule.h" // For export macro
#include "vtkSmartPointer.h"          // for the pipeline smart pointer
#include "vtkWebGPUComputePass.h"     // for compute passes
#include "vtkWebGPUComputePipeline.h" // for the member compute pipeline

VTK_ABI_NAMESPACE_BEGIN

/**
 * This culler culls props that are occluded (behind) other props and that are not visible to the
 * camera because of that. This implementation uses the two-pass hierarchical z-buffer approach.
 *
 * To use this culler, simply instantiate it:
 *
 * vtkNew<vtkWebGPUComputeOcclusionCuller> webgpuOcclusionCuller;
 *
 * set its vtkWebGPURenderWindow:
 *
 * webgpuOcclusionCuller->SetRenderWindow(...)
 *
 * and add it to the cullers of your renderer:
 *
 * renderer->GetCullers()->AddItem(webgpuOcclusionCuller);
 *
 * Note that having both a frustum/coverage culler as well as an occlusion culler is a good idea,
 * they work well together. This means that you can first add a vtkWebGPUFrustumCuller to your
 * renderer (or use the default frustum coverage culler) and then add the
 * vtkWebGPUComputeOcclusionCuller. Having the vtkWebGPUComputeOcclusionCuller in second position
 * may be preferable since occlusion culling tends to be more expensive than frustum culling. This
 * means that executing the frustum culling first may cut the work that will be done by the
 * occlusion culler, resulting in better performance.
 */
class VTKRENDERINGWEBGPU_EXPORT vtkWebGPUComputeOcclusionCuller : public vtkCuller
{
public:
  vtkTypeMacro(vtkWebGPUComputeOcclusionCuller, vtkCuller);
  static vtkWebGPUComputeOcclusionCuller* New();

  void PrintSelf(ostream& os, vtkIndent indent);

  /**
   * Sets which render window this occlusion culler is going to work on
   */
  void SetRenderWindow(vtkWebGPURenderWindow* webGpuRenderWindow);

  /**
   * Culls props and returns the number of props that still need to be rendered after the culling
   */
  virtual double Cull(vtkRenderer* ren, vtkProp** propList, int& listLength, int& initialized);

protected:
  vtkWebGPUComputeOcclusionCuller();
  ~vtkWebGPUComputeOcclusionCuller();
  vtkWebGPUComputeOcclusionCuller(const vtkWebGPUComputeOcclusionCuller&) = delete;
  void operator=(const vtkWebGPUComputeOcclusionCuller&) = delete;

private:
  /**
   * Sets up the first compute pass for copying the depth buffer of the render window to the first
   * mip level (level 0) of the hierarchical Z-buffer
   */
  void SetupDepthBufferCopyPass();

  /**
   * Sets up the compute pass for compute the max-mipmaps of the depth buffer
   */
  void SetupMipmapsPass();

  /**
   * Sets up the buffer used in the culling pass
   */
  void SetupCullingPass();

  /**
   * First render pass that renders the props that were visible last frame and that passed the
   * previous culling tests (if any). This pass is needed to fill the z-buffer.
   *
   * Return a list of the props that were not rendered for the first (filling the depth buffer) but
   * that need to be tested for culling (they are in the propList given to the Cull() call).
   */
  void FirstPassRender(vtkRenderer* ren, vtkProp** propList, int listLength);

  /**
   * Copies the depth buffer filled by the rendering of the props of last frame into the mipmap
   * level 0 of the hierarchical z-buffer
   */
  void CopyDepthBuffer();

  /**
   * Computes the depth buffer max-mipmaps
   */
  void DepthMipmaps();

  /**
   * Culls the actors using the depth buffer mipmaps computed in the previous pass
   */
  void PropCulling(vtkRenderer* ren, vtkProp** propList, int& listLength);

  /**
   * Reuploads the camera MVP matrix to its GPU buffer
   */
  void UpdateCameraMVPBuffer(vtkRenderer* ren);

  /**
   * Resizes the various bounds buffers (inputBounds, outputBoundsIndices) and updates their data
   */
  void UpdateBoundsBuffers(vtkProp** propList, int listLength);

  /**
   * Sets-up the hierarchical z-buffer mipmapped texture
   */
  void CreateHiZBuffer();

  /**
   * Computes the number of mip levels for the given width and height and returns that number.
   *
   * The widths and heights of all the mip levels are also stored in MipmapWidths and MipmapHeights
   */
  int ComputeMipLevelsSizes(int width, int height);

  /**
   * Resizes the hi zbuffer texture to the given new width and height. The level 0 of the new
   * texture isn't initialized and the mipmaps are not immediately recomputed.
   */
  void ResizeHiZBuffer(uint32_t newWidth, uint32_t newHeight);

  /**
   * Recomputes the number of mipmaps of the hi-z buffer for the given newWidth and newHeight and
   * recreates all the texture views on the mipmap levels of the hi-z buffer
   */
  void ResizeHiZBufferMipmapsChain(uint32_t newWidth, uint32_t newHeight);

  /**
   * Creates the texture view of the hierarchical z buffer for copying the depth buffer of the
   * render window into it
   */
  void FinishSetupDepthCopyPass();

  /**
   * Creates the texture views for all the mipmap levels that are going to be needed to downsample
   * the depth buffer
   */
  void FinishSetupMipmapsPass();

  /**
   * Adds the texture view of the hierarchical z buffer to the culling pass
   */
  void FinishSetupCullingPass();

  /**
   * Callback to read the number of props that passed the culling test
   */
  static void ReadIndicesCountCallback(const void* mappedData, void* indicesCount);

  /**
   * Callback for reading the props that passed the culling test and store them in the propList of
   * the renderer (passed through the userdata of the callback).
   *
   * For a prop to be written to the propList that will be rendered, it needs to have passed the
   * culling test but also not having been rendered in the first pass (because rendering it twice is
   * useless so we're not adding a prop that was alredy rendered to the list of props that need to
   * be rendered).
   */
  static void FillObjectsToDrawCallback(const void* mappedData, void* data);

  /**
   * Callback that reads the indices of the props that were culled by the occlusion culling. These
   * props are then removed from the list of "props rendered last frame"  of the wgpu renderer
   */
  static void OutputIndicesCulledCallback(const void* mappedData, void* data);

  /**
   * Callback called when the render window of this occlusion culler is resized. This callback
   * resizes the hierarchical z-buffer
   */
  static void WindowResizedCallback(
    vtkObject* caller, unsigned long eid, void* clientdata, void* calldata);

  // Occlusion culling pipeline
  vtkSmartPointer<vtkWebGPUComputePipeline> OcclusionCullingPipeline;

  // Pass that copies the depth buffer of the render window into the mip level 0 of the hierarchical
  // z-buffer
  vtkSmartPointer<vtkWebGPUComputePass> DepthBufferCopyPass;

  // Index of the hierarchical z buffer in the depth buffer copy compute pass
  int HiZBufferTextureIndexCopyPass = -1;
  // Index of the hierarchical z buffer in the depth buffer mipmaps pass
  int HiZBufferTextureIndexMipmapsPass = -1;
  // Index of the hierarchical z buffer in the culling pass
  int HiZBufferTextureIndexCullingPass = -1;
  // All the views necessary for the computation of the depth buffer mipmaps
  std::vector<vtkSmartPointer<vtkWebGPUComputeTextureView>> HiZBufferMipmapViews;
  // Texture view indices within the DepthMipmapsPass compute pass
  std::vector<int> HiZBufferMipmapViewsIndices;
  // Total number of mipmaps of the hierarchical z buffer
  int HiZBufferMipmapCount = -1;
  // Widths of the successive mipmaps of the hierarchical z buffer
  std::vector<int> MipmapWidths;
  // Heights of the successive mipmaps of the hierarchical z buffer
  std::vector<int> MipmapHeights;
  // We need to keep the uniform buffer here because we can only create it once the RenderWindow has
  // set its Device to the pipeline of this occlusion culler. We will add the buffer to the pipeline
  // on the first frame, when we're sure that the Device has been set

  // Pass that downsamples the mipmap level 0 of the depth buffer into as many mipmap levels as
  // possible
  vtkSmartPointer<vtkWebGPUComputePass> DepthMipmapsPass;
  // Pass that does the culling of the actors against the hierarchial z-buffer
  vtkSmartPointer<vtkWebGPUComputePass> CullingPass;

  // Index of the bounds buffer in the culling pass
  int CullingPassBoundsBufferIndex = -1;
  // Index of the buffer that contains the indices of the props that passed the culling test in the
  // culling pass
  int CullingPassOutputIndicesBufferIndex = -1;
  // How many props that were sent to the culling shader passed the culling test
  int CullingPassOutputIndicesCountBufferIndex = -1;
  // Index of the buffer that contains the indices of the props that were culled. Needed to update
  // the visibility of the props in the PropsRendered array of the WGPURenderer
  int CullingPassOutputIndicesCulledBufferIndex = -1;
  // How many props were culled by the culling pass
  int CullingPassOutputIndicesCulledCountBufferIndex = -1;
  // Index of the buffer that contains the number of bounds to cull in the culling pass
  int CullingPassBoundsCountBufferIndex = -1;
  // Index of the buffer that contains the view projection matrix in the culling pass
  int CullingPassMVPMatrixBufferIndex = -1;

  // Structure passed to the callbacks for reading the results of the culling pass
  struct FillObjectsToDrawCallbackMapData
  {
    // How many props passed the culling test.
    // This is a pointer to the 'listLength' parameter of the Cull() method
    int* listLength = 0;
    // Prop list of the renderer that needs to be updated
    vtkProp** propList;

    // WebGPU renderer
    // Used for accessing the 'rendered last frame' list
    vtkWebGPURenderer* renderer;
  };

  // Structure passed to the callbacks for reading the results of the culling pass
  struct OutputIndicesCulledMapData
  {
    // Point to the renderer for removing the props that were culled from the list of "props
    // rendered last frame"
    vtkWebGPURenderer* renderer;

    // Prop list of the renderer
    vtkProp** propList;

    // How many props were culled
    int culledCount;
  };

  // If this is the first frame, every object is going to be rendered b the FirstPassRender to fill
  // the z-buffer
  bool FirstFrame = true;
  // Whether or not we're done initializing the compute passes of this culler
  bool Initialized = false;
  // Render window whose depth buffer we're going to use for the culling
  vtkWebGPURenderWindow* WebGPURenderWindow = nullptr;

  // Callback command for when the render window that this occlusion culler is attached to is
  // resized
  vtkSmartPointer<vtkCallbackCommand> WindowResizedCallbackCommand;
};

VTK_ABI_NAMESPACE_END

#endif
