// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

#ifndef vtkWebGPUComputePass_h
#define vtkWebGPUComputePass_h

#include "vtkObject.h"                         // for superclass
#include "vtkWebGPUComputeRenderBuffer.h"      // for compute render buffers used by the pipeline
#include "vtkWebGPUComputeRenderTexture.h"     // for compute render textures used by the pipeline
#include "vtkWebGPUComputeTextureView.h"       // for texture view management
#include "vtkWebGPUInternalsBindGroup.h"       // for bind group utilitary methods
#include "vtkWebGPUInternalsBindGroupLayout.h" // for bind group layouts utilitary methods
#include "vtkWebGPUInternalsBuffer.h"          // for internal buffer utils
#include "vtkWebGPUInternalsShaderModule.h"    // for wgpu::ShaderModule
#include "vtk_wgpu.h"                          // for webgpu

#include <unordered_map>
#include <unordered_set>

class vtkWebGPUComputePipeline;

VTK_ABI_NAMESPACE_BEGIN

/**
 * A compute pass is an abstraction for offloading computation from the CPU onto the GPU using
 * WebGPU compute shaders.
 *
 * The basic usage of a compute pass outside a rendering pipeline is:
 *  - Create a vtkWebGPUComputePipeline
 *  - Obtain a compute pass from this compute pipeline
 *  - Set its shader source code
 *  - Set its shader entry point
 *  - Create the vtkWebGPUComputeBuffers that contain the data manipulated by the compute pass
 *  - Add the buffers to the compute pass
 *  - Set the number of workgroups
 *  - Dispatch the compute shader
 *  - ReadBufferFromGPU() to make results from the GPU available to the CPU
 *  - Update() the pipeline so that the compute pass is executed
 *
 * Integrated into a rendering pipeline, the only difference in the usage of the class is going to
 * be the creation of the buffers. You will not create the vtkWebGPUComputeBuffer yourself but
 * rather acquire one (or many) by calling AcquirePointAttributeComputeRenderBuffer() on a
 * vtkWebGPURenderer. The returned buffers can then be added to the pipeline with AddRenderBuffer().
 * Other steps are identical.
 * This remark also applies to render textures.
 */
class VTKRENDERINGWEBGPU_EXPORT vtkWebGPUComputePass : public vtkObject
{
public:
  /**
   * Note to the VTK user: A compute pass should always be acquired through
   * vtkWebGPUComputePipeline::CreateComputePass(). You should not create compute pass through
   * vtkNew<> or vtkSmartPointer<> directly.
   */
  static vtkWebGPUComputePass* New();
  vtkTypeMacro(vtkWebGPUComputePass, vtkObject);

  /*
   * Callback called when the asynchronous mapping of a buffer is done
   * and data is ready to be copied.
   * This callback takes three parameters:
   *
   * - A first void pointer to the data mapped from the GPU ready to be copied
   * - A second void pointer pointing to user data, which can essentially be anything
   *      needed by the callback to copy the data to the CPU
   */
  using MapAsyncCallback = std::function<void(const void*, void*)>;

  /*
   * Callback called when the asynchronous mapping of a texture is done
   * and data is ready to be copied.
   * This callback takes three parameters:
   *
   * - A void pointer to the data mapped from the GPU ready to be copied.
   *
   * - An integer representing how many bytes per row the mapped data contains. This is
   * useful because some padding has probably be done on the buffer to satisfy WebGPU size
   * constraints. At the time of writing, buffers for texture mapping need a number of bytes per row
   * that is a multiple of 256 bytes. This means that for a texture of 300x300 RGBA (300 * 4 = 1200
   * bytes per row), there will be 80 bytes of additional padding to achieve 1280 bytes per row
   * which is a multiple of 256. In this case, the integer argument of the callback will contain the
   * value '1280' and it is then the responsibility of the user to only read relevant data (i.e. the
   * 1200 first bytes of each row since the 80 last bytes are irrelevant padding).
   *
   * - A second void pointer pointing to user data, which can essentially be anything
   *      needed by the callback to copy the data to the CPU
   */
  using TextureMapAsyncCallback = std::function<void(const void*, int, void*)>;

  ///@{
  /**
   * Get/set the device used by this compute pass (usually the device of the compute pipeline
   * holding this compute pass)
   */
  wgpu::Device GetDevice() { return this->Device; }
  void SetDevice(wgpu::Device device) { this->Device = device; }
  ///@}

  ///@{
  /**
   * Get/set the compute pipeline to which this compute pass belongs to
   */
  vtkGetMacro(AssociatedPipeline, vtkWebGPUComputePipeline*);
  vtkSetMacro(AssociatedPipeline, vtkWebGPUComputePipeline*);
  ///@}

  ///@{
  /**
   * Set/get the WGSL source of the shader
   */
  vtkGetMacro(ShaderSource, std::string);
  vtkSetMacro(ShaderSource, std::string);
  ///@}

  void SetShaderSourceFromPath(const char* shaderFilePath);

  ///@{
  /**
   * Set/get the entry (name of the function) of the WGSL compute shader
   */
  vtkGetMacro(ShaderEntryPoint, std::string);
  vtkSetMacro(ShaderEntryPoint, std::string);
  ///@}

  ///@{
  /**
   * Set/get the label of the compute pass.
   * This label will be printed along with error/warning logs
   * to help with debugging
   */
  vtkGetMacro(Label, std::string&);
  void SetLabel(const std::string& label);
  ///@}

  /**
   * Adds a buffer to the pipeline and uploads its data to the device.
   *
   * Returns the index of the buffer that can for example be used as input to the
   * ReadBufferFromGPU() function
   */
  int AddBuffer(vtkSmartPointer<vtkWebGPUComputeBuffer> buffer);

  /**
   * Adds a render buffer to the pipeline. A render buffer can be obtained from
   * vtkWebGPURenderWindow::AcquireDepthBufferRenderTexture()
   */
  void AddRenderBuffer(vtkSmartPointer<vtkWebGPUComputeRenderBuffer> renderBuffer);

  /**
   * Adds a render texture to the pipeline. A render texture can be obtained from
   * vtkWebGPUPolyDataMapper::AcquirePointXXXXRenderBuffer() or
   * vtkWebGPUPolyDataMapper::AcquireCellXXXXRenderBuffer()
   */
  void AddRenderTexture(vtkSmartPointer<vtkWebGPUComputeRenderTexture> renderTexture);

  /**
   * Adds a texture to the pipeline and upload its data to the device
   *
   * Returns the index of the texture that can for example be used as input to the
   * ReadTextureFromGPU() function
   */
  int AddTexture(vtkSmartPointer<vtkWebGPUComputeTexture> texture);

  /**
   * Returns a new texture view on the given texture (given by the index) that can be configured and
   * then added to the compute pass by AddTextureView()
   */
  vtkSmartPointer<vtkWebGPUComputeTextureView> CreateTextureView(int textureIndex);

  /**
   * Adds a texture view to the compute pass and returns its index
   */
  int AddTextureView(vtkSmartPointer<vtkWebGPUComputeTextureView> textureView);

  /**
   * This function allows the usage of multiple texture views on a single binding point (group /
   * binding combination) in the shader (although not at the same time). It acts as AddTextureView()
   * if no texture view was bound to the group/binding in the first place.
   *
   * For example, consider that your shader has the following binding:
   * @group(0) @binding(0) var inputTexture: texture_2d<f32>;
   *
   * Depending on your needs, you may want to execute this compute pass twice but with a different
   * texture as input to the shader each time. To achieve that, you would create 2 TextureViews on
   * the 2 Textures that you want your shader to manipulate and call RebindTextureView() on your
   * second texture view index before Dispatching the second compute pass so that it the texture
   * view sampled in the shader samples the second texture.
   */
  void RebindTextureView(int group, int binding, int textureViewIndex);

  /**
   * Deletes all the texture views of a given texture (given by its index)
   */
  void DeleteTextureViews(int textureIndex);

  /**
   * Returns the size in bytes of a buffer
   */
  unsigned int GetBufferByteSize(int bufferIndex);

  /**
   * Resizes a buffer of the pipeline.
   *
   * @warning: After the resize, the data of the buffer is undefined and should be updated by a
   * call to UpdateBufferData()
   */
  void ResizeBuffer(int bufferIndex, vtkIdType newByteSize);

  // ///@{
  // /**
  //  * Resizes a texture of the pipeline to the given new dimensions. If one of the given
  //  dimensions
  //  * given is -1 (the depth for example), the texture will keep its original size for this
  //  * dimension.
  //  *
  //  * @warning: After the resize, the data of the texture is undefined.
  //  */
  // void ResizeTexture(int textureIndex, uint32_t newWidth, uint32_t newHeight, uint32_t newDepth);
  // void ResizeTexture(int textureIndex, uint32_t newWidth, uint32_t newHeight);
  // void ResizeTexture(int textureIndex, uint32_t* newDimsXYZ);
  // ///@}

  /**
   * Retrieves the compute texture associated with the given texture index
   *
   * @warning The texture will need to be recreated by calling RecreateComputeTexture for all the
   * changes done to this compute texture to take effect
   */
  vtkSmartPointer<vtkWebGPUComputeTexture> GetComputeTexture(int textureIndex);

  /**
   * Recreates a compute texture. Must be called if the compute texture has been modified (after a
   * call to GetComputeTexure for example) for the  changes to take effect.
   */
  void RecreateComputeTexture(int textureIndex);

  /*
   * This function maps the buffer, making it accessible to the CPU. This is
   * an asynchronous operation, meaning that the given callback will be called
   * when the mapping is done.
   *
   * The buffer data can then be read from the callback and stored
   * in a buffer (std::vector<>, vtkDataArray, ...) passed in via the userdata pointer for example
   */
  void ReadBufferFromGPU(
    int bufferIndex, vtkWebGPUComputePass::MapAsyncCallback callback, void* userdata);

  /**
   *
   * This function maps the texture into a linear memory block, making it accessible to the CPU.
   * This is an asynchronous operation, meaning that the given callback will be called when the
   * mapping is done.
   *
   * The texture data can then be read from the callback and stored
   * in a buffer (std::vector<>, vtkDataArray, ...) passed in via the userdata pointer for example
   */
  void ReadTextureFromGPU(int textureIndex, int mipLevel,
    vtkWebGPUComputePass::TextureMapAsyncCallback callback, void* userdata);

  /**
   * Updates the data of a buffer.
   * The given data is expected to be at most the size of the buffer.
   * If N bytes are given to update but the buffer size is > N, only the first N bytes
   * will be updated, the rest will remain unchanged.
   * The data is immediately available to the GPU (no call to vtkWebGPUComputePipeline::Update() is
   * necessary for this call to take effect)
   *
   * @note: This method can be used even if the buffer was initially configured with std::vector
   * data and the given data can safely be destroyed directly after calling this function.
   *
   */
  template <typename T>
  void UpdateBufferData(int bufferIndex, const std::vector<T>& newData)
  {
    if (!CheckBufferIndex(bufferIndex, std::string("UpdataBufferData")))
    {
      return;
    }

    vtkWebGPUComputeBuffer* buffer = this->Buffers[bufferIndex];
    vtkIdType byteSize = buffer->GetByteSize();
    vtkIdType givenSize = newData.size() * sizeof(T);

    if (givenSize > byteSize)
    {
      vtkLog(ERROR,
        "std::vector data given to UpdateBufferData with index "
          << bufferIndex << " is too big. " << givenSize
          << "bytes were given but the buffer is only " << byteSize
          << " bytes long. No data was updated by this call.");

      return;
    }

    wgpu::Buffer wgpuBuffer = this->WebGPUBuffers[bufferIndex];
    this->Device.GetQueue().WriteBuffer(wgpuBuffer, 0, newData.data(), newData.size() * sizeof(T));
  }

  /**
   * Similar to the overload without offset of this function.
   * The offset is used to determine where in the buffer to reupload data.
   * Useful when only a portion of the buffer needs to be reuploaded.
   */
  template <typename T>
  void UpdateBufferData(int bufferIndex, vtkIdType byteOffset, const std::vector<T>& data)
  {
    if (!CheckBufferIndex(bufferIndex, std::string("UpdataBufferData with offset")))
    {
      return;
    }

    vtkWebGPUComputeBuffer* buffer = this->Buffers[bufferIndex];
    vtkIdType byteSize = buffer->GetByteSize();
    vtkIdType givenSize = data.size() * sizeof(T);

    if (givenSize + byteOffset > byteSize)
    {
      vtkLog(ERROR,
        "std::vector data given to UpdateBufferData with index "
          << bufferIndex << " and offset " << byteOffset << " is too big. " << givenSize
          << "bytes and offset " << byteOffset << " were given but the buffer is only " << byteSize
          << " bytes long. No data was updated by this call.");

      return;
    }

    wgpu::Buffer wgpuBuffer = this->WebGPUBuffers[bufferIndex];
    this->Device.GetQueue().WriteBuffer(
      wgpuBuffer, byteOffset, data.data(), data.size() * sizeof(T));
  }

  /**
   * Updates the data of a buffer with a vtkDataArray.
   * The given data is expected to be at most the size of the buffer.
   * If N bytes are given to update but the buffer size is > N, only the first N bytes
   * will be updated, the rest will remain unchanged.
   * The data is immediately available to the GPU (no call to vtkWebGPUComputePipeline::Update() is
   * necessary for this call to take effect
   *
   * @note: This method can be used even if the buffer was initially configured with std::vector
   * data and the given data can safely be destroyed directly after calling this function.
   */
  void UpdateBufferData(int bufferIndex, vtkDataArray* newData);

  /**
   * Similar to the overload without offset of this function.
   * The offset is used to determine where in the buffer to reupload data.
   * Useful when only a portion of the buffer needs to be reuploaded.
   */
  void UpdateBufferData(int bufferIndex, vtkIdType byteOffset, vtkDataArray* newData);

  /**
   * Uploads the given data to the texture starting at pixel (0, 0)
   */
  template <typename T>
  void UpdateTextureData(int textureIndex, const std::vector<T>& data)
  {
    if (!CheckTextureIndex(textureIndex, "UpdateTextureData"))
    {
      return;
    }

    wgpu::Texture wgpuTexture;
    vtkSmartPointer<vtkWebGPUComputeTexture> texture;
    wgpuTexture = this->WebGPUTextures[textureIndex];
    texture = this->Textures[textureIndex];

    if (data.size() * sizeof(T) > texture->GetByteSize())
    {
      vtkLog(ERROR,
        "The given data is larger than what the texture \""
          << texture->GetLabel() << "\" with byte size: " << texture->GetByteSize());

      return;
    }

    wgpu::Extent3D textureExtents = { texture->GetWidth(), texture->GetHeight(),
      texture->GetDepth() };

    wgpu::ImageCopyTexture copyTexture;
    copyTexture.aspect = wgpu::TextureAspect::All;
    copyTexture.mipLevel = 0;
    copyTexture.nextInChain = nullptr;
    copyTexture.origin = { 0, 0, 0 };
    copyTexture.texture = wgpuTexture;

    wgpu::TextureDataLayout textureDataLayout;
    textureDataLayout.nextInChain = nullptr;
    textureDataLayout.bytesPerRow = texture->GetBytesPerPixel() * textureExtents.width;
    textureDataLayout.offset = 0;
    textureDataLayout.rowsPerImage = textureExtents.height;

    // Uploading from std::vector or vtkDataArray if one of the two is present
    this->Device.GetQueue().WriteTexture(
      &copyTexture, data.data(), data.size() * sizeof(T), &textureDataLayout, &textureExtents);
  }

  ///@{
  /*
   * Set/get the number of workgroups in each dimension that are used by each Dispatch() call.
   */
  void SetWorkgroups(int groupsX, int groupsY, int groupsZ);
  ///@}

  /**
   * Dispatch the compute pass with (X, Y, Z) = (groupX, groupsY, groupZ) groups
   */
  void Dispatch();

private:
  friend class vtkWebGPUComputePipeline;
  friend class vtkWebGPUHelpers;
  friend class vtkWebGPURenderWindow;
  friend class vtkWebGPURenderer;

  /**
   * Given a buffer, creates the associated bind group layout entry
   * that will be used when creating the bind group layouts and returns it
   */
  wgpu::BindGroupLayoutEntry CreateBindGroupLayoutEntry(
    uint32_t binding, vtkWebGPUComputeBuffer::BufferMode mode);

  /**
   * Given a texture and its view, creates the associated bind group layout entry and returns it
   */
  wgpu::BindGroupLayoutEntry CreateBindGroupLayoutEntry(uint32_t binding,
    vtkSmartPointer<vtkWebGPUComputeTexture> computeTexture,
    vtkSmartPointer<vtkWebGPUComputeTextureView> textureView);

  /**
   * Overload mainly used for creating the layout entry of render textures where we don't have a
   * vtkWebGPUComputeTextureView object and where the view is assumed to be very close in
   * configuration to the texture so the mode of the texture is used for the texture view for
   * example and returns it
   */
  wgpu::BindGroupLayoutEntry CreateBindGroupLayoutEntry(uint32_t binding,
    vtkSmartPointer<vtkWebGPUComputeTexture> computeTexture,
    wgpu::TextureViewDimension textureViewDimension);

  /**
   * Given a buffer, creates the associated bind group entry
   * that will be used when creating the bind groups and returns it
   */
  wgpu::BindGroupEntry CreateBindGroupEntry(wgpu::Buffer buffer, uint32_t binding,
    vtkWebGPUComputeBuffer::BufferMode mode, uint32_t offset);

  /**
   * Given a texture view, creates the associated bind group entry
   * that will be used when creating the bind groups and returns it
   */
  wgpu::BindGroupEntry CreateBindGroupEntry(uint32_t binding, wgpu::TextureView textureView);

  /**
   * Compiles the shader source given into a WGPU shader module
   */
  void CreateShaderModule();

  /**
   * Creates all the bind groups and bind group layouts of this compute pipeline from the buffers
   * that have been added so far.
   */
  void CreateBindGroupsAndLayouts();

  /**
   * Creates the bind group layout of a given list of buffers (that must all belong to the same bind
   * group)
   */
  static wgpu::BindGroupLayout CreateBindGroupLayout(
    const wgpu::Device& device, const std::vector<wgpu::BindGroupLayoutEntry>& layoutEntries);

  /**
   * Creates the bind group entries given a list of buffers
   */
  std::vector<wgpu::BindGroupEntry> CreateBindGroupEntries(
    const std::vector<vtkWebGPUComputeBuffer*>& buffers);

  /**
   * Checks if a given index is suitable for indexing this->Buffers. Logs an error if the index is
   * negative or greater than the number of buffer of the compute pass. The callerFunctionName
   * parameter is using to give more information on what function used an invalid buffer index
   *
   * Returns true if the buffer index is valid, false if it's not.
   */
  bool CheckBufferIndex(int bufferIndex, const std::string& callerFunctionName);

  /**
   * Checks if a given index is suitable for indexing this->Textures. Logs an error if the index is
   * negative or greater than the number of texture of the compute pass. The callerFunctionName
   * parameter is using to give more information on what function used an invalid texture index
   *
   * Returns true if the texture index is valid, false if it's not.
   */
  bool CheckTextureIndex(int textureIndex, const std::string& callerFunctionName);

  /**
   * Checks if a given index is suitable for indexing this->TextureViews. Logs an error if the index
   * is negative or greater than the number of texture views of the compute pass. The
   * callerFunctionName parameter is using to give more information on what function used an invalid
   * texture view index
   *
   * Returns true if the texture view index is valid, false if it's not.
   */
  bool CheckTextureViewIndex(int textureViewIndex, const std::string& callerFunctionName);

  /**
   * Makes some various (and obvious) checks to ensure that the buffer is ready to be created.
   *
   * Returns true if the buffer is correct.
   * If the buffer is incorrect, returns false and logs the error with the ERROR verbosity
   */
  bool CheckBufferCorrectness(vtkWebGPUComputeBuffer* buffer);

  /**
   * Makes sure the texture is correct with regards to its properties (size, ...)
   */
  bool CheckTextureCorrectness(vtkWebGPUComputeTexture* texture);

  /**
   * Makes sure the texture view is correct with regards to its properties (binding, group, ...)
   */
  bool CheckTextureViewCorrectness(vtkWebGPUComputeTextureView* textureView);

  /**
   * Destroys and recreates a buffer with the given newByteSize
   * Only the wgpu::Buffer object is recreated so the binding/group of the group doesn't change
   */
  void RecreateBuffer(int bufferIndex, vtkIdType newByteSize);

  /**
   * Recreates the bind group and bind group entry of a buffer (given by its index)
   *
   * The function is useful after recreating a wgpu::Buffer, the bind group entry (and the bind
   * group) will need to be updated because the wgpu::Buffer object has changed. This function thus
   * assumes that the new buffer can be found in WebGPUBuffers[bufferIndex]
   */
  void RecreateBufferBindGroup(int bufferIndex);

  /**
   * Destroys and recreates the texture with the given index.
   */
  void RecreateTexture(int textureIndex);

  /**
   * Recreates all the texture views of a texture given its index.
   *
   * Useful when a texture has been recreated, meaning that the wgpu::Texture of this compute pass
   * has changed --> the texture view do not point to a correct texture anymore and need to be
   * recreated
   */
  void RecreateTextureViews(int textureIndex);

  /**
   * Utilitary method to create a wgpu::TextureView from a ComputeTextureView and the texture this
   * wgpu::TextureView is going to be a view off
   */
  wgpu::TextureView CreateWebGPUTextureView(
    vtkSmartPointer<vtkWebGPUComputeTextureView> textureView, wgpu::Texture wgpuTexture);

  /**
   * Makes sure that the compute texture given in parameter internally points to the given
   * newWgpuTexture. If this is not initially the case, it will be true after the call to this
   * function. Also, all texture views of this texture will now be views of the given newWgpuTexture
   *
   * This is useful when recreating the compute texture from another compute pass: the compute
   * pipeline will be responsible for calling on all its compute passes to make sure that if a
   * compute pass was using the texture that was recreated, it now uses the recreated texture and
   * not the old one
   */
  void UpdateComputeTextureAndViews(
    vtkSmartPointer<vtkWebGPUComputeTexture> texture, wgpu::Texture newWgpuTexture);

  /**
   * After recreating a wgpu::Buffer, the bind group entry (and the bind group) will need to be
   * updated. This
   */
  void RecreateTextureBindGroup(int textureIndex);

  /**
   * Binds the buffer to the pipeline at the WebGPU level.
   * To use once the buffer has been properly set up.
   */
  void SetupRenderBuffer(vtkSmartPointer<vtkWebGPUComputeRenderBuffer> renderBuffer);

  /**
   * Binds the texture to the pipeline at the WebGPU level.
   * To use once the texture has been properly set up.
   */
  void SetupRenderTexture(vtkSmartPointer<vtkWebGPUComputeRenderTexture> renderTexture,
    wgpu::TextureViewDimension textureViewDimension, wgpu::TextureView textureView);

  /**
   * Recreates a render texture given a new textureView and possibly new parameters as specified in
   * the 'renderTexture' parameter
   *
   * @warning This method assumes that the render texture was already setup in the pipeline by a
   * previous call to SetupRenderTexture(...) i.e. this method can only do RE-creation, not
   * creation.
   */
  void RecreateRenderTexture(vtkSmartPointer<vtkWebGPUComputeRenderTexture> renderTexture,
    wgpu::TextureViewDimension textureViewDimension, wgpu::TextureView textureView);

  /**
   * Creates the compute pipeline that will be used to dispatch the compute shader
   */
  void CreateWebGPUComputePipeline();

  /**
   * Creates the compute pipeline layout associated with the bind group layouts of this compute
   * pipeline
   *
   * @warning: The bind group layouts must have been created by CreateBindGroups() prior to calling
   * this function
   */
  wgpu::PipelineLayout CreateWebGPUComputePipelineLayout();

  /**
   * Creates and returns a command encoder
   */
  wgpu::CommandEncoder CreateCommandEncoder();

  /**
   * Creates a compute pass encoder from a command encoder
   */
  wgpu::ComputePassEncoder CreateComputePassEncoder(const wgpu::CommandEncoder& commandEncoder);

  /**
   * Encodes the compute pass and dispatches the workgroups
   *
   * @warning: The bind groups and the compute pipeline must have been created prior to calling this
   * function
   */
  void WebGPUDispatch(unsigned int groupsX, unsigned int groupsY, unsigned int groupsZ);

  /**
   * Finishes the encoding of a command encoder and submits the resulting command buffer
   * to the queue
   */
  void SubmitCommandEncoderToQueue(const wgpu::CommandEncoder& commandEncoder);

  /**
   * Internal method used to convert the user friendly BufferMode to the internal enum
   * wgpu::BufferUsage
   */
  static wgpu::BufferUsage ComputeBufferModeToBufferUsage(vtkWebGPUComputeBuffer::BufferMode mode);

  /**
   * Internal method used to convert the user friendly BufferMode to the internal enum
   * wgpu::BufferBindingType
   */
  static wgpu::BufferBindingType ComputeBufferModeToBufferBindingType(
    vtkWebGPUComputeBuffer::BufferMode mode);

  /**
   * Internal method used to convert the user friendly Dimension enum to its wgpu::TextureDimension
   * equivalent
   */
  static wgpu::TextureDimension ComputeTextureDimensionToWebGPU(
    vtkWebGPUComputeTexture::TextureDimension dimension);

  /**
   * The API currently assumes that the view created on a texture is unique and completely matches
   * the texture. This means that the texture view has the same extents and the same dimension. This
   * function thus does a simple mapping between the dimension of the texture
   * (vtkWebGPUComputeTexture::TextureDimension) and that of the texture view
   * (wgpu::TextureViewDimension).
   */
  static wgpu::TextureViewDimension ComputeTextureDimensionToViewDimension(
    vtkWebGPUComputeTexture::TextureDimension dimension);

  /**
   * Internal method used to convert the user friendly TextureFormat enum to its wgpu::TextureFormat
   * equivalent
   *
   * The texture label parameter is used for error logging.
   */
  static wgpu::TextureFormat ComputeTextureFormatToWebGPU(
    vtkWebGPUComputeTexture::TextureFormat format);

  /**
   * Internal method used to convert the user friendly TextureMode enum to its wgpu::TextureUsage
   * equivalent.
   *
   * The texture label parameter is used for error logging.
   */
  static wgpu::TextureUsage ComputeTextureModeToUsage(
    vtkWebGPUComputeTexture::TextureMode mode, const std::string& textureLabel);

  /**
   * Internal method used to get the StorageTextureAccess mode associated with a TextureMode
   *
   * The texture label parameter is used for error logging.
   */
  static wgpu::StorageTextureAccess ComputeTextureModeToShaderStorage(
    vtkWebGPUComputeTexture::TextureMode mode, const std::string& textureLabel);

  /**
   * Internal method used to get the StorageTextureAccess mode associated with a TextureViewMode
   *
   * The texture view label parameter is used for error logging.
   */
  static wgpu::StorageTextureAccess ComputeTextureViewModeToShaderStorage(
    vtkWebGPUComputeTextureView::TextureViewMode mode, const std::string& textureViewLabel);

  /**
   * Internal method used to convert the user friendly TextureSampleType enum to its
   * wgpu::TextureSampleType equivalent.
   */
  static wgpu::TextureSampleType ComputeTextureSampleTypeToWebGPU(
    vtkWebGPUComputeTexture::TextureSampleType sampleType);

  /**
   * Internal method used to convert the user friendly TextureAspect enum to its
   * wgpu::TextureAspect equivalent.
   */
  static wgpu::TextureAspect ComputeTextureViewAspectToWebGPU(
    vtkWebGPUComputeTextureView::TextureViewAspect aspect);

  /**
   * Whether or not the shader module, binds groups, layouts and the wgpu::ComputePipeline have
   * been created already
   */
  bool Initialized = false;
  // Wheter or not the binds group / layouts have changed since they were last created and so they
  // need to be recreated. Defaults to true since the bind group / layouts are initially not created
  // so they are invalid.
  bool BindGroupOrLayoutsInvalidated = true;

  // Device of the compute pipeline this pass belongs to.Used to submit commands.
  wgpu::Device Device = nullptr;

  // The compute pipeline this compute pass belongs to.
  vtkWebGPUComputePipeline* AssociatedPipeline = nullptr;

  wgpu::ShaderModule ShaderModule;
  // List of the bind groups, used to set the bind groups of the compute pass at each dispatch
  std::vector<wgpu::BindGroup> BindGroups;
  // Maps a bind group index to to the list of bind group entries for this group. These
  // entries will be used at the creation of the bind groups
  std::unordered_map<int, std::vector<wgpu::BindGroupEntry>> BindGroupEntries;
  std::vector<wgpu::BindGroupLayout> BindGroupLayouts;
  // Maps a bind group index to to the list of bind group layout entries for this group.
  // These layout entries will be used at the creation of the bind group layouts
  std::unordered_map<int, std::vector<wgpu::BindGroupLayoutEntry>> BindGroupLayoutEntries;
  // WebGPU compute shader pipeline
  wgpu::ComputePipeline ComputePipeline;

  // Compute buffers
  std::vector<vtkSmartPointer<vtkWebGPUComputeBuffer>> Buffers;
  // WebGPU buffers associated with the compute buffers, in the same order
  std::vector<wgpu::Buffer> WebGPUBuffers;

  // Compute textures of this compute pass
  std::vector<vtkSmartPointer<vtkWebGPUComputeTexture>> Textures;
  // Compute render textures of this compute pass
  std::vector<vtkSmartPointer<vtkWebGPUComputeRenderTexture>> RenderTextures;
  // Maps the compute render texture of this compute pass to the internal wgpu::Texture that they
  // use
  std::unordered_map<vtkSmartPointer<vtkWebGPUComputeRenderTexture>, wgpu::Texture>
    RenderTexturesToWebGPUTexture;
  // A map of the compute textures of this compute pass associated with all the texture views of it
  // that have been created
  std::unordered_map<vtkSmartPointer<vtkWebGPUComputeTexture>,
    std::unordered_set<vtkSmartPointer<vtkWebGPUComputeTextureView>>>
    ComputeTextureToViews;
  // List of the texture views add by the user to this compute pass. Can be used to find a texture
  // view from its index (which the user manipulates)
  std::vector<vtkSmartPointer<vtkWebGPUComputeTextureView>> TextureViews;

  // WebGPU textures associated with the compute texture in the same order
  std::vector<wgpu::Texture> WebGPUTextures;
  // WebGPU textures views associated with the compute texture views in the same order
  std::unordered_map<vtkSmartPointer<vtkWebGPUComputeTextureView>, wgpu::TextureView>
    TextureViewsToWebGPUTextureViews;

  std::string ShaderSource;
  std::string ShaderEntryPoint;

  // How many groups to launch when dispatching the compute
  unsigned int GroupsX = 0, GroupsY = 0, GroupsZ = 0;

  // Label used for the wgpu compute pipeline of this VTK compute pipeline
  std::string Label = "WebGPU compute pass";
  // Label used for the wgpu command encoders created and used by this VTK compute pipeline
  std::string WGPUCommandEncoderLabel = "WebGPU command encoder \"" + this->Label + "\"";
  std::string WGPUComputePipelineLabel = "WebGPU pipeline \"" + this->Label + "\"";
};

VTK_ABI_NAMESPACE_END

#endif
