// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause

#include "vtkWebGPUComputePass.h"
#include "assert.h"
#include "vtkImageData.h"
#include "vtkNew.h"
#include "vtkObjectFactory.h"
#include "vtkPNGWriter.h"
#include "vtkWebGPUComputePipeline.h"
#include "vtkWebGPUInternalsComputeBuffer.h"
#include "vtkWebGPUInternalsTexture.h"
#include "vtksys/FStream.hxx"

VTK_ABI_NAMESPACE_BEGIN

vtkStandardNewMacro(vtkWebGPUComputePass);

/**
 * Superclass of the InternalMapXXXAsyncData structures
 */
struct InternalMapAsyncData
{
  // Buffer currently being mapped
  wgpu::Buffer buffer;
  // Label of the buffer currently being mapped. Used for printing errors
  std::string bufferLabel;
  // Size of the buffer being mapped in bytes
  vtkIdType byteSize;

  // Userdata passed to userCallback. This is typically the structure that contains the CPU-side
  // buffer into which the data of the mapped buffer will be copied
  void* userdata;
};

/**
 * Structure used to pass data to the asynchronous callback of wgpu::Buffer.MapAsync()
 */
struct InternalMapBufferAsyncData : public InternalMapAsyncData
{
  // The callback given by the user that will be called once the buffer is mapped. The user will
  // usually use their callback to copy the data from the mapped buffer into a CPU-side buffer that
  // will then use the result of the compute shader in the rest of the application
  vtkWebGPUComputePass::MapAsyncCallback userCallback;
};

struct InternalMapTextureAsyncData : public InternalMapAsyncData
{
  // Bytes per row of the padded buffer that contains the mapped texture data
  int bytesPerRow;
  // Callback given by the user
  vtkWebGPUComputePass::TextureMapAsyncCallback userCallback;
};

// TODO (Tom) write printself()
// os << indent << "Initialized: " << this->Initialized << std::endl;
//   os << indent << "ShaderModule: " << this->ShaderModule.Get() << std::endl;
//   os << indent << this->BindGroups.size() << " binds groups: " << std::endl;
//   for (const wgpu::BindGroup& bindGroup : this->BindGroups)
//   {
//     os << indent << "\t- " << bindGroup.Get() << std::endl;
//   }

//   os << indent << this->BindGroupEntries.size() << " binds group entries: " << std::endl;
//   for (const auto& bindGroupEntry : this->BindGroupEntries)
//   {
//     os << indent << "\t Bind group " << bindGroupEntry.first << std::endl;
//     os << indent << "\t (binding/buffer/offset/size)" << std::endl;
//     for (wgpu::BindGroupEntry entry : bindGroupEntry.second)
//     {
//       os << indent << "\t- " << entry.binding << " / " << entry.buffer.Get() << " / "
//          << entry.offset << " / " << entry.size << std::endl;
//     }
//   }

//   os << indent << this->BindGroupLayouts.size() << " bind group layouts:" << std::endl;
//   for (const wgpu::BindGroupLayout& bindGroupLayout : this->BindGroupLayouts)
//   {
//     os << indent << "\t- " << bindGroupLayout.Get() << std::endl;
//   }

//   os << indent << this->BindGroupLayoutEntries.size()
//      << " binds group layouts entries: " << std::endl;
//   for (const auto& bindLayoutGroupEntry : this->BindGroupLayoutEntries)
//   {
//     os << indent << "\t Bind group layout " << bindLayoutGroupEntry.first << std::endl;
//     os << indent << "\t (binding/buffer type/visibility)" << std::endl;
//     for (wgpu::BindGroupLayoutEntry entry : bindLayoutGroupEntry.second)
//     {
//       os << indent << "\t- " << entry.binding << " / " <<
//       static_cast<uint32_t>(entry.buffer.type)
//          << " / " << static_cast<uint32_t>(entry.visibility) << std::endl;
//     }
//   }

//   os << indent << "WGPU Compute pipeline: " << this->ComputePipeline.Get() << std::endl;

//   os << indent << this->Buffers.size() << "buffers: " << std::endl;
//   for (vtkWebGPUComputeBuffer* buffer : this->Buffers)
//   {
//     os << indent << "\t- " << buffer << std::endl;
//   }

//   os << indent << this->WebGPUBuffers.size() << "WGPU Buffers:" << std::endl;
//   for (wgpu::Buffer buffer : this->WebGPUBuffers)
//   {
//     os << indent << "\t- " << buffer.Get() << std::endl;
//   }

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::SetLabel(const std::string& label)
{
  this->Label = label;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::SetShaderSourceFromPath(const char* shaderFilePath)
{
  if (!vtksys::SystemTools::FileExists(shaderFilePath))
  {
    vtkLogF(ERROR, "Given shader file path '%s' doesn't exist", shaderFilePath);

    return;
  }

  vtksys::ifstream inputFileStream(shaderFilePath);
  assert(inputFileStream);
  std::string source(
    (std::istreambuf_iterator<char>(inputFileStream)), std::istreambuf_iterator<char>());

  this->SetShaderSource(source);
}

int vtkWebGPUComputePass::AddBuffer(vtkSmartPointer<vtkWebGPUComputeBuffer> buffer)
{
  // Giving the buffer a default label if it doesn't have one already
  if (buffer->GetLabel().empty())
  {
    buffer->SetLabel("Buffer " + std::to_string(this->Buffers.size()));
  }

  std::string bufferLabel = buffer->GetLabel();
  const char* bufferLabelCStr = bufferLabel.c_str();

  if (!this->CheckBufferCorrectness(buffer))
  {
    return -1;
  }

  wgpu::Buffer wgpuBuffer;
  vtkWebGPUComputeBuffer::BufferMode mode = buffer->GetMode();
  if (!this->AssociatedPipeline->GetRegisteredBuffer(buffer, wgpuBuffer))
  {
    // If this buffer wasn't already registered in the pipeline by another compute pass, creating
    // the buffer. Otherwise, wgpuBuffer has been set to the already existing buffer

    wgpu::BufferUsage usage;
    vtkIdType byteSize;

    usage = ComputeBufferModeToBufferUsage(mode);
    byteSize = buffer->GetByteSize();
    wgpuBuffer = vtkWebGPUInternalsBuffer::CreateABuffer(
      this->Device, byteSize, usage, false, bufferLabelCStr);

    // Uploading from std::vector or vtkDataArray if one of the two is present
    if (buffer->GetDataPointer() != nullptr)
    {
      this->Device.GetQueue().WriteBuffer(
        wgpuBuffer, 0, buffer->GetDataPointer(), buffer->GetByteSize());
    }
    else if (buffer->GetDataArray() != nullptr)
    {
      vtkWebGPUInternalsComputeBuffer::UploadFromDataArray(
        this->Device, wgpuBuffer, buffer->GetDataArray());
    }

    this->AssociatedPipeline->RegisterBuffer(buffer, wgpuBuffer);
  }

  // Adding the buffer to the lists
  this->Buffers.push_back(buffer);
  this->WebGPUBuffers.push_back(wgpuBuffer);

  // Creating the layout entry and the bind group entry for this buffer. These entries will be used
  // later when creating the bind groups / bind group layouts
  uint32_t group = buffer->GetGroup();
  uint32_t binding = buffer->GetBinding();

  wgpu::BindGroupLayoutEntry bglEntry = this->CreateBindGroupLayoutEntry(binding, mode);
  wgpu::BindGroupEntry bgEntry = this->CreateBindGroupEntry(wgpuBuffer, binding, mode, 0);

  this->BindGroupLayoutEntries[group].push_back(bglEntry);
  this->BindGroupEntries[group].push_back(bgEntry);

  // Returning the index of the buffer
  return this->Buffers.size() - 1;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::AddRenderBuffer(
  vtkSmartPointer<vtkWebGPUComputeRenderBuffer> renderBuffer)
{
  renderBuffer->SetAssociatedComputePass(this);

  this->Buffers.push_back(renderBuffer);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::AddRenderTexture(
  vtkSmartPointer<vtkWebGPUComputeRenderTexture> renderTexture)
{
  renderTexture->SetAssociatedComputePass(this);

  this->RenderTextures.push_back(renderTexture);
}

//------------------------------------------------------------------------------
int vtkWebGPUComputePass::AddTexture(vtkSmartPointer<vtkWebGPUComputeTexture> texture)
{
  wgpu::Extent3D textureExtents = { texture->GetWidth(), texture->GetHeight(),
    texture->GetDepth() };

  if (!CheckTextureCorrectness(texture))
  {
    return -1;
  }

  std::string textureLabel = texture->GetLabel();
  wgpu::Texture wgpuTexture;

  // Check if this texture has already been created for another compute pass and has been registered
  // in the compute pipeline. If not, we need to create it
  if (!this->AssociatedPipeline->GetRegisteredTexture(texture, wgpuTexture))
  {
    wgpu::TextureUsage usage = ComputeTextureModeToUsage(texture->GetMode(), texture->GetLabel());
    wgpu::TextureFormat format = ComputeTextureFormatToWebGPU(texture->GetFormat());
    wgpu::TextureDimension dimension = ComputeTextureDimensionToWebGPU(texture->GetDimension());
    int mipLevelCount = texture->GetMipLevelCount();

    wgpuTexture = vtkWebGPUInternalsTexture::CreateATexture(
      this->Device, textureExtents, dimension, format, usage, mipLevelCount, textureLabel.c_str());

    texture->SetByteSize(textureExtents.width * textureExtents.height *
      textureExtents.depthOrArrayLayers * texture->GetBytesPerPixel());

    // Uploading from std::vector or vtkDataArray if one of the two is present
    ////////////////////////
    ////////////////////////
    // TODO use same logic as in vtk/master for the choice between std::vector and vtkDataArray
    ////////////////////////
    ////////////////////////
    ////////////////////////
    if (texture->GetDataPointer() != nullptr)
    {
      vtkWebGPUInternalsTexture::Upload(this->Device, wgpuTexture,
        texture->GetBytesPerPixel() * textureExtents.width, texture->GetByteSize(),
        texture->GetDataPointer());
    }
    else if (texture->GetDataArray() != nullptr)
    {
      vtkWebGPUInternalsTexture::UploadFromDataArray(this->Device, wgpuTexture,
        texture->GetBytesPerPixel() * textureExtents.width, texture->GetDataArray());
    }

    // The texture view isn't created immediately so we're registering with a null textureView for
    // now
    this->AssociatedPipeline->RegisterTexture(texture, wgpuTexture);
  }

  this->Textures.push_back(texture);
  this->WebGPUTextures.push_back(wgpuTexture);

  return this->Textures.size() - 1;
}

//------------------------------------------------------------------------------
vtkSmartPointer<vtkWebGPUComputeTextureView> vtkWebGPUComputePass::GetTextureView(int textureIndex)
{
  if (!CheckTextureIndex(textureIndex, "GetTextureView"))
  {
    return nullptr;
  }

  vtkSmartPointer<vtkWebGPUComputeTexture> texture = this->Textures[textureIndex];

  vtkSmartPointer<vtkWebGPUComputeTextureView> textureView =
    vtkSmartPointer<vtkWebGPUComputeTextureView>::New();

  textureView->SetDimension(texture->GetDimension());
  textureView->SetFormat(texture->GetFormat());
  textureView->SetAssociatedTextureIndex(textureIndex);

  return textureView;
}

//------------------------------------------------------------------------------
int vtkWebGPUComputePass::AddTextureView(vtkSmartPointer<vtkWebGPUComputeTextureView> textureView)
{
  int associatedTextureIndex = textureView->GetAssociatedTextureIndex();
  if (associatedTextureIndex == -1)
  {
    vtkLog(ERROR,
      "The texture view with label \""
        << textureView->GetLabel()
        << "\" has no assicated texture index. Make sure you obtained the textureView by calling "
           "vtkWebGPUComputePass::GetTextureView().");

    return -1;
  }

  vtkSmartPointer<vtkWebGPUComputeTexture> texture = this->Textures[associatedTextureIndex];
  wgpu::Texture wgpuTexture = this->WebGPUTextures[associatedTextureIndex];
  wgpu::TextureView wgpuTextureView = this->CreateWebGPUTextureView(textureView, wgpuTexture);

  // Note that here, group and binding may be -1 if the texture view wasn't given a group/binding
  // combination. This is valid if the user intends to rebind the texture view to a group / binding
  // later. If the user actually forgot to set the group / binding, and doesn't rebind the texture
  // view, the compute pass will crash when dispatching anyway so the error will be caught at some
  // point
  vtkIdType group = textureView->GetGroup();
  vtkIdType binding = textureView->GetBinding();
  if (group > -1 && binding > -1)
  {
    // Only creating the bind group layout and bind group if the group and binding are valid,
    // they will be created by RebindTextureView otherwise
    wgpu::BindGroupLayoutEntry bglEntry;
    wgpu::BindGroupEntry bgEntry;
    bglEntry = this->CreateBindGroupLayoutEntry(binding, texture, textureView);
    bgEntry = this->CreateBindGroupEntry(binding, wgpuTextureView);

    this->BindGroupLayoutEntries[group].push_back(bglEntry);
    this->BindGroupEntries[group].push_back(bgEntry);
  }

  this->ComputeTextureToViews[texture].push_back(textureView);
  this->TextureViews.push_back(textureView);
  this->TextureViewsToWebGPUTextureViews[textureView] = wgpuTextureView;

  return this->TextureViews.size() - 1;
}

//------------------------------------------------------------------------------
wgpu::TextureView vtkWebGPUComputePass::CreateWebGPUTextureView(
  vtkSmartPointer<vtkWebGPUComputeTextureView> textureView, wgpu::Texture wgpuTexture)
{
  std::string textureViewLabel = textureView->GetLabel().c_str();
  wgpu::TextureViewDimension textureViewDimension =
    ComputeTextureDimensionToViewDimension(textureView->GetDimension());
  // Creating a "full" view of the texture
  wgpu::TextureAspect textureViewAspect =
    ComputeTextureViewAspectToWebGPU(textureView->GetAspect());
  wgpu::TextureFormat textureViewFormat = ComputeTextureFormatToWebGPU(textureView->GetFormat());
  int baseMipLevel = textureView->GetBaseMipLevel();
  int mipLevelCount = textureView->GetMipLevelCount();

  return vtkWebGPUInternalsTexture::CreateATextureView(this->Device, wgpuTexture,
    textureViewDimension, textureViewAspect, textureViewFormat, baseMipLevel, mipLevelCount,
    textureViewLabel);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::UpdateComputeTextureAndViews(
  vtkSmartPointer<vtkWebGPUComputeTexture> texture, wgpu::Texture newWgpuTexture)
{
  int textureIndex = 0;

  // Finding the index of the texture that needs to be updated as well as updating it with the
  // newWgpuTexture
  for (const vtkSmartPointer<vtkWebGPUComputeTexture>& computePassTexture : this->Textures)
  {
    if (computePassTexture == texture)
    {
      this->WebGPUTextures[textureIndex] = newWgpuTexture;

      break;
    }

    textureIndex++;
  }

  if (textureIndex == this->Textures.size())
  {
    // The texture isn't in the pipeline, nothing to update

    return;
  }

  // Updating the views that were using this texture
  for (vtkSmartPointer<vtkWebGPUComputeTextureView> textureView :
    this->ComputeTextureToViews[texture])
  {
    wgpu::TextureView newTextureView = CreateWebGPUTextureView(textureView, newWgpuTexture);
    this->TextureViewsToWebGPUTextureViews[textureView] = newTextureView;

    // Finding the bind group / bind group layout entries that need to be recreated
    uint32_t binding = textureView->GetBinding();
    uint32_t group = textureView->GetGroup();

    int entryIndex = 0;
    for (wgpu::BindGroupLayoutEntry& bglEntry : this->BindGroupLayoutEntries[group])
    {
      if (bglEntry.binding == binding)
      {
        break;
      }

      entryIndex++;
    }

    if (entryIndex == this->BindGroupLayoutEntries[group].size())
    {
      // The texture view wasn't found in the bindings. This may not be an error if the user intends
      // to rebind the texture views later i.e. if the user has 5 views of the same texture for
      // example but only 2 bindings in the shader. The user may then want to rebind one of the five
      // texture view to one of the two bindings in the shader. This means that texture views not
      // currently bound to the shader will not be found in the bindings and we get here.
      // No bind groups to recreate for this texture view, moving on to the next

      continue;
    }

    // Now that we have the index of the entries that need to be recreated, we can recreate them
    // with the newTextureView
    wgpu::BindGroupLayoutEntry newBglEntry =
      this->CreateBindGroupLayoutEntry(binding, texture, textureView);
    wgpu::BindGroupEntry newBgEntry = this->CreateBindGroupEntry(binding, newTextureView);

    this->BindGroupLayoutEntries[group][entryIndex] = newBglEntry;
    this->BindGroupEntries[group][entryIndex] = newBgEntry;
  }

  this->BindGroupOrLayoutsInvalidated = true;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::RebindTextureView(int group, int binding, int textureViewIndex)
{
  if (!this->CheckTextureViewIndex(textureViewIndex, "RebindTextureView"))
  {
    return;
  }

  vtkSmartPointer<vtkWebGPUComputeTextureView> computeTextureView;
  vtkSmartPointer<vtkWebGPUComputeTexture> computeTexture;
  wgpu::TextureView wgpuTextureView;
  computeTextureView = this->TextureViews[textureViewIndex];
  computeTexture = this->Textures[computeTextureView->GetAssociatedTextureIndex()];
  wgpuTextureView = this->TextureViewsToWebGPUTextureViews[computeTextureView];

  std::vector<wgpu::BindGroupEntry>& bgEntries = this->BindGroupEntries[group];
  std::vector<wgpu::BindGroupLayoutEntry>& bglEntries = this->BindGroupLayoutEntries[group];

  bool found = false;
  // Recreating the bind group layout. We need to find the existing bind group layout enttry for
  // this group / binding to replace it with the new bgl entry
  for (wgpu::BindGroupLayoutEntry& bglEntry : bglEntries)
  {
    if (bglEntry.binding == binding)
    {
      bglEntry = CreateBindGroupLayoutEntry(binding, computeTexture, computeTextureView);
      found = true;
    }
  }

  // Recreating the bind group by finding it first as above for the bgl entry
  for (wgpu::BindGroupEntry& bgEntry : bgEntries)
  {
    if (bgEntry.binding == binding)
    {
      bgEntry = CreateBindGroupEntry(binding, wgpuTextureView);
      found = true;
    }
  }

  if (found)
  {

    this->BindGroupOrLayoutsInvalidated = true;

    return;
  }

  // If we're here, this means that we couldn't find the bind group entry that correspond to the
  // group / binding combination. This means that the texture view wasn't bound by AddTextureView
  // (because the user didn't give a proper group / binding combination at the time) so we're
  // binding it here.
  vtkSmartPointer<vtkWebGPUComputeTextureView> textureView;
  vtkSmartPointer<vtkWebGPUComputeTexture> texture;
  textureView = this->TextureViews[textureViewIndex];
  texture = this->Textures[textureView->GetAssociatedTextureIndex()];

  wgpu::BindGroupLayoutEntry bglEntry;
  wgpu::BindGroupEntry bgEntry;

  bglEntry = this->CreateBindGroupLayoutEntry(binding, texture, textureView);
  bgEntry = this->CreateBindGroupEntry(binding, wgpuTextureView);

  this->BindGroupLayoutEntries[group].push_back(bglEntry);
  this->BindGroupEntries[group].push_back(bgEntry);

  this->BindGroupOrLayoutsInvalidated = true;
}

//------------------------------------------------------------------------------
unsigned int vtkWebGPUComputePass::GetBufferByteSize(int bufferIndex)
{
  if (!this->CheckBufferIndex(bufferIndex, "GetBufferByteSize"))
  {
    return 0;
  }

  return this->WebGPUBuffers[bufferIndex].GetSize();
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::ResizeBuffer(int bufferIndex, vtkIdType newByteSize)
{
  if (!this->CheckBufferIndex(bufferIndex, "ResizeBuffer"))
  {
    return;
  }

  vtkSmartPointer<vtkWebGPUComputeBuffer> buffer = this->Buffers[bufferIndex];

  this->RecreateBuffer(bufferIndex, newByteSize);
  this->RecreateBufferBindGroup(bufferIndex);

  this->AssociatedPipeline->RegisterBuffer(buffer, this->WebGPUBuffers[bufferIndex]);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::RecreateBuffer(int bufferIndex, vtkIdType newByteSize)
{
  vtkSmartPointer<vtkWebGPUComputeBuffer> buffer = this->Buffers[bufferIndex];

  // Updating the byte size
  buffer->SetByteSize(newByteSize);
  wgpu::BufferUsage bufferUsage =
    vtkWebGPUComputePass::ComputeBufferModeToBufferUsage(buffer->GetMode());

  // Recreating the buffer
  const char* bufferLabel = buffer->GetLabel().c_str();
  this->WebGPUBuffers[bufferIndex] = vtkWebGPUInternalsBuffer::CreateABuffer(
    this->Device, newByteSize, bufferUsage, false, bufferLabel);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::RecreateBufferBindGroup(int bufferIndex)
{
  vtkWebGPUComputeBuffer* buffer = this->Buffers[bufferIndex];

  // We also need to recreate the bind group entry (and the bind group below) that corresponded to
  // this buffer.
  // We first need to find the bind group entry that corresponded to this buffer
  std::vector<wgpu::BindGroupEntry>& bgEntries = this->BindGroupEntries[buffer->GetGroup()];
  for (wgpu::BindGroupEntry& entry : bgEntries)
  {
    // We only need to check the binding because we already retrieved all the entries that
    // correspond to the group of the buffer
    if (entry.binding == buffer->GetBinding())
    {
      // Replacing the buffer by the one we just recreated
      entry.buffer = this->WebGPUBuffers[bufferIndex];

      break;
    }
  }

  // We need the bind group layout that the buffer belongs to to recreate the bind group.
  // The bind group layout is only created during a Dispatch().
  // If the user tries to resize the buffer before having called Dispatch(), we cannot recreate the
  // bind group because we don't have the bind group layout yet. This is why we're only recreating
  // the bind group if the group index can be found in the bind group layout vector.
  //
  // If the bind group layout doesn't exist yet and we cannot recreate the bind group, it's ok, the
  // Dispatch() call will do it. What matters in such a situation is that we recreated the buffer
  // with the right size so that the Dispatch() can create the right bind group
  int group = buffer->GetGroup();
  if (group < this->BindGroupLayouts.size())
  {
    this->BindGroups[group] = vtkWebGPUInternalsBindGroup::MakeBindGroup(
      this->Device, this->BindGroupLayouts[group], bgEntries);
  }

  this->BindGroupOrLayoutsInvalidated = true;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::RecreateTexture(int textureIndex, wgpu::Extent3D newExtents)
{
  vtkSmartPointer<vtkWebGPUComputeTexture> texture = this->Textures[textureIndex];

  std::string textureLabel = texture->GetLabel();
  wgpu::TextureDimension dimension = ComputeTextureDimensionToWebGPU(texture->GetDimension());
  wgpu::TextureFormat format = ComputeTextureFormatToWebGPU(texture->GetFormat());
  wgpu::TextureUsage usage = ComputeTextureModeToUsage(texture->GetMode(), textureLabel);
  int mipLevelCount = texture->GetMipLevelCount();

  texture->SetSize(newExtents.width, newExtents.height, newExtents.depthOrArrayLayers);
  this->WebGPUTextures[textureIndex] = vtkWebGPUInternalsTexture::CreateATexture(
    this->Device, newExtents, dimension, format, usage, mipLevelCount, textureLabel);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::ResizeTexture(int textureIndex, uint32_t* newDimsXYZ)
{
  this->ResizeTexture(newDimsXYZ[0], newDimsXYZ[1], newDimsXYZ[2]);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::ResizeTexture(int textureIndex, uint32_t width, uint32_t height)
{
  this->ResizeTexture(textureIndex, width, height, 0);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::ResizeTexture(
  int textureIndex, uint32_t newWidth, uint32_t newHeight, uint32_t newDepth)
{
  if (!this->CheckTextureIndex(textureIndex, "ResizeTexture"))
  {
    return;
  }

  vtkSmartPointer<vtkWebGPUComputeTexture> texture = this->Textures[textureIndex];

  uint32_t width = (newWidth == 0) ? texture->GetWidth() : newWidth;
  uint32_t height = (newHeight == 0) ? texture->GetHeight() : newHeight;
  uint32_t depth = (newDepth == 0) ? texture->GetDepth() : newDepth;
  wgpu::Extent3D newExtents = { width, height, depth };

  this->RecreateTexture(textureIndex, newExtents);
  this->RecreateTextureViews(textureIndex);
  this->RecreateTextureBindGroup(textureIndex);

  // Registering the texture with the new texture recreated by previous calls
  this->AssociatedPipeline->RegisterTexture(texture, this->WebGPUTextures[textureIndex]);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::RecreateTextureViews(int textureIndex)
{
  if (!this->CheckTextureIndex(textureIndex, "RecreateTextureViews"))
  {
    return;
  }

  wgpu::Texture wgpuTexture = this->WebGPUTextures[textureIndex];
  vtkSmartPointer<vtkWebGPUComputeTexture> texture = this->Textures[textureIndex];
  for (vtkSmartPointer<vtkWebGPUComputeTextureView> textureView :
    this->ComputeTextureToViews[texture])
  {
    wgpu::TextureView newWgpuTextureView = CreateWebGPUTextureView(textureView, wgpuTexture);

    this->TextureViewsToWebGPUTextureViews[textureView] = newWgpuTextureView;
  }
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::RecreateTextureBindGroup(int textureIndex)
{
  if (!this->CheckTextureIndex(textureIndex, "RecreateTextureBindGroup"))
  {
    return;
  }

  vtkSmartPointer<vtkWebGPUComputeTexture> texture = this->Textures[textureIndex];

  // We're going to have to recreate the bind group entries for all the texture views that have been
  // created of this texture so we're getting all the views of this texture
  std::vector<vtkSmartPointer<vtkWebGPUComputeTextureView>>& textureViews =
    ComputeTextureToViews.find(texture)->second;

  for (const vtkSmartPointer<vtkWebGPUComputeTextureView>& textureView : textureViews)
  {
    // Finding the bind group entry of the texture view
    std::vector<wgpu::BindGroupEntry>& bgEntries = this->BindGroupEntries[textureView->GetGroup()];

    // Now iterating over all the entries of this group to find the one that has the same binding as
    // the texture view whose entry we're trying to recreate
    for (wgpu::BindGroupEntry& entry : bgEntries)
    {
      if (entry.binding == textureView->GetBinding())
      {
        // Replacing the texture view by the new one (recreated by a previous call to
        // RecreateTexture())
        entry.textureView = this->TextureViewsToWebGPUTextureViews[textureView];

        break;
      }
    }

    // Also recreating the bind group of this texture view. If we cannot find the bind group layout
    // of the current texture view, this means that the bind group layouts haven't been created yet.
    // This is probably because the user is trying to resize a texture before having call
    // Dispatch(): it is the Dispatch() call that creates the bind group layouts
    //
    // In this case, we have nothing to do and it is the Dispatch() call that will create the bind
    // group layouts for us
    //
    // Otherwise, if we could find the bind group layout, we need to recreate the bind group that
    // goes with it
    int group = textureView->GetGroup();
    if (group < this->BindGroupLayouts.size())
    {
      this->BindGroups[group] = vtkWebGPUInternalsBindGroup::MakeBindGroup(
        this->Device, this->BindGroupLayouts[group], bgEntries);
    }
  }

  this->BindGroupOrLayoutsInvalidated = true;
}

//------------------------------------------------------------------------------
wgpu::BindGroupLayoutEntry vtkWebGPUComputePass::CreateBindGroupLayoutEntry(
  uint32_t binding, vtkWebGPUComputeBuffer::BufferMode mode)
{
  wgpu::BufferBindingType bindingType =
    vtkWebGPUComputePass::ComputeBufferModeToBufferBindingType(mode);

  vtkWebGPUInternalsBindGroupLayout::LayoutEntryInitializationHelper bglEntry{ binding,
    wgpu::ShaderStage::Compute, bindingType };

  return bglEntry;
}

//------------------------------------------------------------------------------
wgpu::BindGroupLayoutEntry vtkWebGPUComputePass::CreateBindGroupLayoutEntry(uint32_t binding,
  vtkSmartPointer<vtkWebGPUComputeTexture> computeTexture,
  vtkSmartPointer<vtkWebGPUComputeTextureView> textureView)
{
  wgpu::TextureViewDimension textureViewDimension =
    ComputeTextureDimensionToViewDimension(textureView->GetDimension());

  if (textureView->GetMode() == vtkWebGPUComputeTextureView::TextureViewMode::READ_ONLY)
  {
    // Not a storage texture

    vtkWebGPUInternalsBindGroupLayout::LayoutEntryInitializationHelper bglEntry(binding,
      wgpu::ShaderStage::Compute,
      vtkWebGPUComputePass::ComputeTextureSampleTypeToWebGPU(computeTexture->GetSampleType()),
      textureViewDimension);

    return bglEntry;
  }
  else
  {
    // Storage texture
    wgpu::StorageTextureAccess storageAccess;
    wgpu::TextureFormat textureFormat;

    storageAccess = vtkWebGPUComputePass::ComputeTextureViewModeToShaderStorage(
      textureView->GetMode(), textureView->GetLabel());
    textureFormat = vtkWebGPUComputePass::ComputeTextureFormatToWebGPU(textureView->GetFormat());

    vtkWebGPUInternalsBindGroupLayout::LayoutEntryInitializationHelper bglEntry(
      binding, wgpu::ShaderStage::Compute, storageAccess, textureFormat, textureViewDimension);

    return bglEntry;
  }
}

//------------------------------------------------------------------------------
wgpu::BindGroupLayoutEntry vtkWebGPUComputePass::CreateBindGroupLayoutEntry(uint32_t binding,
  vtkSmartPointer<vtkWebGPUComputeTexture> computeTexture,
  wgpu::TextureViewDimension textureViewDimension)
{
  if (computeTexture->GetMode() == vtkWebGPUComputeTexture::TextureMode::READ_ONLY)
  {
    // Not a storage texture

    vtkWebGPUInternalsBindGroupLayout::LayoutEntryInitializationHelper bglEntry(binding,
      wgpu::ShaderStage::Compute,
      vtkWebGPUComputePass::ComputeTextureSampleTypeToWebGPU(computeTexture->GetSampleType()),
      textureViewDimension);

    return bglEntry;
  }
  else
  {
    // Storage texture

    vtkWebGPUInternalsBindGroupLayout::LayoutEntryInitializationHelper bglEntry(binding,
      wgpu::ShaderStage::Compute,
      vtkWebGPUComputePass::ComputeTextureModeToShaderStorage(
        computeTexture->GetMode(), computeTexture->GetLabel()),
      vtkWebGPUComputePass::ComputeTextureFormatToWebGPU(computeTexture->GetFormat()),
      textureViewDimension);

    return bglEntry;
  }
}

//------------------------------------------------------------------------------
wgpu::BindGroupEntry vtkWebGPUComputePass::CreateBindGroupEntry(wgpu::Buffer wgpuBuffer,
  uint32_t binding, vtkWebGPUComputeBuffer::BufferMode mode, uint32_t offset)
{
  wgpu::BufferBindingType bindingType =
    vtkWebGPUComputePass::ComputeBufferModeToBufferBindingType(mode);

  vtkWebGPUInternalsBindGroup::BindingInitializationHelper bgEntry{ binding, wgpuBuffer, offset };

  return bgEntry.GetAsBinding();
}

//------------------------------------------------------------------------------
wgpu::BindGroupEntry vtkWebGPUComputePass::CreateBindGroupEntry(
  uint32_t binding, wgpu::TextureView textureView)
{
  vtkWebGPUInternalsBindGroup::BindingInitializationHelper bgEntry{ binding, textureView };

  return bgEntry.GetAsBinding();
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::ReadBufferFromGPU(
  int bufferIndex, vtkWebGPUComputePass::MapAsyncCallback callback, void* userdata)
{
  if (!CheckBufferIndex(bufferIndex, "ReadBufferFromGPU"))
  {
    return;
  }

  // We need a buffer that will hold the mapped data.
  // We cannot directly map the output buffer of the compute shader because
  // wgpu::BufferUsage::Storage is incompatible with wgpu::BufferUsage::MapRead. This is a
  // restriction of WebGPU. This means that we have to create a new buffer with the MapRead flag
  // that is not a Storage buffer, copy the storage buffer that we actually want to this new buffer
  // (that has the MapRead usage flag) and then map this buffer to the CPU.
  vtkIdType byteSize = this->Buffers[bufferIndex]->GetByteSize();
  wgpu::Buffer mappedBuffer = vtkWebGPUInternalsBuffer::CreateABuffer(this->Device, byteSize,
    wgpu::BufferUsage::CopyDst | wgpu::BufferUsage::MapRead, false, nullptr);

  // If we were to allocate this callbackData locally on the stack, it would be destroyed when going
  // out of scope (at the end of this function). The callback, called asynchronously would then be
  // refering to data that has been destroyed (since it was allocated locally). This is why we're
  // allocating it dynamically with a new
  InternalMapBufferAsyncData* internalCallbackData = new InternalMapBufferAsyncData;
  internalCallbackData->buffer = mappedBuffer;
  internalCallbackData->bufferLabel = this->Label;
  internalCallbackData->byteSize = byteSize;
  internalCallbackData->userCallback = callback;
  internalCallbackData->userdata = userdata;

  wgpu::CommandEncoder commandEncoder = this->CreateCommandEncoder();
  commandEncoder.CopyBufferToBuffer(
    this->WebGPUBuffers[bufferIndex], 0, internalCallbackData->buffer, 0, byteSize);
  this->SubmitCommandEncoderToQueue(commandEncoder);

  auto internalCallback = [](WGPUBufferMapAsyncStatus status, void* wgpuUserData)
  {
    InternalMapBufferAsyncData* callbackData =
      reinterpret_cast<InternalMapBufferAsyncData*>(wgpuUserData);

    if (status == WGPUBufferMapAsyncStatus::WGPUBufferMapAsyncStatus_Success)
    {
      const void* mappedRange = callbackData->buffer.GetConstMappedRange(0, callbackData->byteSize);
      callbackData->userCallback(mappedRange, callbackData->userdata);

      callbackData->buffer.Unmap();
      // Freeing the callbackData structure as it was dynamically allocated
      delete callbackData;
    }
    else
    {
      vtkLogF(WARNING, "Could not map buffer '%s' with error status: %d",
        callbackData->bufferLabel.empty() ? "(nolabel)" : callbackData->bufferLabel.c_str(),
        status);

      delete callbackData;
    }
  };

  internalCallbackData->buffer.MapAsync(
    wgpu::MapMode::Read, 0, byteSize, internalCallback, internalCallbackData);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::ReadTextureFromGPU(int textureIndex, int mipLevel,
  vtkWebGPUComputePass::TextureMapAsyncCallback callback, void* userdata)
{
  if (!CheckTextureIndex(textureIndex, "ReadTextureFromGPU"))
  {
    return;
  }

  vtkSmartPointer<vtkWebGPUComputeTexture> texture = this->Textures[textureIndex];
  wgpu::Texture wgpuTexture = this->WebGPUTextures[textureIndex];

  // Bytes needs to be a multiple of 256
  vtkIdType bytesPerRow =
    std::ceil(wgpuTexture.GetWidth() * texture->GetBytesPerPixel() / 256.0f) * 256.0f;

  // Creating the buffer that will hold the data of the texture
  wgpu::BufferDescriptor bufferDescriptor;
  bufferDescriptor.label = "Buffer descriptor for mapping texture";
  bufferDescriptor.mappedAtCreation = false;
  bufferDescriptor.nextInChain = nullptr;
  bufferDescriptor.size = bytesPerRow * texture->GetHeight();
  bufferDescriptor.usage = wgpu::BufferUsage::CopyDst | wgpu::BufferUsage::MapRead;

  wgpu::Buffer buffer = this->Device.CreateBuffer(&bufferDescriptor);

  // Parameters for copying the texture
  wgpu::ImageCopyTexture imageCopyTexture;
  imageCopyTexture.mipLevel = mipLevel;
  imageCopyTexture.nextInChain = nullptr;
  imageCopyTexture.origin = { 0, 0, 0 };
  imageCopyTexture.texture = wgpuTexture;

  // Parameters for copying the buffer
  unsigned int mipLevelWidth = std::floor(texture->GetWidth() / std::pow(2, mipLevel));
  unsigned int mipLevelHeight = std::floor(texture->GetHeight() / std::pow(2, mipLevel));
  wgpu::ImageCopyBuffer imageCopyBuffer;
  imageCopyBuffer.buffer = buffer;
  imageCopyBuffer.layout.nextInChain = nullptr;
  imageCopyBuffer.layout.offset = 0;
  imageCopyBuffer.layout.rowsPerImage = mipLevelHeight;
  imageCopyBuffer.layout.bytesPerRow = bytesPerRow;

  // Copying the texture to the buffer
  wgpu::CommandEncoder commandEncoder = this->CreateCommandEncoder();
  wgpu::Extent3D copySize = { mipLevelWidth, mipLevelHeight, texture->GetDepth() };
  commandEncoder.CopyTextureToBuffer(&imageCopyTexture, &imageCopyBuffer, &copySize);

  // Submitting the comand
  wgpu::CommandBuffer commandBuffer = commandEncoder.Finish();
  this->Device.GetQueue().Submit(1, &commandBuffer);

  auto bufferMapCallback = [](WGPUBufferMapAsyncStatus status, void* userdata)
  {
    InternalMapTextureAsyncData* mapData = reinterpret_cast<InternalMapTextureAsyncData*>(userdata);

    if (status == WGPUBufferMapAsyncStatus_Success)
    {
      const void* mappedRange = mapData->buffer.GetConstMappedRange(0, mapData->byteSize);
      mapData->userCallback(mappedRange, mapData->bytesPerRow, mapData->userdata);

      mapData->buffer.Unmap();
      // Freeing the callbackData structure as it was dynamically allocated
      delete mapData;
    }
    else
    {
      vtkLogF(WARNING, "Could not map texture '%s' with error status: %d",
        mapData->bufferLabel.empty() ? "(nolabel)" : mapData->bufferLabel.c_str(), status);

      // Freeing the callbackData structure as it was dynamically allocated
      delete mapData;
    }
  };

  // Now mapping the buffer that contains the texture data to the CPU
  // Dynamically allocating here because we callbackData to stay alive even after exiting this
  // function (because buffer.MapAsync is asynchronous). buffer.MapAsync() also takes a raw pointer
  // so we cannot use smart pointers here
  InternalMapTextureAsyncData* callbackData = new InternalMapTextureAsyncData;
  callbackData->buffer = buffer;
  callbackData->bufferLabel = "ReadTextureFromGPU map buffer";
  callbackData->byteSize = bufferDescriptor.size;
  callbackData->bytesPerRow = bytesPerRow;
  callbackData->userCallback = callback;
  callbackData->userdata = userdata;

  buffer.MapAsync(wgpu::MapMode::Read, 0, bufferDescriptor.size, bufferMapCallback, callbackData);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::UpdateBufferData(int bufferIndex, vtkDataArray* newData)
{
  if (!CheckBufferIndex(bufferIndex, std::string("UpdateBufferData")))
  {
    return;
  }

  vtkWebGPUComputeBuffer* buffer = this->Buffers[bufferIndex];
  vtkIdType byteSize = buffer->GetByteSize();
  vtkIdType givenSize = newData->GetNumberOfValues() * newData->GetDataTypeSize();

  if (givenSize > byteSize)
  {
    vtkLog(ERROR,
      "std::vector data given to UpdateBufferData with index "
        << bufferIndex << " is too big. " << givenSize << "bytes were given but the buffer is only "
        << byteSize << " bytes long. No data was updated by this call.");

    return;
  }

  wgpu::Buffer wgpuBuffer = this->WebGPUBuffers[bufferIndex];

  vtkWebGPUInternalsComputeBuffer::UploadFromDataArray(this->Device, wgpuBuffer, newData);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::UpdateBufferData(
  int bufferIndex, vtkIdType byteOffset, vtkDataArray* newData)
{
  if (!CheckBufferIndex(bufferIndex, std::string("UpdateBufferData with offset")))
  {
    return;
  }

  vtkWebGPUComputeBuffer* buffer = this->Buffers[bufferIndex];
  vtkIdType byteSize = buffer->GetByteSize();
  vtkIdType givenSize = newData->GetNumberOfValues() * newData->GetDataTypeSize();

  if (givenSize + byteOffset > byteSize)
  {
    vtkLog(ERROR,
      "vtkDataArray data given to UpdateBufferData with index "
        << bufferIndex << " and offset " << byteOffset << " is too big. " << givenSize
        << "bytes and offset " << byteOffset << " were given but the buffer is only " << byteSize
        << " bytes long. No data was updated by this call.");

    return;
  }

  wgpu::Buffer wgpuBuffer = this->WebGPUBuffers[bufferIndex];

  vtkWebGPUInternalsComputeBuffer::UploadFromDataArray(
    this->Device, wgpuBuffer, byteOffset, newData);
}

//------------------------------------------------------------------------------
bool vtkWebGPUComputePass::CheckBufferIndex(int bufferIndex, const std::string& callerFunctionName)
{
  if (bufferIndex < 0)
  {
    vtkLog(ERROR,
      "Negative bufferIndex given to "
        << callerFunctionName << ". Make sure to use an index that was returned by AddBuffer().");

    return false;
  }
  else if (bufferIndex >= this->Buffers.size())
  {
    vtkLog(ERROR,
      "Invalid bufferIndex given to "
        << callerFunctionName << ". Index was '" << bufferIndex << "' while there are "
        << this->Buffers.size()
        << " available buffers. Make sure to use an index that was returned by AddBuffer().");

    return false;
  }

  return true;
}

//------------------------------------------------------------------------------
bool vtkWebGPUComputePass::CheckTextureIndex(
  int textureIndex, const std::string& callerFunctionName)
{
  if (textureIndex < 0)
  {
    vtkLog(ERROR,
      "Negative textureIndex given to "
        << callerFunctionName << ". Make sure to use an index that was returned by AddTexture().");

    return false;
  }
  else if (textureIndex >= this->Textures.size())
  {
    vtkLog(ERROR,
      "Invalid textureIndex given to "
        << callerFunctionName << ". Index was '" << textureIndex << "' while there are "
        << this->Textures.size()
        << " available textures. Make sure to use an index that was returned by AddTexture().");

    return false;
  }

  return true;
}

//------------------------------------------------------------------------------
bool vtkWebGPUComputePass::CheckTextureViewIndex(
  int textureViewIndex, const std::string& callerFunctionName)
{
  if (textureViewIndex < 0)
  {
    vtkLog(ERROR,
      "Negative textureViewIndex given to "
        << callerFunctionName
        << ". Make sure to use an index that was returned by AddTextureView().");

    return false;
  }
  else if (textureViewIndex >= this->TextureViewsToWebGPUTextureViews.size())
  {
    vtkLog(ERROR,
      "Invalid textureViewIndex given to " << callerFunctionName << ". Index was '"
                                           << textureViewIndex << "' while there are "
                                           << this->TextureViewsToWebGPUTextureViews.size()
                                           << " available texture views. Make sure to use an index "
                                              "that was returned by AddTextureView().");

    return false;
  }

  return true;
}

//------------------------------------------------------------------------------
bool vtkWebGPUComputePass::CheckBufferCorrectness(vtkWebGPUComputeBuffer* buffer)
{
  const char* bufferLabel = buffer->GetLabel().c_str();

  if (buffer->GetGroup() == -1)
  {
    vtkLogF(
      ERROR, "The group of the buffer with label \"%s\" hasn't been initialized", bufferLabel);
    return false;
  }
  else if (buffer->GetBinding() == -1)
  {
    vtkLogF(
      ERROR, "The binding of the buffer with label \"%s\" hasn't been initialized", bufferLabel);
    return false;
  }
  else if (buffer->GetByteSize() == 0)
  {
    vtkLogF(ERROR, "The buffer with label \"%s\" has a size of 0. Did you forget to set its size?",
      bufferLabel);
    return false;
  }
  else
  {
    // Checking that the buffer isn't already used
    for (vtkWebGPUComputeBuffer* existingBuffer : this->Buffers)
    {
      if (buffer->GetBinding() == existingBuffer->GetBinding() &&
        buffer->GetGroup() == existingBuffer->GetGroup())
      {
        vtkLog(ERROR,
          "The buffer with label" << bufferLabel << " is bound to binding " << buffer->GetBinding()
                                  << " but that binding is already used by buffer with label \""
                                  << existingBuffer->GetLabel() << "\" in bind group "
                                  << buffer->GetGroup());

        return false;
      }
    }
  }

  return true;
}

//------------------------------------------------------------------------------
bool vtkWebGPUComputePass::CheckTextureCorrectness(vtkWebGPUComputeTexture* texture)
{
  const char* textureLabel = texture->GetLabel().c_str();

  if (texture->GetWidth() == 0 || texture->GetHeight() == 0 || texture->GetDepth() == 0)
  {
    vtkLog(ERROR,
      "The texture with label "
        << textureLabel
        << " had one of its size (width, heigh or depth) 0. Did you forget to call SetSize()?");

    return false;
  }

  return true;
}

//------------------------------------------------------------------------------
bool vtkWebGPUComputePass::CheckTextureViewCorrectness(vtkWebGPUComputeTextureView* textureView)
{
  std::string textureViewLabel = textureView->GetLabel();

  if (textureView->GetBinding() == -1)
  {
    vtkLog(ERROR,
      "The texture with label "
        << textureViewLabel
        << " had its binding uninitialized. Did you forget to call SetBinding()?");

    return false;
  }
  else if (textureView->GetGroup() == -1)
  {
    vtkLog(ERROR,
      "The texture with label "
        << textureViewLabel << " had its group uninitialized. Did you forget to call SetGroup()?");

    return false;
  }
  else
  {
    // Checking that the buffer isn't already used
    for (auto texViewEntry : this->TextureViewsToWebGPUTextureViews)
    {
      vtkSmartPointer<vtkWebGPUComputeTextureView> existingTextureView = texViewEntry.first;

      if (textureView->GetBinding() == existingTextureView->GetBinding() &&
        textureView->GetGroup() == existingTextureView->GetGroup())
      {
        vtkLog(ERROR,
          "The texture with label"
            << textureViewLabel << " is bound to binding " << textureView->GetBinding()
            << " but that binding is already used by texture with label \""
            << existingTextureView->GetLabel() << "\" in bind group " << textureView->GetGroup());

        return false;
      }
    }
  }

  return true;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::SetWorkgroups(int groupsX, int groupsY, int groupsZ)
{
  this->GroupsX = groupsX;
  this->GroupsY = groupsY;
  this->GroupsZ = groupsZ;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::Dispatch()
{
  if (!this->Initialized)
  {
    this->CreateShaderModule();

    this->Initialized = true;
  }

  if (this->BindGroupOrLayoutsInvalidated || !this->Initialized)
  {
    this->CreateBindGroupsAndLayouts();
    this->CreateWebGPUComputePipeline();

    this->BindGroupOrLayoutsInvalidated = false;
  }

  this->WebGPUDispatch(this->GroupsX, this->GroupsY, this->GroupsZ);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::WebGPUDispatch(
  unsigned int groupsX, unsigned int groupsY, unsigned int groupsZ)
{
  if (groupsX * groupsY * groupsZ == 0)
  {
    vtkLogF(ERROR,
      "Invalid number of workgroups when dispatching compute pipeline \"%s\". Work groups sizes "
      "(X, Y, Z) were: (%d, %d, %d) but no dimensions can be 0.",
      this->Label.c_str(), groupsX, groupsY, groupsZ);

    return;
  }

  wgpu::CommandEncoder commandEncoder = this->CreateCommandEncoder();

  wgpu::ComputePassEncoder computePassEncoder = CreateComputePassEncoder(commandEncoder);
  computePassEncoder.SetPipeline(this->ComputePipeline);
  for (int bindGroupIndex = 0; bindGroupIndex < this->BindGroups.size(); bindGroupIndex++)
  {
    computePassEncoder.SetBindGroup(bindGroupIndex, this->BindGroups[bindGroupIndex], 0, nullptr);
  }
  computePassEncoder.DispatchWorkgroups(groupsX, groupsY, groupsZ);
  computePassEncoder.End();

  this->SubmitCommandEncoderToQueue(commandEncoder);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::CreateShaderModule()
{
  this->ShaderModule =
    vtkWebGPUInternalsShaderModule::CreateFromWGSL(this->Device, this->ShaderSource);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::CreateBindGroupsAndLayouts()
{
  this->BindGroupLayouts.clear();
  this->BindGroups.clear();

  this->BindGroupLayouts.resize(this->BindGroupLayoutEntries.size());
  this->BindGroups.resize(this->BindGroupLayoutEntries.size());

  for (const auto& mapEntry : this->BindGroupLayoutEntries)
  {
    int bindGroupIndex = mapEntry.first;

    const std::vector<wgpu::BindGroupLayoutEntry>& bglEntries =
      this->BindGroupLayoutEntries[bindGroupIndex];
    const std::vector<wgpu::BindGroupEntry>& bgEntries = this->BindGroupEntries[bindGroupIndex];

    this->BindGroupLayouts[bindGroupIndex] = CreateBindGroupLayout(this->Device, bglEntries);
    this->BindGroups[bindGroupIndex] = vtkWebGPUInternalsBindGroup::MakeBindGroup(
      this->Device, BindGroupLayouts[bindGroupIndex], bgEntries);
  }
}

//------------------------------------------------------------------------------
wgpu::BindGroupLayout vtkWebGPUComputePass::CreateBindGroupLayout(
  const wgpu::Device& device, const std::vector<wgpu::BindGroupLayoutEntry>& layoutEntries)
{
  wgpu::BindGroupLayout bgl =
    vtkWebGPUInternalsBindGroupLayout::MakeBindGroupLayout(device, layoutEntries);
  return bgl;
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::SetupRenderBuffer(
  vtkSmartPointer<vtkWebGPUComputeRenderBuffer> renderBuffer)
{
  if (renderBuffer->GetWebGPUBuffer() == nullptr)
  {
    vtkLog(ERROR,
      "The given render buffer with label \""
        << renderBuffer->GetLabel()
        << "\" does not have an assigned WebGPUBuffer meaning that it will not reuse an existing "
           "buffer of the render pipeline. The issue probably is that SetWebGPUBuffer() wasn't "
           "called.");

    return;
  }

  this->WebGPUBuffers.push_back(renderBuffer->GetWebGPUBuffer());

  // Creating the entries for this existing buffer
  vtkIdType group = renderBuffer->GetGroup();
  vtkIdType binding = renderBuffer->GetBinding();
  vtkWebGPUComputeBuffer::BufferMode mode = renderBuffer->GetMode();

  wgpu::BindGroupLayoutEntry bglEntry;
  wgpu::BindGroupEntry bgEntry;
  bglEntry = this->CreateBindGroupLayoutEntry(binding, mode);
  bgEntry = this->CreateBindGroupEntry(renderBuffer->GetWebGPUBuffer(), binding, mode, 0);

  this->BindGroupLayoutEntries[group].push_back(bglEntry);
  this->BindGroupEntries[group].push_back(bgEntry);

  // Creating the uniform buffer that will contain the offset and the length of the data held by
  // the render buffer
  std::vector<unsigned int> uniformData = { renderBuffer->GetRenderBufferOffset(),
    renderBuffer->GetRenderBufferElementCount() };
  vtkNew<vtkWebGPUComputeBuffer> offsetSizeUniform;
  offsetSizeUniform->SetMode(vtkWebGPUComputeBuffer::BufferMode::UNIFORM_BUFFER);
  offsetSizeUniform->SetGroup(renderBuffer->GetRenderUniformsGroup());
  offsetSizeUniform->SetBinding(renderBuffer->GetRenderUniformsBinding());
  offsetSizeUniform->SetData(uniformData);

  this->AddBuffer(offsetSizeUniform);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::SetupRenderTexture(
  vtkSmartPointer<vtkWebGPUComputeRenderTexture> renderTexture,
  wgpu::TextureViewDimension textureViewDimension, wgpu::TextureView textureView)
{
  if (renderTexture->GetWebGPUTexture() == nullptr)
  {
    vtkLog(ERROR,
      "The given render texture with label \""
        << renderTexture->GetLabel()
        << "\" does not have an assigned WebGPUTexture meaning that it will not reuse an "
           "existing "
           "texture of the render pipeline. The issue probably is that SetWebGPUTexture() wasn't "
           "called.");

    return;
  }

  // Creating the entries for this existing render texture
  uint32_t group = renderTexture->GetGroup();
  uint32_t binding = renderTexture->GetBinding();
  wgpu::BindGroupLayoutEntry bglEntry;
  wgpu::BindGroupEntry bgEntry;
  bglEntry = this->CreateBindGroupLayoutEntry(binding, renderTexture, textureViewDimension);
  bgEntry = this->CreateBindGroupEntry(binding, textureView);

  this->BindGroupLayoutEntries[group].push_back(bglEntry);
  this->BindGroupEntries[group].push_back(bgEntry);
  this->BindGroupOrLayoutsInvalidated = true;

  this->RenderTexturesToWebGPUTexture[renderTexture] = renderTexture->GetWebGPUTexture();
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::RecreateRenderTexture(
  vtkSmartPointer<vtkWebGPUComputeRenderTexture> renderTexture,
  wgpu::TextureViewDimension textureViewDimension, wgpu::TextureView textureView)
{
  if (renderTexture->GetWebGPUTexture() == nullptr)
  {
    vtkLog(ERROR,
      "The given render texture with label \""
        << renderTexture->GetLabel()
        << "\" does not have an assigned WebGPUTexture meaning that it will not reuse an "
           "existing "
           "texture of the render pipeline. The issue probably is that SetWebGPUTexture() wasn't "
           "called.");

    return;
  }

  // Creating the entries for this existing render texture
  uint32_t group = renderTexture->GetGroup();
  uint32_t binding = renderTexture->GetBinding();

  // Finding the index of the bind group layout / bind group entry that corresponds to the
  // previously created render texture
  int entryIndex = 0;
  for (wgpu::BindGroupLayoutEntry& existingBglEntry : this->BindGroupLayoutEntries[group])
  {
    if (existingBglEntry.binding == renderTexture->GetBinding())
    {
      break;
    }

    // Incrementing the index to know which bind group / bind group layout entry we're going to
    // override
    entryIndex++;
  }

  if (entryIndex == this->BindGroupLayoutEntries.size())
  {
    // We couldn't find the entry
    vtkLog(ERROR,
      "Couldn't find the bind group layout entry of the render texture with label \""
        << renderTexture->GetLabel()
        << "\". Did you forget to call SetupRenderTexture() before trying to recreate the "
           "texture?");

    return;
  }

  wgpu::BindGroupLayoutEntry bglEntry =
    this->CreateBindGroupLayoutEntry(binding, renderTexture, textureViewDimension);
  wgpu::BindGroupEntry bgEntry = this->CreateBindGroupEntry(binding, textureView);
  this->BindGroupLayoutEntries[group][entryIndex] = bglEntry;
  this->BindGroupEntries[group][entryIndex] = bgEntry;
  this->BindGroupOrLayoutsInvalidated = true;

  this->RenderTexturesToWebGPUTexture[renderTexture] = renderTexture->GetWebGPUTexture();
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::CreateWebGPUComputePipeline()
{
  wgpu::ComputePipelineDescriptor computePipelineDescriptor;
  computePipelineDescriptor.compute.constantCount = 0;
  computePipelineDescriptor.compute.constants = nullptr;
  computePipelineDescriptor.compute.entryPoint = this->ShaderEntryPoint.c_str();
  computePipelineDescriptor.compute.module = this->ShaderModule;
  computePipelineDescriptor.compute.nextInChain = nullptr;
  computePipelineDescriptor.label = this->WGPUComputePipelineLabel.c_str();
  computePipelineDescriptor.layout = this->CreateWebGPUComputePipelineLayout();

  this->ComputePipeline = this->Device.CreateComputePipeline(&computePipelineDescriptor);
}

//------------------------------------------------------------------------------
wgpu::PipelineLayout vtkWebGPUComputePass::CreateWebGPUComputePipelineLayout()
{
  wgpu::PipelineLayoutDescriptor computePipelineLayoutDescriptor;
  computePipelineLayoutDescriptor.bindGroupLayoutCount = this->BindGroupLayouts.size();
  computePipelineLayoutDescriptor.bindGroupLayouts = this->BindGroupLayouts.data();
  computePipelineLayoutDescriptor.nextInChain = nullptr;

  return this->Device.CreatePipelineLayout(&computePipelineLayoutDescriptor);
}

wgpu::CommandEncoder vtkWebGPUComputePass::CreateCommandEncoder()
{
  wgpu::CommandEncoderDescriptor commandEncoderDescriptor;
  commandEncoderDescriptor.label = this->WGPUCommandEncoderLabel.c_str();

  return this->Device.CreateCommandEncoder(&commandEncoderDescriptor);
}

//------------------------------------------------------------------------------
wgpu::ComputePassEncoder vtkWebGPUComputePass::CreateComputePassEncoder(
  const wgpu::CommandEncoder& commandEncoder)
{
  wgpu::ComputePassDescriptor computePassDescriptor;
  computePassDescriptor.nextInChain = nullptr;
  computePassDescriptor.timestampWrites = 0;
  return commandEncoder.BeginComputePass(&computePassDescriptor);
}

//------------------------------------------------------------------------------
void vtkWebGPUComputePass::SubmitCommandEncoderToQueue(const wgpu::CommandEncoder& commandEncoder)
{
  wgpu::CommandBuffer commandBuffer = commandEncoder.Finish();
  this->Device.GetQueue().Submit(1, &commandBuffer);
}

//------------------------------------------------------------------------------
wgpu::BufferUsage vtkWebGPUComputePass::ComputeBufferModeToBufferUsage(
  vtkWebGPUComputeBuffer::BufferMode mode)
{
  switch (mode)
  {
    case vtkWebGPUComputeBuffer::BufferMode::READ_ONLY_COMPUTE_STORAGE:
    case vtkWebGPUComputeBuffer::READ_WRITE_COMPUTE_STORAGE:
      return wgpu::BufferUsage::CopyDst | wgpu::BufferUsage::Storage;

    case vtkWebGPUComputeBuffer::BufferMode::READ_WRITE_MAP_COMPUTE_STORAGE:
      return wgpu::BufferUsage::CopySrc | wgpu::BufferUsage::CopyDst | wgpu::BufferUsage::Storage;

    case vtkWebGPUComputeBuffer::BufferMode::UNIFORM_BUFFER:
      return wgpu::BufferUsage::CopyDst | wgpu::BufferUsage::Uniform;

    default:
      vtkLog(ERROR, "Unhandled compute buffer mode in ComputeBufferModeToBufferUsage: " << mode);
      return wgpu::BufferUsage::None;
  }
}

//------------------------------------------------------------------------------
wgpu::BufferBindingType vtkWebGPUComputePass::ComputeBufferModeToBufferBindingType(
  vtkWebGPUComputeBuffer::BufferMode mode)
{
  switch (mode)
  {
    case vtkWebGPUComputeBuffer::BufferMode::READ_ONLY_COMPUTE_STORAGE:
      return wgpu::BufferBindingType::ReadOnlyStorage;

    case vtkWebGPUComputeBuffer::BufferMode::READ_WRITE_COMPUTE_STORAGE:
    case vtkWebGPUComputeBuffer::BufferMode::READ_WRITE_MAP_COMPUTE_STORAGE:
      return wgpu::BufferBindingType::Storage;

    case vtkWebGPUComputeBuffer::BufferMode::UNIFORM_BUFFER:
      return wgpu::BufferBindingType::Uniform;

    default:
      vtkLog(
        ERROR, "Unhandled compute buffer mode in ComputeBufferModeToBufferBindingType: " << mode);
      return wgpu::BufferBindingType::Undefined;
  }
}

//------------------------------------------------------------------------------
wgpu::TextureFormat vtkWebGPUComputePass::ComputeTextureFormatToWebGPU(
  vtkWebGPUComputeTexture::TextureFormat format)
{
  switch (format)
  {
    case vtkWebGPUComputeTexture::TextureFormat::RGBA8_UNORM:
      return wgpu::TextureFormat::RGBA8Unorm;

    case vtkWebGPUComputeTexture::TextureFormat::R32_FLOAT:
      return wgpu::TextureFormat::R32Float;

    default:
      vtkLog(ERROR, "Unhandled texture format in ComputeTextureFormatToWebGPU: " << format);
      return wgpu::TextureFormat::Undefined;
  }
}

//------------------------------------------------------------------------------
wgpu::TextureDimension vtkWebGPUComputePass::ComputeTextureDimensionToWebGPU(
  vtkWebGPUComputeTexture::TextureDimension dimension)
{
  switch (dimension)
  {
    case vtkWebGPUComputeTexture::TextureDimension::DIMENSION_1D:
      return wgpu::TextureDimension::e1D;

    case vtkWebGPUComputeTexture::TextureDimension::DIMENSION_2D:
      return wgpu::TextureDimension::e2D;

    case vtkWebGPUComputeTexture::TextureDimension::DIMENSION_3D:
      return wgpu::TextureDimension::e3D;

    default:
      vtkLog(ERROR,
        "Unhandled texture dimension in ComputeTextureDimensionToWebGPU: "
          << dimension << ". Assuming DIMENSION_2D.");
      return wgpu::TextureDimension::e2D;
  }
}

//------------------------------------------------------------------------------
wgpu::TextureViewDimension vtkWebGPUComputePass::ComputeTextureDimensionToViewDimension(
  vtkWebGPUComputeTexture::TextureDimension dimension)
{
  switch (dimension)
  {
    case vtkWebGPUComputeTexture::TextureDimension::DIMENSION_1D:
      return wgpu::TextureViewDimension::e1D;

    case vtkWebGPUComputeTexture::TextureDimension::DIMENSION_2D:
      return wgpu::TextureViewDimension::e2D;

    case vtkWebGPUComputeTexture::TextureDimension::DIMENSION_3D:
      return wgpu::TextureViewDimension::e3D;

    default:
      vtkLog(ERROR,
        "Unhandled texture view dimension in ComputeTextureDimensionToViewDimension: "
          << dimension << ". Assuming DIMENSION_2D.");
      return wgpu::TextureViewDimension::e2D;
  }
}

//------------------------------------------------------------------------------
wgpu::TextureUsage vtkWebGPUComputePass::ComputeTextureModeToUsage(
  vtkWebGPUComputeTexture::TextureMode mode, const std::string& textureLabel)
{
  switch (mode)
  {
    case vtkWebGPUComputeTexture::TextureMode::READ_ONLY:
      return wgpu::TextureUsage::TextureBinding | wgpu::TextureUsage::CopyDst;

    case vtkWebGPUComputeTexture::TextureMode::WRITE_ONLY_STORAGE:
      return wgpu::TextureUsage::StorageBinding | wgpu::TextureUsage::CopySrc;

    case vtkWebGPUComputeTexture::READ_WRITE_STORAGE:
      return wgpu::TextureUsage::TextureBinding | wgpu::TextureUsage::StorageBinding |
        wgpu::TextureUsage::CopySrc | wgpu::TextureUsage::CopyDst;

    default:
      vtkLog(ERROR,
        "Compute texture \"" << textureLabel
                             << "\" has undefined mode. Did you forget to call "
                                "vtkWebGPUComputeTexture::SetMode()?");

      return wgpu::TextureUsage::None;
  }
}

//------------------------------------------------------------------------------
wgpu::StorageTextureAccess vtkWebGPUComputePass::ComputeTextureModeToShaderStorage(
  vtkWebGPUComputeTexture::TextureMode mode, const std::string& textureLabel)
{
  switch (mode)
  {
    case vtkWebGPUComputeTexture::TextureMode::READ_ONLY:
      return wgpu::StorageTextureAccess::ReadOnly;

    case vtkWebGPUComputeTexture::TextureMode::WRITE_ONLY_STORAGE:
      return wgpu::StorageTextureAccess::WriteOnly;

    case vtkWebGPUComputeTexture::READ_WRITE_STORAGE:
      return wgpu::StorageTextureAccess::ReadWrite;

    default:
      vtkLog(ERROR,
        "Compute texture \"" << textureLabel
                             << "\" has undefined mode. Did you forget to call "
                                "vtkWebGPUComputeTexture::SetMode()?");

      return wgpu::StorageTextureAccess::Undefined;
  }
}

//------------------------------------------------------------------------------
wgpu::StorageTextureAccess vtkWebGPUComputePass::ComputeTextureViewModeToShaderStorage(
  vtkWebGPUComputeTextureView::TextureViewMode mode, const std::string& textureViewLabel)
{
  switch (mode)
  {
    case vtkWebGPUComputeTextureView::TextureViewMode::READ_ONLY:
      return wgpu::StorageTextureAccess::ReadOnly;

    case vtkWebGPUComputeTextureView::TextureViewMode::WRITE_ONLY_STORAGE:
      return wgpu::StorageTextureAccess::WriteOnly;

    case vtkWebGPUComputeTextureView::TextureViewMode::READ_WRITE_STORAGE:
      return wgpu::StorageTextureAccess::ReadWrite;

    default:
      vtkLog(ERROR,
        "Compute texture view \"" << textureViewLabel
                                  << "\" has undefined mode. Did you forget to call "
                                     "vtkWebGPUComputeTextureView::SetMode()?");

      return wgpu::StorageTextureAccess::Undefined;
  }
}

//------------------------------------------------------------------------------
wgpu::TextureSampleType vtkWebGPUComputePass::ComputeTextureSampleTypeToWebGPU(
  vtkWebGPUComputeTexture::TextureSampleType sampleType)
{
  switch (sampleType)
  {
    case vtkWebGPUComputeTexture::TextureSampleType::FLOAT:
      return wgpu::TextureSampleType::Float;

    case vtkWebGPUComputeTexture::TextureSampleType::UNFILTERABLE_FLOAT:
      return wgpu::TextureSampleType::UnfilterableFloat;

    case vtkWebGPUComputeTexture::TextureSampleType::DEPTH:
      return wgpu::TextureSampleType::Depth;

    case vtkWebGPUComputeTexture::TextureSampleType::SIGNED_INT:
      return wgpu::TextureSampleType::Sint;

    case vtkWebGPUComputeTexture::TextureSampleType::UNSIGNED_INT:
      return wgpu::TextureSampleType::Uint;

    default:
      vtkLog(
        ERROR, "Unhandled texture sampleType in ComputeTextureSampleTypeToWebGPU: " << sampleType);
      return wgpu::TextureSampleType::Undefined;
  }
}

//------------------------------------------------------------------------------
wgpu::TextureAspect vtkWebGPUComputePass::ComputeTextureViewAspectToWebGPU(
  vtkWebGPUComputeTextureView::TextureViewAspect aspect)
{
  switch (aspect)
  {
    case vtkWebGPUComputeTextureView::TextureViewAspect::ASPECT_ALL:
      return wgpu::TextureAspect::All;

    case vtkWebGPUComputeTextureView::TextureViewAspect::ASPECT_DEPTH:
      return wgpu::TextureAspect::DepthOnly;

    case vtkWebGPUComputeTextureView::TextureViewAspect::ASPECT_STENCIL:
      return wgpu::TextureAspect::StencilOnly;

    default:
      vtkLog(ERROR,
        "Unhandled texture view aspect in ComputeTextureViewAspectToWebGPU: "
          << aspect << ". Assuming ASPECT_ALL.");
      return wgpu::TextureAspect::All;
  }
}

VTK_ABI_NAMESPACE_END
