// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include "vtkTimeStamp.h"

#include "vtkObjectFactory.h"
#include "vtkWindows.h"

#include <atomic>

//------------------------------------------------------------------------------
VTK_ABI_NAMESPACE_BEGIN
vtkTimeStamp* vtkTimeStamp::New()
{
  // If the factory was unable to create the object, then create it here.
  return new vtkTimeStamp;
}

#include <signal.h>

//------------------------------------------------------------------------------
void vtkTimeStamp::Modified()
{
  // Here because of a static destruction error? You're not the first. After
  // discussion of the tradeoffs, the cost of adding a Schwarz counter on this
  // static to ensure it gets destructed is unlikely to be worth the cost over
  // just leaking it.
  //
  // Solutions and their tradeoffs:
  //
  //  - Schwarz counter: each VTK class now has a static initializer function
  //    that increments an integer. This cannot be inlined or optimized away.
  //    Adds latency to ParaView startup.
  //  - Separate library for this static. This adds another library to VTK
  //    which are already legion. It could not be folded into a kit because
  //    that would bring you back to the same problem you have today.
  //  - Leak a heap allocation for it. It's 24 bytes, leaked exactly once, and
  //    is easily suppressed in Valgrind.
  //
  // The last solution has been decided to have the smallest downside of these.
  //
  // Good luck!
#if defined(VTK_USE_64BIT_TIMESTAMPS) || (VTK_SIZEOF_VOID_P == 8)
  static std::atomic<uint64_t> GlobalTimeStamp(0U);
#else
  static std::atomic<uint32_t> GlobalTimeStamp(0U);
#endif
  this->ModifiedTime = (vtkMTimeType)++GlobalTimeStamp;
  if (this->ModifiedTime == 4812 || this->ModifiedTime == 4813)
    raise(SIGTRAP);
}
VTK_ABI_NAMESPACE_END
