/*****************************************************************************
*
* Copyright (c) 2000 - 2018, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-442911
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

#include <ExodusPluginInfo.h>
#include <avtExodusFileFormat.h>
#include <avtMTSDFileFormatInterface.h>
#include <avtGenericDatabase.h>
#include <avtExodusOptions.h>

// ****************************************************************************
//  Method:  ExodusCommonPluginInfo::GetDatabaseType
//
//  Purpose:
//    Returns the type of a Exodus database.
//
//  Programmer:  generated by xml2info
//  Creation:    omitted
//
// ****************************************************************************
DatabaseType
ExodusCommonPluginInfo::GetDatabaseType()
{
    return DB_TYPE_MTSD;
}

// ****************************************************************************
//  Method: ExodusCommonPluginInfo::SetupDatabase
//
//  Purpose:
//      Sets up a Exodus database.
//
//  Arguments:
//      list    A list of file names.
//      nList   The number of timesteps in list.
//      nBlocks The number of blocks in the list.
//
//  Returns:    A Exodus database from list.
//
//  Programmer: Jeremy Meredith
//  Creation:   August 27, 2002
//
//  Modifications:
//
//    Hank Childs, Tue Feb 25 13:56:42 PST 2003
//    Account for .EXODUS, .nemesis, and .NEMESIS.
//
//    Hank Childs, Thu Jul 22 11:29:31 PDT 2004
//    Register the file list 
//
//    Jeremy Meredith, Thu Jan 28 12:28:07 EST 2010
//    MTSD now accepts grouping multiple files into longer sequences, so
//    its interface has changed to accept both a number of timestep groups
//    and a number of blocks.
//
//    Brad Whitlock, Tue Feb 2 10:06:34 PDT 2010
//    I changed the code so when nBlocks==-1 it will treat the list of files
//    as domains so multiblock files work again.
//
//    Mark C. Miller, Wed Jan  8 18:07:37 PST 2014
//    I enabled this to work for cases in which we have timestep groups *and*
//    many files as well as the most common case where nBlocks is number of
//
//    files in the list.
//    Mark C. Miller, Thu Dec 18 12:59:32 PST 2014
//    Added stuff to pass around zero index instance.
// ****************************************************************************
#include <string>

avtDatabase *
ExodusCommonPluginInfo::SetupDatabase(const char *const *list, int nList, int nBlock)
{
    char  **reallist  = NULL;
    bool    read_list_from_text_file = false;
    int     listcount = 0;
    int     bang_nBlocks = -1;
    int     fileListId = -1;
    int     nTimestepGroups = 1;

    if (nList == 1)
        read_list_from_text_file = avtDatabase::GetFileListFromTextFile(list[0],
            reallist, listcount, &bang_nBlocks);

    if (read_list_from_text_file)
    {
        if (bang_nBlocks > 0)
            nBlock = bang_nBlocks;
        else
            nBlock = listcount;
        list = reallist;
        nList = listcount;
    }
    else
    {
        nBlock = nList;
    }

    fileListId = avtExodusFileFormat::RegisterFileList(list, nList);
    nTimestepGroups = nList / nBlock;

    avtMTSDFileFormat ***ffl = new avtMTSDFileFormat**[nTimestepGroups];
    avtExodusFileFormat *zidx = 0;
    for (int i = 0 ; i < nTimestepGroups ; i++)
    {
        ffl[i] = new avtMTSDFileFormat*[nBlock];
        for (int j = 0 ; j < nBlock ; j++)
        {
            avtExodusFileFormat *exo = new avtExodusFileFormat(list[i*nBlock+j], readOptions);
            if (zidx == 0) zidx = exo;
            exo->SetZeroIndexInstance(zidx);
            exo->SetFileList(fileListId);
            ffl[i][j] = exo; 
        }
    }

    avtMTSDFileFormatInterface *inter 
        = new avtMTSDFileFormatInterface(ffl, nTimestepGroups, nBlock);

    if (reallist)
    {
        for (int i = 0 ; i < listcount ; i++)
            delete [] reallist[i];
        delete [] reallist;
    }

    return new avtGenericDatabase(inter);
}


// ****************************************************************************
//  Method: ExodusCommonPluginInfo::GetReadOptions
//
//  Purpose:
//      Gets the read options.
//
//  Programmer: generated by xml2info
//  Creation:   omitted
//
// ****************************************************************************

DBOptionsAttributes *
ExodusCommonPluginInfo::GetReadOptions() const
{
    return GetExodusReadOptions();
}
// ****************************************************************************
//  Method: ExodusCommonPluginInfo::GetWriteOptions
//
//  Purpose:
//      Gets the write options.
//
//  Programmer: generated by xml2info
//  Creation:   omitted
//
// ****************************************************************************

DBOptionsAttributes *
ExodusCommonPluginInfo::GetWriteOptions() const
{
    return GetExodusWriteOptions();
}
// ****************************************************************************
//  Method: ExodusCommonPluginInfo::GetLicense
//
//  Purpose:
//      Gets the write options.
//
//  Programmer: generated by xml2info
//  Creation:   omitted
//
// ****************************************************************************

std::string
ExodusCommonPluginInfo::GetLicense() const
{
    return std::string();
}
