/*****************************************************************************
*
* Copyright (c) 2000 - 2017, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-442911
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

#include <VTKPluginInfo.h>
#include <avtVTKFileFormat.h>
#include <avtSTMDFileFormatInterface.h>
#include <avtSTSDFileFormatInterface.h>
#include <avtGenericDatabase.h>
#include <avtVTKOptions.h>

VTKCommonPluginInfo::VTKCommonPluginInfo() : CommonDatabasePluginInfo(), VTKGeneralPluginInfo()
{
    // Assume SD by default.
    dbType = DB_TYPE_STSD;
}

// ****************************************************************************
//  Method:  VTKCommonPluginInfo::GetDatabaseType
//
//  Purpose:
//    Returns the type of a VTK database.
//
//  Programmer:  generated by xml2info
//  Creation:    omitted
//
// ****************************************************************************
DatabaseType
VTKCommonPluginInfo::GetDatabaseType()
{
    return dbType;
}

// ****************************************************************************
//  Method: VTKCommonPluginInfo::SetupDatabase
//
//  Purpose:
//      Sets up a VTK database.
//
//  Arguments:
//      list    A list of file names.
//      nList   The number of timesteps in list.
//      nBlocks The number of blocks in the list.
//
//  Returns:    A VTK database from list.
//
//  Programmer: generated by xml2info
//  Creation:   omitted
//
//  Modifications:
//    Brad Whitlock, Mon Oct 22 17:03:44 PDT 2012
//    I changed the code so it switches the interface based on how many domains
//    are present in the 1st file.
//
// ****************************************************************************
avtDatabase *
VTKCommonPluginInfo::SetupDatabase(const char *const *list,
                                   int nList, int nBlock)
{
    avtDatabase *db = NULL;

    // Figure out how many domains there are in the 1st file.
    avtVTKFileReader *reader = new avtVTKFileReader(list[0], readOptions);
    int nDomains = 1;
    TRY
    {
        nDomains = reader->GetNumberOfDomains();
    }
    CATCHALL
    {
        delete reader;
        RETHROW;
    }
    ENDTRY

    // Return the proper interface.
    if(nDomains > 1)
    {
        dbType = DB_TYPE_STMD;

        // STMD case
        avtSTMDFileFormat **ffl = new avtSTMDFileFormat*[nList];
        for (int i = 0 ; i < nList ; i++)
        {
            if(i == 0)
                ffl[i] = new avtVTK_STMDFileFormat(list[i], readOptions, reader);
            else
                ffl[i] = new avtVTK_STMDFileFormat(list[i], readOptions);
        }
        avtSTMDFileFormatInterface *inter 
           = new avtSTMDFileFormatInterface(ffl, nList);
        db = new avtGenericDatabase(inter);
    }
    else
    {
        dbType = DB_TYPE_STSD;

        // STSD case
        int nTimestep = nList / nBlock;
        avtSTSDFileFormat ***ffl = new avtSTSDFileFormat**[nTimestep];
        for (int i = 0 ; i < nTimestep ; i++)
        {
            ffl[i] = new avtSTSDFileFormat*[nBlock];
            for (int j = 0 ; j < nBlock ; j++)
            {
                if(i == 0 && j == 0)
                    ffl[i][j] = new avtVTK_STSDFileFormat(list[i*nBlock + j], readOptions, reader);
                else 
                    ffl[i][j] = new avtVTK_STSDFileFormat(list[i*nBlock + j], readOptions);
            }
        }
        avtSTSDFileFormatInterface *inter 
            = new avtSTSDFileFormatInterface(ffl, nTimestep, nBlock);
        db = new avtGenericDatabase(inter);
    }

    return db;
}

// ****************************************************************************
//  Method: VTKCommonPluginInfo::GetReadOptions
//
//  Purpose:
//      Gets the read options.
//
//  Programmer: generated by xml2info
//  Creation:   omitted
//
// ****************************************************************************

DBOptionsAttributes *
VTKCommonPluginInfo::GetReadOptions() const
{
    return GetVTKReadOptions();
}

// ****************************************************************************
//  Method: VTKCommonPluginInfo::GetWriteOptions
//
//  Purpose:
//      Gets the write options.
//
//  Programmer: generated by xml2info
//  Creation:   omitted
//
// ****************************************************************************

DBOptionsAttributes *
VTKCommonPluginInfo::GetWriteOptions() const
{
    return GetVTKWriteOptions();
}
