/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangeHexahedron.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkLagrangeHexahedron
// .SECTION Description
// .SECTION See Also

#ifndef vtkLagrangeHexahedron_h
#define vtkLagrangeHexahedron_h

#include "vtkCommonDataModelModule.h" // For export macro
#include "vtkNonLinearCell.h"
#include "vtkSmartPointer.h" // For member variable.
#include "vtkCellType.h" // For GetCellType.
#include "vtkNew.h" // For member variable.

class vtkCellData;
class vtkDoubleArray;
class vtkHexahedron;
class vtkIdList;
class vtkPointData;
class vtkPoints;
class vtkVector3d;
class vtkVector3i;

class VTKCOMMONDATAMODEL_EXPORT vtkLagrangeHexahedron : public vtkNonLinearCell
{
public:
  static vtkLagrangeHexahedron* New();
  vtkTypeMacro(vtkLagrangeHexahedron,vtkNonLinearCell);
  void PrintSelf(ostream& os, vtkIndent indent);

  int GetCellType() { return VTK_LAGRANGE_HEXAHEDRON; }
  int GetCellDimension() { return 3; }
  int RequiresInitialization() { return 1; }
  int GetNumberOfEdges() { return 12; }
  int GetNumberOfFaces() { return 6; }
  vtkCell* GetEdge(int edgeId);
  vtkCell* GetFace(int faceId);

  void Initialize();

  int CellBoundary(int subId, double pcoords[3], vtkIdList* pts);
  int EvaluatePosition(
    double x[3], double* closestPoint,
    int& subId, double pcoords[3],
    double& dist2, double* weights);
  void EvaluateLocation(
    int& subId, double pcoords[3], double x[3],
    double* weights);
  void Contour(
    double value, vtkDataArray* cellScalars,
    vtkIncrementalPointLocator* locator, vtkCellArray* verts,
    vtkCellArray* lines, vtkCellArray* polys,
    vtkPointData* inPd, vtkPointData* outPd,
    vtkCellData* inCd, vtkIdType cellId, vtkCellData* outCd);
  void Clip(
    double value, vtkDataArray* cellScalars,
    vtkIncrementalPointLocator* locator, vtkCellArray* polys,
    vtkPointData* inPd, vtkPointData* outPd,
    vtkCellData* inCd, vtkIdType cellId, vtkCellData* outCd,
    int insideOut);
  int IntersectWithLine(
    double p1[3], double p2[3], double tol, double& t,
    double x[3], double pcoords[3], int& subId);
  int Triangulate(int index, vtkIdList* ptIds, vtkPoints* pts);
  void Derivatives(
    int subId, double pcoords[3], double* values,
    int dim, double* derivs);
  double* GetParametricCoords();

  double GetParametricDistance(double pcoords[3]);

  const int* GetOrder();
  int GetOrder(int i) { return this->GetOrder()[i]; }

  void InterpolateFunctions(double pcoords[3], double* weights);
  void InterpolateDerivs(double pcoords[3], double* derivs);

  bool SubCellCoordinatesFromId(vtkVector3i& ijk, int subId);
  bool SubCellCoordinatesFromId(int& i, int& j, int& k, int subId);
  int PointIndexFromIJK(int i, int j, int k);
  bool TransformApproxToCellParams(int subCell, double* pcoords);

protected:
  vtkLagrangeHexahedron();
  ~vtkLagrangeHexahedron();

  vtkHexahedron* GetApprox();
  void PrepareApproxData(vtkPointData* pd, vtkCellData* cd, vtkIdType cellId, vtkDataArray* cellScalars);
  vtkHexahedron* GetApproximateHex(
    int subId, vtkDataArray* scalarsIn = NULL, vtkDataArray* scalarsOut = NULL);

  int Order[4];
  vtkSmartPointer<vtkPoints> PointParametricCoordinates;
  vtkSmartPointer<vtkHexahedron> Approx;
  vtkSmartPointer<vtkPointData> ApproxPD;
  vtkSmartPointer<vtkCellData> ApproxCD;
  vtkNew<vtkDoubleArray> CellScalars;
  vtkNew<vtkDoubleArray> Scalars;
  vtkNew<vtkPoints> TmpPts;
  vtkNew<vtkIdList> TmpIds;

private:
  vtkLagrangeHexahedron(const vtkLagrangeHexahedron&);  // Not implemented.
  void operator=(const vtkLagrangeHexahedron&);  // Not implemented.
};

#endif // vtkLagrangeHexahedron_h
