/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkPlotCheater.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

// .NAME vtkPlotCheater - Class for drawing an a cheater plot.
//
// .SECTION Description
// This class allows to draw a cheater plot given three columns from
// a vtkTable. The first two columns represent the independent variables, and
// the third one specifies the dependent variable.
//
// .SECTION See Also
// vtkPlotPoints

#ifndef vtkPlotCheater_h
#define vtkPlotCheater_h

#include "vtkChartsCoreModule.h" // For export macro
#include "vtkPlotPoints.h"

class vtkPen;

class VTKCHARTSCORE_EXPORT vtkPlotCheater : public vtkPlotPoints
{
public:
  vtkTypeMacro(vtkPlotCheater, vtkPlotPoints);
  virtual void PrintSelf(ostream &os, vtkIndent indent);

  // Description:
  // Creates a new Cheater Plot object.
  static vtkPlotCheater *New();

  // Description:
  // Perform any updates to the item that may be necessary before rendering.
  // The scene should take care of calling this on all items before their
  // Paint function is invoked.
  virtual void Update();

  // Description:
  // Paint event for the XY plot, called whenever the chart needs to be drawn.
  virtual bool Paint(vtkContext2D *painter);

  // Description:
  // Paint legend event for the XY plot, called whenever the legend needs the
  // plot items symbol/mark/line drawn. A rect is supplied with the lower left
  // corner of the rect (elements 0 and 1) and with width x height (elements 2
  // and 3). The plot can choose how to fill the space supplied.
  virtual bool PaintLegend(vtkContext2D *painter, const vtkRectf& rect,
                           int legendIndex);

  // Description:
  // Get the plot labels. If this array has a length greater than 1 the index
  // refers to the stacked objects in the plot. See vtkPlotBar for example.
  virtual vtkStringArray *GetLabels();

  // Description:
  // Generate and return the tooltip label string for this plot
  // The segmentIndex parameter is ignored, except for vtkPlotBar
  virtual vtkStdString GetTooltipLabel(const vtkVector2d &plotPos,
                                       vtkIdType seriesIndex,
                                       vtkIdType segmentIndex);

  // Description:
  // Set the input, we are expecting a vtkTable with three columns. The first
  // and second columns represent the independent parameters. The third
  // column represents the dependent parameter. Inherited method will call the
  // last SetInputData method with default paramaters.
  virtual void SetInputData(vtkTable *table);
  virtual void SetInputData(vtkTable *table, const vtkStdString &yColumn,
                            const vtkStdString &densityColumn);
  virtual void SetInputData(vtkTable *table, const vtkStdString &xColumn,
                            const vtkStdString &yColumn,
                            const vtkStdString &densityColumn);

  virtual void SetInputData(vtkTable *table, vtkIdType xColumn,
                            vtkIdType yColumn,
                            vtkIdType densityColumn);

  // Description:
  // Set/get the vtkPen object that controls how this plot draws boundary lines.
  void SetLinePen(vtkPen* pen);
  vtkGetObjectMacro(LinePen, vtkPen);

  // Description:
  // Set/get the vtkPen object that controls how this plot draws points.
  // Those are just helpers functions:
 //  this pen is actually the default Plot pen.
  void SetPointPen(vtkPen* pen) { this->SetPen(pen); }
  vtkPen* GetPointPen() { return this->GetPen(); }

protected:
  vtkPlotCheater();
  ~vtkPlotCheater();

  void UpdateTableCache(vtkDataArray*);

  vtkPoints2D* CheaterPoints;
  vtkPen* LinePen;

private:
  vtkPlotCheater(const vtkPlotCheater &) VTK_DELETE_FUNCTION;
  void operator=(const vtkPlotCheater &) VTK_DELETE_FUNCTION;
};

#endif //vtkPlotCheater_h
