#!/usr/bin/env python3

""" backport-commits-for-milestone.py:

Given a milestone, scrape VTK's gitlab repository for merged MRs with the
milestone assigned to them. For each of these MRs, check if it has been
backported to VTK's release branch. If any have not, create a branch and
cherry-pick these merge requests onto it. If a commit is a merge commit, skip it
(these must be handled by hand) and notify the user of these outstanding MRs.

Prior to running, make sure your repository is up to date with vtk (`git pull`).
"""

import argparse
import git
import gitlab
import json
import os

arg_parser = argparse.ArgumentParser()
arg_parser.add_argument('-t', '--token',
                        help='gitlab api token (gitlab->settings->access tokens)',
                        required=True)

arg_parser.add_argument('-m', '--milestone',
                        help='milestone to query for merge requests',
                        default="")

args = arg_parser.parse_args()

# Access the repository containing this script
repo = git.Repo('.')

# Check that the repository loaded correctly
if not repo.bare:
    # record tthe name of the remote branch associated with 'gitlab' (this is
    # the user's repository)
    gitlab_url = repo.remotes['gitlab'].url
else:
    print(
        'Could not load repository. Please run this script from within the checked-out repository')
    exit(1)

# Check out the release branch
repo.git.checkout('release')

# Access gitlab using a token
gl = gitlab.Gitlab('https://gitlab.kitware.com/', private_token=args.token)
gl.auth()

# VTK is project # 13 in gitlab
vtk_vtk = gl.projects.get(13)

# Grab the merged mrs with the milestone of interest
try:
    ms = [m for m in vtk_vtk.milestones.list() if m.title == args.milestone][0]
except IndexError:
    print('Cannot find milestone %s in repository' % args.milestone)
    exit(1)
merged_requests = [mr for mr in ms.merge_requests() if mr.state == 'merged']

# Check if a MR's sha is in the release branch
def check_in_release(repo, mr):
    output = os.popen('git log --first-parent --grep \'Merge-request: !%s\' origin/release' %
                      mr.iid).read()
    return output != ''

# Check if a MR contains a merge commit. A merge commit will have more than one
# parent
def check_third_party(mr):
    output = os.popen('git show --pretty=%%P %s' % mr.sha).read().partition('\n')[0]
    if len(output.split(' ')) > 1:
        return True
    return False

# Create a branch based off of release for performing the backports
repo.git.checkout('release', b='backport-tagged-commits')

cherry_picked_mrs = []
skipped_mrs = []

for mr in merged_requests:
    if not check_in_release(repo, mr):
        if not check_third_party(mr):
            os.popen('git cherry-pick --strategy=recursive -x -X theirs %s' % mr.sha).read()
            cherry_picked_mrs.append(mr)
        else:
            skipped_mrs.append(mr)

print('\nThe following merge requests were added to branch \'backport-tagged-commits\':\n')
print('\n'.join([mr.sha for mr in cherry_picked_mrs]) + '\n')

print('\nThe following merge requests were skipped, and must be backported by hand:\n')
print('\n'.join([mr.sha for mr in skipped_mrs]) + '\n')

