/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangePyramid.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkLagrangePyramid.h"

#include "vtkObjectFactory.h"
#include "vtkMath.h"
#include "vtkLine.h"
#include "vtkLagrangeCurve.h"
#include "vtkLagrangeTriangle.h"
#include "vtkLagrangeQuadrilateral.h"
#include "vtkPyramid.h"
#include "vtkDoubleArray.h"
#include "vtkPoints.h"

namespace
{
  // The linearized pyramid is comprised of five linearized faces. Each face has
  // either three or four vertices.
  static const vtkIdType FaceVertices[5][4] = {{0,1,2,3},
                                               {0,1,4,-1}, {1,2,4,-1},
                                               {2,3,4,-1}, {3,0,4,-1}};

  // The linearized pyramid is comprised of eight linearized edges. Each edge
  // has two vertices.
  static const vtkIdType EdgeVertices[8][2] = {{0,1},{1,2},{2,3},{3,0},
                                               {1,4},{4,0},{2,4},{3,4}};

  // The rational coordinates of the five vertices of the linear pyramid.
  static const vtkIdType LinearVertices[5][3] = {{0,0,0}, {1,0,0}, {1,1,0},
                                                 {0,1,0}, {0,0,1}};


  // Each linearized pyramid edge has two rational coordinates at their extrema
  // and varies the other one. These are the coordinates that are at their
  // extrema for each edge, as well as their extremum (min or max).
  static const vtkIdType EdgeFixedCoords[8][2][2] =
  {{{1,0},{2,0}},{{0,1},{2,0}},
   {{1,1},{2,0}},{{0,0},{2,0}},
   {{0,1},{1,0}},{{0,0},{1,0}},
   {{0,1},{1,1}},{{0,0},{1,1}}};

  // Each linearized pyramid edge has two rational coordinates at their extrema
  // and varies the other one. These are the coordinates that vary for each
  // edge, as well as the direction in which it varies (positively or
  // negatively).
  static const vtkIdType EdgeCountingCoord[8][2] = {{0,1},{1,1},{0,-1},{1,-1},
                                                    {2,1},{2,-1},{2,1},{2,1}};

  // Each linearized pyramid face has either a rational coordinate at its
  // minimum or a pair of rational coordinates that sum to its maximum. We
  // identify them here.
  static const vtkIdType FaceFixedCoords[5][2] = {{2,0},{1,0},{0,1},
                                                  {1,1},{0,0}};
}

vtkStandardNewMacro(vtkLagrangePyramid);
//----------------------------------------------------------------------------
vtkLagrangePyramid::vtkLagrangePyramid() :
  EdgeIds(NULL),
  ParametricCoordinates(NULL)
{
  this->Order = 0;

  this->Edge = vtkLagrangeCurve::New();
  this->Triangle = vtkLagrangeTriangle::New();
  this->Quad = vtkLagrangeQuadrilateral::New();
  this->Pyramid = vtkPyramid::New();
  this->Scalars = vtkDoubleArray::New();
  this->Scalars->SetNumberOfTuples(5);

  this->Points->SetNumberOfPoints(5);
  this->PointIds->SetNumberOfIds(5);
  for (int i = 0; i < 5; i++)
    {
    this->Points->SetPoint(i, 0.0, 0.0, 0.0);
    this->PointIds->SetId(i,0);
    }
}

//----------------------------------------------------------------------------
vtkLagrangePyramid::~vtkLagrangePyramid()
{
  delete [] this->ParametricCoordinates;
  delete [] this->EdgeIds;
  this->Edge->Delete();
  this->Triangle->Delete();
  this->Quad->Delete();
  this->Pyramid->Delete();
  this->Scalars->Delete();
}

//----------------------------------------------------------------------------
vtkCell *vtkLagrangePyramid::GetEdge(int edgeId)
{
  // vtkIdType order = this->GetOrder();

  // vtkIdType bindex[4] = {0,0,0,0};
  // bindex[EdgeVertices[edgeId][0]] = order;
  // for (vtkIdType i=0;i<=order;i++)
  //   {
  //   this->EdgeIds[i] = this->ToIndex(bindex);
  //   bindex[EdgeVertices[edgeId][0]]--;
  //   bindex[EdgeVertices[edgeId][1]]++;
  //   }
  // this->Edge->vtkCell::Initialize(order + 1, this->EdgeIds, this->Points);
  return this->Edge;
}

//----------------------------------------------------------------------------
vtkCell *vtkLagrangePyramid::GetFace(int faceId)
{
  // assert(faceId >= 0 && faceId < 4);

  // vtkIdType order = this->GetOrder();

  // vtkIdType nPoints = (order + 1)*(order + 2)/2;
  // this->Face->GetPointIds()->SetNumberOfIds(nPoints);
  // this->Face->GetPoints()->SetNumberOfPoints(nPoints);

  // vtkIdType tetBCoords[4], triBCoords[3];
  // for (vtkIdType p = 0; p < nPoints; p++)
  //   {
  //   vtkLagrangeTriangle::BarycentricIndex(p,triBCoords,order);

  //   for (vtkIdType coord = 0; coord < 3; coord++)
  //     {
  //     tetBCoords[FaceBCoords[faceId][coord]] = triBCoords[coord];
  //     }
  //   tetBCoords[FaceMinCoord[faceId]] = 0;

  //   vtkIdType pointIndex = vtkLagrangePyramid::Index(tetBCoords, order);
  //   this->Face->GetPointIds()->SetId(p,pointIndex);
  //   this->Face->GetPoints()->SetPoint(p,this->Points->GetPoint(pointIndex));
  //   }
  // this->Face->Initialize();

  return this->Quad;
}

//----------------------------------------------------------------------------
void vtkLagrangePyramid::Initialize()
{
  vtkIdType order = this->ComputeOrder();

  if (this->Order != order)
    {
    // Reset our caches
    this->Order = order;

    delete [] this->EdgeIds;
    this->EdgeIds = new vtkIdType[order+1];
    }
}

//----------------------------------------------------------------------------
int vtkLagrangePyramid::CellBoundary(int subId, double pcoords[3],
                                   vtkIdList *pts)
{
  // TODO: Fill me in!
  return 0;
}

//----------------------------------------------------------------------------
int vtkLagrangePyramid::EvaluatePosition(double* x, double* closestPoint,
                                       int& subId, double pcoords[3],
                                       double& minDist2, double *weights)
{
  // double pc[3], pcoordsMin[3], dist2, tempWeights[3], closest[3];
  // int returnStatus=0, status, ignoreId;
  // vtkIdType minBIndices[4][4], bindices[4][4], minIndices[4], pointIndices[4];

  // vtkIdType order = this->GetOrder();
  // vtkIdType numberOfSubtetras = this->GetNumberOfSubtetras();

  // minDist2 = VTK_DOUBLE_MAX;
  // for (vtkIdType subCellId = 0; subCellId < numberOfSubtetras; subCellId++)
  //   {
  //   this->SubtetraBarycentricPointIndices(subCellId, bindices);

  //   for (vtkIdType i=0;i<4;i++)
  //     {
  //     pointIndices[i] = this->ToIndex(bindices[i]);
  //     this->Face->Points->SetPoint(i, this->Points->GetPoint(pointIndices[i]));
  //     }

  //   status = this->Face->EvaluatePosition(x, closest, ignoreId, pc, dist2,
  //                                         tempWeights);

  //   if ( status != -1 && dist2 < minDist2 )
  //     {
  //     returnStatus = status;
  //     minDist2 = dist2;
  //     subId = subCellId;
  //     pcoordsMin[0] = pc[0];
  //     pcoordsMin[1] = pc[1];
  //     pcoordsMin[2] = pc[2];
  //     for (vtkIdType i=0;i<4;i++)
  //       {
  //       minIndices[i] = pointIndices[i];
  //       for (vtkIdType j=0;j<4;j++)
  //         {
  //         minBIndices[i][j] = bindices[i][j];
  //         }
  //       }
  //     }
  //   }

  // // adjust parametric coordinates
  // if ( returnStatus != -1 )
  //   {
  //   for (vtkIdType i=0;i<3;i++)
  //     {
  //     pcoords[i]= (minBIndices[0][i] +
  //                  pcoordsMin[0]*(minBIndices[1][i] - minBIndices[0][i]) +
  //                  pcoordsMin[1]*(minBIndices[2][i] - minBIndices[0][i]) +
  //                  pcoordsMin[2]*(minBIndices[3][i] - minBIndices[0][i]))/order;
  //     }

  //   if(closestPoint!=0)
  //     {
  //     // Compute both closestPoint and weights
  //     this->EvaluateLocation(subId,pcoords,closestPoint,weights);
  //     }
  //   else
  //     {
  //     // Compute weights only
  //     this->InterpolateFunctions(pcoords,weights);
  //     }
  //   }

  // return returnStatus;
  return 0;
}

//----------------------------------------------------------------------------
void vtkLagrangePyramid::EvaluateLocation(int& vtkNotUsed(subId),
                                        double pcoords[3], double x[3],
                                        double *weights)
{
  // x[0] = x[1] = x[2] = 0.;

  // this->InterpolateFunctions(pcoords,weights);

  // double p[3];
  // vtkIdType nPoints = this->GetPoints()->GetNumberOfPoints();
  // for (vtkIdType idx = 0; idx < nPoints; idx++)
  //   {
  //   this->Points->GetPoint(idx,p);
  //   for (vtkIdType jdx = 0; jdx < 3; jdx++)
  //     {
  //     x[jdx] += p[jdx]*weights[idx];
  //     }
  //   }
}

//----------------------------------------------------------------------------
void vtkLagrangePyramid::Contour(double value,
                               vtkDataArray* cellScalars,
                               vtkIncrementalPointLocator* locator,
                               vtkCellArray *verts,
                               vtkCellArray* lines,
                               vtkCellArray* polys,
                               vtkPointData* inPd,
                               vtkPointData* outPd,
                               vtkCellData* inCd,
                               vtkIdType cellId,
                               vtkCellData* outCd)
{
  // vtkIdType bindices[4][4];
  // vtkIdType numberOfSubtetras = this->GetNumberOfSubtetras();

  // for (vtkIdType subCellId = 0; subCellId < numberOfSubtetras; subCellId++)
  //   {
  //   this->SubtetraBarycentricPointIndices(subCellId, bindices);

  //   for (vtkIdType i=0;i<4;i++)
  //     {
  //     vtkIdType pointIndex = this->ToIndex(bindices[i]);
  //     this->Pyramid->Points->SetPoint(i, this->Points->GetPoint(pointIndex));
  //     if ( outPd )
  //       {
  //       this->Pyramid->PointIds->SetId(i, this->PointIds->GetId(pointIndex));
  //       }
  //     this->Scalars->SetTuple(i, cellScalars->GetTuple(pointIndex));
  //     }

  //   this->Pyramid->Contour(value, this->Scalars, locator, verts,
  //                        lines, polys, inPd, outPd, inCd, cellId, outCd);

  //   }
}

//----------------------------------------------------------------------------
void vtkLagrangePyramid::Clip(double value,
                            vtkDataArray* cellScalars,
                            vtkIncrementalPointLocator* locator,
                            vtkCellArray* polys,
                            vtkPointData* inPd,
                            vtkPointData* outPd,
                            vtkCellData* inCd,
                            vtkIdType cellId,
                            vtkCellData* outCd,
                            int insideOut)
{
  // vtkIdType bindices[4][4];
  // vtkIdType numberOfSubtetras = this->GetNumberOfSubtetras();

  // for (vtkIdType subCellId = 0; subCellId < numberOfSubtetras; subCellId++)
  //   {
  //   this->SubtetraBarycentricPointIndices(subCellId, bindices);

  //   for (vtkIdType i=0;i<4;i++)
  //     {
  //     vtkIdType pointIndex = this->ToIndex(bindices[i]);
  //     this->Pyramid->Points->SetPoint(i, this->Points->GetPoint(pointIndex));
  //     if ( outPd )
  //       {
  //       this->Pyramid->PointIds->SetId(i, this->PointIds->GetId(pointIndex));
  //       }
  //     this->Scalars->SetTuple(i, cellScalars->GetTuple(pointIndex));
  //     }

  //   this->Pyramid->Clip(value, this->Scalars, locator, polys, inPd, outPd,
  //                     inCd, cellId, outCd, insideOut);
  //   }
}

//----------------------------------------------------------------------------
int vtkLagrangePyramid::IntersectWithLine(double* p1,
                                        double* p2,
                                        double tol,
                                        double& t,
                                        double* x,
                                        double* pcoords,
                                        int& subId)
{
  // int subTest;

  // t = VTK_DOUBLE_MAX;
  // double tTmp;
  // double xMin[3], pcoordsMin[3];

  // for (int i=0;i<this->GetNumberOfFaces();i++)
  //   {
  //   if (this->GetFace(i)->IntersectWithLine(p1, p2, tol, tTmp, xMin, pcoordsMin,
  //                                           subTest) && tTmp < t)
  //     {
  //     for (vtkIdType j=0;j<3;j++)
  //       {
  //       x[j] = xMin[j];
  //       pcoords[FaceBCoords[i][j]] = pcoordsMin[j];
  //       }
  //     pcoords[FaceMinCoord[i]] = 0.;
  //     t = tTmp;
  //     }
  //   }
  // return (t == VTK_DOUBLE_MAX ? 0 : 1);
  return 0;
}

//----------------------------------------------------------------------------
int vtkLagrangePyramid::Triangulate(int vtkNotUsed(index), vtkIdList *ptIds,
                                  vtkPoints *pts)
{
  // pts->Reset();
  // ptIds->Reset();

  // vtkIdType bindices[4][4];
  // vtkIdType numberOfSubtetras = this->GetNumberOfSubtetras();

  // for (vtkIdType subCellId = 0; subCellId < numberOfSubtetras; subCellId++)
  //   {
  //   this->SubtetraBarycentricPointIndices(subCellId, bindices);

  //   for (vtkIdType i=0;i<4;i++)
  //     {
  //     vtkIdType pointIndex = this->ToIndex(bindices[i]);
  //     ptIds->InsertId(4*subCellId+i,this->PointIds->GetId(pointIndex));
  //     pts->InsertPoint(4*subCellId+i,this->Points->GetPoint(pointIndex));
  //     }
  //   }
  return 1;
}

//----------------------------------------------------------------------------
void vtkLagrangePyramid::Derivatives(int vtkNotUsed(subId),
                                   double vtkNotUsed(pcoords)[3],
                                   double *vtkNotUsed(values),
                                   int vtkNotUsed(dim),
                                   double *vtkNotUsed(derivs))
{
  // TODO: Fill me in!
  return;
}

//----------------------------------------------------------------------------

double* vtkLagrangePyramid::GetParametricCoords()
{
  if (!this->ParametricCoordinates)
    {
    vtkIdType order = this->GetOrder();
    double order_d = static_cast<vtkIdType>(order);

    vtkIdType nPoints = (order + 1)*(order + 2)*(2*order + 3)/6;
    this->ParametricCoordinates = new double[3*nPoints];

    vtkIdType pIdx = 0;
    vtkIdType rindex[3];
    for (vtkIdType p = 0; p < nPoints; p++)
      {
      this->ToRationalIndex(p,rindex);
      // this->ParametricCoordinates[pIdx] = -1. + 2.*rindex[0]/order_d;
      // this->ParametricCoordinates[pIdx + 1] = -1. + 2.*rindex[1]/order_d;
      // this->ParametricCoordinates[pIdx + 2] = rindex[2]/order_d;
      this->ParametricCoordinates[pIdx] = rindex[0]/order_d;
      this->ParametricCoordinates[pIdx + 1] = rindex[1]/order_d;
      this->ParametricCoordinates[pIdx + 2] = rindex[2]/order_d;
      pIdx += 3;
      }
    }

  return this->ParametricCoordinates;
}

//----------------------------------------------------------------------------
int vtkLagrangePyramid::GetParametricCenter(double pcoords[3])
{
  pcoords[0] = pcoords[1] = 0.4;
  pcoords[2] = 0.2;
  return 0;
}

//----------------------------------------------------------------------------
double vtkLagrangePyramid::GetParametricDistance(double pcoords[3])
{
  // int i;
  // double pDist, pDistMax=0.0;
  // double pc[4];

  // pc[0] = pcoords[0];
  // pc[1] = pcoords[1];
  // pc[2] = pcoords[2];
  // pc[3] = 1.0 - pcoords[0] - pcoords[1] - pcoords[2];

  // for (i=0; i<4; i++)
  //   {
  //   if ( pc[i] < 0.0 )
  //     {
  //     pDist = -pc[i];
  //     }
  //   else if ( pc[i] > 1.0 )
  //     {
  //     pDist = pc[i] - 1.0;
  //     }
  //   else //inside the cell in the parametric direction
  //     {
  //     pDist = 0.0;
  //     }
  //   if ( pDist > pDistMax )
  //     {
  //     pDistMax = pDist;
  //     }
  //   }

  // return pDistMax;
  return 0;
}
#if 0
namespace
{
// vtkIdType factorial(vtkIdType n, vtkIdType answer = 1)
// {
//   if (n == 0)
//     {
//     return answer;
//     }
//   return factorial(n-1,answer*n);
// }

// double JacobiP(vtkIdType alpha, vtkIdType beta, vtkIdType n, double x)
// {
//   double ans = factorial(n + alpha)*factorial(n + beta);
//   for (vtkIdType s = 0; s <= n; s++)
//     {
//     double eta = (x - 1.)*.5;
//     double xi  = (x + 1.)*.5;
//     ans *= (pow(eta,n-s)*pow(xi,s)) /
//       (factorial(s)*factorial(n+alpha-s)*factorial(beta+s)*factorial(n-s));
//     }
//   return ans;
// }

void ToInfCoords(const double* p, double* pInf)
{
  double prefac = 1./(1. - p[2]);
  pInf[0] = p[0]*prefac;
  pInf[1] = p[1]*prefac;
  pInf[2] = p[2]*prefac;
}

void FromInfCoords(const double* pInf, double* p)
{
  double prefac = 1./(1. + pInf[2]);
  p[0] = pInf[0]*prefac;
  p[1] = pInf[1]*prefac;
  p[2] = pInf[2]*prefac;
}

// double LagrangeP(double* pcoords, vtkIdType order, vtkIdType* ijk)
// {
//   double ans = 1.;
//   double xyz[3] = {pcoords[0]*order, pcoords[1]*order, pcoords[2]*(order-1.)};
//   for (vtkIdType ord=0;ord<=order;ord++)
//     {
//     for (vtkIdType m=0;m<2;m++)
//       {
//       if (ord != ijk[m])
//         {
//         ans *= (xyz[m] - ord)/(ijk[m] - ord);
//         }
//       }
//     if (ord != ijk[2] && ord != order)
//       {
//         ans *= (xyz[2] - ord)/(ijk[2] - ord);
//       }
//     }
//   return ans;
// }

// double LagrangeP(double* p, vtkIdType order, vtkIdType* ijk, bool verbose = false)
// {
// This version uses Lagrange polynomials in the semi-infinite space.

//   double ans = 1.;

//   double order_d = static_cast<double>(order);
//   double pInf[3];
//   ToInfCoords(p,pInf);
//   double pNode[3] = {ijk[0]/order_d,ijk[1]/order_d,ijk[2]/order_d};
//   double pNodeInf[3];
//   ToInfCoords(pNode,pNodeInf);

//   if (ijk[2] == order)
//     {
//       return pow(pInf[2]/(1. - pInf[2]),order);
//     }

//   double prefac = 1./(1. - pInf[2]);
//   for (vtkIdType ord=0;ord<=order;ord++)
//     {
//       if (verbose)
//       {
//         std::cout<<prefac<<" "<<ord<<" "<<pNodeInf[0]<<" "<<pNodeInf[1]<<" "<<pNodeInf[2]<<std::endl;
//       }
//     if (ord != ijk[0])
//       {
//       if (verbose)
//       {
//         std::cout<<ord<<" "<<ijk[0]<<": multiplying by ("<<pInf[0]<<" - "<<prefac*ord<<")/("<<pNodeInf[0]<<" - "<<prefac*ord<<")"<<std::endl;
//       }
//       ans *= (pInf[0] - prefac*ord)/(pNodeInf[0] - prefac*ord);
//       }
//     if (ord != ijk[1])
//       {
//       if (verbose)
//       {
//         std::cout<<ord<<" "<<ijk[1]<<": multiplying by ("<<pInf[1]<<" - "<<prefac*ord<<")/("<<pNodeInf[1]<<" - "<<prefac*ord<<")"<<std::endl;
//       }
//       ans *= (pInf[1] - prefac*ord)/(pNodeInf[1] - prefac*ord);
//       }
//     if (ord != ijk[2] && ord != order)
//       {
//       if (verbose)
//       {
//         std::cout<<ord<<" "<<ijk[2]<<": multiplying by ("<<pInf[2]<<" - "<<prefac*ord<<")/("<<pNodeInf[2]<<" - "<<prefac*ord<<")"<<std::endl;
//       }
//       ans *= (pInf[2] - prefac*ord)/(pNodeInf[2] - prefac*ord);
//       }
//     }

//   ans *= pow(prefac,order);

//   return ans;
// }

double LagrangeP(double* p, vtkIdType order, vtkIdType* ijk, bool verbose = false)
{
  double ans = 1.;

  double order_d = static_cast<double>(order);
  double pInf[3];
  ToInfCoords(p,pInf);
  double pNode[3] = {ijk[0]/order_d,ijk[1]/order_d,ijk[2]/order_d};
  double pNodeInf[3];
  ToInfCoords(pNode,pNodeInf);

  if (ijk[2] == order)
    {
      return pow(pInf[2]/(1. - pInf[2]),order);
    }

  double prefac = 1./(1. - pInf[2]);
  for (vtkIdType ord=0;ord<=order;ord++)
    {
      if (verbose)
      {
        std::cout<<prefac<<" "<<ord<<" "<<pNodeInf[0]<<" "<<pNodeInf[1]<<" "<<pNodeInf[2]<<std::endl;
      }
    if (ord != ijk[0])
      {
      if (verbose)
      {
        std::cout<<ord<<" "<<ijk[0]<<": multiplying by ("<<pInf[0]<<" - "<<prefac*ord<<")/("<<pNodeInf[0]<<" - "<<prefac*ord<<")"<<std::endl;
      }
      ans *= (pInf[0] - prefac*ord)/(pNodeInf[0] - prefac*ord);
      }
    if (ord != ijk[1])
      {
      if (verbose)
      {
        std::cout<<ord<<" "<<ijk[1]<<": multiplying by ("<<pInf[1]<<" - "<<prefac*ord<<")/("<<pNodeInf[1]<<" - "<<prefac*ord<<")"<<std::endl;
      }
      ans *= (pInf[1] - prefac*ord)/(pNodeInf[1] - prefac*ord);
      }
    if (ord != ijk[2] && ord != order)
      {
      if (verbose)
      {
        std::cout<<ord<<" "<<ijk[2]<<": multiplying by ("<<pInf[2]<<" - "<<prefac*ord<<")/("<<pNodeInf[2]<<" - "<<prefac*ord<<")"<<std::endl;
      }
      ans *= (pInf[2] - prefac*ord)/(pNodeInf[2] - prefac*ord);
      }
    }

  ans *= pow(prefac,order);

  return ans;
}

double LagrangeP(double pcoord, vtkIdType order, vtkIdType i,bool verbose=false)
{
      if (verbose)
  std::cout<<pcoord<<" "<<order<<" "<<i<<std::endl;
  double ans = 1.;
  double x = pcoord * (order >= 1 ? order : 1);
  for (vtkIdType ord=0;ord<=order;ord++)
    {
    if (ord != i)
      {
      if (verbose)
        std::cout<<"multiplying by ("<<x<<" - "<<ord<<")/("<<i<<" - "<<ord<<")"<<std::endl;
      ans *= (x - ord)/(i - ord);
      }
    }
  return ans;
}
}

//----------------------------------------------------------------------------
void vtkLagrangePyramid::InterpolateFunctions(double pcoords[3], double* weights)
{
  vtkIdType n = this->GetOrder();

  vtkIdType nPoints = this->GetPoints()->GetNumberOfPoints();

  std::cout<<"Function evaluation @ ("<<pcoords[0]<<","<<pcoords[1]<<","<<pcoords[2]<<")"<<std::endl;

  double xyz[3] = {pcoords[0], pcoords[1], pcoords[2]};
  // TO DO: apply L'Hopital's rule instead of this mess
  if (fabs(1.-pcoords[2]) < 1.e-14)
    {
    xyz[2] = 1. - 1.e-14;
    }

  // Affine coordinates
  if (n == 1)
    {
    double prefac = 1. / (1. - xyz[2]);
    double mu0 = 1. - xyz[0] - xyz[2];
    double mu1 = xyz[0];
    double nu0 = 1. - xyz[1] - xyz[2];
    double nu1 = xyz[1];
    weights[0] = prefac*mu0*nu0;
    weights[1] = prefac*mu1*nu0;
    weights[2] = prefac*mu1*nu1;
    weights[3] = prefac*mu0*nu1;
    weights[4] = xyz[2];
  std::cout<<"precomputed: ("<<weights[0]<<","<<weights[1]<<","<<weights[2]<<","<<weights[3]<<","<<weights[4]<<")"<<std::endl;
    // return;
    }

  // double sum = 0.;

  for (vtkIdType idx = 0; idx < nPoints; idx++)
    {
    weights[idx] = 1.;

    vtkIdType ijk[3];
    this->ToRationalIndex(idx,ijk);

    // std::cout<<"multiplying by ("<<n<<","<<ijk[0]<<","<<ijk[1]<<","<<ijk[2]<<") : "<<LagrangeP(xyz,n,ijk) /
    //   pow(1. + xyz[2],ijk[0] + ijk[1] + ijk[2])<<std::endl;

    double pInf[3];
    ToInfCoords(xyz, pInf);

    // if (ijk[2] != n)
      {
        bool verbose = true;
      // if (idx >=9 && idx <= 12 && ijk[2] > 0.)
      // else
      // {
      //   verbose = false;
      //   }

        // std::cout<<"Idx "<<idx<<": "<<ijk[0]/static_cast<double>(n)<<" "<<ijk[1]/static_cast<double>(n)<<" "<<ijk[2]/static_cast<double>(n)<<std::endl;

      // weights[idx] = (LagrangeP(pInf[0], n - ijk[2], ijk[0]) *
      //                 LagrangeP(pInf[1], n - ijk[2], ijk[1]) *
      //                 LagrangeP(pInf[2], n - 1, ijk[2],verbose)) / pow(1. + pInf[2], n);

      // weights[idx] = (LagrangeP(pInf[0], n, ijk[0],verbose) *
      //                 LagrangeP(pInf[1], n, ijk[1],verbose) *
      //                 LagrangeP(pInf[2], n - 1, ijk[2],verbose)) / pow(1. + pInf[2], n);
      // sum += weights[idx];
      weights[idx] = LagrangeP(xyz, n, ijk,verbose);
      std::cout<<"Idx: "<<idx<<" "<<weights[idx]<<std::endl;
      }

    // else
    // if (ijk[2] == n)
    //   {
    //   weights[idx] *= pow(pInf[2]/(1. + pInf[2]), n);
    //   }

    // weights[idx] *= pow(pInf[2]/(1. + pInf[2]), n);

   // vtkIdType max = std::max(ijk[0],ijk[1]);

    // weights[idx] += (JacobiP(0, 0, ijk[0], x/(1.-z)) *
    //                  JacobiP(0, 0, ijk[1], y/(1.-z)) *
    //                  pow(1.-z,max) *
    //                  JacobiP(2*max + 2, 0, ijk[2], 2.*z - 1.));


    }
  // for (unsigned i=0;i<nPoints;i++)
  //   weights[i] /= sum;

  std::cout<<"Lagrange: ("<<weights[0];
  for (unsigned i=1;i<nPoints;i++)
    std::cout<<","<<weights[i];
  std::cout<<")"<<std::endl;
  std::cout<<std::endl;
}
#endif

//----------------------------------------------------------------------------
void vtkLagrangePyramid::InterpolateFunctions(double pcoords[3], double* weights)
{
  std::cout<<"("<<pcoords[0]<<","<<pcoords[1]<<","<<pcoords[2]<<")"<<std::endl;

  vtkIdType k = this->GetOrder();

  double xi =   pcoords[0];
  double eta =  pcoords[1];
  double zeta = pcoords[2];
  if (fabs(1. - zeta) < 1.e-14)
    {
    zeta -= 1.e-14;
    }

  double alpha_xi  = (1. - xi - zeta);
  double alpha_eta = (1. - eta - zeta);
  double norm = pow(1. - zeta, -(2 - k));

  std::cout<<"xi, eta, alpha_xi, alpha_eta: "<<xi<<" "<<eta<<" "<<alpha_xi<<" "<<alpha_eta<<std::endl;

  // vertices
  vtkIdType index = 0;
    {
    weights[index++] = norm*alpha_xi*alpha_eta;
    weights[index++] = norm*xi*alpha_eta;
    weights[index++] = norm*xi*eta;
    weights[index++] = norm*alpha_xi*eta;
    weights[index++] = pow(zeta,k);
    }

  // base edges
  for (vtkIdType edge = 0; edge < 4; edge++)
    {
    double x = (edge == 0 || edge == 3 ? alpha_xi : xi);
    double y = (edge == 0 || edge == 1 ? alpha_eta : eta);
    double x1 = (edge == 0 || edge == 3 ? xi : alpha_xi);
    double norm1 = x1 / (1. - zeta);
    double prefac = norm1*norm;
    for (vtkIdType idx = 0; idx < (k - 1); idx++)
      {
      std::cout<<prefac<<" * "<<x<<" * "<<y<<" * "<<x1<<std::endl;
      weights[index++] = prefac*x*y*x1;
      x1 *= xi;
      prefac *= norm1;
      }
    }

  vtkIdType nPoints = this->GetPoints()->GetNumberOfPoints();
  std::cout<<"  ("<<weights[0];
  for (vtkIdType i=1;i<nPoints;i++)
    std::cout<<","<<weights[i];
  std::cout<<")"<<std::endl;
}

//----------------------------------------------------------------------------
void vtkLagrangePyramid::InterpolateDerivs(double pcoords[3], double* derivs)
{

}

//----------------------------------------------------------------------------
vtkIdType vtkLagrangePyramid::ComputeOrder()
{
  // when order = n, # points = (n+1)*(n+2)*(2*n+3)/6

  vtkIdType order = 1;
  vtkIdType nPoints = this->GetPoints()->GetNumberOfPoints();
  vtkIdType nPointsForOrder = 5;

  while (nPointsForOrder < nPoints)
  {
    order++;
    nPointsForOrder = (order+1)*(order+2)*(2*order+3)/6;
  }

  assert (nPoints == nPointsForOrder);
  return order;
}

//----------------------------------------------------------------------------
namespace
{
// TODO: these functions shouldn't have to exist. This should be pulled from
// vtkLagrangeQuad.
void QuadIJK(vtkIdType index, vtkIdType* ijk, vtkIdType order)
{
  assert(order >= 1);

  ijk[2] = 0;

  vtkIdType max = order;
  vtkIdType min = 0;

  // scope into the correct quad
  while (index >= 4*order && index != 0 && order != 1)
    {
    index -= 4*order;
    max--;
    min++;
    order -= 2;
    }

  if (index < 4)
    {
    // we are on a vertex
    for (vtkIdType coord = 0; coord < 2; coord++)
      {
      ijk[coord] = (LinearVertices[index][coord] == 1 ? max : min);
      }
    }
  else if (index - 4 < 4*(order - 1))
    {
    // we are on an edge
    vtkIdType edgeId = (index - 4) / (order - 1);
    vtkIdType vertexId = (index - 4) % (order - 1);

    for (vtkIdType coord = 0; coord < 2; coord++)
      {
      ijk[coord] =
        min + (LinearVertices[EdgeVertices[edgeId][0]][coord] *
               (max - min - 1 - vertexId) +
               LinearVertices[EdgeVertices[edgeId][1]][coord] *
               (1 + vertexId));
      }
    }
}

vtkIdType QuadIndex(const vtkIdType* ijk, vtkIdType order)
{
  vtkIdType index = 0;

  assert(order >= 1);
  assert(ijk[0] >= 0 && ijk[0] <= order && ijk[1] >= 0 && ijk[1] <= order);

  vtkIdType max = order;
  vtkIdType min = 0;

  vtkIdType ijk_min = std::min(ijk[0], ijk[1]);
  vtkIdType ijk_max = std::max(ijk[0], ijk[1]);

  // scope into the correct quad
  while (ijk_min > min && ijk_max < max)
    {
    index += 4*order;
    max--;
    min++;
    order -= 2;
    }

  for (vtkIdType vertex = 0; vertex < 4; vertex++)
    {
    if (ijk[0] == min + LinearVertices[vertex][0]*(max-min) &&
        ijk[1] == min + LinearVertices[vertex][1]*(max-min))
      {
      // we are on a vertex
      return index;
      }
    index++;
    }

  static const vtkIdType EdgeSelection[4][2] = {{1,0},{0,1},{1,1},{0,0}};

  for (vtkIdType edge = 0; edge < 4; edge++)
    {
    if (ijk[EdgeSelection[edge][0]] == min + EdgeSelection[edge][1]*(max-min))
      {
      return index + (edge < 2 ?
                      ijk[(EdgeSelection[edge][0]+1)%2] - (min + 1) :
                      max - 1 - ijk[(EdgeSelection[edge][0]+1)%2]);
      }
    index += max - (min + 1);
    }

  return index;
}

}

//----------------------------------------------------------------------------
void vtkLagrangePyramid::ToRationalIndex(vtkIdType index, vtkIdType* rindex)
{
  vtkIdType order = this->GetOrder();

  assert(order >= 1);

  vtkIdType max = order;
  vtkIdType min = 0;

  // scope into the correct tetra
  while (index >= 3*order*order + 2 && index != 0 && order != 1)
    {
    index -= 3*order*order + 2;
    max -= 2;
    min++;
    order -= 3;
    }

  if (index < 5)
    {
    // we are on a vertex
    for (vtkIdType coord = 0; coord < 3; coord++)
      {
      rindex[coord] = (LinearVertices[index][coord] == 1 ? max : min);
      }
    return;
    }
  else if (index - 5 < 8*(order - 1))
    {
    // we are on an edge
    vtkIdType edgeId = (index - 5) / (order - 1);
    vtkIdType vertexId = (index - 5) % (order - 1);

    for (vtkIdType coord = 0; coord < 3; coord++)
      {
      rindex[coord] =
        min + (LinearVertices[EdgeVertices[edgeId][0]][coord] *
               (max - min - 1 - vertexId) +
               LinearVertices[EdgeVertices[edgeId][1]][coord] *
               (1 + vertexId));
      }
    return;
    }
  else if (index - 5 - 8*(order - 1) < (order-1)*(order-1))
    {
    // we are on the quad face
    vtkIdType vertexId = index - 5 - 8*(order - 1);

    if (order == 2)
      {
      rindex[0] = rindex[1] = rindex[2] = 0;
      }
    else
      {
      QuadIJK(vertexId,rindex,order-2);
      }
    for (vtkIdType coord = 0; coord < 3; coord++)
      {
      rindex[coord] += min + (coord == 2 ? 0 : 1);
      }
      return;
    }
  else
    {
    // we are on one of the triangle faces
    vtkIdType idx = index - 5 - 8*(order - 1) - (order-1)*(order-1);
    vtkIdType faceId = (idx)/((order - 2)*(order - 1)/2);
    vtkIdType vertexId = (idx)%((order - 2)*(order - 1)/2);

    vtkIdType projectedBIndex[3];
    if (order == 3)
      {
      projectedBIndex[0] = projectedBIndex[1] = projectedBIndex[2] = 0;
      }
    else
      {
      vtkLagrangeTriangle::BarycentricIndex(vertexId, projectedBIndex,
                                            order-3);
      }

    rindex[faceId%2] = projectedBIndex[0] + min + 1;
    rindex[(faceId+1)%2] = ((faceId + 1)%4 < 2 ? min
                            : max - projectedBIndex[1] - 1);
    rindex[2] = min + 1 + projectedBIndex[1];
    return;
    }
}

//----------------------------------------------------------------------------
vtkIdType vtkLagrangePyramid::ToIndex(const vtkIdType* rindex)
{
  vtkIdType order = this->GetOrder();

  assert(order >= 1);
  assert(rindex[0] >= 0 && rindex[0] <= order &&
         rindex[1] >= 0 && rindex[1] <= order &&
         rindex[2] >= 0 &&
         rindex[2] + rindex[0] <= order + 1 &&
         rindex[2] + rindex[1] <= order + 1);

  vtkIdType index = 0;
  vtkIdType max = order;
  vtkIdType min = 0;

  vtkIdType r_min = std::min(std::min(std::min(rindex[0], rindex[1]),rindex[2]),
                             std::min(order - (rindex[0] + rindex[2]),
                                      order - (rindex[1] + rindex[2])));

  // scope into the correct pyramid
  while (r_min > min)
    {
    index += 3*order*order + 2;
    max -= 2;
    min++;
    order -= 3;
    }

  for (vtkIdType vertex = 0; vertex < 5; vertex++)
    {
    bool vtx = true;
    for (vtkIdType coord = 0; coord < 3; coord++)
      {
      vtx &= rindex[coord] == (LinearVertices[vertex][coord] == 1 ? max : min);
      }
    if (vtx)
      {
      // we are on a vertex
      return index;
      }
    index++;
    }

  for (vtkIdType edge = 0; edge < 8; edge++)
    {
    vtkIdType extremum = max - (edge < 4 ? 0 : rindex[2] - min);

    if (rindex[EdgeFixedCoords[edge][0][0]] ==
        (EdgeFixedCoords[edge][0][1] == 1 ? extremum : min) &&
        rindex[EdgeFixedCoords[edge][1][0]] ==
        (EdgeFixedCoords[edge][1][1] == 1 ? extremum : min))
      {
      // we are on an edge
      return index + (EdgeCountingCoord[edge][1] == 1 ?
                      rindex[EdgeCountingCoord[edge][0]] - (min + 1) :
                      max - 1 - rindex[EdgeCountingCoord[edge][0]]);
      }
    index += max - (min + 1);
    }

  if (rindex[FaceFixedCoords[0][0]] == FaceFixedCoords[0][1] + min)
    {
    vtkIdType projectedIJK[3] = {rindex[0] - (min + 1),
                                 rindex[1] - (min + 1), 0};
    return index + ( order > 2 ? QuadIndex(projectedIJK, order - 2) : 0);
    }

  index += (order-1)*(order-1);

  for (vtkIdType face = 1; face < 5; face++)
    {
    if (rindex[FaceFixedCoords[face][0]] == (FaceFixedCoords[face][1] == 1 ?
                                             max - (rindex[2] - min) : min))
      {
      // we are on a face
      vtkIdType projectedBIndex[3];
      projectedBIndex[0] = rindex[(face+1)%2] - min;
      projectedBIndex[1] = rindex[2] - min;
      projectedBIndex[2] = order - projectedBIndex[0] - projectedBIndex[1];
      // we must subtract the indices of the face's vertices and edges, which
      // total to 3*order
      return (index + vtkLagrangeTriangle::Index(projectedBIndex, order)
              - 3*order);
      }
    index += (order+1)*(order+2)/2 - 3*order;
    }
  return index;
}

//----------------------------------------------------------------------------
void vtkLagrangePyramid::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
}
