/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangeTetra.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkLagrangeTetra
// .SECTION Description
// .SECTION See Also

#ifndef vtkLagrangianTetra_h
#define vtkLagrangianTetra_h

#include "vtkCommonDataModelModule.h" // For export macro
#include "vtkNonLinearCell.h"

class vtkTetra;
class vtkLagrangianTriangle;
class vtkDoubleArray;

class VTKCOMMONDATAMODEL_EXPORT vtkLagrangianTetra : public vtkNonLinearCell
{
public:
  static vtkLagrangianTetra *New();
  vtkTypeMacro(vtkLagrangianTetra,vtkNonLinearCell);
  void PrintSelf(ostream& os, vtkIndent indent);

  int GetCellType() { return VTK_LAGRANGIAN_TETRAHEDRON; }
  int GetCellDimension() { return 3; }
  int RequiresInitialization() { return 1; }
  int GetNumberOfEdges() { return 6; }
  int GetNumberOfFaces() { return 4; }
  vtkCell *GetEdge(int edgeId);
  vtkCell *GetFace(int faceId);

  void Initialize();

  int CellBoundary(int subId, double pcoords[3], vtkIdList *pts);
  int EvaluatePosition(double x[3], double* closestPoint,
                       int& subId, double pcoords[3],
                       double& dist2, double *weights);
  void EvaluateLocation(int& subId, double pcoords[3], double x[3],
                        double *weights);
  void Contour(double value, vtkDataArray *cellScalars,
               vtkIncrementalPointLocator *locator, vtkCellArray *verts,
               vtkCellArray *lines, vtkCellArray *polys,
               vtkPointData *inPd, vtkPointData *outPd,
               vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd);
  void Clip(double value, vtkDataArray *cellScalars,
            vtkIncrementalPointLocator *locator, vtkCellArray *polys,
            vtkPointData *inPd, vtkPointData *outPd,
            vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd,
            int insideOut);
  int IntersectWithLine(double p1[3], double p2[3], double tol, double& t,
                        double x[3], double pcoords[3], int& subId);
  int Triangulate(int index, vtkIdList *ptIds, vtkPoints *pts);
  void Derivatives(int subId, double pcoords[3], double *values,
                   int dim, double *derivs);
  double* GetParametricCoords();

  int GetParametricCenter(double pcoords[3]);
  double GetParametricDistance(double pcoords[3]);

  void InterpolateFunctions(double pcoords[3], double* weights);
  void InterpolateDerivs(double pcoords[3], double* derivs);

  vtkIdType GetOrder();

protected:
  vtkLagrangianTetra();
  ~vtkLagrangianTetra();

  vtkIdType NumberOfSubtetras();

  // Description:
  // Given the index of the subtriangle, compute the barycentric indices of
  // the subtriangle's vertices.
  void SubtetraBarycentricPointIndices(vtkIdType cellIndex,
                                       vtkIdType (&pointBIndices)[4][4]);
  void TetraFromOctahedron(vtkIdType cellIndex,
                           const vtkIdType (&octBIndices)[6][4],
                           vtkIdType (&tetraBIndices)[4][4]);

  vtkLagrangianTriangle *Face;
  vtkTetra *Tetra;
  vtkDoubleArray *Scalars; //used to avoid New/Delete in contouring/clipping

  double* LegacyParametricCoordinates;

 public:
  void ToBarycentricIndex(vtkIdType index, vtkIdType* bindex);
  vtkIdType ToIndex(const vtkIdType* bindex);

  static void BarycentricIndex(vtkIdType index, vtkIdType* bindex,
                                      vtkIdType order);
  static vtkIdType Index(const vtkIdType* bindex, vtkIdType order);

  void WorldToBarycentric(double* coords, double* bcoords);
  void BarycentricToWorld(const double* bcoords, double* coords);

private:
  vtkLagrangianTetra(const vtkLagrangianTetra&);  // Not implemented.
  void operator=(const vtkLagrangianTetra&);  // Not implemented.
};

#endif
