/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangeQuadrilateral.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkLagrangeQuadrilateral
// .SECTION Description
// .SECTION See Also

#ifndef vtkLagrangeQuadrilateral_h
#define vtkLagrangeQuadrilateral_h

#include "vtkCommonDataModelModule.h" // For export macro
#include "vtkNonLinearCell.h"
#include "vtkCellType.h"
#include "vtkSmartPointer.h"
#include "vtkNew.h"

#include <vector>

class vtkQuad;
class vtkIdList;
class vtkPoints;
class vtkDoubleArray;
class vtkVector3d;
class vtkVector3i;

class VTKCOMMONDATAMODEL_EXPORT vtkLagrangeQuadrilateral : public vtkNonLinearCell
{
public:
  static vtkLagrangeQuadrilateral* New();
  vtkTypeMacro(vtkLagrangeQuadrilateral,vtkNonLinearCell);
  void PrintSelf(ostream& os, vtkIndent indent);

  int GetCellType() { return VTK_LAGRANGE_QUADRILATERAL; }
  int GetCellDimension() { return 2; }
  int RequiresInitialization() { return 1; }
  int GetNumberOfEdges() { return 4; }
  int GetNumberOfFaces() { return 0; }
  vtkCell* GetEdge(int edgeId);
  vtkCell* GetFace(int faceId) { return NULL; }

  void Initialize();

  int CellBoundary(int subId, double pcoords[3], vtkIdList* pts);
  int EvaluatePosition(
    double x[3], double* closestPoint,
    int& subId, double pcoords[3],
    double& dist2, double* weights);
  void EvaluateLocation(
    int& subId, double pcoords[3], double x[3],
    double* weights);
  void Contour(
    double value, vtkDataArray* cellScalars,
    vtkIncrementalPointLocator* locator, vtkCellArray* verts,
    vtkCellArray* lines, vtkCellArray* polys,
    vtkPointData* inPd, vtkPointData* outPd,
    vtkCellData* inCd, vtkIdType cellId, vtkCellData* outCd);
  void Clip(
    double value, vtkDataArray* cellScalars,
    vtkIncrementalPointLocator* locator, vtkCellArray* polys,
    vtkPointData* inPd, vtkPointData* outPd,
    vtkCellData* inCd, vtkIdType cellId, vtkCellData* outCd,
    int insideOut);
  int IntersectWithLine(
    double p1[3], double p2[3], double tol, double& t,
    double x[3], double pcoords[3], int& subId);
  int Triangulate(int index, vtkIdList* ptIds, vtkPoints* pts);
  void Derivatives(
    int subId, double pcoords[3], double* values,
    int dim, double* derivs);
  double* GetParametricCoords();

  double GetParametricDistance(double pcoords[3]);

  const int* GetOrder();
  int GetOrder(int i) { return this->GetOrder()[i]; }

  void InterpolateFunctions(double pcoords[3], double* weights);
  void InterpolateDerivs(double pcoords[3], double* derivs);

  bool SubCellCoordinatesFromId(vtkVector3i& ijk, int subId);
  bool SubCellCoordinatesFromId(int& i, int& j, int& k, int subId);
  int PointIndexFromIJK(int i, int j, int k);
  bool TransformApproxToCellParams(int subCell, double* pcoords);

protected:
  vtkLagrangeQuadrilateral();
  ~vtkLagrangeQuadrilateral();

  vtkQuad* GetApprox();
  vtkQuad* GetApproximateQuad(
    int subId, vtkDataArray* scalarsIn = NULL, vtkDataArray* scalarsOut = NULL);

  int Order[3];
  vtkSmartPointer<vtkPoints> PointParametricCoordinates;
  vtkSmartPointer<vtkQuad> Approx;
  vtkNew<vtkDoubleArray> Scalars;
  vtkNew<vtkPoints> TmpPts;
  vtkNew<vtkIdList> TmpIds;

private:
  vtkLagrangeQuadrilateral(const vtkLagrangeQuadrilateral&);  // Not implemented.
  void operator=(const vtkLagrangeQuadrilateral&);  // Not implemented.
};

#endif // vtkLagrangeQuadrilateral_h
