#ifndef __vtkLagrangeInterpolation_h
#define __vtkLagrangeInterpolation_h

#include "vtkObject.h"
#include "vtkSmartPointer.h" // For API.
#include "vtkCommonDataModelModule.h" // For export macro.

#include <vector>

class vtkPoints;
class vtkVector2i;
class vtkVector3d;

class VTKCOMMONDATAMODEL_EXPORT vtkLagrangeInterpolation : public vtkObject
{
public:
  static vtkLagrangeInterpolation* New();
  virtual void PrintSelf(ostream& os, vtkIndent indent);
  vtkTypeMacro(vtkLagrangeInterpolation,vtkObject);

  enum Constants {
    MaxDegree = 10 // The maximum degree that VTK will support.
  };

  template<int N, typename T>
  void EvaluateField(
    double* fieldOut,
    double pcoord[N],
    const int order[N], const vtkIdType* conn, T* fieldIn);

  template<int N, typename T>
  void EvaluateFieldAndDerivatives(
    double* fieldOut, double* derivsOut,
    double pcoord[N],
    const int order[N], const vtkIdType* conn, T* fieldIn);

  static void EvaluateShapeFunctions(int order, double pcoord, double* shape);
  static void EvaluateShapeAndGradient(int order, double pcoord, double* shape, double* grad);

  static void Tensor2ShapeFunctions(const int order[2], const double* pcoords, double* shape);
  static void Tensor2ShapeDerivatives(const int order[2], const double* pcoords, double* derivs);

  static void Tensor3ShapeFunctions(const int order[3], const double* pcoords, double* shape);
  static void Tensor3ShapeDerivatives(const int order[3], const double* pcoords, double* derivs);

  template<int N>
  static void TensorShapeFunctions(const int order[N], const double* pcoords, double* derivs);
  template<int N>
  static void TensorShapeDerivatives(const int order[N], const double* pcoords, double* derivs);

  static vtkVector3d GetParametricHexCoordinates(int vertexId);
  static vtkVector2i GetPointIndicesBoundingHexEdge(int edgeId);
  static int GetVaryingParameterOfHexEdge(int edgeId);
  static const int* GetPointIndicesBoundingHexFace(int faceId);
  static const int* GetEdgeIndicesBoundingHexFace(int faceId);
  static vtkVector2i GetVaryingParametersOfHexFace(int faceId);

  static void AppendCurveCollocationPoints(vtkSmartPointer<vtkPoints>& pts, const int order[1]);
  static void AppendQuadrilateralCollocationPoints(vtkSmartPointer<vtkPoints>& pts, const int order[2]);
  static void AppendHexahedronCollocationPoints(vtkSmartPointer<vtkPoints>& pts, const int order[3]);

  template<int N>
  static int NumberOfIntervals(const int order[N]);
protected:
  vtkLagrangeInterpolation();
  virtual ~vtkLagrangeInterpolation();

  void PrepareForOrder(const int o[3]);

  std::vector<double> ShapeSpace;
  std::vector<double> DerivSpace;

private:
  vtkLagrangeInterpolation(const vtkLagrangeInterpolation&); // Not implemented.
  void operator = (const vtkLagrangeInterpolation&); // Not implmented.
};

template<int N, typename T>
void vtkLagrangeInterpolation::EvaluateField(
  double* fieldOut,
  double pcoord[N],
  const int order[N], const vtkIdType* conn, T* fieldIn)
{
  vtkLagrangeInterpolation::TensorShapeFunctions<N>(order, pcoord, &this->ShapeSpace[0]);
}

template<int N, typename T>
void vtkLagrangeInterpolation::EvaluateFieldAndDerivatives(
  double* fieldOut, double* derivsOut,
  double pcoord[N],
  const int order[N], const vtkIdType* conn, T* fieldIn)
{
  vtkLagrangeInterpolation::TensorShapeDerivatives<N>(order, pcoord, &this->ShapeSpace[0]);
}

template<int N>
int vtkLagrangeInterpolation::NumberOfIntervals(const int order[N])
{
  int ni = 1;
  for (int n = 0; n < N; ++n)
    {
    ni *= order[n];
    }
  return ni;
}

#endif // __vtkLagrangeInterpolation_h
