/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangeTriangle.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkLagrangeTriangle
// .SECTION Description
// .SECTION See Also

#ifndef vtkLagrangeTriangle_h
#define vtkLagrangeTriangle_h

#include "vtkCommonDataModelModule.h" // For export macro
#include "vtkNonLinearCell.h"

class vtkDoubleArray;
class vtkLagrangeCurve;
class vtkTriangle;

class VTKCOMMONDATAMODEL_EXPORT vtkLagrangeTriangle : public vtkNonLinearCell
{
public:
  static vtkLagrangeTriangle *New();
  vtkTypeMacro(vtkLagrangeTriangle,vtkNonLinearCell);
  void PrintSelf(ostream& os, vtkIndent indent);

  int GetCellType() { return VTK_LAGRANGE_TRIANGLE; }
  int GetCellDimension() { return 2; }
  int RequiresInitialization() { return 1; }
  int GetNumberOfEdges() { return 3; }
  int GetNumberOfFaces() { return 0; }
  vtkCell *GetEdge(int edgeId);
  vtkCell *GetFace(int) { return 0; }

  void Initialize();

  int CellBoundary(int subId, double pcoords[3], vtkIdList *pts);
  int EvaluatePosition(double x[3], double* closestPoint,
                       int& subId, double pcoords[3],
                       double& dist2, double *weights);
  void EvaluateLocation(int& subId, double pcoords[3], double x[3],
                        double *weights);
  void Contour(double value, vtkDataArray *cellScalars,
               vtkIncrementalPointLocator *locator, vtkCellArray *verts,
               vtkCellArray *lines, vtkCellArray *polys,
               vtkPointData *inPd, vtkPointData *outPd,
               vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd);
  void Clip(double value, vtkDataArray *cellScalars,
            vtkIncrementalPointLocator *locator, vtkCellArray *polys,
            vtkPointData *inPd, vtkPointData *outPd,
            vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd,
            int insideOut);
  int IntersectWithLine(double p1[3], double p2[3], double tol, double& t,
                        double x[3], double pcoords[3], int& subId);
  int Triangulate(int index, vtkIdList *ptIds, vtkPoints *pts);
  void Derivatives(int subId, double pcoords[3], double *values,
                   int dim, double *derivs);
  double* GetParametricCoords();
  static void ComputeParametricCoords(double*,vtkIdType);

  int GetParametricCenter(double pcoords[3]);
  double GetParametricDistance(double pcoords[3]);

  void InterpolateFunctions(double pcoords[3], double* weights);
  void InterpolateDerivs(double pcoords[3], double* derivs);

  vtkIdType GetOrder() const { return this->Order; }
  vtkIdType ComputeOrder();

protected:
  vtkLagrangeTriangle();
  ~vtkLagrangeTriangle();

  vtkIdType GetNumberOfSubtriangles() const {return this->NumberOfSubtriangles;}
  vtkIdType ComputeNumberOfSubtriangles();

  // Description:
  // Given the index of the subtriangle, compute the barycentric indices of
  // the subtriangle's vertices.
  void SubtriangleBarycentricPointIndices(vtkIdType cellIndex,
                                          vtkIdType (&pointBIndices)[3][3]);

  vtkLagrangeCurve *Edge;
  vtkTriangle *Face;
  vtkDoubleArray *Scalars; //used to avoid New/Delete in contouring/clipping
  vtkIdType Order;
  vtkIdType NumberOfSubtriangles;
  vtkIdType* BarycentricIndexMap;
  vtkIdType* IndexMap;
  double* ParametricCoordinates;
  vtkIdType* SubtriangleIndexMap;

 public:
  void ToBarycentricIndex(vtkIdType index, vtkIdType* bindex);
  vtkIdType ToIndex(const vtkIdType* bindex);

  static void BarycentricIndex(vtkIdType index, vtkIdType* bindex,
                               vtkIdType order);
  static vtkIdType Index(const vtkIdType* bindex, vtkIdType order);

  void WorldToBarycentric(double* coords, double* bcoords);
  void BarycentricToWorld(const double* bcoords, double* coords);

  static double eta(vtkIdType n, vtkIdType chi, double sigma);
  static double d_eta(vtkIdType n, vtkIdType chi,double sigma,
                      vtkIdType x=0,double answer=0.);

private:
  vtkLagrangeTriangle(const vtkLagrangeTriangle&);  // Not implemented.
  void operator=(const vtkLagrangeTriangle&);  // Not implemented.
};

#endif
