/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangianTriangle.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkLagrangianTriangle.h"

#include "vtkObjectFactory.h"
#include "vtkMath.h"
#include "vtkLine.h"
#include "vtkTriangle.h"
#include "vtkDoubleArray.h"
#include "vtkPoints.h"

vtkStandardNewMacro(vtkLagrangianTriangle);
//----------------------------------------------------------------------------
vtkLagrangianTriangle::vtkLagrangianTriangle() :
  LegacyParametricCoordinates(NULL)
{
  this->Face = vtkTriangle::New();
  this->Scalars = vtkDoubleArray::New();
  this->Scalars->SetNumberOfTuples(3);

  this->Points->SetNumberOfPoints(3);
  this->PointIds->SetNumberOfIds(3);
  for (int i = 0; i < 3; i++)
    {
    this->Points->SetPoint(i, 0.0, 0.0, 0.0);
    this->PointIds->SetId(i,0);
    }
}

//----------------------------------------------------------------------------
vtkLagrangianTriangle::~vtkLagrangianTriangle()
{
  delete this->LegacyParametricCoordinates;
  this->Face->Delete();
  this->Scalars->Delete();
}

//----------------------------------------------------------------------------
vtkCell *vtkLagrangianTriangle::GetEdge(int edgeId)
{
  // TODO: Fill me in!
  return 0;
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::Initialize()
{

}

//----------------------------------------------------------------------------
vtkIdType vtkLagrangianTriangle::NumberOfSubtriangles()
{
  vtkIdType order = this->GetOrder();
  return order*order;
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::SubtriangleBarycentricPointIndices(
  vtkIdType cellIndex, vtkIdType (&pointBIndices)[3][3])
{
  assert(cellIndex < this->NumberOfSubtriangles());

  vtkIdType order = this->GetOrder();
  vtkIdType nRightSideUp = order*(order+1)/2;

  if (order == 1)
    {
    pointBIndices[0][0] = 0;
    pointBIndices[0][1] = 0;
    pointBIndices[0][2] = 1;
    pointBIndices[1][0] = 1;
    pointBIndices[1][1] = 0;
    pointBIndices[1][2] = 0;
    pointBIndices[2][0] = 0;
    pointBIndices[2][1] = 1;
    pointBIndices[2][2] = 0;
    return;
    }

  if (cellIndex < nRightSideUp)
    {
    // there are nRightSideUp subtriangles whose orientation is the same as the
    // parent triangle. We traverse them here.
    vtkLagrangianTriangle::BarycentricIndex(cellIndex,pointBIndices[0],order-1);
    pointBIndices[0][2] += 1;
    pointBIndices[1][0] = pointBIndices[0][0] + 1;
    pointBIndices[1][1] = pointBIndices[0][1];
    pointBIndices[1][2] = pointBIndices[0][2] - 1;
    pointBIndices[2][0] = pointBIndices[0][0];
    pointBIndices[2][1] = pointBIndices[0][1] + 1;
    pointBIndices[2][2] = pointBIndices[0][2] - 1;
    }
  else
    {
    // the remaining subtriangles are inverted with respect to the parent
    // triangle. We traverse them here.
    if (order == 2)
      {
      pointBIndices[0][0] = 1;
      pointBIndices[0][1] = 1;
      pointBIndices[0][2] = 0;
      pointBIndices[1][0] = 0;
      pointBIndices[1][1] = 1;
      pointBIndices[1][2] = 1;
      pointBIndices[2][0] = 1;
      pointBIndices[2][1] = 0;
      pointBIndices[2][2] = 1;
      }
    else
      {
      vtkLagrangianTriangle::BarycentricIndex(cellIndex - nRightSideUp,
                                              pointBIndices[1],order-2);
      pointBIndices[1][1] += 1;
      pointBIndices[1][2] += 1;

      pointBIndices[2][0] = pointBIndices[1][0] + 1;
      pointBIndices[2][1] = pointBIndices[1][1] - 1;
      pointBIndices[2][2] = pointBIndices[1][2];
      pointBIndices[0][0] = pointBIndices[1][0] + 1;
      pointBIndices[0][1] = pointBIndices[1][1];
      pointBIndices[0][2] = pointBIndices[1][2] - 1;
      }
    }
}

//----------------------------------------------------------------------------
int vtkLagrangianTriangle::CellBoundary(int subId, double pcoords[3],
                                       vtkIdList *pts)
{
  // TODO: Fill me in!
  return 0;
}

//----------------------------------------------------------------------------
int vtkLagrangianTriangle::EvaluatePosition(double* x, double* closestPoint,
                                           int& subId, double pcoords[3],
                                           double& minDist2, double *weights)
{
  double pc[3], dist2, tempWeights[3], closest[3];
  int returnStatus=0, status, ignoreId;
  vtkIdType minBIndices[3][3], bindices[3][3], minIndices[3], pointIndices[3];

  vtkIdType order = this->GetOrder();
  vtkIdType numberOfSubtriangles = this->NumberOfSubtriangles();

  minDist2 = VTK_DOUBLE_MAX;
  for (vtkIdType subCellId = 0; subCellId < numberOfSubtriangles; subCellId++)
    {
    this->SubtriangleBarycentricPointIndices(subCellId, bindices);

    for (vtkIdType i=0;i<3;i++)
      {
      pointIndices[i] = this->ToIndex(bindices[i]);
      this->Face->Points->SetPoint(i, this->Points->GetPoint(pointIndices[i]));
      }

    status = this->Face->EvaluatePosition(x, closest, ignoreId, pc, dist2,
                                          tempWeights);

    if ( status != -1 && dist2 < minDist2 )
      {
      returnStatus = status;
      minDist2 = dist2;
      subId = subCellId;
      pcoords[0] = pc[0];
      pcoords[1] = pc[1];
      for (vtkIdType i=0;i<3;i++)
        {
        minIndices[i] = pointIndices[i];
        for (vtkIdType j=0;j<3;j++)
          {
          minBIndices[i][j] = bindices[i][j];
          }
        }
      }
    }

  // adjust parametric coordinates
  if ( returnStatus != -1 )
    {
      pcoords[0] = ((1. - pcoords[0])*minBIndices[0][0] +
                    pcoords[0]*minBIndices[1][0])/order;
      pcoords[1] = ((1. - pcoords[1])*minBIndices[0][1] +
                    pcoords[1]*minBIndices[1][1])/order;
      pcoords[2] = 0.0;

    if(closestPoint!=0)
      {
      // Compute both closestPoint and weights
      this->EvaluateLocation(subId,pcoords,closestPoint,weights);
      }
    else
      {
      // Compute weights only
      this->InterpolateFunctions(pcoords,weights);
      }
    }

  return returnStatus;
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::EvaluateLocation(int& vtkNotUsed(subId),
                                             double pcoords[3],
                                             double x[3], double *weights)
{
  x[0] = x[1] = x[2] = 0.;

  this->InterpolateFunctions(pcoords,weights);

  double p[3];
  vtkIdType nPoints = this->GetPoints()->GetNumberOfPoints();
  for (vtkIdType idx = 0; idx < nPoints; idx++)
    {
    this->Points->GetPoint(idx,p);
    for (vtkIdType jdx = 0; jdx < 3; jdx++)
      {
      x[jdx] += p[jdx]*weights[idx];
      }
    }
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::Contour(double value,
                                   vtkDataArray* cellScalars,
                                   vtkIncrementalPointLocator* locator,
                                   vtkCellArray *verts,
                                   vtkCellArray* lines,
                                   vtkCellArray* polys,
                                   vtkPointData* inPd,
                                   vtkPointData* outPd,
                                   vtkCellData* inCd,
                                   vtkIdType cellId,
                                   vtkCellData* outCd)
{
  vtkIdType bindices[3][3];
  vtkIdType numberOfSubtriangles = this->NumberOfSubtriangles();

  for (vtkIdType subCellId = 0; subCellId < numberOfSubtriangles; subCellId++)
    {
    this->SubtriangleBarycentricPointIndices(subCellId, bindices);

    for (vtkIdType i=0;i<3;i++)
      {
      vtkIdType pointIndex = this->ToIndex(bindices[i]);
      this->Face->Points->SetPoint(i, this->Points->GetPoint(pointIndex));
      if ( outPd )
        {
        this->Face->PointIds->SetId(i, this->PointIds->GetId(pointIndex));
        }
      this->Scalars->SetTuple(i, cellScalars->GetTuple(pointIndex));
      }

    this->Face->Contour(value, this->Scalars, locator, verts,
                        lines, polys, inPd, outPd, inCd, cellId, outCd);

    }
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::Clip(double value,
                                vtkDataArray* cellScalars,
                                vtkIncrementalPointLocator* locator,
                                vtkCellArray* polys,
                                vtkPointData* inPd,
                                vtkPointData* outPd,
                                vtkCellData* inCd,
                                vtkIdType cellId,
                                vtkCellData* outCd,
                                int insideOut)
{
  vtkIdType bindices[3][3];
  vtkIdType numberOfSubtriangles = this->NumberOfSubtriangles();

  for (vtkIdType subCellId = 0; subCellId < numberOfSubtriangles; subCellId++)
    {
    this->SubtriangleBarycentricPointIndices(subCellId, bindices);

    for (vtkIdType i=0;i<3;i++)
      {
      vtkIdType pointIndex = this->ToIndex(bindices[i]);
      this->Face->Points->SetPoint(i, this->Points->GetPoint(pointIndex));
      if ( outPd )
        {
        this->Face->PointIds->SetId(i, this->PointIds->GetId(pointIndex));
        }
      this->Scalars->SetTuple(i, cellScalars->GetTuple(pointIndex));
      }

    this->Face->Clip(value, this->Scalars, locator, polys, inPd, outPd,
                     inCd, cellId, outCd, insideOut);
    }
}

//----------------------------------------------------------------------------
int vtkLagrangianTriangle::IntersectWithLine(double* p1,
                                            double* p2,
                                            double tol,
                                            double& t,
                                            double* x,
                                            double* pcoords,
                                            int& subId)
{
  vtkIdType bindices[3][3];
  vtkIdType numberOfSubtriangles = this->NumberOfSubtriangles();
  int subTest;

  t = VTK_DOUBLE_MAX;
  double tTmp;
  double xMin[3], pcoordsMin[3];

  for (vtkIdType subCellId = 0; subCellId < numberOfSubtriangles; subCellId++)
    {
    this->SubtriangleBarycentricPointIndices(subCellId, bindices);

    for (vtkIdType i=0;i<3;i++)
      {
      vtkIdType pointIndex = this->ToIndex(bindices[i]);
      this->Face->Points->SetPoint(i, this->Points->GetPoint(pointIndex));
      }

    if (this->Face->IntersectWithLine(p1, p2, tol, tTmp, xMin, pcoordsMin,
                                      subTest) && tTmp < t)
      {
      for (vtkIdType i=0;i<3;i++)
        {
        x[i] = xMin[i];
        pcoords[i] = pcoordsMin[i];
        }
      t = tTmp;
      }
    }
  return (t == VTK_DOUBLE_MAX ? 0 : 1);
}

//----------------------------------------------------------------------------
int vtkLagrangianTriangle::Triangulate(int vtkNotUsed(index), vtkIdList *ptIds,
                                      vtkPoints *pts)
{
  pts->Reset();
  ptIds->Reset();

  vtkIdType bindices[3][3];
  vtkIdType numberOfSubtriangles = this->NumberOfSubtriangles();

  for (vtkIdType subCellId = 0; subCellId < numberOfSubtriangles; subCellId++)
    {
    this->SubtriangleBarycentricPointIndices(subCellId, bindices);

    for (vtkIdType i=0;i<3;i++)
      {
      vtkIdType pointIndex = this->ToIndex(bindices[i]);
      ptIds->InsertId(3*subCellId+i,this->PointIds->GetId(pointIndex));
      pts->InsertPoint(3*subCellId+i,this->Points->GetPoint(pointIndex));
      }
    }

  return 1;
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::Derivatives(int vtkNotUsed(subId),
                                       double vtkNotUsed(pcoords)[3],
                                       double *vtkNotUsed(values),
                                       int vtkNotUsed(dim),
                                       double *vtkNotUsed(derivs))
{
  // TODO: Fill me in!
  return;
}

//----------------------------------------------------------------------------
double* vtkLagrangianTriangle::GetParametricCoords()
{
  if (!this->LegacyParametricCoordinates)
    {
    vtkIdType order = this->GetOrder();

    vtkIdType nPoints = (order + 1)*(order + 2)/2;
    this->LegacyParametricCoordinates = new double[3*nPoints];
    ComputeParametricCoords(this->LegacyParametricCoordinates,order);
    }
  return this->LegacyParametricCoordinates;
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::ComputeParametricCoords(double* coords,
                                                    vtkIdType order)
{
  double max = static_cast<double>(order);
  double min = 0.;
  vtkIdType pIdx = 0;
  double p[3];
  vtkIdType ord;
  for (ord = order; ord > 0; ord-=3)
    {
    // add the vertex points
    for (vtkIdType dim=0;dim<3;dim++)
      {
      coords[pIdx + dim]       = min/order;
      coords[pIdx + (dim+1)%3] = min/order;
      coords[pIdx + (dim+2)%3] = max/order;
      pIdx += 3;
      }

    // add the edge points
    if (ord > 1)
      {
      for (vtkIdType dim=0;dim<3;dim++)
        {
        p[dim] = p[(dim+1)%3] = min/order;
        p[(dim+2)%3] = max/order;
        for (vtkIdType i=0;i<ord-1;i++)
          {
          p[dim] += 1./order;
          p[(dim+2)%3] -= 1./order;
          for (vtkIdType j=0;j<3;j++)
            {
            coords[pIdx + j] = p[j];
            }
          pIdx += 3;
          }
        }
      }
    max -= 2.;
    min += 1.;
    }

  if (ord == 0)
    {
    coords[pIdx]     = min/order;
    coords[pIdx + 1] = min/order;
    coords[pIdx + 2] = min/order;
    pIdx += 3;
    }

  // For parametric coordinates, we project the barycentric coordinates
  // onto the z=0 plane
  for (vtkIdType i=2;i<pIdx;i+=3)
    {
    coords[i] = 0.;
    }
}

//----------------------------------------------------------------------------
int vtkLagrangianTriangle::GetParametricCenter(double pcoords[3])
{
  pcoords[0] = pcoords[1] = 1./3.;
  pcoords[2] = 0.;
  return 0;
}

//----------------------------------------------------------------------------
double vtkLagrangianTriangle::GetParametricDistance(double pcoords[3])
{
  int i;
  double pDist, pDistMax=0.0;
  double pc[3];

  pc[0] = pcoords[0];
  pc[1] = pcoords[1];
  pc[2] = 1.0 - pcoords[0] - pcoords[1];

  for (i=0; i<3; i++)
    {
    if ( pc[i] < 0.0 )
      {
      pDist = -pc[i];
      }
    else if ( pc[i] > 1.0 )
      {
      pDist = pc[i] - 1.0;
      }
    else //inside the cell in the parametric direction
      {
      pDist = 0.0;
      }
    if ( pDist > pDistMax )
      {
      pDistMax = pDist;
      }
    }

  return pDistMax;
}

//----------------------------------------------------------------------------
double vtkLagrangianTriangle::eta(vtkIdType n, vtkIdType chi, double sigma)
{
  double result = 1.;
  for (vtkIdType i=1;i<=chi;i++)
    {
    result *= (n*sigma - i + 1.)/i;
    }
  return result;
}

//----------------------------------------------------------------------------
double vtkLagrangianTriangle::d_eta(vtkIdType n, vtkIdType chi, double sigma)
{
  if (chi == 0)
    {
    return 0.;
    }
  else
    {
    double chi_d = static_cast<double>(chi);
    return (n/chi_d*eta(n,chi-1,sigma) +
            (n*sigma-chi_d+1.)/chi_d*d_eta(n,chi-1,sigma));
    }
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::InterpolateFunctions(double pcoords[3],
                                                 double* weights)
{
  // Adapted from P. Silvester, "High-Order Polynomial Triangular Finite
  // Elements for Potential Problems". Int. J. Engng Sci. Vol. 7, pp. 849-861.
  // Pergamon Press, 1969.

  double tau[3] = {pcoords[0], pcoords[1], 1. - pcoords[0] - pcoords[1]};

  vtkIdType n = this->GetOrder();

  for (vtkIdType idx=0; idx<this->GetPoints()->GetNumberOfPoints(); idx++)
    {
    weights[idx] = 1.;
    vtkIdType lambda[3];
    this->ToBarycentricIndex(idx,lambda);

    for (vtkIdType dim=0;dim<3;dim++)
      {
        weights[idx] *= eta(n, lambda[dim], tau[dim]);
      }
    }
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::InterpolateDerivs(double pcoords[3],
                                              double* derivs)
{
  double tau[3] = {pcoords[0], pcoords[1], 1. - pcoords[0] - pcoords[1]};

  vtkIdType n = this->GetOrder();

  vtkIdType nPoints = this->GetPoints()->GetNumberOfPoints();

  for (vtkIdType idx = 0; idx < nPoints; idx++)
    {
    vtkIdType lambda[3];
    this->ToBarycentricIndex(idx,lambda);

    double eta_alpha = eta(n,lambda[0],tau[0]);
    double eta_beta = eta(n,lambda[1],tau[1]);
    double eta_gamma = eta(n,lambda[2],tau[2]);

    double d_eta_alpha = d_eta(n,lambda[0],tau[0]);
    double d_eta_beta = d_eta(n,lambda[1],tau[1]);
    double d_eta_gamma = d_eta(n,lambda[2],tau[2]);

    double d_f_d_tau1 = (d_eta_alpha*eta_beta*eta_gamma -
                         eta_alpha*eta_beta*d_eta_gamma);
    double d_f_d_tau2 = (eta_alpha*d_eta_beta*eta_gamma -
                         eta_alpha*eta_beta*d_eta_gamma);
    // double d_f_d_tau3 = (eta_alpha*eta_beta*d_eta_gamma -
    //                      d_eta_alpha*eta_beta*eta_gamma -
    //                      eta_alpha*d_eta_beta*eta_gamma);

    derivs[idx] = d_f_d_tau1;
    derivs[nPoints + idx] = d_f_d_tau2;
    }
}

//----------------------------------------------------------------------------
vtkIdType vtkLagrangianTriangle::GetOrder()
{
  // when order = n, # points = (n+1)*(n+2)/2
  // We almost certainly want to cache this
  return (sqrt(8*this->GetPoints()->GetNumberOfPoints() + 1) - 3)/2;
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::ToBarycentricIndex(vtkIdType index,
                                               vtkIdType* bindex)
{
  return vtkLagrangianTriangle::BarycentricIndex(index,bindex,this->GetOrder());
}

//----------------------------------------------------------------------------
vtkIdType vtkLagrangianTriangle::ToIndex(const vtkIdType* bindex)
{
  return vtkLagrangianTriangle::Index(bindex, this->GetOrder());
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::BarycentricIndex(vtkIdType index, vtkIdType* bindex,
                                             vtkIdType order)
{
  // "Barycentric index" is a triplet of integers, each running from 0 to
  // <Order>. It is the index of a point on the triangle in barycentric
  // coordinates.

  assert(order >= 1);

  vtkIdType max = order;
  vtkIdType min = 0;

  // scope into the correct triangle
  while (index != 0 && index >= 3*order)
    {
    index -= 3*order;
    max-=2;
    min++;
    order -= 3;
    }

  if (index < 3)
    {
    // we are on a vertex
    bindex[index] = bindex[(index+1)%3] = min;
    bindex[(index+2)%3] = max;
    }
  else
    {
    // we are on an edge
    index -= 3;
    vtkIdType dim = index/(order - 1);
    vtkIdType offset = (index - dim*(order - 1));
    bindex[(dim+1)%3] = min;
    bindex[(dim+2)%3] = (max - 1) - offset;
    bindex[dim] = (min + 1) + offset;
    }
}

//----------------------------------------------------------------------------
vtkIdType vtkLagrangianTriangle::Index(const vtkIdType* bindex, vtkIdType order)
{
  vtkIdType index = 0;

  assert(order >= 1);
  assert(bindex[0] + bindex[1] + bindex[2] == order);

  vtkIdType max = order;
  vtkIdType min = 0;

  vtkIdType bmin = std::min(std::min(bindex[0], bindex[1]), bindex[2]);

  // scope into the correct triangle
  while (bmin > min)
    {
    index += 3*order;
    max-=2;
    min++;
    order -= 3;
    }

  for (vtkIdType dim = 0; dim < 3; dim++)
    {
    if (bindex[(dim+2)%3] == max)
      {
      // we are on a vertex
      return index;
      }
    index++;
    }

  for (vtkIdType dim = 0; dim < 3; dim++)
    {
    if (bindex[(dim+1)%3] == min)
      {
      // we are on an edge
      return index + bindex[dim] - (min + 1);
      }
    index += max - (min + 1);
    }

  return index;
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::WorldToBarycentric(double* coords,
                                               double* bcoords)
{
  double p0[3], p1[3], p2[3];
  this->GetPoints()->GetPoint(0, p0);
  this->GetPoints()->GetPoint(1, p1);
  this->GetPoints()->GetPoint(2, p2);

  double area = vtkTriangle::TriangleArea(p0, p1, p2);

  bcoords[0] = vtkTriangle::TriangleArea(coords, p1, p2) / area;
  bcoords[1] = vtkTriangle::TriangleArea(p0, coords, p2) / area;
  bcoords[2] = vtkTriangle::TriangleArea(p0, p1, coords) / area;
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::BarycentricToWorld(const double* bcoords,
                                               double* coords)
{
  double p0[3], p1[3], p2[3];
  this->GetPoints()->GetPoint(0, p0);
  this->GetPoints()->GetPoint(1, p1);
  this->GetPoints()->GetPoint(2, p2);

  vtkTriangle::TriangleCenter(p0,p1,p2,coords);
  for (vtkIdType i=0;i<3;i++)
    {
    coords[i] += bcoords[0]*p0[i] + bcoords[1]*p1[i] + bcoords[2]*p2[i];
    }
}

//----------------------------------------------------------------------------
void vtkLagrangianTriangle::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
}
