/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangePyramid.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME vtkLagrangePyramid
// .SECTION Description
// .SECTION See Also

#ifndef vtkLagrangePyramid_h
#define vtkLagrangePyramid_h

#include "vtkCommonDataModelModule.h" // For export macro
#include "vtkNonLinearCell.h"

class vtkPyramid;
class vtkLagrangeCurve;
class vtkLagrangeTriangle;
class vtkLagrangeQuadrilateral;
class vtkDoubleArray;

class VTKCOMMONDATAMODEL_EXPORT vtkLagrangePyramid : public vtkNonLinearCell
{
public:
  static vtkLagrangePyramid *New();
  vtkTypeMacro(vtkLagrangePyramid,vtkNonLinearCell);
  void PrintSelf(ostream& os, vtkIndent indent);

  int GetCellType() { return VTK_LAGRANGE_TETRAHEDRON; }
  int GetCellDimension() { return 3; }
  int RequiresInitialization() { return 1; }
  int GetNumberOfEdges() { return 7; }
  int GetNumberOfFaces() { return 5; }
  vtkCell *GetEdge(int edgeId);
  vtkCell *GetFace(int faceId);

  void Initialize();

  int CellBoundary(int subId, double pcoords[3], vtkIdList *pts);
  int EvaluatePosition(double x[3], double* closestPoint,
                       int& subId, double pcoords[3],
                       double& dist2, double *weights);
  void EvaluateLocation(int& subId, double pcoords[3], double x[3],
                        double *weights);
  void Contour(double value, vtkDataArray *cellScalars,
               vtkIncrementalPointLocator *locator, vtkCellArray *verts,
               vtkCellArray *lines, vtkCellArray *polys,
               vtkPointData *inPd, vtkPointData *outPd,
               vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd);
  void Clip(double value, vtkDataArray *cellScalars,
            vtkIncrementalPointLocator *locator, vtkCellArray *polys,
            vtkPointData *inPd, vtkPointData *outPd,
            vtkCellData *inCd, vtkIdType cellId, vtkCellData *outCd,
            int insideOut);
  int IntersectWithLine(double p1[3], double p2[3], double tol, double& t,
                        double x[3], double pcoords[3], int& subId);
  int Triangulate(int index, vtkIdList *ptIds, vtkPoints *pts);
  void Derivatives(int subId, double pcoords[3], double *values,
                   int dim, double *derivs);
  double* GetParametricCoords();

  int GetParametricCenter(double pcoords[3]);
  double GetParametricDistance(double pcoords[3]);

  void InterpolateFunctions(double pcoords[3], double* weights);
  void InterpolateDerivs(double pcoords[3], double* derivs);

  vtkIdType GetOrder() const { return this->Order; }
  vtkIdType ComputeOrder();

  void ToRationalIndex(vtkIdType index, vtkIdType *rindex);
  vtkIdType ToIndex(const vtkIdType *rindex);

protected:
  vtkLagrangePyramid();
  ~vtkLagrangePyramid();

  vtkLagrangeCurve *Edge;
  vtkLagrangeTriangle *Triangle;
  vtkLagrangeQuadrilateral *Quad;
  vtkPyramid *Pyramid;
  vtkDoubleArray *Scalars; //used to avoid New/Delete in contouring/clipping
  vtkIdType Order;
  double* ParametricCoordinates;
  vtkIdType* EdgeIds;

private:
  vtkLagrangePyramid(const vtkLagrangePyramid&);  // Not implemented.
  void operator=(const vtkLagrangePyramid&);  // Not implemented.
};

#endif
