//=============================================================================
//
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//=============================================================================

#include "smtk/simulation/adh/ExportHotStartFile.h"

#include "smtk/mesh/core/PointField.h"

#include <fstream>

namespace smtk
{
namespace simulation
{
namespace adh
{

void ExportHotStartFile::operator()(
  const std::string& meshHotStartFileName,
  smtk::mesh::ResourcePtr& meshResource,
  const smtk::attribute::ResourcePtr& exportSpecification,
  double startTime,
  std::size_t startTimeUnits)
{
  // Construct a filestream for the geometry. The contents of this file are
  // generated entirely by this method, so we rewrite the file each time the
  // method is called.
  std::fstream hotStartFile(meshHotStartFileName.c_str(), std::ios::out | std::fstream::trunc);

  // Access the top-level meshset's point fields
  std::set<smtk::mesh::PointField> pointFields = meshResource->meshes().pointFields();

  // We are only interested in point fields that match the allowed hotstart
  // names, so we remove any extraneous fields.
  std::set<std::string> allowedNames = { { "IH", "IT", "ioh", "iov", "icon" } };
  for (auto it = pointFields.begin(); it != pointFields.end(); )
  {
    if (allowedNames.find(it->name()) == allowedNames.end())
    {
      it = pointFields.erase(it);
    }
    else
    {
      ++it;
    }
  }

  // For each point field...
  for (const smtk::mesh::PointField& pointField : pointFields)
  {
    //...fill in the file's preamble...
    hotStartFile << "DATASET" << std::endl;
    hotStartFile << "OBJTYPE \"mesh3d\"" << std::endl;
    hotStartFile << "BEGSCL" << std::endl;
    hotStartFile << "ND " << meshResource->meshes().points().size() << std::endl;
    hotStartFile << "NC " << meshResource->meshes().cells().size() << std::endl;
    hotStartFile << "NAME " << pointField.name() << std::endl;
    hotStartFile << "TS " << startTimeUnits << " " << startTime << std::endl;

    //...access the point field data...
    std::vector<double> retrievedFieldValues = pointField.get<double>();

    //...and write the data to file.
    for (std::size_t i = 0; i < pointField.size(); ++i)
    {
      for (std::size_t j = 0; j < pointField.dimension(); ++j)
      {
        if (i != 0)
        {
          hotStartFile << " ";
        }
        hotStartFile << retrievedFieldValues[i * pointField.dimension() + j];
      }
      hotStartFile << std::endl;
    }

    hotStartFile << "ENDDS" << std::endl;
  }

  hotStartFile.close();
}

}
}
}
