//=============================================================================
//
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//=============================================================================

#include "smtk/simulation/adh/ExportAdHMesh.h"

#include "smtk/mesh/adh/MeshPartition.h"

#include "smtk/attribute/Resource.h"

#include "smtk/mesh/core/ForEachTypes.h"
#include "smtk/mesh/core/PointField.h"

#include "smtk/mesh/moab/Interface.h"

#include "moab/Core.hpp"
#include "moab/Interface.hpp"

#include <pybind11/embed.h>

#include <fstream>
#include <iomanip>

namespace
{

std::string to_CardType(smtk::mesh::CellType type)
{

  if (type == smtk::mesh::Line)
  {
    return std::string("E2L");
  }
  if (type == smtk::mesh::Triangle)
  {
    return std::string("E3T");
  }
  else if (type == smtk::mesh::Quad)
  {
    return std::string("E4Q");
  }
  else if (type == smtk::mesh::Tetrahedron)
  {
    return std::string("E4T");
  }
  else if (type == smtk::mesh::Pyramid)
  {
    return std::string("E5P");
  }
  else if (type == smtk::mesh::Wedge)
  {
    return std::string("E6W");
  }
  else if (type == smtk::mesh::Hexahedron)
  {
    return std::string("E8H");
  }
  return std::string("B4D");
}

// When writing out a triangle or quad region the cell must be written
// in counter clockwise orientation. We are presuming that for 2d meshes
// the triangles are all planar, so we can use the shoelace formula
// to determine if the points are in clockwise order.
// https://en.wikipedia.org/wiki/Shoelace_formula
double shoelace(const std::vector<double>& coordinates)
{
  double sum = 0;
  std::size_t nPoints = coordinates.size() / 3;
  for (std::size_t j = 0; j < nPoints; ++j)
  {
    std::size_t c1 = j;
    std::size_t c2 = ((j + 1) % nPoints);

    double x1 = coordinates[c1 * 3];
    double y1 = coordinates[c1 * 3 + 1];

    double x2 = coordinates[c2 * 3];
    double y2 = coordinates[c2 * 3 + 1];

    sum += (x2 - x1) * (y2 + y1);
  }
  return sum;
}

class IterateCells : public smtk::mesh::CellForEach
{
public:
  IterateCells(std::ofstream& meshGeometryFile,
               std::ofstream& boundaryConditionFile,
               const std::unordered_map<smtk::mesh::Handle, std::size_t>& pointMap,
               const smtk::mesh::HandleRange& boundaryVolumeCells,
               const std::vector<smtk::mesh::adh::MeshPartition>& boundaryMeshes,
               const smtk::mesh::InterfacePtr& interface)
    : smtk::mesh::CellForEach()
    , m_meshGeometryFile(meshGeometryFile)
    , m_boundaryConditionFile(boundaryConditionFile)
    , m_pointMap(pointMap)
    , m_boundaryVolumeCells(boundaryVolumeCells)
    , m_boundaryMeshes(boundaryMeshes)
    , m_interface(std::dynamic_pointer_cast<smtk::mesh::moab::Interface>(
                    interface)->moabInterface())
    , m_cellIndex(1)
    , m_partitionIndex(1)
    {
    }

  void forCell(const smtk::mesh::Handle& cellId, smtk::mesh::CellType cellType,
               int numPts) override
    {
      // Write cell connectivity.
      {
        m_meshGeometryFile << to_CardType(cellType) << " \t " << m_cellIndex << " ";

        // when writing out a triangle or quad region the cell must be written
        // in counter clockwise orientation. We are presuming that for 2d meshes
        // the triangles are all planar, so we can use the shoelace formula
        // to determine if the points are in clockwise order.
        //  https://en.wikipedia.org/wiki/Shoelace_formula
        if ((cellType == smtk::mesh::Triangle || cellType == smtk::mesh::Quad) &&
            shoelace(coordinates()) > 0.)
        {
          for (int i = numPts - 1; i >= 0; --i)
          {
            m_meshGeometryFile << std::setw(8) << m_pointMap.at(pointId(i)) << " ";
          }
        }
        else
        {
          for (int i = 0; i < numPts; ++i)
          {
            m_meshGeometryFile << std::setw(8) << m_pointMap.at(pointId(i)) << " ";
          }
        }
        m_meshGeometryFile << std::setw(8) << m_partitionIndex << std::endl;
      }

      // Check if the cell is part of the boundary definition.
      if (smtk::mesh::rangeContains(m_boundaryVolumeCells, cellId))
      {
        // If it is involved in the boundary definition, extract its
        // 2-dimensional adjacencies...
        std::vector<::moab::EntityHandle> adjacencies;
        m_interface->get_adjacencies(&cellId, 1, 2, false, adjacencies);

        // ...iterate each adjacency...
        for (auto& adjacency : adjacencies)
        {
          // ...determine to which boundary it belongs...
          for (std::size_t boundaryIndex = 1; boundaryIndex <= m_boundaryMeshes.size(); ++boundaryIndex)
          {
            if (smtk::mesh::rangeContains(m_boundaryMeshes[boundaryIndex - 1].cells().range(),
                                          adjacency))
            {
              // ...access the boundary cell's "side number" (the canonical
              // ordering information side number)...
              int sideNumber, sense, offset;
              m_interface->side_number(cellId, adjacency, sideNumber, sense, offset);

              // ...and write out the cell index, side number and boundary
              // index.
              m_boundaryConditionFile << "FCS" << std::setw(8) << m_cellIndex << " "
                                      << std::setw(8) << sideNumber << " "
                                      << std::setw(8) << boundaryIndex << std::endl;
            }
          }
        }
      }
      ++m_cellIndex;
    }

  void setPartitionIndex(std::size_t partitionIndex) { m_partitionIndex = partitionIndex; }

private:

  // Used to write out the geometry file
  std::ofstream& m_meshGeometryFile;
  std::size_t m_partitionIndex;

  // Used to write out the boundary file
  std::ofstream& m_boundaryConditionFile;
  const smtk::mesh::HandleRange& m_boundaryVolumeCells;
  const std::vector<smtk::mesh::adh::MeshPartition>& m_boundaryMeshes;
  ::moab::Interface* m_interface;

  // Used for both
  const std::unordered_map<smtk::mesh::Handle, std::size_t>& m_pointMap;
  std::size_t m_cellIndex;
};

class IteratePoints : public smtk::mesh::PointForEach
{
public:
  IteratePoints(std::ofstream& meshGeometryFile,
               const std::unordered_map<smtk::mesh::Handle, std::size_t>& pointMap)
    : smtk::mesh::PointForEach()
    , m_meshGeometryFile(meshGeometryFile)
    , m_pointMap(pointMap)
    , m_pointIndex(1)
    {
    }

  void forPoints(const smtk::mesh::HandleRange& pointIds, std::vector<double>& xyz, bool&) override
    {
      // Write points to file.
      {
        std::size_t counter = 0;
        for (auto i = smtk::mesh::rangeElementsBegin(pointIds);
             i != smtk::mesh::rangeElementsEnd(pointIds); ++i, counter += 3)
        {
          m_meshGeometryFile << "ND \t " << std::setw(8) << m_pointMap.at(*i) << " "
                             << std::fixed << std::setw(12) << xyz[counter]
                             << " " << std::setw(12) << xyz[counter + 1] << " "
                             << std::setw(12) << xyz[counter + 2] << std::endl;
        }
      }
    }

private:

  // Used to write out the geometry file
  std::ofstream& m_meshGeometryFile;
  std::size_t m_partitionIndex;

  const std::unordered_map<smtk::mesh::Handle, std::size_t>& m_pointMap;
  std::size_t m_pointIndex;
};

void writeBCPreamble(const std::string& boundaryConditionFileName)
{
  using namespace pybind11::literals;
  auto locals = pybind11::dict("bc_path"_a=boundaryConditionFileName.c_str());

  const char * bcPreamblePy = R"PYTHONSNIPPET(
with open(bc_path, 'w') as output:
    output.write('hello!')
)PYTHONSNIPPET";

  pybind11::exec(bcPreamblePy, pybind11::globals(), locals);

}

}

namespace smtk
{
namespace simulation
{
namespace adh
{

void ExportAdHMesh::operator()(
  const std::string& geometryFileName,
  const std::string& boundaryConditionFileName,
  smtk::mesh::ResourcePtr& meshResource,
  smtk::model::ResourcePtr& modelResource,
  const std::string& modelPropertyName,
  const std::vector<smtk::attribute::AttributePtr>& boundaryConditions)
{
  // First, partition the mesh according to the model property name.
  std::vector<smtk::mesh::adh::MeshPartition> volumeMeshes =
    smtk::mesh::adh::partitionByModelProperty(
      meshResource, modelResource, modelPropertyName, smtk::mesh::Dims3);

  // Then, partition the surface mesh according to the boundary condition attributes.
  std::vector<smtk::mesh::adh::MeshPartition> boundaryMeshes =
    smtk::mesh::adh::partitionByAttributeAssociation(
      meshResource, boundaryConditions, smtk::mesh::Dims2);

  // Collect all of the points needed to describe the output mesh.

  std::size_t nCells, nPoints;

  // No default constructor! Give it dummy parameters and set it in the braced
  // logic below.
  smtk::mesh::PointSet points(meshResource, smtk::mesh::HandleRange());
  {
    smtk::mesh::CellSet cs = volumeMeshes[0].cells();
    const std::size_t size = volumeMeshes.size();
    for (std::size_t i = 1; i < size; ++i)
    {
      cs.append(volumeMeshes[i].cells());
    }
    points = cs.points();
    nCells = cs.size();
    nPoints = points.size();
  }

  // Construct a map connecting point handles to AdH's 1-based indexing. This is
  // usually done implicitly in smtk::mesh::extractTessellation; it is
  // repeatedly called when exporting region-partitioned meshes to XMS format.
  // By explicitly constructing this map once here, we avoid the overhead of
  // multiple point iterations.
  std::unordered_map<smtk::mesh::Handle, std::size_t> pointMap;
  {
    auto it = smtk::mesh::rangeElementsBegin(points.range());
    auto end = smtk::mesh::rangeElementsEnd(points.range());
    for (std::size_t counter = 1; it != end; ++it, ++counter)
    {
      pointMap[*it] = counter;
    }
  }

  // Collect all of the volume cells associated with the boundary surfaces. When
  // we loop over every volume cell, we will check if each cell is in this group
  // before performing the more expensive adjacency and canonical index
  // calculations.
  smtk::mesh::HandleRange boundaryVolumeCells;
  {
    smtk::mesh::MeshSet ms = boundaryMeshes[0].m_meshSet;
    const std::size_t size = boundaryMeshes.size();
    for (std::size_t i = 1; i < size; ++i)
    {
      ms = smtk::mesh::set_union(ms, boundaryMeshes[i].m_meshSet);
    }
    meshResource->interface()->computeAdjacenciesOfDimension(
      ms.range(), smtk::mesh::Dims3, boundaryVolumeCells);
  }

  // Construct a filestream for the geometry. The contents of this file are
  // generated entirely by this method, so we rewrite the file each time the
  // method is called.
  std::fstream geometryFile(geometryFileName.c_str(), std::ios::out | std::fstream::trunc);

  // Fill in the file's preamble
  geometryFile << "MESH3D" << std::endl;
  geometryFile << "#NELEM " << nCells << std::endl;
  geometryFile << "#NNODE " << nPoints << std::endl;


  writeBCPreamble(boundaryConditionFileName);

  // std::fstream boundaryConditionFile(meshBoundaryConditionFileName.c_str(), std::ios::out  | std::fstream::app);

  // IterateCells iterateCells();
}

void ExportAdHMesh::operator()(const std::string& testFileName)
{
  writeBCPreamble(testFileName);
}

void ExportAdHMesh::operator()(pybind11::object py_scope)
{
  if (!pybind11::hasattr(py_scope, "sim_atts"))
  {
    std::cerr << "could not find sim_atts" << std::endl;
    return;
  }

  smtk::attribute::ResourcePtr sim_atts =
    py_scope.attr("sim_atts").cast<smtk::attribute::ResourcePtr>();
  std::cout<<"sim_atts:" <<sim_atts<<std::endl;
  std::cout<<"sim_atts name: "<<sim_atts->name()<<std::endl;

  if (!pybind11::hasattr(py_scope, "mesh_collection"))
  {
    std::cerr << "could not find mesh_collection" << std::endl;
    return;
  }

  smtk::mesh::ResourcePtr mesh_collection =
    py_scope.attr("mesh_collection").cast<smtk::mesh::ResourcePtr>();
  std::cout<<"mesh_collection:" <<mesh_collection<<std::endl;
  std::cout<<"mesh_collection name: "<<mesh_collection->name()<<std::endl;

  std::string output_filebase =
    py_scope.attr("output_filebase").cast<std::string>();
  std::cout<<"output_filebase: " <<output_filebase<<std::endl;
}

// void ExportAdHMesh::operator()(
//   const std::string& geometryFileName,
//   const std::string& boundaryConditionFileName,
//   smtk::mesh::ResourcePtr& meshResource,
//   smtk::model::ResourcePtr& modelResource,
//   const std::string& modelPropertyName,
//   const std::vector<smtk::attribute::AttributePtr>& boundaryConditions)


}
}
}
