//=============================================================================
//
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//=============================================================================

#include "smtk/mesh/adh/MeshPartition.h"

#include "smtk/model/EntityRef.h"

namespace smtk
{
namespace mesh
{
namespace adh
{

MeshPartition::MeshPartition(const smtk::mesh::MeshSet& ms, int partitionId,
                           smtk::mesh::DimensionType dim)
  : m_meshSet(ms)
  , m_dim(dim)
  , m_partitionId(partitionId)
{
}

std::vector<MeshPartition> partitionByModelEntity(
  smtk::mesh::ResourcePtr meshResource, smtk::mesh::DimensionType type)
{
  std::vector<MeshPartition> meshPartitions;
  smtk::mesh::MeshSet meshes = meshResource->meshes(type);
  if (meshes.is_empty())
  { //if we have no meshes, stop
    return meshPartitions;
  }

  smtk::model::EntityRefArray entities;
  meshes.modelEntities(entities);

  if (entities.size() <= 1)
  { //explicitly done so that no model relationship is equal to everything
    //being in the same partition
    meshPartitions.push_back(MeshPartition(meshes, 1, type));
  }
  else
  {
    meshPartitions.reserve(entities.size());
    int partition = 1;

    typedef smtk::model::EntityRefArray::const_iterator it;
    for (it i = entities.begin(); i != entities.end(); ++i)
    {
      smtk::mesh::MeshSet subset = meshResource->findAssociatedMeshes(*i, type);
      if (!subset.is_empty())
      {
        meshPartitions.push_back(MeshPartition(subset, partition, type));
        ++partition;
      }
    }
  }
  return meshPartitions;
}

std::vector<MeshPartition> partitionByModelProperty(smtk::mesh::ResourcePtr meshResource,
                                                    smtk::model::ResourcePtr resource,
                                                    const std::string& modelPropertyName,
                                                    smtk::mesh::DimensionType type)
{
  std::vector<MeshPartition> meshPartitions;
  smtk::mesh::MeshSet meshes = meshResource->meshes(type);
  if (meshes.is_empty())
  { //if we have no meshes, stop
    return meshPartitions;
  }

  smtk::model::EntityRefArray entities;
  meshes.modelEntities(entities);

  typedef smtk::model::EntityRefArray::const_iterator it;
  for (it i = entities.begin(); i != entities.end(); ++i)
  {
    const smtk::model::IntegerList& values =
      resource->integerProperty(i->entity(), modelPropertyName);
    smtk::mesh::MeshSet subset = meshResource->findAssociatedMeshes(*i, type);
    if (values.size() == 1 && !subset.is_empty())
    { //only accept model properties that have single values
      //since that is what we think partition id's should be
      const int& partition = values[0];
      meshPartitions.push_back(MeshPartition(subset, partition, type));
    }
  }

  return meshPartitions;
}

std::vector<MeshPartition> partitionByAttributeAssociation(
  smtk::mesh::ResourcePtr meshResource,
  const std::vector<smtk::attribute::AttributePtr>& attributes,
  smtk::mesh::DimensionType type)
{
  std::vector<MeshPartition> meshPartitions;
  smtk::mesh::MeshSet meshes = meshResource->meshes(type);
  if (meshes.is_empty())
  { //if we have no meshes, stop
    return meshPartitions;
  }

  for (std::size_t partition = 0; partition < attributes.size(); ++partition)
  {
    auto attribute = attributes[partition];
    auto associations = attribute->associations();
    auto entities =
      associations->as<smtk::model::EntityRefArray>([](smtk::resource::PersistentObjectPtr obj) {
          return smtk::model::EntityRef(std::dynamic_pointer_cast<smtk::model::Entity>(obj));
        });

    meshes.modelEntities(entities);

    typedef smtk::model::EntityRefArray::const_iterator it;
    for (it i = entities.begin(); i != entities.end(); ++i)
    {
      smtk::mesh::MeshSet subset = meshResource->findAssociatedMeshes(*i, type);
      if (!subset.is_empty())
      {
        meshPartitions.push_back(MeshPartition(subset, partition, type));
      }
    }
  }

  return meshPartitions;
}

}
}
}
