/*=========================================================================

   Program: ParaView
   Module:    pqSpreadSheetViewSelectionModel.cxx

   Copyright (c) 2005-2008 Sandia Corporation, Kitware Inc.
   All rights reserved.

   ParaView is a free software; you can redistribute it and/or modify it
   under the terms of the ParaView license version 1.2. 

   See License_v1.2.txt for the full ParaView license.
   A copy of this license can be obtained by contacting
   Kitware Inc.
   28 Corporate Drive
   Clifton Park, NY 12065
   USA

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

========================================================================*/
#include "pqSpreadSheetViewSelectionModel.h"

// Server Manager Includes.
#include "vtkProcessModule.h"
#include "vtkPVDataInformation.h"
#include "vtkSelection.h"
#include "vtkSelectionNode.h"
#include "vtkSMProxyManager.h"
#include "vtkSMSourceProxy.h"
#include "vtkSMVectorProperty.h"

// Qt Includes.
#include <QtDebug>

// ParaView Includes.
#include "pqDataRepresentation.h"
#include "pqOutputPort.h"
#include "pqPipelineSource.h"
#include "pqServer.h"
#include "pqSMAdaptor.h"
#include "pqSpreadSheetViewModel.h"

static uint qHash(pqSpreadSheetViewModel::vtkIndex index)
{
  return qHash(index.Tuple[2]);
}

class pqSpreadSheetViewSelectionModel::pqInternal
{
public:
  pqSpreadSheetViewModel* Model;
};

//-----------------------------------------------------------------------------
pqSpreadSheetViewSelectionModel::pqSpreadSheetViewSelectionModel(
  pqSpreadSheetViewModel* amodel, QObject* _parent)
: Superclass(amodel, _parent)
{
  this->UpdatingSelection = false;
  this->Internal = new pqInternal();
  this->Internal->Model = amodel;

  QObject::connect(amodel, 
    SIGNAL(selectionChanged(const QItemSelection&)),
    this, SLOT(serverSelectionChanged(const QItemSelection&)));
}

//-----------------------------------------------------------------------------
pqSpreadSheetViewSelectionModel::~pqSpreadSheetViewSelectionModel()
{
  delete this->Internal;
}

//-----------------------------------------------------------------------------
void pqSpreadSheetViewSelectionModel::serverSelectionChanged(
  const QItemSelection& sel)
{
  this->UpdatingSelection = true;
  this->select(sel, QItemSelectionModel::ClearAndSelect);
  this->UpdatingSelection = false;
}

//-----------------------------------------------------------------------------
void pqSpreadSheetViewSelectionModel::select(const QItemSelection& sel, 
    QItemSelectionModel::SelectionFlags command)
{
  this->Superclass::select(sel, command);
  if (this->UpdatingSelection || command == QItemSelectionModel::NoUpdate)
    {
    return;
    }

  // Update VTK Selection.
  // * Obtain the currently set sel on the selected source (if none, a
  //    new one is created).
  // * We then update the ids selected on the
  vtkSmartPointer<vtkSMSourceProxy> selSource ;
  selSource.TakeReference(this->getSelectionSource());
  if (!selSource)
    {
    emit this->selection(0);
    return;
    }

  vtkSMVectorProperty* vp = vtkSMVectorProperty::SafeDownCast(
    selSource->GetProperty("IDs"));
  QList<QVariant> ids = pqSMAdaptor::getMultipleElementProperty(vp);
  int numElemsPerCommand = vp->GetNumberOfElementsPerCommand();
  if (command & QItemSelectionModel::Clear)
    {
    ids.clear();
    }
  if (command & QItemSelectionModel::Select || 
    command & QItemSelectionModel::Deselect || 
    command & QItemSelectionModel::Toggle)
    {
    // Get the (process id, index) pairs for the indices indicated in the 
    // selection.
    QSet<pqSpreadSheetViewModel::vtkIndex> vtkIndices = 
      this->Internal->Model->getVTKIndices(sel.indexes());

    QSet<pqSpreadSheetViewModel::vtkIndex> curIndices;
    for (int cc=0; (cc+numElemsPerCommand) <= ids.size(); )
      {
      pqSpreadSheetViewModel::vtkIndex index(0, -1, 0);
      if (numElemsPerCommand == 3)
        {
        index.Tuple[0] = ids[cc].value<vtkIdType>(); cc++;
        index.Tuple[1] = ids[cc].value<vtkIdType>(); cc++;
        index.Tuple[2] = ids[cc].value<vtkIdType>(); cc++;
        }
      else // numElemsPerCommand == 2
        {
        index.Tuple[1] = ids[cc].value<vtkIdType>(); cc++;
        index.Tuple[2] = ids[cc].value<vtkIdType>(); cc++;
        }
      curIndices.insert(index);
      }

    if (command & QItemSelectionModel::Select)
      {
      curIndices += vtkIndices;
      }
    if (command & QItemSelectionModel::Deselect)
      {
      curIndices -= vtkIndices;
      }
    if (command & QItemSelectionModel::Toggle)
      {
      QSet<pqSpreadSheetViewModel::vtkIndex> toSelect = 
        vtkIndices - curIndices;
      QSet<pqSpreadSheetViewModel::vtkIndex> toDeselect =
        vtkIndices - toSelect;
      curIndices -= toDeselect;
      curIndices += toSelect;
      }

    ids.clear();
    QSet<pqSpreadSheetViewModel::vtkIndex>::iterator iter;
    for (iter = curIndices.begin(); iter != curIndices.end(); ++iter)
      {
      if (numElemsPerCommand == 3)
        {
        ids.push_back(iter->Tuple[0]);
        ids.push_back(iter->Tuple[1]);
        ids.push_back(iter->Tuple[2]);
        }
      else //numElemsPerCommand == 2
        {
        ids.push_back(iter->Tuple[1]);
        ids.push_back(iter->Tuple[2]);
        }
      }
    }

  if (ids.size() == 0)
    {
    selSource = 0;
    }
  else
    {
    pqSMAdaptor::setMultipleElementProperty(
      vp, ids);
    selSource->UpdateVTKObjects();
    }

  emit this->selection(selSource);
}

//-----------------------------------------------------------------------------
// Locate the selection source currently set on the representation being shown.
// If no selection exists, or selection present is not "updatable" by this
// model, we create a new selection.
vtkSMSourceProxy* pqSpreadSheetViewSelectionModel::getSelectionSource()
{
  pqDataRepresentation* repr = this->Internal->Model->getRepresentation();
  if (!repr)
    {
    return 0;
    }

  int field_type = this->Internal->Model->getFieldType();
  if (field_type != vtkDataObject::FIELD_ASSOCIATION_POINTS &&
    field_type != vtkDataObject::FIELD_ASSOCIATION_CELLS)
    {
    return 0;
    }

  // Convert field_type to selection field type.
  field_type = (field_type == vtkDataObject::FIELD_ASSOCIATION_POINTS)?
    vtkSelectionNode::POINT : vtkSelectionNode::CELL;

  pqOutputPort* opport = repr->getOutputPortFromInput();
  vtkSMSourceProxy* selsource = opport->getSelectionInput(); 

  // We may be able to simply update the currently existing selection, if any.
  bool updatable = (selsource != 0);

  // If field types differ, not updatable.
  if (updatable && pqSMAdaptor::getElementProperty(
      selsource->GetProperty("FieldType")).toInt() != field_type)
    {
    updatable = false;
    }

  // Determine what selection proxy name we want. If the name differs then not
  // updatable.
  const char* proxyname = "IDSelectionSource";
  vtkPVDataInformation* dinfo = opport->getDataInformation(false);
  const char* cdclassname = dinfo->GetCompositeDataClassName();
  if (cdclassname && strcmp(cdclassname, "vtkHierarchicalBoxDataSet") == 0)
    {
    proxyname = "HierarchicalDataIDSelectionSource";
    }
  else if (cdclassname && strcmp(cdclassname, "vtkMultiBlockDataSet") == 0)
    {
    proxyname = "CompositeDataIDSelectionSource";
    }

  if (updatable && strcmp(selsource->GetXMLName(), proxyname) != 0)
    {
    updatable = false;
    }

  if (updatable)
    {
    selsource->Register(0);
    }
  else
    {
    vtkSMProxyManager* pxm = vtkSMProxyManager::GetProxyManager();
    selsource = 
      vtkSMSourceProxy::SafeDownCast(pxm->NewProxy("sources", proxyname));
    selsource->SetConnectionID(repr->getServer()->GetConnectionID());
    selsource->SetServers(vtkProcessModule::DATA_SERVER);
    pqSMAdaptor::setElementProperty(
      selsource->GetProperty("FieldType"), field_type);
    selsource->UpdateVTKObjects();
    }

  return selsource;
}
