/*=========================================================================

  Program:   ParaView
  Module:    vtkPVIOSettings.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPVIOSettings.h"

#include "vtkDataArraySelection.h"
#include "vtkObjectFactory.h"
#include "vtkSMProxyManager.h"
#include "vtkSMReaderFactory.h"
#include "vtkSMSession.h"
#include "vtkStringArray.h"

#include <cassert>
#include <string>
#include <vector>

vtkSmartPointer<vtkPVIOSettings> vtkPVIOSettings::Instance;

class vtkPVIOSettings::vtkInternals
{
public:
  std::vector<std::string> ExcludedNameFilters;
  vtkNew<vtkStringArray> AllNameFilters;
};

//----------------------------------------------------------------------------
vtkPVIOSettings* vtkPVIOSettings::New()
{
  vtkPVIOSettings* instance = vtkPVIOSettings::GetInstance();
  assert(instance);
  instance->Register(nullptr);
  return instance;
}

//----------------------------------------------------------------------------
vtkPVIOSettings* vtkPVIOSettings::GetInstance()
{
  if (!vtkPVIOSettings::Instance)
  {
    vtkPVIOSettings* instance = new vtkPVIOSettings();
    instance->InitializeObjectBase();
    vtkPVIOSettings::Instance.TakeReference(instance);
  }
  return vtkPVIOSettings::Instance;
}

//----------------------------------------------------------------------------
vtkPVIOSettings::vtkPVIOSettings()
{
  this->Internals = new vtkInternals();
}

//----------------------------------------------------------------------------
vtkPVIOSettings::~vtkPVIOSettings()
{
  delete this->Internals;
}

//----------------------------------------------------------------------------
void vtkPVIOSettings::SetNumberOfExcludedNameFilters(int n)
{
  if (n != this->GetNumberOfExcludedNameFilters())
  {
    this->Internals->ExcludedNameFilters.resize(n);
    this->Modified();
  }
}

//----------------------------------------------------------------------------
int vtkPVIOSettings::GetNumberOfExcludedNameFilters()
{
  return static_cast<int>(this->Internals->ExcludedNameFilters.size());
}

//----------------------------------------------------------------------------
void vtkPVIOSettings::SetExcludedNameFilter(int i, const char* expression)
{
  if (i >= 0 && i < this->GetNumberOfExcludedNameFilters())
  {
    if (this->Internals->ExcludedNameFilters[i] != expression)
    {
      this->Internals->ExcludedNameFilters[i] = expression;
      this->Modified();
    }
  }
  else
  {
    vtkErrorMacro("Index out of range: " << i);
  }
}

//----------------------------------------------------------------------------
const char* vtkPVIOSettings::GetExcludedNameFilter(int i)
{
  if (i >= 0 && i < this->GetNumberOfExcludedNameFilters())
  {
    return this->Internals->ExcludedNameFilters[i].c_str();
  }
  else
  {
    vtkErrorMacro("Index out of range: " << i);
  }

  return nullptr;
}

//------------------------------------------------------------------------------
vtkStringArray* vtkPVIOSettings::GetAllNameFilters()
{
  return this->Internals->AllNameFilters;
}

//----------------------------------------------------------------------------
void vtkPVIOSettings::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}
