// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
/**
 * @class   vtkDGWdg
 * @brief   Metadata for a discontinuous Galerkin wedge.
 *
 * Currently, only a linear shape is supported but this
 * may change to arbitrary order.
 */

#ifndef vtkDGWdg_h
#define vtkDGWdg_h

#include "vtkFiltersCellGridModule.h" // for export macro

#include "vtkDeRhamCell.h"
#include "vtkStringToken.h" // for vtkStringToken::Hash

VTK_ABI_NAMESPACE_BEGIN
class vtkCellGrid;
class vtkDataSetAttributes;
class vtkCellAttribute;

class VTKFILTERSCELLGRID_EXPORT vtkDGWdg : public vtkDeRhamCell
{
public:
  static vtkDGWdg* New();

  vtkTypeMacro(vtkDGWdg, vtkDeRhamCell);
  vtkInheritanceHierarchyOverrideMacro(vtkDGWdg);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  ///@{
  /// Methods to query cell shape and reference parameterization.
  ///
  /// These methods do not return information about particular
  /// cells or sides specified by the cell-grid's arrays; only
  /// metadata about the cell.
  bool IsInside(const vtkVector3d& rst, double tolerance) override;
  Shape GetShape() const override { return Shape::Wedge; }
  int GetDimension() const override { return Dimension; }
  const std::array<double, 3>& GetCornerParameter(int corner) const override;
  int GetNumberOfSideTypes() const override;
  std::pair<int, int> GetSideRangeForType(int sideType) const override;
  int GetNumberOfSidesOfDimension(int dimension) const override;
  const std::vector<vtkIdType>& GetSideConnectivity(int side) const override;
  const std::vector<vtkIdType>& GetSidesOfSide(int side) const override;
  Shape GetSideShape(int side) const override;
  ///@}

  vtkTypeFloat32Array* GetReferencePoints() const override;
  vtkTypeInt32Array* GetSideConnectivity() const override;
  vtkTypeInt32Array* GetSideOffsetsAndShapes() const override;

  static constexpr int Dimension = 3;
  const std::array<std::array<double, 3>, 6> Parameters = { {
    { 0., 0., -1. },  // node 0
    { +1., 0., -1. }, // node 1
    { 0., +1., -1. }, // node 2
    { 0., 0., +1. },  // node 3
    { +1., 0., +1. }, // node 4
    { 0., +1., +1. }  // node 5
  } };
  const std::array<std::vector<vtkIdType>, 21> Sides = { {
    { 0, 1, 2, 3, 4, 5 }, // wedge itself
    { 0, 1, 4, 3 },       // face 0 (-S normal)
    { 1, 2, 5, 4 },       // face 1 (+RS normal)
    { 0, 3, 5, 2 },       // face 2 (-R normal)
    { 0, 2, 1 },          // face 4 (-T normal)
    { 3, 4, 5 },          // face 5 (+T normal)
    { 0, 1 },             // edge 0 (-S-T planes, +R dir)
    { 1, 2 },             // edge 1 (+RS-T planes, -R+S dir)
    { 0, 2 },             // edge 2 (-R-T planes, +S dir)
    { 0, 3 },             // edge 3 (-R-S planes, +T dir)
    { 1, 4 },             // edge 4 (+RS-S planes, +T dir)
    { 2, 5 },             // edge 5 (-R+RS planes, +T dir)
    { 3, 4 },             // edge 6 (-S+T planes, +R dir)
    { 4, 5 },             // edge 7 (+RS+T planes, -R+S dir)
    { 5, 3 },             // edge 8 (+R+T planes, +S dir)
    { 0 },                // vertex 0
    { 1 },                // vertex 1
    { 2 },                // vertex 2
    { 3 },                // vertex 3
    { 4 },                // vertex 4
    { 5 }                 // vertex 5
  } };
  const std::array<std::vector<vtkIdType>, 21> SidesOfSides = { { { 0, 1, 2, 3, 4 },
    { 5, 9, 11, 8 }, { 6, 10, 12, 9 }, { 8, 13, 10, 7 }, { 7, 6, 5 }, { 11, 12, 13 }, { 14, 15 },
    { 15, 16 }, { 14, 16 }, { 14, 17 }, { 15, 18 }, { 16, 19 }, { 17, 18 }, { 18, 19 }, { 19, 17 },
    {}, {}, {}, {}, {}, {} } };
  const std::array<int, Dimension + 3> SideOffsets = { { 0, 1, 4, 6, 15, 21 } };
  const std::array<Shape, Dimension + 3> SideShapes = { { Shape::Wedge,
    Shape::Quadrilateral, Shape::Triangle, Shape::Edge, Shape::Vertex, Shape::None } };

protected:
  vtkDGWdg();
  ~vtkDGWdg() override;

private:
  vtkDGWdg(const vtkDGWdg&) = delete;
  void operator=(const vtkDGWdg&) = delete;
};

VTK_ABI_NAMESPACE_END
#endif
