// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include "vtkDGWdg.h"

#include "vtkCellAttribute.h"
#include "vtkCellGrid.h"
#include "vtkDataSetAttributes.h"
#include "vtkObjectFactory.h"
#include "vtkStringToken.h"
#include "vtkTypeFloat32Array.h"
#include "vtkTypeInt32Array.h"

VTK_ABI_NAMESPACE_BEGIN

using namespace vtk::literals;

vtkStandardNewMacro(vtkDGWdg);

// TODO: Use IMPLEMENTABLE/IMPLEMENTS and vtkObjectFactory or equivalent.
static bool registerType = vtkCellMetadata::RegisterType<vtkDGWdg>();

vtkDGWdg::vtkDGWdg()
{
  this->CellSpec.SourceShape = this->GetShape();
}

vtkDGWdg::~vtkDGWdg() = default;

void vtkDGWdg::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

bool vtkDGWdg::IsInside(const vtkVector3d& rst, double tolerance)
{
  tolerance = std::abs(tolerance);
  double pb = 1 + tolerance;
  double nb = -1 - tolerance;
  double n0 = -tolerance;
  double rs = 1.0 - rst[0] - rst[1];
  return rst[0] >= n0 && rst[0] <= pb && rst[1] >= n0 && rst[1] <= pb && rs >= n0 && rs <= pb &&
    rst[2] >= nb && rst[2] <= pb;
}

const std::array<double, 3>& vtkDGWdg::GetCornerParameter(int corner) const
{
  if (corner < 0 || corner >= 6)
  {
    static std::array<double, 3> dummy{ 0., 0., 0. }; // Maybe NaN would be better?
    return dummy;
  }
  return this->Parameters[corner];
}

int vtkDGWdg::GetNumberOfSideTypes() const
{
  return static_cast<int>(vtkDGWdg::SideOffsets.size() - 2);
}

std::pair<int, int> vtkDGWdg::GetSideRangeForType(int sideType) const
{
  if (sideType < -1)
  {
    return std::make_pair(SideOffsets[1] - 1, SideOffsets[vtkDGWdg::Dimension + 2] - 1);
  }
  if (sideType > vtkDGWdg::Dimension)
  {
    return std::make_pair(-1, -1);
  }
  return std::make_pair(SideOffsets[sideType + 1] - 1, SideOffsets[sideType + 2] - 1);
}

int vtkDGWdg::GetNumberOfSidesOfDimension(int dimension) const
{
  if (dimension < 0 || dimension >= this->Dimension)
  {
    return 0;
  }
  return this->SideOffsets[Dimension - dimension + 1] - this->SideOffsets[Dimension - dimension];
}

const std::vector<vtkIdType>& vtkDGWdg::GetSideConnectivity(int side) const
{
  if (side < -1 || side >= 21)
  {
    static std::vector<vtkIdType> dummy;
    return dummy;
  }
  return this->Sides[side + 1];
}

const std::vector<vtkIdType>& vtkDGWdg::GetSidesOfSide(int side) const
{
  if (side < -1 || side >= 21)
  {
    static std::vector<vtkIdType> dummy;
    return dummy;
  }
  return this->SidesOfSides[side + 1];
}

vtkTypeFloat32Array* vtkDGWdg::GetReferencePoints() const
{
  static vtkNew<vtkTypeFloat32Array> refPts;
  if (refPts->GetNumberOfTuples() == 0)
  {
    this->FillReferencePoints(refPts);
    refPts->SetName("WdgReferencePoints");
  }
  return refPts;
}

vtkTypeInt32Array* vtkDGWdg::GetSideConnectivity() const
{
  static vtkNew<vtkTypeInt32Array> sideConn;
  if (sideConn->GetNumberOfTuples() == 0)
  {
    this->FillSideConnectivity(sideConn);
    sideConn->SetName("WdgSideConn");
  }
  return sideConn;
}

vtkDGWdg::Shape vtkDGWdg::GetSideShape(int side) const
{
  if (side < -1)
  {
    return None;
  }
  for (std::size_t ii = 0; ii < SideOffsets.size() - 1; ++ii)
  {
    if (side + 1 < vtkDGWdg::SideOffsets[ii + 1])
    {
      return vtkDGWdg::SideShapes[ii];
    }
  }
  return None;
}

vtkTypeInt32Array* vtkDGWdg::GetSideOffsetsAndShapes() const
{
  static vtkNew<vtkTypeInt32Array> sideOffsetsAndShapes;
  if (sideOffsetsAndShapes->GetNumberOfTuples() == 0)
  {
    this->FillSideOffsetsAndShapes(sideOffsetsAndShapes);
    sideOffsetsAndShapes->SetName("WdgOffsetsAndShapes");
  }
  return sideOffsetsAndShapes;
}

VTK_ABI_NAMESPACE_END
