/*=========================================================================

  Program:   ParaView
  Module:    vtkBivariateNoiseMapper.h

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class vtkBivariateNoiseMapper
 * @brief Mapper to visualize bivariate data with noise
 *
 * The vtkBivariateNoiseMapper allow to visualize bivariate data with noise.
 * A second array can be passed to this mapper. The values of this array will be used
 * to control the noise intensity: the bigger the values are, the bigger the intensity
 * of the noise is (values are clamped between 0 and 1; a value of 0 corresponds to no
 * noise at all.)
 *
 * The user can play with two parameters to control the noise:
 * - the FrequencyModifier parameter controls the size of the cells of the noise,
 * - the AmplitudeModifier parameter controls how much the values of the 2nd array
 *   affect the amplitude of the noise.
 *
 * The vtkBivariateNoiseMapper only accept point data arrays.
 *
 * The noise function use in this mapper had been taken from ShaderToy.
 * Please see : https://www.shadertoy.com/view/3lXfzH
 *
 * @sa vtkOpenGLPolyDataMapper vtkBivariateNoiseRepresentation
 */

#ifndef vtkBivariateNoiseMapper_h
#define vtkBivariateNoiseMapper_h

#include "vtkBivariateRepresentationsModule.h" // for export macro
#include "vtkCompositePolyDataMapper2.h"

#include <memory> // for unique_ptr

class VTKBIVARIATEREPRESENTATIONS_EXPORT vtkBivariateNoiseMapper
  : public vtkCompositePolyDataMapper2
{
public:
  static vtkBivariateNoiseMapper* New();
  vtkTypeMacro(vtkBivariateNoiseMapper, vtkCompositePolyDataMapper2);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  void SetFrequencyModifier(double mod);
  void SetAmplitudeModifier(double mod);

protected:
  vtkBivariateNoiseMapper();
  ~vtkBivariateNoiseMapper() override;

  /**
   * Override the creation of the helper to create a dedicated mapper helper.
   * This helper inheritate indirectly from vtkOpenGLPolydataMapper and contains
   * most of the rendering code specific to the vtkBivariateNoiseMapper.
   */
  vtkCompositeMapperHelper2* CreateHelper() override;

  /**
   * Overriden to pass the 2nd array to the helper.
   * The 2nd array should be passed to this mapper through the
   * SetInputArrayToProcess method.
   */
  void CopyMapperValuesToHelper(vtkCompositeMapperHelper2* helper) override;

private:
  vtkBivariateNoiseMapper(const vtkBivariateNoiseMapper&) = delete;
  void operator=(const vtkBivariateNoiseMapper&) = delete;

  struct vtkInternals;
  std::unique_ptr<vtkInternals> Internals;

  friend class vtkBivariateNoiseMapperHelper;
};

#endif
