/*=========================================================================

   Program: ParaView
   Module:    pqAnimationViewWidget.cxx

   Copyright (c) 2005-2008 Sandia Corporation, Kitware Inc.
   All rights reserved.

   ParaView is a free software; you can redistribute it and/or modify it
   under the terms of the ParaView license version 1.2.

   See License_v1.2.txt for the full ParaView license.
   A copy of this license can be obtained by contacting
   Kitware Inc.
   28 Corporate Drive
   Clifton Park, NY 12065
   USA

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR
CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "pqAnimationViewWidget.h"
#include "ui_pqPythonAnimationCue.h"

#include <QComboBox>
#include <QDialog>
#include <QDialogButtonBox>
#include <QDoubleValidator>
#include <QFormLayout>
#include <QIntValidator>
#include <QLabel>
#include <QLineEdit>
#include <QPointer>
#include <QPushButton>
#include <QToolButton>
#include <QVBoxLayout>
#include <QtDebug>

#include "pqActiveObjects.h"
#include "pqAnimatablePropertiesComboBox.h"
#include "pqAnimatableProxyComboBox.h"
#include "pqAnimationCue.h"
#include "pqAnimationKeyFrame.h"
#include "pqAnimationModel.h"
#include "pqAnimationScene.h"
#include "pqAnimationTimeWidget.h"
#include "pqAnimationTrack.h"
#include "pqAnimationWidget.h"
#include "pqApplicationCore.h"
#include "pqComboBoxDomain.h"
#include "pqCoreUtilities.h"
#include "pqKeyFrameEditor.h"
#include "pqPipelineTimeKeyFrameEditor.h"
#include "pqPropertyLinks.h"
#include "pqRenderView.h"
#include "pqSMAdaptor.h"
#include "pqServer.h"
#include "pqServerManagerModel.h"
#include "pqSetName.h"
#include "pqSignalAdaptors.h"
#include "pqTimeKeeper.h"
#include "pqTimelineScrollbar.h"
#include "pqUndoStack.h"

#include "vtkCamera.h"
#include "vtkPVGeneralSettings.h"
#include "vtkSMProperty.h"
#include "vtkSMPropertyHelper.h"
#include "vtkSMProxy.h"
#include "vtkSMRenderViewProxy.h"
#include "vtkSMTrace.h"

#include <cassert>

#if VTK_MODULE_ENABLE_ParaView_pqPython
#include "pqPythonSyntaxHighlighter.h"
#endif

//-----------------------------------------------------------------------------
/**
 * Small custom class to clamp the displayed string according to the range when input is
 * outside the range.
 */
class pqIntClampedValidator : public QIntValidator
{
public:
  pqIntClampedValidator(double bottom, double top, QObject* parent)
    : QIntValidator(bottom, top, parent)
  {
  }

  QValidator::State validate(QString& s, int&) const override
  {
    if (s.isEmpty() || s == "-")
    {
      return QValidator::Intermediate;
    }

    QLocale locale;
    bool ok;
    int d = locale.toInt(s, &ok);
    return (ok && d >= bottom() && d <= top()) ? QValidator::Acceptable : QValidator::Invalid;
  }
};

//-----------------------------------------------------------------------------
class pqAnimationViewWidget::pqInternal
{
public:
  pqInternal() = default;

  ~pqInternal() = default;

  QPointer<pqAnimationScene> Scene;
  pqAnimationWidget* AnimationWidget = nullptr;
  pqAnimationTimeWidget* AnimationTimeWidget = nullptr;
  typedef QMap<QPointer<pqAnimationCue>, pqAnimationTrack*> TrackMapType;
  TrackMapType TrackMap;
  QPointer<QDialog> Editor;
  QComboBox* PlayMode = nullptr;
  QLineEdit* StartTime = nullptr;
  QLabel* StartTimeLabel = nullptr;
  QLineEdit* EndTime = nullptr;
  QLabel* EndTimeLabel = nullptr;
  QLabel* DurationLabel = nullptr;
  QLineEdit* Duration = nullptr;
  QLabel* StrideLabel = nullptr;
  QLineEdit* Stride = nullptr;
  pqPropertyLinks Links;
  pqPropertyLinks DurationLink;
  pqAnimatableProxyComboBox* CreateSource = nullptr;
  pqAnimatablePropertiesComboBox* CreateProperty = nullptr;
  QToolButton* LockEndTime = nullptr;
  QToolButton* LockStartTime = nullptr;
  vtkSMProxy* SelectedCueProxy = nullptr;
  vtkSMProxy* SelectedDataProxy = nullptr;
  int SequenceStrideCache = 1;
  int TimestepStrideCache = 1;

  pqAnimationTrack* findTrack(pqAnimationCue* cue)
  {
    TrackMapType::iterator iter;
    iter = this->TrackMap.find(cue);
    if (iter != this->TrackMap.end())
    {
      return iter.value();
    }
    return nullptr;
  }
  pqAnimationCue* findCue(pqAnimationTrack* track)
  {
    TrackMapType::iterator iter;
    for (iter = this->TrackMap.begin(); iter != this->TrackMap.end(); ++iter)
    {
      if (iter.value() == track)
      {
        return iter.key();
      }
    }
    return nullptr;
  }
  QString cueName(pqAnimationCue* cue)
  {
    if (this->cameraCue(cue))
    {
      vtkSMProxy* pxy = cue->getAnimatedProxy();
      pqServerManagerModel* model = pqApplicationCore::instance()->getServerManagerModel();
      if (pqProxy* animation_pqproxy = model->findItem<pqProxy*>(pxy))
      {
        return QString("Camera - %1").arg(animation_pqproxy->getSMName());
      }

      return "Camera";
    }
    else if (this->pythonCue(cue))
    {
      return "Python";
    }
    else
    {
      pqServerManagerModel* model = pqApplicationCore::instance()->getServerManagerModel();

      vtkSMProxy* pxy = cue->getAnimatedProxy();
      vtkSMProperty* pty = cue->getAnimatedProperty();
      QString p = QCoreApplication::translate("ServerManagerXML", pty->GetXMLLabel());
      if (pqSMAdaptor::getPropertyType(pty) == pqSMAdaptor::MULTIPLE_ELEMENTS)
      {
        p = QString("%1 (%2)").arg(p).arg(cue->getAnimatedPropertyIndex());
      }

      if (pqProxy* animation_pqproxy = model->findItem<pqProxy*>(pxy))
      {
        return QString("%1 - %2").arg(animation_pqproxy->getSMName()).arg(p);
      }

      // could be a helper proxy
      QString helper_key;
      if (pqProxy* pqproxy = pqProxy::findProxyWithHelper(pxy, helper_key))
      {
        vtkSMProperty* prop = pqproxy->getProxy()->GetProperty(helper_key.toUtf8().data());
        if (prop)
        {
          return QString("%1 - %2 - %3")
            .arg(pqproxy->getSMName())
            .arg(QCoreApplication::translate("ServerManagerXML", prop->GetXMLLabel()))
            .arg(p);
        }
        return QString("%1 - %2").arg(pqproxy->getSMName()).arg(p);
      }
    }
    return QString("<%1>").arg(tr("unrecognized"));
  }
  // returns if this is a cue for animating a camera
  bool cameraCue(pqAnimationCue* cue)
  {
    if (cue && QString("CameraAnimationCue") == cue->getProxy()->GetXMLName())
    {
      return true;
    }
    return false;
  }

  /// returns true if the cue is a python cue.
  bool pythonCue(pqAnimationCue* cue)
  {
    if (QString("PythonAnimationCue") == cue->getProxy()->GetXMLName())
    {
      return true;
    }
    return false;
  }

  int numberOfTicks()
  {
    vtkSMProxy* pxy = this->Scene->getProxy();
    QString mode = pqSMAdaptor::getEnumerationProperty(pxy->GetProperty("PlayMode")).toString();

    int num = 0;

    if (mode == "Sequence")
    {
      num = pqSMAdaptor::getElementProperty(pxy->GetProperty("NumberOfFrames")).toInt();
    }
    else if (mode == "Snap To TimeSteps")
    {
      num = this->Scene->getTimeSteps().size();
    }
    return num;
  }

  QList<double> ticks()
  {
    vtkSMProxy* pxy = this->Scene->getProxy();
    QString mode = pqSMAdaptor::getEnumerationProperty(pxy->GetProperty("PlayMode")).toString();
    if (mode == "Snap To TimeSteps")
    {
      return this->Scene->getTimeSteps();
    }
    return QList<double>();
  }
};

//-----------------------------------------------------------------------------
pqAnimationViewWidget::pqAnimationViewWidget(QWidget* _parent)
  : QWidget(_parent)
{
  this->Internal = new pqAnimationViewWidget::pqInternal();
  QVBoxLayout* vboxlayout = new QVBoxLayout(this);
  vboxlayout->setMargin(2);
  vboxlayout->setSpacing(2);

  QHBoxLayout* hboxlayout = new QHBoxLayout;
  vboxlayout->addLayout(hboxlayout);
  hboxlayout->setMargin(0);
  hboxlayout->setSpacing(2);

  hboxlayout->addWidget(new QLabel(tr("Mode:"), this));
  this->Internal->PlayMode = new QComboBox(this) << pqSetName("PlayMode");
  this->Internal->PlayMode->addItem(tr("Snap to Timesteps"));
  hboxlayout->addWidget(this->Internal->PlayMode);
  this->Internal->AnimationTimeWidget = new pqAnimationTimeWidget(this);
  this->Internal->AnimationTimeWidget->setPlayModeReadOnly(true);
  hboxlayout->addWidget(this->Internal->AnimationTimeWidget);

  this->Internal->StartTimeLabel = new QLabel(tr("Start Time:"), this);
  hboxlayout->addWidget(this->Internal->StartTimeLabel);
  this->Internal->StartTime = new QLineEdit(this) << pqSetName("StartTime");
  this->Internal->StartTime->setMinimumWidth(30);
  this->Internal->StartTime->setValidator(new QDoubleValidator(this->Internal->StartTime));
  hboxlayout->addWidget(this->Internal->StartTime);
  this->Internal->LockStartTime = new QToolButton(this) << pqSetName("LockStartTime");
  this->Internal->LockStartTime->setIcon(QIcon(":pqWidgets/Icons/pqLock24.png"));
  this->Internal->LockStartTime->setToolTip("<html>" +
    tr("Lock the start time to keep ParaView from changing it "
       "as available data times change") +
    "</html>");
  this->Internal->LockStartTime->setStatusTip("<html>" +
    tr("Lock the start time to keep ParaView from changing it "
       "as available data times change") +
    "</html>");
  this->Internal->LockStartTime->setCheckable(true);
  hboxlayout->addWidget(this->Internal->LockStartTime);
  this->Internal->EndTimeLabel = new QLabel(tr("End Time:"), this);
  hboxlayout->addWidget(this->Internal->EndTimeLabel);
  this->Internal->EndTime = new QLineEdit(this) << pqSetName("EndTime");
  this->Internal->EndTime->setMinimumWidth(30);
  this->Internal->EndTime->setValidator(new QDoubleValidator(this->Internal->EndTime));
  hboxlayout->addWidget(this->Internal->EndTime);
  this->Internal->LockEndTime = new QToolButton(this) << pqSetName("LockEndTime");
  this->Internal->LockEndTime->setIcon(QIcon(":pqWidgets/Icons/pqLock24.png"));
  this->Internal->LockEndTime->setToolTip("<html>" +
    tr("Lock the end time to keep ParaView from changing it "
       "as available data times change") +
    "</html>");
  this->Internal->LockEndTime->setStatusTip("<html>" +
    tr("Lock the end time to keep ParaView from changing it "
       "as available data times change") +
    "</html>");
  this->Internal->LockEndTime->setCheckable(true);
  hboxlayout->addWidget(this->Internal->LockEndTime);
  this->Internal->DurationLabel = new QLabel(this);
  hboxlayout->addWidget(this->Internal->DurationLabel);
  this->Internal->Duration = new QLineEdit(this) << pqSetName("Duration");
  this->Internal->Duration->setMinimumWidth(30);
  this->Internal->Duration->setValidator(
    new QIntValidator(1, static_cast<int>(~0u >> 1), this->Internal->Duration));
  hboxlayout->addWidget(this->Internal->Duration);
  hboxlayout->addSpacing(5);
  this->Internal->StrideLabel = new QLabel(tr("Stride"), this);
  hboxlayout->addWidget(this->Internal->StrideLabel);
  this->Internal->Stride = new QLineEdit("1", this) << pqSetName("Stride");
  this->Internal->Stride->setMinimumWidth(30);
  this->Internal->Stride->setValidator(new pqIntClampedValidator(1, 1, this->Internal->Stride));
  hboxlayout->addWidget(this->Internal->Stride, 1, Qt::AlignLeft);
  hboxlayout->addStretch();

  this->Internal->AnimationWidget = new pqAnimationWidget(this) << pqSetName("pqAnimationWidget");
  this->Internal->AnimationWidget->animationModel()->setInteractive(true);
  this->Internal->AnimationWidget->animationModel()->setTimePrecision(
    vtkPVGeneralSettings::GetInstance()->GetAnimationTimePrecision());
  this->Internal->AnimationWidget->animationModel()->setTimeNotation(
    vtkPVGeneralSettings::GetInstance()->GetAnimationTimeNotation());

  pqCoreUtilities::connect(vtkPVGeneralSettings::GetInstance(), vtkCommand::ModifiedEvent, this,
    SLOT(generalSettingsChanged()));

  QWidget* w = this->Internal->AnimationWidget->createDeleteWidget();

  this->Internal->CreateSource = new pqAnimatableProxyComboBox(w) << pqSetName("ProxyCombo");
#if VTK_MODULE_ENABLE_ParaView_pqPython
  this->Internal->CreateSource->addProxy(0, "Python", nullptr);
#endif
  this->Internal->CreateProperty = new pqAnimatablePropertiesComboBox(w)
    << pqSetName("PropertyCombo");
  this->Internal->CreateSource->setSizeAdjustPolicy(QComboBox::AdjustToContents);
  this->Internal->CreateProperty->setSizeAdjustPolicy(QComboBox::AdjustToContents);
  QHBoxLayout* l = new QHBoxLayout(w);
  l->setMargin(0);
  l->addSpacing(6);
  l->addWidget(this->Internal->CreateSource);
  l->addWidget(this->Internal->CreateProperty);
  l->addStretch();

  QObject::connect(this->Internal->AnimationWidget, SIGNAL(trackSelected(pqAnimationTrack*)), this,
    SLOT(trackSelected(pqAnimationTrack*)));
  QObject::connect(this->Internal->AnimationWidget, SIGNAL(deleteTrackClicked(pqAnimationTrack*)),
    this, SLOT(deleteTrack(pqAnimationTrack*)));
  QObject::connect(this->Internal->AnimationWidget, SIGNAL(enableTrackClicked(pqAnimationTrack*)),
    this, SLOT(toggleTrackEnabled(pqAnimationTrack*)));
  QObject::connect(
    this->Internal->AnimationWidget, SIGNAL(createTrackClicked()), this, SLOT(createTrack()));

  QObject::connect(this->Internal->AnimationWidget->animationModel(),
    SIGNAL(currentTimeSet(double)), this, SLOT(setCurrentTime(double)));
  QObject::connect(this->Internal->AnimationWidget->animationModel(),
    SIGNAL(keyFrameTimeChanged(pqAnimationTrack*, pqAnimationKeyFrame*, int, double)), this,
    SLOT(setKeyFrameTime(pqAnimationTrack*, pqAnimationKeyFrame*, int, double)));

  QObject::connect(
    &pqActiveObjects::instance(), SIGNAL(viewChanged(pqView*)), this, SLOT(setActiveView(pqView*)));

  QObject::connect(&pqActiveObjects::instance(), SIGNAL(sourceChanged(pqPipelineSource*)), this,
    SLOT(setCurrentSelection(pqPipelineSource*)));

  QObject::connect(&pqActiveObjects::instance(), SIGNAL(serverChanged(pqServer*)), this,
    SLOT(onSceneCuesChanged()));

  QObject::connect(this->Internal->CreateSource, SIGNAL(currentProxyChanged(vtkSMProxy*)), this,
    SLOT(setCurrentProxy(vtkSMProxy*)));

  QObject::connect(this->Internal->Stride, &QLineEdit::editingFinished, this,
    &pqAnimationViewWidget::onStrideChanged);
  QObject::connect(this->Internal->Duration, &QLineEdit::editingFinished, this,
    &pqAnimationViewWidget::updateStrideRange);

  pqTimelineScrollbar* timelineScrollbar = new pqTimelineScrollbar(this);
  timelineScrollbar->linkSpacing(this->Internal->AnimationWidget);
  timelineScrollbar->setAnimationModel(this->Internal->AnimationWidget->animationModel());

  vboxlayout->addWidget(timelineScrollbar);

  vboxlayout->addWidget(this->Internal->AnimationWidget);
}

//-----------------------------------------------------------------------------
pqAnimationViewWidget::~pqAnimationViewWidget()
{
  delete this->Internal;
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::setScene(pqAnimationScene* scene)
{
  if (this->Internal->Scene)
  {
    this->Internal->Links.removeAllPropertyLinks();
    QObject::disconnect(this->Internal->Scene, nullptr, this, nullptr);

    pqComboBoxDomain* d0 = this->Internal->PlayMode->findChild<pqComboBoxDomain*>("ComboBoxDomain");
    delete d0;
    pqSignalAdaptorComboBox* adaptor =
      this->Internal->PlayMode->findChild<pqSignalAdaptorComboBox*>("ComboBoxAdaptor");
    delete adaptor;
  }
  this->Internal->Scene = scene;
  if (this->Internal->Scene)
  {
    pqComboBoxDomain* d0 =
      new pqComboBoxDomain(this->Internal->PlayMode, scene->getProxy()->GetProperty("PlayMode"));
    d0->setObjectName("ComboBoxDomain");
    pqSignalAdaptorComboBox* adaptor = new pqSignalAdaptorComboBox(this->Internal->PlayMode);
    adaptor->setObjectName("ComboBoxAdaptor");
    this->Internal->Links.addTraceablePropertyLink(adaptor, "currentText",
      SIGNAL(currentTextChanged(const QString&)), scene->getProxy(),
      scene->getProxy()->GetProperty("PlayMode"));

    // connect time widget to the scene
    this->Internal->AnimationTimeWidget->setAnimationScene(scene);
    // connect start time
    this->Internal->Links.addTraceablePropertyLink(this->Internal->StartTime, "text",
      SIGNAL(editingFinished()), scene->getProxy(), scene->getProxy()->GetProperty("StartTime"));
    // connect end time
    this->Internal->Links.addTraceablePropertyLink(this->Internal->EndTime, "text",
      SIGNAL(editingFinished()), scene->getProxy(), scene->getProxy()->GetProperty("EndTime"));
    // connect lock start time.
    this->Internal->Links.addTraceablePropertyLink(this->Internal->LockStartTime, "checked",
      SIGNAL(toggled(bool)), scene->getProxy(), scene->getProxy()->GetProperty("LockStartTime"));
    this->Internal->Links.addTraceablePropertyLink(this->Internal->LockEndTime, "checked",
      SIGNAL(toggled(bool)), scene->getProxy(), scene->getProxy()->GetProperty("LockEndTime"));

    QObject::connect(
      scene, &pqAnimationScene::cuesChanged, this, &pqAnimationViewWidget::onSceneCuesChanged);
    QObject::connect(scene, &pqAnimationScene::clockTimeRangesChanged, this,
      &pqAnimationViewWidget::updateSceneTimeRange);
    QObject::connect(
      scene, &pqAnimationScene::timeStepsChanged, this, &pqAnimationViewWidget::updateTicks);
    QObject::connect(
      scene, &pqAnimationScene::timeStepsChanged, this, &pqAnimationViewWidget::updateStrideRange);
    QObject::connect(
      scene, &pqAnimationScene::frameCountChanged, this, &pqAnimationViewWidget::updateTicks);
    QObject::connect(
      scene, &pqAnimationScene::animationTime, this, &pqAnimationViewWidget::updateSceneTime);
    QObject::connect(
      scene, &pqAnimationScene::playModeChanged, this, &pqAnimationViewWidget::updatePlayMode);
    QObject::connect(
      scene, &pqAnimationScene::playModeChanged, this, &pqAnimationViewWidget::updateTicks);
    QObject::connect(
      scene, &pqAnimationScene::playModeChanged, this, &pqAnimationViewWidget::updateSceneTime);
    QObject::connect(
      scene, &pqAnimationScene::timeLabelChanged, this, &pqAnimationViewWidget::onTimeLabelChanged);

    this->updateSceneTimeRange();
    this->updateSceneTime();
    this->updatePlayMode();
    this->updateTicks();
    this->onTimeLabelChanged();
    this->updateStrideRange();
  }
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::onSceneCuesChanged()
{
  if (!this->Internal->Scene)
  {
    // No scene, so do nothing
    return;
  }

  QSet<pqAnimationCue*> cues = this->Internal->Scene->getCues();
  pqAnimationModel* animModel = this->Internal->AnimationWidget->animationModel();

  pqInternal::TrackMapType oldCues = this->Internal->TrackMap;
  pqInternal::TrackMapType::iterator iter;

  // add new tracks
  Q_FOREACH (pqAnimationCue* cue, cues)
  {
    if (cue == nullptr)
    {
      continue;
    }
    QString completeName = this->Internal->cueName(cue);

    iter = this->Internal->TrackMap.find(cue);

    if (iter == this->Internal->TrackMap.end())
    {
      pqAnimationTrack* track = animModel->addTrack();
      if (completeName.startsWith("TimeKeeper"))
      {
        track->setDeletable(false);
      }
      this->Internal->TrackMap.insert(cue, track);
      track->setProperty(completeName);
      QObject::connect(
        cue, &pqAnimationCue::keyframesModified, this, [=]() { this->keyFramesChanged(cue); });
      QObject::connect(cue, SIGNAL(enabled(bool)), track, SLOT(setEnabled(bool)));
      track->setEnabled(cue->isEnabled());

      // this ensures that the already present keyframes are loaded currently
      // (which happens when loading state files).
      this->keyFramesChanged(cue);
    }
    else
    {
      oldCues.remove(cue);
    }
  }

  // remove old tracks
  for (iter = oldCues.begin(); iter != oldCues.end(); iter++)
  {
    animModel->removeTrack(iter.value());
    this->Internal->TrackMap.remove(iter.key());
    if (iter.key())
    {
      QObject::disconnect(iter.key(), SIGNAL(keyframesModified()), nullptr, nullptr);
    }
  }
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::keyFramesChanged(QObject* cueObject)
{
  pqAnimationCue* cue = qobject_cast<pqAnimationCue*>(cueObject);
  pqAnimationTrack* track = this->Internal->findTrack(cue);

  QList<vtkSMProxy*> keyFrames = cue->getKeyFrames();

  bool camera = this->Internal->cameraCue(cue);

  // clean out old ones
  while (track->count())
  {
    track->removeKeyFrame(track->keyFrame(0));
  }

  for (int j = 0; j < keyFrames.count() - 1; j++)
  {
    QIcon icon;
    QVariant startValue;
    QVariant endValue;

    double startTime =
      pqSMAdaptor::getElementProperty(keyFrames[j]->GetProperty("KeyTime")).toDouble();
    double endTime =
      pqSMAdaptor::getElementProperty(keyFrames[j + 1]->GetProperty("KeyTime")).toDouble();

    if (!camera)
    {
      QVariant interpolation =
        pqSMAdaptor::getEnumerationProperty(keyFrames[j]->GetProperty("Type"));
      if (interpolation == "Boolean")
        interpolation = "Step";
      else if (interpolation == "Sinusoid")
        interpolation = "Sinusoidal";
      QString iconstr =
        QString(":pqWidgets/Icons/pq%1%2.png").arg(interpolation.toString()).arg(16);
      icon = QIcon(iconstr);

      startValue = pqSMAdaptor::getElementProperty(keyFrames[j]->GetProperty("KeyValues"));
      endValue = pqSMAdaptor::getElementProperty(keyFrames[j + 1]->GetProperty("KeyValues"));
    }

    pqAnimationKeyFrame* newFrame = track->addKeyFrame();
    newFrame->setNormalizedStartTime(startTime);
    newFrame->setNormalizedEndTime(endTime);
    newFrame->setStartValue(startValue);
    newFrame->setEndValue(endValue);
    newFrame->setIcon(QIcon(icon));
  }
  pqAnimationModel* animModel = this->Internal->AnimationWidget->animationModel();
  animModel->zoomTrack(track);
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::updateSceneTimeRange()
{
  pqAnimationModel* animModel = this->Internal->AnimationWidget->animationModel();
  QPair<double, double> timeRange = this->Internal->Scene->getClockTimeRange();
  animModel->setStartTime(timeRange.first);
  animModel->setEndTime(timeRange.second);
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::updateSceneTime()
{
  double time = this->Internal->Scene->getAnimationTime();

  pqAnimationModel* animModel = this->Internal->AnimationWidget->animationModel();
  animModel->setCurrentTime(time);
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::setCurrentTime(double t)
{
  BEGIN_UNDO_EXCLUDE();

  vtkSMProxy* animationScene = this->Internal->Scene->getProxy();
  {
    // Use another scope to prevent modifications to the TimeKeeper from
    // being traced.
    SM_SCOPED_TRACE(PropertiesModified).arg("proxy", animationScene);
    vtkSMPropertyHelper(animationScene, "AnimationTime").Set(t);
  }
  animationScene->UpdateVTKObjects();

  END_UNDO_EXCLUDE();
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::setKeyFrameTime(
  pqAnimationTrack* track, pqAnimationKeyFrame* kf, int edge, double time)
{
  pqAnimationCue* cue = this->Internal->findCue(track);
  if (!cue)
  {
    return;
  }
  QList<vtkSMProxy*> keyFrames = cue->getKeyFrames();
  int i = 0;
  for (i = 0; i < track->count(); i++)
  {
    if (track->keyFrame(i) == kf)
    {
      break;
    }
  }
  if (edge)
  {
    i++;
  }

  if (i < keyFrames.size())
  {
    SM_SCOPED_TRACE(PropertiesModified).arg(keyFrames[i]);
    QPair<double, double> timeRange = this->Internal->Scene->getClockTimeRange();
    double normTime = (time - timeRange.first) / (timeRange.second - timeRange.first);
    pqSMAdaptor::setElementProperty(keyFrames[i]->GetProperty("KeyTime"), normTime);
    keyFrames[i]->UpdateVTKObjects();
  }
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::trackSelected(pqAnimationTrack* track)
{
  pqAnimationCue* cue = this->Internal->findCue(track);
  if (!cue)
  {
    return;
  }

  if (this->Internal->Editor)
  {
    this->Internal->Editor->raise();
    return;
  }

  if (track->property().toString().startsWith("TimeKeeper"))
  {
    this->Internal->Editor = new pqPipelineTimeKeyFrameEditor(this->Internal->Scene, cue, nullptr);
    this->Internal->Editor->resize(600, 400);
  }
  else if (this->Internal->pythonCue(cue))
  {
    QDialog dialog(this);
    Ui::PythonAnimationCue ui;
    ui.setupUi(&dialog);
#if VTK_MODULE_ENABLE_ParaView_pqPython
    pqPythonSyntaxHighlighter* highlighter = new pqPythonSyntaxHighlighter(ui.script, *ui.script);
    highlighter->ConnectHighligter();
#endif
    ui.script->setPlainText(vtkSMPropertyHelper(cue->getProxy(), "Script").GetAsString());
    if (dialog.exec() == QDialog::Accepted)
    {
      vtkSMPropertyHelper(cue->getProxy(), "Script").Set(ui.script->toPlainText().toUtf8().data());
      cue->getProxy()->UpdateVTKObjects();
    }
    return;
  }
  else
  {
    int mode = -1;
    if (cue->getProxy()->GetProperty("Mode"))
    {
      mode = vtkSMPropertyHelper(cue->getProxy(), "Mode").GetAsInt();
    }

    this->Internal->Editor = new QDialog;
    QVBoxLayout* l = new QVBoxLayout(this->Internal->Editor);
    QDialogButtonBox* buttons =
      new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel);

    // FOLLOW_DATA mode
    if (mode == 2)
    {
      // show a combo-box allowing the user to select the data source to follow
      QFormLayout* layout_ = new QFormLayout;
      pqAnimatableProxyComboBox* comboBox = new pqAnimatableProxyComboBox(this->Internal->Editor);
      this->Internal->SelectedCueProxy = cue->getProxy();
      this->Internal->SelectedDataProxy =
        vtkSMPropertyHelper(cue->getProxy(), "DataSource").GetAsProxy();
      comboBox->setCurrentIndex(comboBox->findProxy(this->Internal->SelectedDataProxy));
      connect(comboBox, SIGNAL(currentProxyChanged(vtkSMProxy*)), this,
        SLOT(selectedDataProxyChanged(vtkSMProxy*)));
      connect(
        this->Internal->Editor, SIGNAL(accepted()), this, SLOT(changeDataProxyDialogAccepted()));
      layout_->addRow(tr("Data Source to Follow:"), comboBox);
      l->addLayout(layout_);
      this->Internal->Editor->setWindowTitle(tr("Select Data Source"));
    }
    else
    {
      pqKeyFrameEditor* editor = new pqKeyFrameEditor(this->Internal->Scene, cue,
        tr("Editing ") + this->Internal->cueName(cue), this->Internal->Editor);

      l->addWidget(editor);

      auto apply = buttons->addButton(QDialogButtonBox::Apply);

      connect(this->Internal->Editor, SIGNAL(accepted()), editor, SLOT(writeKeyFrameData()));
      QObject::connect(apply, &QPushButton::clicked, editor, &pqKeyFrameEditor::writeKeyFrameData);

      QObject::connect(apply, &QPushButton::clicked, [=]() { apply->setEnabled(false); });
      QObject::connect(editor, &pqKeyFrameEditor::modified, [=]() { apply->setEnabled(true); });

      this->Internal->Editor->setWindowTitle(tr("Animation Keyframes"));
      this->Internal->Editor->resize(600, 400);
    }

    connect(buttons, SIGNAL(accepted()), this->Internal->Editor, SLOT(accept()));
    connect(buttons, SIGNAL(rejected()), this->Internal->Editor, SLOT(reject()));
    l->addWidget(buttons);
  }

  this->Internal->Editor->setAttribute(Qt::WA_QuitOnClose, false);
  this->Internal->Editor->setAttribute(Qt::WA_DeleteOnClose);

  this->Internal->Editor->show();
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::updatePlayMode()
{
  pqAnimationModel* animModel = this->Internal->AnimationWidget->animationModel();
  vtkSMProxy* pxy = this->Internal->Scene->getProxy();

  QString mode = pqSMAdaptor::getEnumerationProperty(pxy->GetProperty("PlayMode")).toString();

  this->Internal->DurationLink.removeAllPropertyLinks();

  this->Internal->AnimationTimeWidget->setPlayMode(mode);

  if (mode == "Real Time")
  {
    QString promptMessage =
      tr("'Real time' mode is deprecated and maybe removed in a near future.\n Prefer 'Snap to "
         "Timestep' or 'Sequence' if you need to interpolate between existing timesteps.");
    pqCoreUtilities::promptUser("pqAnimationViewWidget::updatePlayMode", QMessageBox::Warning,
      tr("Real Time mode is deprecated."), promptMessage, QMessageBox::Ok | QMessageBox::Save);

    animModel->setMode(pqAnimationModel::Real);

    this->Internal->Stride->setVisible(false);
    this->Internal->StrideLabel->setVisible(false);
    this->Internal->Duration->setVisible(true);
    this->Internal->DurationLabel->setVisible(true);
    this->Internal->StartTime->setVisible(true);
    this->Internal->StartTimeLabel->setVisible(true);
    this->Internal->EndTime->setVisible(true);
    this->Internal->EndTimeLabel->setVisible(true);
    this->Internal->LockStartTime->setVisible(true);
    this->Internal->LockEndTime->setVisible(true);

    this->Internal->StartTime->setEnabled(true);
    this->Internal->EndTime->setEnabled(true);
    this->Internal->AnimationTimeWidget->setEnabled(true);
    this->Internal->Duration->setEnabled(true);
    this->Internal->DurationLabel->setEnabled(true);
    this->Internal->StrideLabel->setEnabled(false);
    this->Internal->Stride->setEnabled(false);

    this->Internal->DurationLabel->setText(tr("Duration (s):"));
    this->Internal->DurationLink.addTraceablePropertyLink(this->Internal->Duration, "text",
      SIGNAL(editingFinished()), this->Internal->Scene->getProxy(),
      this->Internal->Scene->getProxy()->GetProperty("Duration"));
  }
  else if (mode == "Sequence")
  {
    animModel->setMode(pqAnimationModel::Sequence);

    this->Internal->Stride->setVisible(true);
    this->Internal->StrideLabel->setVisible(true);
    this->Internal->Duration->setVisible(true);
    this->Internal->DurationLabel->setVisible(true);
    this->Internal->StartTime->setVisible(true);
    this->Internal->StartTimeLabel->setVisible(true);
    this->Internal->EndTime->setVisible(true);
    this->Internal->EndTimeLabel->setVisible(true);
    this->Internal->LockStartTime->setVisible(true);
    this->Internal->LockEndTime->setVisible(true);

    this->Internal->StartTime->setEnabled(true);
    this->Internal->EndTime->setEnabled(true);
    this->Internal->Duration->setEnabled(true);
    this->Internal->DurationLabel->setEnabled(true);
    this->Internal->StrideLabel->setEnabled(true);
    this->Internal->Stride->setEnabled(true);

    this->Internal->DurationLabel->setText(tr("No. Frames:"));
    this->Internal->DurationLink.addTraceablePropertyLink(this->Internal->Duration, "text",
      SIGNAL(editingFinished()), this->Internal->Scene->getProxy(),
      this->Internal->Scene->getProxy()->GetProperty("NumberOfFrames"));
    this->Internal->Stride->setText(QString::number(this->Internal->SequenceStrideCache));
    Q_EMIT this->Internal->Stride->editingFinished();
  }
  else if (mode == "Snap To TimeSteps")
  {
    animModel->setMode(pqAnimationModel::Custom);

    this->Internal->Stride->setVisible(true);
    this->Internal->StrideLabel->setVisible(true);
    this->Internal->Duration->setVisible(false);
    this->Internal->DurationLabel->setVisible(false);
    this->Internal->StartTime->setVisible(false);
    this->Internal->StartTimeLabel->setVisible(false);
    this->Internal->EndTime->setVisible(false);
    this->Internal->EndTimeLabel->setVisible(false);
    this->Internal->LockStartTime->setVisible(false);
    this->Internal->LockEndTime->setVisible(false);

    this->Internal->Stride->setEnabled(true);
    this->Internal->StrideLabel->setEnabled(true);
    this->Internal->Duration->setEnabled(false);
    this->Internal->DurationLabel->setEnabled(false);
    this->Internal->StartTime->setEnabled(false);
    this->Internal->EndTime->setEnabled(false);

    this->Internal->Stride->setText(QString::number(this->Internal->TimestepStrideCache));
    Q_EMIT this->Internal->Stride->editingFinished();
  }
  else
  {
    qWarning("Unrecognized play mode");
  }
  this->updateStrideRange();
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::updateTicks()
{
  pqAnimationModel* animModel = this->Internal->AnimationWidget->animationModel();
  if (animModel->mode() == pqAnimationModel::Custom)
  {
    QList<double> ticks = this->Internal->ticks();
    double* dticks = new double[ticks.size() + 1];
    for (int cc = 0; cc < ticks.size(); cc++)
    {
      dticks[cc] = ticks[cc];
    }
    animModel->setTickMarks(ticks.size(), dticks);
    delete[] dticks;
  }
  else
  {
    animModel->setTicks(this->Internal->numberOfTicks());
  }
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::updateStrideRange()
{
  int newMax = 1;
  vtkSMProxy* pxy = this->Internal->Scene->getProxy();
  const QString& mode =
    pqSMAdaptor::getEnumerationProperty(pxy->GetProperty("PlayMode")).toString();
  if (mode == "Sequence")
  {
    newMax = this->Internal->Duration->text().toInt();
  }
  else if (mode == "Snap To TimeSteps")
  {
    newMax = this->Internal->Scene->getTimeSteps().size();
  }
  auto* widget = this->Internal->Stride;
  widget->setValidator(new pqIntClampedValidator(1, newMax, widget));
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::toggleTrackEnabled(pqAnimationTrack* track)
{
  pqAnimationCue* cue = this->Internal->findCue(track);
  if (!cue)
  {
    return;
  }
  BEGIN_UNDO_SET(tr("Toggle Animation Track"));
  SM_SCOPED_TRACE(PropertiesModified).arg(cue->getProxy());
  cue->setEnabled(!track->isEnabled());
  END_UNDO_SET();
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::deleteTrack(pqAnimationTrack* track)
{
  pqAnimationCue* cue = this->Internal->findCue(track);
  if (!cue)
  {
    return;
  }
  BEGIN_UNDO_SET(tr("Remove Animation Track"));
  SM_SCOPED_TRACE(Delete).arg(cue->getProxy());
  this->Internal->Scene->removeCue(cue);
  END_UNDO_SET();
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::setActiveView(pqView* view)
{
  pqRenderView* rview = qobject_cast<pqRenderView*>(view);
  this->Internal->CreateSource->removeProxy("Camera");
  if (rview && this->Internal->CreateSource->findText("Camera") == -1)
  {
    this->Internal->CreateSource->addProxy(0, "Camera", rview->getProxy());
  }
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::setCurrentSelection(pqPipelineSource* pxy)
{
  if (pxy)
  {
    int idx = this->Internal->CreateSource->findProxy(pxy->getProxy());
    if (idx != -1)
    {
      this->Internal->CreateSource->setCurrentIndex(idx);
    }
  }
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::setCurrentProxy(vtkSMProxy* pxy)
{
  if (vtkSMRenderViewProxy::SafeDownCast(pxy))
  {
    this->Internal->CreateProperty->setSourceWithoutProperties(pxy);
    // add camera animation modes as properties for creating the camera
    // animation track.
    this->Internal->CreateProperty->addSMProperty("Follow Path", "path", 0);
    this->Internal->CreateProperty->addSMProperty("Follow Data", "data", 0);
    this->Internal->CreateProperty->addSMProperty("Interpolate cameras", "camera", 0);
  }
  else
  {
    this->Internal->CreateProperty->setSource(pxy);
  }
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::createTrack()
{
  vtkSMRenderViewProxy* ren =
    vtkSMRenderViewProxy::SafeDownCast(this->Internal->CreateSource->getCurrentProxy());
  // Need to create new cue for this property.
  vtkSMProxy* curProxy = this->Internal->CreateProperty->getCurrentProxy();
  QString pname = this->Internal->CreateProperty->getCurrentPropertyName();
  int pindex = this->Internal->CreateProperty->getCurrentIndex();

  // used for camera tracks.
  QString mode = this->Internal->CreateProperty->getCurrentPropertyName();

  if (ren)
  {
    curProxy = ren;
    pname = QString();
    pindex = 0;
  }

  if (!curProxy)
  {
// curProxy == nullptr is only used for "Python" track for now. Of course,
// we only support that when python is enabled.
// we allow creating as many python tracks as needed, hence we don't check
// if there exists a track already (which is the case with others).
#if VTK_MODULE_ENABLE_ParaView_pqPython
    this->createPythonTrack();
#endif
    return;
  }

  // check that we don't already have one
  Q_FOREACH (pqAnimationCue* cue, this->Internal->TrackMap.keys())
  {
    if (cue->getAnimatedProxy() == nullptr)
    {
      continue; // skip Python tracks.
    }
    if (cue->getAnimatedProxy() == curProxy &&
      cue->getAnimatedProxy()->GetPropertyName(cue->getAnimatedProperty()) == pname &&
      cue->getAnimatedPropertyIndex() == pindex)
    {
      return;
    }
  }

  BEGIN_UNDO_SET(tr("Add Animation Track"));

  // This will create the cue and initialize it with default keyframes.
  pqAnimationCue* cue = this->Internal->Scene->createCue(
    curProxy, pname.toUtf8().data(), pindex, ren ? "CameraAnimationCue" : "KeyFrameAnimationCue");

  SM_SCOPED_TRACE(CreateAnimationTrack).arg("cue", cue->getProxy());

  if (ren)
  {
    if (mode == "path")
    {
      // Setup default animation to revolve around the selected objects (if any)
      // in a plane normal to the current view-up vector.
      pqSMAdaptor::setElementProperty(
        cue->getProxy()->GetProperty("Mode"), 1); // PATH-based animation.
    }
    else if (mode == "data")
    {
      pqSMAdaptor::setElementProperty(
        cue->getProxy()->GetProperty("Mode"), 2); // DATA-based animation.

      // set the data source for the follow-data animation
      pqPipelineSource* source = pqActiveObjects::instance().activeSource();
      if (source)
      {
        pqSMAdaptor::setProxyProperty(
          cue->getProxy()->GetProperty("DataSource"), source->getProxy());
      }
    }
    else
    {
      pqSMAdaptor::setElementProperty(
        cue->getProxy()->GetProperty("Mode"), 0); // non-PATH-based animation.

      pqSMAdaptor::setElementProperty(cue->getProxy()->GetProperty("Interpolation"), 1);
    }
    cue->getProxy()->UpdateVTKObjects();
  }

  END_UNDO_SET();
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::createPythonTrack()
{
#if VTK_MODULE_ENABLE_ParaView_pqPython
  BEGIN_UNDO_SET(tr("Add Animation Track"));

  pqAnimationCue* cue = this->Internal->Scene->createCue("PythonAnimationCue");
  assert(cue != nullptr);
  (void)cue;
  END_UNDO_SET();
#else
  qCritical() << "Python support not enabled. Please recompile ParaView "
                 "with Python enabled.";
#endif
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::onTimeLabelChanged()
{
  QString timeName = "Time";
  if (this->Internal->Scene)
  {
    timeName = pqSMAdaptor::getElementProperty(
      this->Internal->Scene->getServer()->getTimeKeeper()->getProxy()->GetProperty("TimeLabel"))
                 .toString();
  }

  // Update labels
  this->Internal->StartTimeLabel->setText(tr("Start %1:").arg(timeName));
  this->Internal->EndTimeLabel->setText(tr("End %1:").arg(timeName));
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::selectedDataProxyChanged(vtkSMProxy* proxy)
{
  this->Internal->SelectedDataProxy = proxy;
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::changeDataProxyDialogAccepted()
{
  if (!this->Internal->SelectedCueProxy)
  {
    return;
  }

  // set the proxy property
  vtkSMProxy* currentDataProxy =
    vtkSMPropertyHelper(this->Internal->SelectedCueProxy, "DataSource").GetAsProxy();
  if (this->Internal->SelectedDataProxy != currentDataProxy)
  {
    vtkSMPropertyHelper(this->Internal->SelectedCueProxy, "DataSource")
      .Set(this->Internal->SelectedDataProxy);
    this->Internal->SelectedCueProxy->UpdateVTKObjects();
  }
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::generalSettingsChanged()
{
  this->Internal->AnimationWidget->animationModel()->setTimePrecision(
    vtkPVGeneralSettings::GetInstance()->GetAnimationTimePrecision());
  this->Internal->AnimationWidget->animationModel()->setTimeNotation(
    vtkPVGeneralSettings::GetInstance()->GetAnimationTimeNotation());
}

//-----------------------------------------------------------------------------
void pqAnimationViewWidget::onStrideChanged()
{
  int strideValue = this->Internal->Stride->text().toInt();
  vtkSMProxy* proxy = this->Internal->Scene->getProxy();
  SM_SCOPED_TRACE(PropertiesModified).arg(proxy);
  vtkSMPropertyHelper(proxy->GetProperty("Stride"), false).Set(strideValue);
  proxy->UpdateProperty("Stride");

  const QString& mode = this->Internal->PlayMode->currentText();
  if (mode == "Sequence")
  {
    this->Internal->SequenceStrideCache = strideValue;
  }
  else if (mode == "Snap To TimeSteps")
  {
    this->Internal->TimestepStrideCache = strideValue;
  }
}
