// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

// Modifications:
//   Tom Fogal, Sat Feb  7 18:21:59 EST 2009
//   Added missing includes and modernize them.

#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <string>
#include <vector>
#include <silo.h>
#include <visitstream.h>

// suppress the following since silo uses char * in its API
#if defined(__clang__)
# pragma clang diagnostic ignored "-Wwritable-strings"
#elif defined(__GNUC__)
# pragma GCC diagnostic ignored "-Wwrite-strings"
#endif

using std::cerr;
using std::endl;

//
// Pasted from an XPM file that I made in gimp.
//
    
/* XPM */
static char * lowfrac_xpm[] = {
"300 100 11 1",
"     c None",
".    c #FFFFFF",
"+    c #555555",
"@    c #8E8E8E",
"#    c #1D1D1D",
"$    c #000000",
"%    c #E3E3E3",
"&    c #727272",
"*    c #AAAAAA",
"=    c #393939",
"-    c #C7C7C7",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"....................+++++++++.................................................................................@++#$+.......................................................................%&+$#+%....................................................%*....................................................",
"....................++#$$$#++.................................................................................@+=$$+......................................................................%#$+*&$#%..................................................%#$=...................................................",
".......................$$$.......................................................................................$$+......................................................................=$+..&$$*.........................................#*.......*$$$...................................................",
".......................$$$.......................................................................................$$+.....................................................................-$$-..&$$-.........................................$*........+$@...................................................",
".......................$$$.......................................................................................$$+.....................................................................@$$....*-.........................................-$*..............................................................",
".......................$$$.......................................................................................$$+.....................................................................+$$...............................................&$*..............................................................",
".......................$$$.......................................................................................$$+.....................................................................+$$..............................................*$$*..............................................................",
".......................$$$.................-**......******%%******.%*****........-******..-****-.....%**%........$$+...-++++-..%@++++...........-*-.....-*-...........%**%.............**=$$***.........-*%......%***%...........%**%...-&$$$&***%....-*-........-**..............-**.......................",
".......................$$$...............*##+=$&....+#$$#+--+$$$=+.-+$$$+........@#$$$=+..@=$$$@...%=$++$=%......$$+...@+#$$*..-+=$$$...@+++$*-#$$$#%.%=$$$#%.......-=$++$=%...........++#$$+++.@++=$*.+$$$-...*##++$$#*.......*#$=+$=%.@+=$$=+++-.+$$$$+......*##+=$&....@+++$*%=$$$#*.....................",
".......................$$$..............@$=%..&$#%...%$$+....@$$%....@$*...........#$$......$#%...-#$-..-$$*.....$$+.....*$$*.....+$$...@+#$$@$&*&$$#%#+*@$$#%.....*$#%..*$#%............+$$....@+#$$*++*$$+..@$@....+$$-.....*$$-..*$#%..*$$*.....-*@$$+.....@$=%..&$#%..@+#$$@#=*@#$$-....................",
".......................$$$.............-$$-....#$+....=$#....*$$&....&#............@$$@....%$@....#$&....&$#%....$$+.....*$$*.....+$$.....*$$$*...@$$#&...@$$&....-$$@....&$=............+$$......*$$&#.*$$&..$$#....%$$+....-$$@...#$$*..*$$*........$$+....-$$-....#$+....*$$$*...%$$=....................",
".......................$$$.............=$$.....+$$%...*$$*...&$$#....$@.............#$=....@#....@$$*....*$$@....$$+.....*$$*.....+$$.....*$$=.....$$$.....$$+....=$$.....*$$*...........+$$......*$$$@..&&%..#$$.....$$+....=$$....#$$-..*$$*........$$+....=$$.....+$$%...*$$=.....=$$....................",
".......................$$$............%$$+.....*$$@....$$=...#=$$-..*$%.............&$$-...#&....=$$......$$=....$$+.....*$$*.....+$$.....*$$*.....$$=.....$$+....$$+.....*$$&...........+$$......*$$$........%@-....*$$+...%$$=....%*-...*$$*........$$+...%$$+.....*$$@...*$$&.....+$$....................",
".......................$$$.........*$.*$$+.....*$$+....&$$-.*$.#$+..=+..............%$$=..-$%....$$$......$$$....$$+.....*$$*.....+$$.....*$$*.....$$+.....$$+...*$$#+++++#$$+...........+$$......*$$+..........%@+$$#$$+...*$$+..........*$$*........$$+...*$$+.....*$$+...*$$*.....+$$....................",
".......................$$$.........+=.*$$+.....*$$+....-$$&.=+.&$$%%$-...............&$$%.=+.....$$$......$$$....$$+.....*$$*.....+$$.....*$$*.....$$+.....$$+...*$$=********-...........+$$......*$$+........%+$#@-..$$+...*$$+..........*$$*........$$+...*$$+.....*$$+...*$$*.....+$$....................",
".......................$$$.........#+.-$$+.....*$$&.....=$$%$-.%$$@&#................-$$&%$-.....#$$......$$#....$$+.....*$$*.....=$$.....*$$*.....$$+.....$$+...*$$+....................+$$......*$$*.......%#$+.....$$+...*$$=..........*$$*...&-...$$+...-$$+.....*$$&...*$$*.....+$$....................",
".......................$$$........-$+..#$#.....&$$-.....@$$##...=$##@.................=$$+#......&$$-....-$$&....$$+.....*$$*.....$$$.....*$$*.....$$+.....$$+....$$$.......*-...........+$$......*$$*.......=$$%....%$$+....$$$%.....*-..*$$*...$*...$$+....#$#.....&$$-...*$$*.....+$$....................",
".......................$$$........=$+..&$$%....=$=.......$$$@...*$$$%.................-$$$@......-$$@....@$$%....$$+.....*$$@....@$$$.....*$$*.....$$+.....$$+....&$$@.....%#@...........+$$......*$$*.......$$$%....+$$+....&$$&....%#@..*$$*..-$-...$$+....&$$%....=$=....*$$*.....+$$....................",
".......................$$$......-=$$&...#$+...-$$%.......&$$%....#$+...................#$$........&$#%...#$@.....$$+......$$#-.%&#=$$-....*$$*.....$$+.....$$+....%#$$&%..@#+............+$$......*$$*.......#$$+%.-+$#$#.-%.%#$$&%.-#=...%$$=%%+=....$$+.....#$+...-$$%....*$$*.....+$$....................",
"....................++#$$$#++++#$$$$*...%+$+*@##-........-$+.....&$*...................@$&.........@$#**=$@...%*&$$#@*....*$$$$$#*+$$$$*-*=$$=*-%*&$$#@-.*@$$#@*...%=$$$$$$+...........*@#$$&*-.-*=$$=**.....*$$$$$$#-&$$$#%..%#$$$$$#%....+$$$$#%.%*&$$#@*...%+$+*@##-...-*=$$=*-.*@#$$&*%.................",
"....................++++++++++++++++-.....-&++*...........&-.....%+.....................+%..........%@++@%....-+++++++.....%@++@%.-****%@++++++@-++++++@.+++++++.....-&++&-............+++++++@.@+++++++......%@++&*...@+&%.....*+++*.......*++@%..-+++++++.....-&++*.....@++++++@.+++++++-.................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................",
"............................................................................................................................................................................................................................................................................................................"};


template <class T>
T *remake(T *ptr, int oldsize, int size)
{
    T *retval = new T[size];
    T *iptr = retval;
    for(int i = 0; i < oldsize; ++i)
        *iptr++ = ptr[i];
    delete [] ptr;
    return retval;
}

// ****************************************************************************
// Class: MaterialList
//
// Purpose:
//   Keeps track of mixed material information.
//
// Notes:
//
// Programmer: Brad Whitlock
// Creation:   Fri Jun 21 13:53:35 PST 2002
//
// Modifications:
//   Brad Whitlock, Wed Mar 17 16:43:34 PST 2004
//   I fixed an off by one error.
//
// ****************************************************************************

class MaterialList
{
public:
    MaterialList() : matNames()
    {
        have_mixed = false;
        mix_zone = NULL;
        mix_mat = NULL;
        mix_vf = NULL;
        mix_next = NULL;
        matlist = NULL;

        // initialize private members.
        _array_size = 0;
        _array_index = 1;
        _array_growth = 1000;
    }

    ~MaterialList()
    {
       delete [] matlist;
       if (have_mixed)
       {
           delete [] mix_zone;
           delete [] mix_mat;
           delete [] mix_vf;
           delete [] mix_next;
       }
    }

    void AddMaterial(const std::string &mat)
    {
        matNames.push_back(mat);
    }

    void AddClean(int siloZone, int matNumber)
    {
        matlist[siloZone] = matNumber;
    }

    void AddMixed(int siloZone, int *matNumbers, double *matVf, int nMats)
    {
        int i;

        /* Grow the arrays if they will not fit nMats materials. */
        Resize(nMats);

        /* Record the mixed zone as a negative offset into the mix arrays. */
        matlist[siloZone] = -_array_index;

        /* Update the mix arrays. */
        for(i = 0; i < nMats; ++i)
        {
            int index = _array_index - 1;

            mix_zone[index] = siloZone;
            mix_mat[index]  = matNumbers[i];
            mix_vf[index]   = matVf[i];

            if(i < nMats - 1)
                mix_next[index] = index + 2;
            else
                mix_next[index] = 0;

            ++(_array_index);
        }

        /* indicate that we have mixed materials. */
        have_mixed = true;
    }

    void AllocClean(int nZones)
    {
        matlist = new int[nZones];
    }

    int GetMixedSize() const { return _array_index - 1; };

    void WriteMaterial(DBfile *db, const char *matvarname, const char *meshName, int nx,
 int ny, int nz)
    {
        int mdims[3] = {nx,ny,nz};

        /* Create a 1..nTotalMaterials material number array. */
        int *allmats = new int[matNames.size()];
        for(size_t i = 0; i < matNames.size(); ++i)
            allmats[i] = i + 1;

        DBoptlist *optList = DBMakeOptlist(2);

        // Add material names.
        char **matnames = new char *[5];
        for(size_t i = 0; i < matNames.size(); ++i)
            matnames[i] = (char *)matNames[i].c_str();
        DBAddOption(optList, DBOPT_MATNAMES, matnames);

        if (have_mixed)
        {
            DBPutMaterial(db, (char *)matvarname, (char *)meshName,
                          matNames.size(), allmats,
                          matlist, mdims, 3, mix_next,
                          mix_mat, mix_zone,
                          (float*)mix_vf , GetMixedSize(),
                          DB_DOUBLE, optList);
        }
        else
        {
            DBPutMaterial(db, (char *)matvarname, (char *)meshName,
                          matNames.size(), allmats,
                          matlist, mdims, 3, NULL,
                          NULL, NULL, NULL , 0,
                          DB_INT, optList);
        }

        DBFreeOptlist(optList);
        delete [] matnames;
        delete [] allmats;
    }
private:
    void Resize(int nMats)
    {
        if(_array_index + nMats >= _array_size)
        {
            int new_size = _array_size + _array_growth;

            if(_array_size == 0)
            {
                /* Reallocate arrays in large increments. */
                mix_zone = new int[new_size];
                mix_mat  = new int[new_size];
                mix_vf   = new double[new_size];
                mix_next = new int[new_size];
            }
            else
            {
                /* Reallocate arrays in large increments. */
                mix_zone = remake(mix_zone, _array_size, new_size);
                mix_mat  = remake(mix_mat, _array_size,  new_size);
                mix_vf   = remake(mix_vf, _array_size, new_size);
                mix_next = remake(mix_next, _array_size, new_size);
            }

            _array_size = new_size;
        }
    }

    int    have_mixed;
    int    *mix_zone;
    int    *mix_mat;
    double *mix_vf;
    int    *mix_next;
    int    *matlist;
    int    _array_size;
    int    _array_index;
    int    _array_growth;
    std::vector<std::string> matNames;
};

// ****************************************************************************
// Function:  main
//
// Purpose:
//   Main for lowfrac.C.
//
// Notes:      This program creates a material and hides a material with a
//             small volume fraction in it. The purpose is to show how we
//             can use VisIt's new Isovolume MIR to get the material with the
//             low volume fraction out of a material where the other materials
//             pretty much completely mask the mixing due to the lowVF material.
//
// Programmer: Brad Whitlock
// Creation:   Mon Oct 3 13:39:44 PST 2005
//
// Modifications:
//   
//   Hank Childs, Fri Jan  8 08:59:16 PST 2010
//   Contribute gcc 4.4 patch from Jed Brown.
//
// ****************************************************************************

int
main(int argc, char *argv[])
{
    int i=1, driver = DB_PDB;
    DBShowErrors(DB_ALL, NULL);
    while (i < argc)
    {
        if (strcmp(argv[i], "DB_HDF5") == 0)
            driver = DB_HDF5;
        else if (strcmp(argv[i], "DB_PDB") == 0)
            driver = DB_PDB;
        else
           fprintf(stderr,"Uncrecognized driver name \"%s\"\n", argv[i]);
        i++;
    }

    // Turn lowfrac from XPM to floats
    int w, h, ncolors, tmp;
    if(sscanf(lowfrac_xpm[0], "%d %d %d %d", &w, &h, &ncolors, &tmp) != 4)
    {
        cerr << "Bad XPM!" << endl;
        return -1;
    }
    // Create gray-scale color LUT.
    char  *colorSym = new char[ncolors];
    float *colorLUT = new float[ncolors];
    for(i = 0; i < ncolors; ++i)
    {
        const char *s = lowfrac_xpm[i+1];
        colorSym[i] = s[0];

        if(strstr(s, "None") != 0)
            colorLUT[i] = 0.;
        else
        {
            const char *c = strstr(s, "c ");
            if(c != 0)
            {
                char hex[100];
                int hexval;
                sprintf(hex, "0x%s", c + 7);
                if(sscanf(hex, "%x", &hexval) == 1)
                    colorLUT[i] = float(hexval) / 255.;
                else
                    colorLUT[i] = 0.;
            }
            else
                colorLUT[i] = 0.;
        }
    }
    float *lowfrac = new float[w * h];
    float *fptr = lowfrac;
    int y, x;
    for(y = 0; y < h; ++y)
    {
        char *s = lowfrac_xpm[ncolors + h-y];
        for(x = 0; x < w; ++x)
        {
            int index = -1;
            for(int ci = 0; ci < ncolors; ++ci)
            {
                if(s[x] == colorSym[ci])
                {
                    index = ci;
                    break;
                }
            }
            if(index == -1)
            {
                cerr << "Invalid XPM!" << endl;
                return -2;
            }
            *fptr++ = 1. - colorLUT[index];
        }
    }

    //
    // Create some volume fraction arrays that we'll aggregate into
    // a material later.
    //
    float *m1 = new float[w*h];
    float *m2 = new float[w*h];
    float *m3 = new float[w*h];
    float *m4 = new float[w*h];
    int   *region = new int[w * h];
    int   *mix1mask = new int[w * h];
    int   *mix2mask = new int[w * h];

    const float xExtents[] = {0., 2. * 3.14159};
    const float dX = xExtents[1] - xExtents[0];
    const float yExtents[] = {-1.2, 1.2};
    const float dY = yExtents[1] - yExtents[0];
    const float cellX = dX / float(w+1);
    const float cellY = dY / float(h+1);
    const int NSAMPLES = 4;
    for(y = 0; y < h; ++y)
    {
        float y0 = y * cellY + yExtents[0];
        float y1 = y0 + cellY;
        for(x = 0; x < w; ++x)
        {
            float x0 = x * cellX + xExtents[0];
            float x1 = x0 + cellX;

            // Sum up the mat1, mat2 for this cell.
            float m1sum = 0.;
            float m2sum = 0.;
            float m3sum = 0.;
            float m4sum = 0.;

            for(int nysamp = 0; nysamp < NSAMPLES; ++nysamp)
            {
                float tcy = float(nysamp) / float(NSAMPLES-1);
                float cy = (1. - tcy) * y0 + tcy * y1;
                for(int nxsamp = 0; nxsamp < NSAMPLES; ++nxsamp)
                {
                    float tcx = float(nxsamp) / float(NSAMPLES-1);
                    float cx = (1. - tcx) * x0 + tcx * x1;

                    if(cy > sin(cx))
                        m1sum += 1.;
                    else
                        m2sum += 1.;

                    if(cy > sin(2 * cx + 3.14159 / 6.))
                        m3sum += 1.;
                    else
                        m4sum += 1.;
                }
            }

            m1sum /= float(NSAMPLES*NSAMPLES);
            m2sum /= float(NSAMPLES*NSAMPLES);
            m3sum /= float(NSAMPLES*NSAMPLES);
            m4sum /= float(NSAMPLES*NSAMPLES);

            int index = y * w + x;
            if(m1sum > 0.5)
                region[index] = 0;
            else
                region[index] = 1;

            mix1mask[index] = (m1sum != 0. && m1sum != 1.) ? 1 : 0;

            if(m3sum > 0.5)
                region[index] = region[index] * 2;
            else
                region[index] = region[index] * 2 + 1;
            region[index]++;

            mix2mask[index] = (m3sum != 0. && m3sum != 1.) ? 1 : 0;

            m1[index] = m1sum;
            m2[index] = m2sum;
            m3[index] = m3sum;
            m4[index] = m4sum;
        }
    }

    //
    // Assemble m1, m2, m3, m4, lowfrac into a material.
    //
    MaterialList M;
    M.AddMaterial("A");
    M.AddMaterial("B");
    M.AddMaterial("C");
    M.AddMaterial("D");
    M.AddMaterial("HardToFind");
    M.AllocClean(w * h);
    double matvf[5];
    int    mats[5];
    int    zn = 0;
    for(y = 0; y < h; ++y)
    {
        for(x = 0; x < w; ++x, ++zn)
        {
            int index = y * w + x;
            int nmats = 1;
            int mat = 1;

            if(mix1mask[index] == 0 && mix2mask[index] == 0)
            {
                // Clean.
                mat = region[index];
                nmats = 1;
            }
            else if(mix1mask[index] == 1 && mix2mask[index] == 1)
            {
                double S = m1[index] + m2[index] + m3[index] + m4[index];
                matvf[0] = m1[index] / S;
                matvf[1] = m2[index] / S;
                matvf[2] = m3[index] / S;
                matvf[3] = m4[index] / S;
                mats[0] = 1;
                mats[1] = 2;
                mats[2] = 3;
                mats[3] = 4;
                nmats = 4;
            }
            else if(mix1mask[index] == 1)
            {
                matvf[0] = m1[index];
                matvf[1] = m2[index];
                if(region[index] == 1)
                {
                    mats[0] = 1;
                    mats[1] = 3;
                }
                else if(region[index] == 2)
                {
                    mats[0] = 2;
                    mats[1] = 4;
                }
                else if(region[index] == 3)
                {
                    mats[0] = 1;
                    mats[1] = 3;
                }
                else if(region[index] == 4)
                {
                    mats[0] = 2;
                    mats[1] = 4;
                }
                nmats = 2;
            }
            else if(mix2mask[index] == 1)
            {
                matvf[0] = m3[index];
                matvf[1] = m4[index];
                if(region[index] == 1)
                {
                    mats[0] = 1;
                    mats[1] = 2;
                }
                else if(region[index] == 2)
                {
                    mats[0] = 1;
                    mats[1] = 2;
                }
                else if(region[index] == 3)
                {
                    mats[0] = 3;
                    mats[1] = 4;
                }
                else if(region[index] == 4)
                {
                    mats[0] = 3;
                    mats[1] = 4;
                }
                nmats = 2;
            }

            // Add the low fraction material
            const double MAX_LOW_CONTRIBUTION = 0.08;
            if(lowfrac[index] != 0.)
            {
                double lf = MAX_LOW_CONTRIBUTION * lowfrac[index];
                if(nmats == 1)
                {
                    matvf[0] = 1. - lf;
                    matvf[1] = lf;
                    mats[0] = mat;
                    mats[1] = 5;
                }
                else
                {
                    double scale = 1. - lf;
                    for(int nn = 0; nn < nmats; ++nn)
                        matvf[nn] *= scale;
                    matvf[nmats] = lf;
                    mats[nmats] = 5;
                }

                ++nmats;
            }

            if(nmats == 1)
                M.AddClean(zn, mat);
            else
            {
                M.AddMixed(zn, mats, matvf, nmats);
            }
        }
    }

    //
    // Create the mesh.
    //
    float *xc = new float[w+1];
    float *yc = new float[h+1];
    float *coords[3] = {0,0,0};
    coords[0] = xc;
    coords[1] = yc;
    int ndims = 2;
    int dims[3] = {0,0,0};
    dims[0] = w + 1;
    dims[1] = h + 1;
    for (i = 0; i < w + 1; i++)
        xc[i] = (float(i) / float(w)) * dX + xExtents[0];
    for (i = 0; i < h + 1; i++)
        yc[i] = (float(i) / float(h)) * dY + yExtents[0];

    // Write the file.
    DBfile *dbfile = DBCreate("lowfrac.silo", 0, DB_LOCAL,
                              "Low volume fraction test file", driver);

    // Write the mesh.
    DBoptlist *optlist = DBMakeOptlist(10);
    int    cycle = 0;
    float  time = 1.234;
    double dtime = 1.23456789;
    DBAddOption(optlist, DBOPT_CYCLE, &cycle);
    DBAddOption(optlist, DBOPT_TIME, &time);
    DBAddOption(optlist, DBOPT_DTIME, &dtime);
    char *xaxis="Horizontal";
    char *yaxis="Vertical";
    DBAddOption(optlist, DBOPT_XLABEL, xaxis);
    DBAddOption(optlist, DBOPT_YLABEL, yaxis);
    char *units="Km";
    DBAddOption(optlist, DBOPT_XUNITS, units);
    DBAddOption(optlist, DBOPT_YUNITS, units);

    DBPutQuadmesh(dbfile, "mesh", NULL, coords, dims, ndims,
                  DB_FLOAT, DB_COLLINEAR, optlist);

    // Write the material
    M.WriteMaterial(dbfile, "mat", "mesh", w, h, 1);

    // Write the volume fraction arrays.
    int vdims[3] = {0,0,0};
    vdims[0] = w;
    vdims[1] = h;
#if 0
    DBPutQuadvar1(dbfile, "region", "mesh", (float*)region, vdims, ndims, NULL, 0, 
                  DB_INT, DB_ZONECENT, optlist);
    DBPutQuadvar1(dbfile, "mix1mask", "mesh", (float*)mix1mask, vdims, ndims, NULL, 0, 
                  DB_INT, DB_ZONECENT, optlist);
    DBPutQuadvar1(dbfile, "mix2mask", "mesh", (float*)mix2mask, vdims, ndims, NULL, 0, 
                  DB_INT, DB_ZONECENT, optlist);
#endif
    DBPutQuadvar1(dbfile, "m1", "mesh", m1, vdims, ndims, NULL, 0, 
                  DB_FLOAT, DB_ZONECENT, optlist);
    DBPutQuadvar1(dbfile, "m2", "mesh", m2, vdims, ndims, NULL, 0, 
                  DB_FLOAT, DB_ZONECENT, optlist);
    DBPutQuadvar1(dbfile, "m3", "mesh", m3, vdims, ndims, NULL, 0, 
                  DB_FLOAT, DB_ZONECENT, optlist);
    DBPutQuadvar1(dbfile, "m4", "mesh", m4, vdims, ndims, NULL, 0, 
                  DB_FLOAT, DB_ZONECENT, optlist);
    DBPutQuadvar1(dbfile, "lowfrac", "mesh", lowfrac, vdims, ndims, NULL, 0, 
                  DB_FLOAT, DB_ZONECENT, optlist);
    DBFreeOptlist(optlist);
    DBClose(dbfile);

    // Free arrays
    delete [] xc;
    delete [] yc;
    delete [] colorLUT;
    delete [] colorSym;
    delete [] lowfrac;
    delete [] m1;
    delete [] m2;
    delete [] m3;
    delete [] m4;
    delete [] region;
    delete [] mix1mask;
    delete [] mix2mask;

    return 0;
}
