// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

#include <silo.h>
#include <math.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef _WIN32
#include <unistd.h>
#endif
#include <stdlib.h>

#define ONE_MEG 1048576

// suppress the following since silo uses char * in its API
#if defined(__clang__)
# pragma clang diagnostic ignored "-Wwritable-strings"
#elif defined(__GNUC__)
# pragma GCC diagnostic ignored "-Wwrite-strings"
#endif

static void
build_curve (DBfile *dbfile)
{
   float        x[20], y[2][20] ;
   int          i ;
   DBoptlist    *opts ;

   // Sine and cosine for 1/2 cycle.  Both curves have the same
   // X data points and share the data in the file.
   for (i=0; i<20; i++) {
      x[i] = i * M_PI / 20.0 ;
      y[0][i] = sin (x[i]) ;
      y[1][i] = cos (x[i]) ;
   }

   opts = DBMakeOptlist (10) ;
   char *xaxis = "X Axis";
   char *yaxis = "Y Axis";
   char *xunits = "radians";
   char *xvarnm = "sincurve_xvals";

   DBAddOption(opts, DBOPT_XLABEL, xaxis);
   DBAddOption(opts, DBOPT_YLABEL, yaxis); 
   DBAddOption(opts, DBOPT_XUNITS, xunits);
   DBAddOption(opts, DBOPT_XVARNAME, xvarnm); 

   // Write the `sincurve' curve. The hdf5 driver allows the user to specify
   // the name which will be used to store the x values, but the pdb driver
   // requires us to know where the values were stored.
   DBPutCurve (dbfile, "sincurve", x, y[0], DB_FLOAT, 20, opts);

   // Write the `coscurve' curve. It shares x values with the `sincurve'
   // curve.
   DBPutCurve (dbfile, "coscurve", NULL, y[1], DB_FLOAT, 20, opts) ;
   DBFreeOptlist (opts) ;
}


/*-------------------------------------------------------------------------
 * Function:     main
 *
 * Purpose:      Build a > 2Gig file with something stored at offsets
 *               greater than 2 Gig that VisIt can try to display.
 *
 * Notes:        This can only work on the HDF5 driver
 *
 * Programmer:   Mark C. Miller, March 23, 2006
 *
 *-------------------------------------------------------------------------
 */
int
main(int argc, char *argv[])
{
    
    int            i, j, ndims=1, dims[]={ONE_MEG/sizeof(float)};
    float          val[ONE_MEG/sizeof(float)];
    char          filename[1024];
    DBfile        *dbfile;
    int           driver = DB_PDB;

    
    if (argc == 1)
    {
        fprintf(stderr, "You must specify a filename.\n"
            "Make sure the file (and path) you specify is\n"
            "to a filesystem that supports large files.\n");
        exit(1);
    }
    else
    {
        strncpy(filename, argv[1], sizeof(filename));
        filename[1023] = '\0';

        for (i = 2; i < argc; i++)
        {
            if (strcmp(argv[i], "DB_HDF5") == 0)
                driver = DB_HDF5;
            else if (strcmp(argv[i], "DB_PDB") == 0)
                driver = DB_PDB;
            else
                fprintf(stderr,"Uncrecognized driver name \"%s\"\n", argv[i]);
        }
    }

    DBShowErrors(DB_TOP, NULL);
    DBForceSingle(1);

    // Create a file that contains a simple variables.
    printf("Creating file: \"%s\"\n", filename);
    dbfile = DBCreate(filename, 0, DB_LOCAL, "Large File Test", driver);

    if (dbfile == NULL)
    {
        fprintf(stderr, "Error creating file\n");
        exit(1);
    }

    // This will generate a 2.5 Gigabyte file
    for (j = 0; j < 2500; j++)
    {
        char tmpname[64];

        if (j % 100 == 0)
            printf("Iterations %04d to %04d of %04d\n", j, j+100-1, 2500);

        sprintf(tmpname, "simple_%04d", j);

        for (i = 0; i < dims[0]; i++)
            val[i] = (float) dims[0] * j + i;

        if (DBWrite(dbfile, tmpname, val, dims, ndims, DB_FLOAT) != 0)
        {
            fprintf(stderr, "Error creating file\n");
            exit(1);
        }
    }

    // Put some objects VisIt can process at the end of the file
    build_curve(dbfile);

    DBClose(dbfile);

    return 0;
}
