// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

// ************************************************************************* //
//                         avtTopologyPlot.C                                 //
// ************************************************************************* //

#include <avtTopologyPlot.h>

#include <avtColorTables.h>
#include <avtLevelsLegend.h>
#include <avtLevelsMapper.h>
#include <avtLookupTable.h>
#include <avtTopologyFilter.h>

#include <ColorAttribute.h>
#include <ColorAttributeList.h>
#include <LineAttributes.h>

#include <InvalidColortableException.h>
#include <maptypes.h>

#include <string>
#include <vector>

// ****************************************************************************
//  Method: avtTopologyPlot constructor
//
//  Programmer: haddox1 -- generated by xml2info
//  Creation:   Fri Jun 27 09:00:35 PDT 2003
//
//  Modifications:
//    Eric Brugger, Wed Jul 16 11:27:41 PDT 2003
//    Modified to work with the new way legends are managed.
//
// ****************************************************************************

avtTopologyPlot::avtTopologyPlot()
{
    topologyFilter = new avtTopologyFilter();

    levelsMapper = new avtLevelsMapper();
    levelsLegend = new avtLevelsLegend();
    levelsLegend->SetTitle("Topology");
    // there is no 'range' per se, so turn off range visibility.
    levelsLegend->SetVarRangeVisibility(0);
    levelsLegend->SetColorBarVisibility(1);


    avtLUT = new avtLookupTable();

    opacity[0] = opacity[1] = opacity[2] = opacity[3] = 1.0;

    //
    // This is to allow the legend to reference counted so the behavior can
    // still access it after the plot is deleted.  The legend cannot be
    // reference counted all of the time since we need to know that it is a
    // LevelsLegend.
    //
    levLegendRefPtr = levelsLegend;
}


// ****************************************************************************
//  Method: avtTopologyPlot destructor
//
//  Programmer: haddox1 -- generated by xml2info
//  Creation:   Fri Jun 27 09:00:35 PDT 2003
//
// ****************************************************************************

avtTopologyPlot::~avtTopologyPlot()
{
    if (levelsMapper != NULL)
    {
        delete levelsMapper;
        levelsMapper = NULL;
    }
    if (avtLUT != NULL)
    {
        delete avtLUT;
        avtLUT = NULL;
    }
    if (topologyFilter != NULL)
    {
        delete topologyFilter;
        topologyFilter = NULL;
    }
    //
    // Do not delete the levelsLegend.
    //
}


// ****************************************************************************
//  Method:  avtTopologyPlot::Create
//
//  Purpose:
//    Call the constructor.
//
//  Programmer: haddox1 -- generated by xml2info
//  Creation:   Fri Jun 27 09:00:35 PDT 2003
//
// ****************************************************************************

avtPlot*
avtTopologyPlot::Create()
{
    return new avtTopologyPlot;
}


// ****************************************************************************
//  Method: avtTopologyPlot::GetMapper
//
//  Purpose:
//      Gets a mapper for this plot, it is actually a variable mapper.
//
//  Returns:    The variable mapper typed as its base class mapper.
//
//  Programmer: haddox1 -- generated by xml2info
//  Creation:   Fri Jun 27 09:00:35 PDT 2003
//
// ****************************************************************************

avtMapperBase *
avtTopologyPlot::GetMapper(void)
{
    return levelsMapper;
}


// ****************************************************************************
//  Method: avtTopologyPlot::SetLegend
//
//  Purpose:
//      Turns the legend on or off.
//
//  Arguments:
//      legendOn     true if the legend should be turned on, false otherwise.
//
//  Programmer: Akira Haddox
//  Creation:   June 27, 2003
//
// ****************************************************************************

void
avtTopologyPlot::SetLegend(bool legendOn)
{
    if (legendOn)
        levelsLegend->LegendOn();
    else
        levelsLegend->LegendOff();
}


// ****************************************************************************
//  Method: avtTopologyPlot::SetLineWidth
//
//  Purpose:
//      Sets the line width.
//
//  Programmer: Akira Haddox
//  Creation:   June 26, 2003
//
// ****************************************************************************

void
avtTopologyPlot::SetLineWidth(int lw)
{
    levelsMapper->SetLineWidth(Int2LineWidth(lw));
}


// ****************************************************************************
//  Method: avtTopologyPlot::ApplyOperators
//
//  Purpose:
//      Applies the operators associated with a Topology plot.  
//      The output from this method is a query-able object.
//
//  Arguments:
//      input   The input data object.
//
//  Returns:    The data object after the Topology plot has been applied.
//
//  Programmer: haddox1 -- generated by xml2info
//  Creation:   Fri Jun 27 09:00:35 PDT 2003
//
// ****************************************************************************

avtDataObject_p
avtTopologyPlot::ApplyOperators(avtDataObject_p input)
{
    topologyFilter->SetInput(input);
    return topologyFilter->GetOutput();
}


// ****************************************************************************
//  Method: avtTopologyPlot::ApplyRenderingTransformation
//
//  Purpose:
//      Applies any rendering transformations.
//
//  Arguments:
//      input   The input data object.
//
//  Returns:    The data object after transformations have been applied.
//
//  Programmer: haddox1 -- generated by xml2info
//  Creation:   Fri Jun 27 09:00:35 PDT 2003
//
// ****************************************************************************


avtDataObject_p
avtTopologyPlot::ApplyRenderingTransformation(avtDataObject_p input)
{
    return input;
}


// ****************************************************************************
//  Method: avtTopologyPlot::CustomizeBehavior
//
//  Purpose:
//      Customizes the behavior as appropriate for a Topology plot.
//      This includes behavior like shifting towards or away from the screen.
//
//  Programmer: haddox1 -- generated by xml2info
//  Creation:   Fri Jun 27 09:00:35 PDT 2003
//
//  Kathleen Bonnell, Thu Sep 18 13:40:26 PDT 2003
//  Set behvior's render order for antialiasing to be the same as normal.
// ****************************************************************************

void
avtTopologyPlot::CustomizeBehavior(void)
{
    SetColors();
    levelsLegend->SetLookupTable(avtLUT->GetLookupTable());

    behavior->SetRenderOrder(MUST_GO_LAST);
    behavior->SetAntialiasedRenderOrder(MUST_GO_LAST);

    behavior->SetLegend(levLegendRefPtr);
    behavior->SetShiftFactor(0.5);
}


// ****************************************************************************
//  Method: avtTopologyPlot::SetAtts
//
//  Purpose:
//      Sets the atts for the Topology plot.
//
//  Arguments:
//      atts    The attributes for this Topology plot.
//
//  Programmer: haddox1 -- generated by xml2info
//  Creation:   Fri Jun 27 09:00:35 PDT 2003
//
// ****************************************************************************

void
avtTopologyPlot::SetAtts(const AttributeGroup *a)
{
    const TopologyAttributes *newAtts = (const TopologyAttributes *)a;
    
    needsRecalculation = atts.ChangesRequireRecalculation(*newAtts);
    
    atts = *(newAtts);

    topologyFilter->SetTolerance(atts.GetTolerance());
    topologyFilter->SetHitpercent(atts.GetHitpercent());

    opacity[0] = atts.GetMinOpacity();
    opacity[1] = atts.GetMinPlateauOpacity();
    opacity[2] = atts.GetMaxPlateauOpacity();
    opacity[3] = atts.GetMaxOpacity();
    
    SetColors();
}


// ****************************************************************************
//  Method: avtTopologyPlot::SetColors
//
//  Purpose:
//    Sets up colors for the mapper and legend to use.
//
//  Programmer: Akira Haddox
//  Creation:   June 27, 2003
//
//  Modifications:
//    Eric Brugger, Wed Jul 16 11:27:41 PDT 2003
//    Modified to work with the new way legends are managed.
//
//    Hank Childs, Fri Jun 15 11:57:07 PDT 2007
//    Was using a deprecated method ... changed to a valid one.
//
//    Kathleen Biagas, Thu Oct 16 09:12:03 PDT 2014
//    Send 'needsRecalculation' flag to levelsMapper when setting colors.
//
// ****************************************************************************

void 
avtTopologyPlot::SetColors()
{
    std::vector < std::string > labels(4);
    LevelColorMap levelColorMap;

    labels[0] = avtTopologyFilter::labelNames[0];
    labels[1] = avtTopologyFilter::labelNames[1];
    labels[2] = avtTopologyFilter::labelNames[2];
    labels[3] = avtTopologyFilter::labelNames[3];
    
    ColorAttributeList cal(atts.GetMultiColor());
    int numColors = cal.GetNumColors();
    
    //
    //  Create colors from original color table. 
    //
    unsigned char *colors = new unsigned char[numColors * 4];
    unsigned char *cptr = colors;
    for(int i = 0; i < numColors; i++)
    {
        cal[i].SetAlpha((unsigned char)(255 * opacity[i]));

        *cptr++ = (unsigned char)cal[i].Red();
        *cptr++ = (unsigned char)cal[i].Green();
        *cptr++ = (unsigned char)cal[i].Blue();
        *cptr++ = (unsigned char)cal[i].Alpha();

        //
        //  Create a label-to-color-index mapping 
        //
        levelColorMap.insert(LevelColorMap::value_type(labels[i], i));
    }

    avtLUT->SetLUTColorsWithOpacity(colors, numColors);
    levelsMapper->SetColors(cal, needsRecalculation);
    levelsLegend->SetLevels(labels);

    levelsMapper->SetLabelColorMap(levelColorMap);
    levelsLegend->SetLabelColorMap(levelColorMap);

    delete [] colors;

    SetLineWidth(atts.GetLineWidth());
}


// ****************************************************************************
//  Method: avtTopologyPlot::ReleaseData
//
//  Purpose:
//      Release the problem sized data associated with this plot.
//
//  Programmer: Akira Haddox
//  Creation:   July 1, 2003
//
// ****************************************************************************
 
void
avtTopologyPlot::ReleaseData(void)
{
    avtSurfaceDataPlot::ReleaseData();

    if (topologyFilter != NULL)
    {
        topologyFilter->ReleaseData();
    }
}
