// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

#include "QvisPoincareWindow.h"

#include <PoincareAttributes.h>

#include <QTabWidget>
#include <QCheckBox>
#include <QLabel>
#include <QLayout>
#include <QLineEdit>
#include <QSpinBox>
#include <QButtonGroup>
#include <QComboBox>
#include <QGroupBox>
#include <QRadioButton>
#include <QPushButton>
#include <QFileDialog>
#include <QListWidget>


static void
TurnOn(QWidget *w0, QWidget *w1=NULL);
static void
TurnOff(QWidget *w0, QWidget *w1=NULL);

// ****************************************************************************
// Method: QvisPoincareWindow::QvisPoincareWindow
//
// Purpose:
//   Constructor
//
// Note:       Autogenerated by xml2window.
//
// Programmer: xml2window
// Creation:   omitted
//
// Modifications:
//
// ****************************************************************************

QvisPoincareWindow::QvisPoincareWindow(const int type,
                                       PoincareAttributes *subj,
                                       const QString &caption,
                                       const QString &shortName,
                                       QvisNotepadArea *notepad)
    : QvisOperatorWindow(type, subj, caption, shortName, notepad)
{
    plotType = type;
    atts = subj;
}


// ****************************************************************************
// Method: QvisPoincareWindow::~QvisPoincareWindow
//
// Purpose:
//   Destructor
//
// Note:       Autogenerated by xml2window.
//
// Programmer: xml2window
// Creation:   omitted
//
// Modifications:
//
// ****************************************************************************

QvisPoincareWindow::~QvisPoincareWindow()
{
}


// ****************************************************************************
// Method: QvisPoincareWindow::CreateWindowContents
//
// Purpose:
//   Creates the widgets for the window.
//
// Note:       Autogenerated by xml2window.
//
// Programmer: xml2window
// Creation:   omitted
//
// ****************************************************************************

void
QvisPoincareWindow::CreateWindowContents()
{
    QTabWidget *propertyTabs = new QTabWidget(central);
    topLayout->addWidget(propertyTabs);

    // ----------------------------------------------------------------------
    // Integration tab
    // ----------------------------------------------------------------------
    QWidget *integrationTab = new QWidget(central);
    propertyTabs->addTab(integrationTab, tr("Integration"));
    CreateIntegrationTab(integrationTab);

    // ----------------------------------------------------------------------
    // Analysis tab
    // ----------------------------------------------------------------------
    QWidget *analysisTab = new QWidget(central);
    propertyTabs->addTab(analysisTab, tr("Analysis"));
    CreateAnalysisTab(analysisTab);

    // ----------------------------------------------------------------------
    // Appearance tab
    // ----------------------------------------------------------------------
    QWidget *appearanceTab = new QWidget(central);
    propertyTabs->addTab(appearanceTab, tr("Appearance"));
    CreateAppearanceTab(appearanceTab);

    // ----------------------------------------------------------------------
    // Advanced tab
    // ----------------------------------------------------------------------
    QWidget *advancedTab = new QWidget(central);
    propertyTabs->addTab(advancedTab, tr("Advanced"));
    CreateAdvancedTab(advancedTab);
}

// ****************************************************************************
// Method: QvisPoincareWindow::CreateIntegrationTab
//
// Purpose:
//   Populates the integration tab.
//
// Programmer: Dave Pugmire
// Creation:   Tue Dec 29 14:37:53 EST 2009
//
// Modifications:
//   Kathleen Biagas, Wed Jun  8 17:10:30 PDT 2016
//   Set keyboard tracking to false for spin boxes so that 'valueChanged'
//   signal will only emit when 'enter' is pressed or spinbox loses focus.
//
// ****************************************************************************

void
QvisPoincareWindow::CreateIntegrationTab(QWidget *pageIntegration)
{
    QGridLayout *mainLayout = new QGridLayout(pageIntegration);
    mainLayout->setMargin(5);
    mainLayout->setSpacing(10);

    // Create the source group box.
    QGroupBox *sourceGroup = new QGroupBox(central);
    sourceGroup->setTitle(tr("Source"));
    mainLayout->addWidget(sourceGroup, 0, 0);
//    mainLayout->setStretchFactor(sourceGroup, 100);
    QGridLayout *sourceLayout = new QGridLayout(sourceGroup);
    sourceLayout->setMargin(5);
    sourceLayout->setSpacing(10);

    sourceLayout->addWidget(new QLabel(tr("Source type"), sourceGroup), 0, 0);
    sourceTypeCombo = new QComboBox(sourceGroup);
    sourceTypeCombo->addItem(tr("Point"));
    sourceTypeCombo->addItem(tr("PointList"));
    sourceTypeCombo->addItem(tr("Line"));
    connect(sourceTypeCombo, SIGNAL(activated(int)),
           this, SLOT(sourceTypeChanged(int)));
    sourceLayout->addWidget(sourceTypeCombo, 0, 1);

    // Point Source
    pointSourceLabel = new QLabel(tr("Location"), sourceGroup);
    sourceLayout->addWidget(pointSourceLabel, 0, 2);
    pointSource = new QLineEdit(sourceGroup);
    connect(pointSource, SIGNAL(returnPressed()),
            this, SLOT(pointSourceProcessText()));
    sourceLayout->addWidget(pointSource, 0, 3);

    // Point list.
    pointList = new QListWidget(sourceGroup);
    sourceLayout->addWidget(pointList, 0, 3, 4, 1);
    connect(pointList, SIGNAL(itemDoubleClicked(QListWidgetItem*)), this, SLOT(pointListDoubleClicked(QListWidgetItem*)));
    connect(pointList, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(pointListClicked(QListWidgetItem*)));
    connect(pointList, SIGNAL(currentTextChanged(const QString&)), this, SLOT(textChanged(QString)));

    pointListAddPoint     = new QPushButton(tr("Add Point"), sourceGroup);
    pointListDelPoint     = new QPushButton(tr("Delete Point"), sourceGroup);
    pointListDelAllPoints = new QPushButton(tr("Delete All Points"), sourceGroup );
    pointListReadPoints   = new QPushButton(tr("Read Text File"), sourceGroup);

    sourceLayout->addWidget(pointListAddPoint,     0, 2);
    sourceLayout->addWidget(pointListDelPoint,     1, 2);
    sourceLayout->addWidget(pointListDelAllPoints, 2, 2);
    sourceLayout->addWidget(pointListReadPoints,   3, 2);

    connect(pointListAddPoint, SIGNAL(clicked()), this, SLOT(addPoint()));
    connect(pointListDelPoint, SIGNAL(clicked()), this, SLOT(deletePoint()));
    connect(pointListDelAllPoints, SIGNAL(clicked()), this, SLOT(deletePoints()));
    connect(pointListReadPoints, SIGNAL(clicked()), this, SLOT(readPoints()));


    // Line Source
    lineStartLabel = new QLabel(tr("Start Point"), sourceGroup);
    sourceLayout->addWidget(lineStartLabel, 0, 2);
    lineStart = new QLineEdit(sourceGroup);
    connect(lineStart, SIGNAL(returnPressed()),
            this, SLOT(lineStartProcessText()));
    sourceLayout->addWidget(lineStart, 0, 3);

    lineEndLabel = new QLabel(tr("End Point"), sourceGroup);
    sourceLayout->addWidget(lineEndLabel, 1, 2);
    lineEnd = new QLineEdit(sourceGroup);
    connect(lineEnd, SIGNAL(returnPressed()),
            this, SLOT(lineEndProcessText()));
    sourceLayout->addWidget(lineEnd, 1, 3);

    pointDensityLabel = new QLabel(tr("Point density"), sourceGroup);
    sourceLayout->addWidget(pointDensityLabel, 2, 2);
    pointDensity = new QSpinBox(sourceGroup);
    pointDensity->setKeyboardTracking(false);
    pointDensity->setMinimum(1);
    pointDensity->setMaximum(10000);
    connect(pointDensity, SIGNAL(valueChanged(int)),
            this, SLOT(pointDensityChanged(int)));
    sourceLayout->addWidget(pointDensity, 2, 3);


    // Create the field group box.
    QGroupBox *fieldGroup = new QGroupBox(central);
    fieldGroup->setTitle(tr("Field"));
    mainLayout->addWidget(fieldGroup, 3, 0, 1, 1);
//    mainLayout->setStretchFactor(fieldGroup, 100);
    QGridLayout *fieldLayout = new QGridLayout(fieldGroup);
    fieldLayout->setMargin(5);
    fieldLayout->setSpacing(10);


    fieldLayout->addWidget( new QLabel(tr("Field"), fieldGroup), 0,0);
    fieldType = new QComboBox(fieldGroup);
    fieldType->addItem(tr("Default"));
    fieldType->addItem(tr("Flash"));
    fieldType->addItem(tr("M3D-C1 2D"));
    fieldType->addItem(tr("M3D-C1 3D"));
    fieldType->addItem(tr("Nek5000"));
    fieldType->addItem(tr("Nektar++"));
    fieldType->addItem(tr("NIMROD"));
    connect(fieldType, SIGNAL(activated(int)),
            this, SLOT(fieldTypeChanged(int)));
    fieldLayout->addWidget(fieldType, 0,1);


    // Create the field constant text field.
    fieldConstantLabel = new QLabel(tr("Constant"), fieldGroup);
    fieldConstant = new QLineEdit(fieldGroup);
    connect(fieldConstant, SIGNAL(returnPressed()), this,
            SLOT(fieldConstantProccessText()));
    fieldLayout->addWidget(fieldConstantLabel, 0,2);
    fieldLayout->addWidget(fieldConstant, 0,3);


    // Create the widgets that specify a velocity source.
    velocitySource = new QLineEdit(fieldGroup);
    connect(velocitySource, SIGNAL(returnPressed()),
            this, SLOT(velocitySourceProcessText()));
    velocitySourceLabel = new QLabel(tr("Velocity"), fieldGroup);
    velocitySourceLabel->setBuddy(velocitySource);
    fieldLayout->addWidget(velocitySourceLabel, 1, 2);
    fieldLayout->addWidget(velocitySource, 1, 3);


    // Create the node centering
    // forceNodal = new QCheckBox(tr("Force node centering"), fieldGroup);
    // connect(forceNodal, SIGNAL(toggled(bool)), this, SLOT(forceNodalChanged(bool)));
    // fieldLayout->addWidget(forceNodal, 2, 0);

    // Create the integration group box.
    QGroupBox *integrationGroup = new QGroupBox(central);
    integrationGroup->setTitle(tr("Integration"));
    mainLayout->addWidget(integrationGroup, 6, 0, 4, 2);
//    mainLayout->setStretchFactor(integrationGroup, 100);
    QGridLayout *integrationLayout = new QGridLayout(integrationGroup);
    integrationLayout->setMargin(5);
    integrationLayout->setSpacing(10);


    integrationLayout->addWidget( new QLabel(tr("Integrator"), integrationGroup), 0,0);
    integrationType = new QComboBox(integrationGroup);
    integrationType->addItem(tr("Forward Euler (Single-step)"));
    integrationType->addItem(tr("Leapfrog (Single-step)"));
    integrationType->addItem(tr("Dormand-Prince (Runge-Kutta)"));
    integrationType->addItem(tr("Adams-Bashforth (Multi-step)"));
    integrationType->addItem(tr("Runge-Kutta 4 (Single-step)"));
    integrationType->addItem(tr("M3D-C1 2D Integrator (M3D-C1 2D fields only)"));
    connect(integrationType, SIGNAL(activated(int)),
            this, SLOT(integrationTypeChanged(int)));
    integrationLayout->addWidget(integrationType, 0,1);

    // Create the step length text field.
    maxStepLengthLabel = new QLabel(tr("Step length"), integrationGroup);
    maxStepLength = new QLineEdit(integrationGroup);
    connect(maxStepLength, SIGNAL(returnPressed()),
            this, SLOT(maxStepLengthProcessText()));
    integrationLayout->addWidget(maxStepLengthLabel, 1,0);
    integrationLayout->addWidget(maxStepLength, 1,1);

    limitMaxTimeStep = new QCheckBox(tr("Limit maximum time step"), integrationGroup);
    connect(limitMaxTimeStep, SIGNAL(toggled(bool)), this, SLOT(limitMaxTimeStepChanged(bool)));
    integrationLayout->addWidget(limitMaxTimeStep, 2, 0);

    // Create the step length text field.
    maxTimeStep = new QLineEdit(integrationGroup);
    connect(maxTimeStep, SIGNAL(returnPressed()),
            this, SLOT(maxTimeStepProcessText()));
    integrationLayout->addWidget(maxTimeStep, 2,1);

    QGroupBox *toleranceGroup = new QGroupBox(central);
    toleranceGroup->setTitle(tr("Tolerances: max error for step < max(abstol, reltol*velocity_i) for each component i"));
    integrationLayout->addWidget(toleranceGroup, 4, 0, 2, 3);
    QGridLayout *toleranceLayout = new QGridLayout(toleranceGroup);
    toleranceLayout->setMargin(5);
    toleranceLayout->setSpacing(10);

    // Create the relative tolerance text field.
    relTolLabel = new QLabel(tr("Relative tolerance"), toleranceGroup);
    relTol = new QLineEdit(toleranceGroup);
    connect(relTol, SIGNAL(returnPressed()),
            this, SLOT(relTolProcessText()));
    toleranceLayout->addWidget(relTolLabel, 0, 0);
    toleranceLayout->addWidget(relTol, 0, 1);

    // Create the absolute tolerance text field.
    absTolLabel = new QLabel(tr("Absolute tolerance"), toleranceGroup);
    absTol = new QLineEdit(toleranceGroup);
    connect(absTol, SIGNAL(returnPressed()), this, SLOT(absTolProcessText()));
    toleranceLayout->addWidget(absTolLabel, 1, 0);
    toleranceLayout->addWidget(absTol, 1, 1);

    absTolSizeType = new QComboBox(toleranceGroup);
    absTolSizeType->addItem(tr("Absolute"), 0);
    absTolSizeType->addItem(tr("Fraction of Bounding Box"), 1);
    connect(absTolSizeType, SIGNAL(activated(int)), this, SLOT(absTolSizeTypeChanged(int)));
    toleranceLayout->addWidget(absTolSizeType, 1, 2);

    // Create the coordinate group
//     QGroupBox *coordinateGroup = new QGroupBox(central);
//     coordinateGroup->setTitle(tr("Coordinate System"));
//     mainLayout->addWidget(coordinateGroup, 2, 0);

//     QGridLayout *coordinateLayout = new QGridLayout(coordinateGroup);
//     coordinateLayout->setMargin(5);
//     coordinateLayout->setSpacing(10);

//     coordinateButtonGroup = new QButtonGroup(coordinateGroup);
//     QRadioButton *cartesianButton = new QRadioButton(tr("Cartesian"), coordinateGroup);
//     QRadioButton *cylindricalButton = new QRadioButton(tr("Cylindrical"), coordinateGroup);
//     coordinateButtonGroup->addButton(cartesianButton, 0);
//     coordinateButtonGroup->addButton(cylindricalButton, 1);

//     coordinateLayout->addWidget(cartesianButton, 0, 0);
//     coordinateLayout->addWidget(cylindricalButton, 0, 1);

//     connect(coordinateButtonGroup, SIGNAL(buttonClicked(int)), this,
//          SLOT(coordinateButtonGroupChanged(int)));

    // Create the punctures group box.
    QGroupBox *puncturesGroup = new QGroupBox(central);
    puncturesGroup->setTitle(tr("Punctures"));
    mainLayout->addWidget(puncturesGroup, 10, 0);
//    mainLayout->setStretchFactor(puncturesGroup, 100);

    QGridLayout *puncturesLayout = new QGridLayout(puncturesGroup);
    puncturesLayout->setMargin(5);
    puncturesLayout->setSpacing(10);


    QGroupBox *puncturesMinMaxGroup = new QGroupBox(central);
    puncturesMinMaxGroup->setTitle(tr(""));
    puncturesLayout->addWidget(puncturesMinMaxGroup, 0, 0);
    QGridLayout *puncturesMinMaxLayout = new QGridLayout(puncturesMinMaxGroup);
    puncturesMinMaxLayout->setMargin(5);
    puncturesMinMaxLayout->setSpacing(10);

    minPuncturesLabel = new QLabel(tr("Minimum"), puncturesGroup);
    puncturesMinMaxLayout->addWidget(minPuncturesLabel, 0, 0);
    minPunctures = new QSpinBox(central);
    minPunctures->setKeyboardTracking(false);
    minPunctures->setMinimum(1);
    minPunctures->setMaximum(10000);
    connect(minPunctures, SIGNAL(valueChanged(int)),
            this, SLOT(minPuncturesChanged(int)));
    puncturesMinMaxLayout->addWidget(minPunctures, 0, 1, Qt::AlignLeft);

    maxPuncturesLabel = new QLabel(tr("Maximum"), puncturesGroup);
    puncturesMinMaxLayout->addWidget(maxPuncturesLabel, 0, 2);
    maxPunctures = new QSpinBox(central);
    maxPunctures->setKeyboardTracking(false);
    maxPunctures->setMinimum(1);
    maxPunctures->setMaximum(10000);
    connect(maxPunctures, SIGNAL(valueChanged(int)),
            this, SLOT(maxPuncturesChanged(int)));
    puncturesMinMaxLayout->addWidget(maxPunctures, 0, 3, Qt::AlignLeft);


    QGroupBox *puncturePlotGroup = new QGroupBox(central);
    puncturePlotGroup->setTitle(tr(""));
    puncturesLayout->addWidget(puncturePlotGroup, 2, 0);
    QGridLayout *puncturePlotLayout = new QGridLayout(puncturePlotGroup);
    puncturePlotLayout->setMargin(5);
    puncturePlotLayout->setSpacing(10);

    puncturePlotTypeLabel = new QLabel(tr("Puncture plot type"), central);
    puncturePlotLayout->addWidget(puncturePlotTypeLabel, 1, 0);
    puncturePlotType = new QWidget(central);
    puncturePlotTypeButtonGroup= new QButtonGroup(puncturePlotType);
    QHBoxLayout *puncturePlotTypeLayout = new QHBoxLayout(puncturePlotType);
    puncturePlotTypeLayout->setMargin(0);
    puncturePlotTypeLayout->setSpacing(10);
    QRadioButton *puncturePlotTypeTypePoloidal =
      new QRadioButton(tr("Single"), puncturePlotType);
    puncturePlotTypeButtonGroup->addButton(puncturePlotTypeTypePoloidal,0);
    puncturePlotTypeLayout->addWidget(puncturePlotTypeTypePoloidal);
    QRadioButton *puncturePlotTypeTypeTorodial =
      new QRadioButton(tr("Double"), puncturePlotType);
    puncturePlotTypeButtonGroup->addButton(puncturePlotTypeTypeTorodial,1);
    puncturePlotTypeLayout->addWidget(puncturePlotTypeTypeTorodial);
    connect(puncturePlotTypeButtonGroup, SIGNAL(buttonClicked(int)),
            this, SLOT(puncturePlotTypeChanged(int)));
    puncturePlotLayout->addWidget(puncturePlotType, 1, 1);


    // Create the puncture period tolerance text field.
    puncturePeriodToleranceLabel = new QLabel(tr("Period tolerance"), puncturesGroup);
    puncturePlotLayout->addWidget(puncturePeriodToleranceLabel, 1, 2, Qt::AlignRight);
    puncturePeriodTolerance = new QLineEdit(puncturesGroup);
    connect(puncturePeriodTolerance, SIGNAL(returnPressed()),
            this, SLOT(puncturePeriodToleranceProcessText()));
    puncturePlotLayout->addWidget(puncturePeriodTolerance, 1, 3);


    maxStepsLabel = new QLabel(tr("Maximum number of steps"), puncturesGroup);
puncturePlotLayout->addWidget(maxStepsLabel, 2, 0, 1, 2);
    maxSteps = new QLineEdit(central);
    connect(maxSteps, SIGNAL(returnPressed()),
            this, SLOT(maxStepsProcessText()));
    puncturePlotLayout->addWidget(maxSteps, 2, 2);

    limitMaxTime = new QCheckBox(tr("Limit maximum time elapsed for particles"), puncturesGroup);
    connect(limitMaxTime, SIGNAL(toggled(bool)), this, SLOT(limitMaxTimeChanged(bool)));
    puncturePlotLayout->addWidget(limitMaxTime, 3, 0, 1, 2);
    maxTime = new QLineEdit(central);
    connect(maxTime, SIGNAL(returnPressed()), this, SLOT(maxTimeProcessText()));
    puncturePlotLayout->addWidget(maxTime, 3, 2);


    QGroupBox *puncturePlaneGroup = new QGroupBox(central);
    puncturePlaneGroup->setTitle(tr(""));
    puncturesLayout->addWidget(puncturePlaneGroup, 4, 0);
    QGridLayout *puncturePlaneLayout = new QGridLayout(puncturePlaneGroup);
    puncturePlaneLayout->setMargin(5);
    puncturePlaneLayout->setSpacing(10);

    puncturePlaneLabel = new QLabel(tr("Puncture Plane"), central);
    puncturePlaneLayout->addWidget(puncturePlaneLabel, 1, 0);
    puncturePlane = new QWidget(central);
    puncturePlaneButtonGroup= new QButtonGroup(puncturePlane);

    QHBoxLayout *puncturePlaneTypeLayout = new QHBoxLayout(puncturePlane);
    puncturePlaneTypeLayout->setMargin(0);
    puncturePlaneTypeLayout->setSpacing(10);
    QRadioButton *puncturePlaneTypePoloidal =
      new QRadioButton(tr("Poloidal"), puncturePlane);
    puncturePlaneButtonGroup->addButton(puncturePlaneTypePoloidal,0);
    puncturePlaneTypeLayout->addWidget(puncturePlaneTypePoloidal);
    QRadioButton *puncturePlaneTypeTorodial =
      new QRadioButton(tr("Toroidal"), puncturePlane);
    puncturePlaneButtonGroup->addButton(puncturePlaneTypeTorodial,1);
    puncturePlaneTypeLayout->addWidget(puncturePlaneTypeTorodial);
    connect(puncturePlaneButtonGroup, SIGNAL(buttonClicked(int)),
            this, SLOT(puncturePlaneChanged(int)));
    puncturePlaneLayout->addWidget(puncturePlane, 1, 1);
}

// ****************************************************************************
// Method: QvisPoincareWindow::CreateAnalysisTab
//
// Purpose:
//   Populates the analysis tab.
//
// Programmer: Dave Pugmire
// Creation:   Tue Dec 29 14:37:53 EST 2009
//
// Modifications:
//
// ****************************************************************************

void
QvisPoincareWindow::CreateAnalysisTab(QWidget *pageAnalysis)
{
    QGridLayout *mainLayout = new QGridLayout(pageAnalysis);
    mainLayout->setMargin(5);
    mainLayout->setSpacing(10);

    analysisLabel = new QLabel(tr("Analysis type"), central);
    mainLayout->addWidget(analysisLabel, 0, 0, Qt::AlignTop);
    analysis = new QWidget(central);
    analysisButtonGroup= new QButtonGroup(analysis);
    QHBoxLayout *analysisTypeLayout = new QHBoxLayout(analysis);
    analysisTypeLayout->setMargin(0);
    analysisTypeLayout->setSpacing(10);
    QRadioButton *analysisTypeNone =
      new QRadioButton(tr("None - Punctures only"), analysis);
    analysisButtonGroup->addButton(analysisTypeNone,0);
    analysisTypeLayout->addWidget(analysisTypeNone);
    QRadioButton *analysisTypeNormal =
      new QRadioButton(tr("Full"), analysis);
    analysisButtonGroup->addButton(analysisTypeNormal,1);
    analysisTypeLayout->addWidget(analysisTypeNormal);
    connect(analysisButtonGroup, SIGNAL(buttonClicked(int)),
            this, SLOT(analysisChanged(int)));
    mainLayout->addWidget(analysis, 0, 1, Qt::AlignTop);


   // Create the analysis group box.
    QGroupBox *analysisGroup = new QGroupBox(central);
    analysisGroup->setTitle(tr("Analysis"));
    mainLayout->addWidget(analysisGroup, 1, 0, 3, 3, Qt::AlignTop);

    QGridLayout *analysisLayout = new QGridLayout(analysisGroup);
    analysisLayout->setMargin(5);
    analysisLayout->setSpacing(10);

    maximumToroidalWindingLabel =
      new QLabel(tr("Maximum toroidal winding"), central);
    analysisLayout->addWidget(maximumToroidalWindingLabel, 0, 0);
    maximumToroidalWinding = new QSpinBox(central);
    maximumToroidalWinding->setKeyboardTracking(false);
    maximumToroidalWinding->setMinimum(0);
    maximumToroidalWinding->setMaximum(1000);
    connect(maximumToroidalWinding, SIGNAL(valueChanged(int)),
            this, SLOT(maximumToroidalWindingChanged(int)));
    analysisLayout->addWidget(maximumToroidalWinding, 0, 1);

    overrideToroidalWindingLabel =
      new QLabel(tr("Override toroidal winding"), central);
    analysisLayout->addWidget(overrideToroidalWindingLabel, 1, 0);
    overrideToroidalWinding = new QSpinBox(central);
    overrideToroidalWinding->setKeyboardTracking(false);
    overrideToroidalWinding->setMinimum(0);
    overrideToroidalWinding->setMaximum(1000);
    connect(overrideToroidalWinding, SIGNAL(valueChanged(int)),
            this, SLOT(overrideToroidalWindingChanged(int)));
    analysisLayout->addWidget(overrideToroidalWinding, 1, 1);

    overridePoloidalWindingLabel =
      new QLabel(tr("Override poloidal winding"), central);
    analysisLayout->addWidget(overridePoloidalWindingLabel, 1, 2);
    overridePoloidalWinding = new QSpinBox(central);
    overridePoloidalWinding->setKeyboardTracking(false);
    overridePoloidalWinding->setMinimum(0);
    overridePoloidalWinding->setMaximum(1000);
    connect(overridePoloidalWinding, SIGNAL(valueChanged(int)),
            this, SLOT(overridePoloidalWindingChanged(int)));
    analysisLayout->addWidget(overridePoloidalWinding, 1, 3);

    windingPairConfidenceLabel =
      new QLabel(tr("Winding pair confidence"), central);
    analysisLayout->addWidget(windingPairConfidenceLabel, 2, 0);
    windingPairConfidence = new QLineEdit(central);
    connect(windingPairConfidence, SIGNAL(returnPressed()),
            this, SLOT(windingPairConfidenceProcessText()));
    analysisLayout->addWidget(windingPairConfidence, 2, 1);


    rationalSurfaceFactorLabel =
      new QLabel(tr("Rational surface factor"), central);
    analysisLayout->addWidget(rationalSurfaceFactorLabel, 2, 2);
    rationalSurfaceFactor = new QLineEdit(central);
    connect(rationalSurfaceFactor, SIGNAL(returnPressed()),
            this, SLOT(rationalSurfaceFactorProcessText()));
    analysisLayout->addWidget(rationalSurfaceFactor, 2, 3);

    // Create the rational surface group box.
    QGroupBox *rationalSurfaceGroup = new QGroupBox(central);
    rationalSurfaceGroup->setTitle(tr("Rational Surfaces"));
    mainLayout->addWidget(rationalSurfaceGroup, 4, 0, 1, 3, Qt::AlignTop);

    QGridLayout *rationalSurfaceLayout = new QGridLayout(rationalSurfaceGroup);
    rationalSurfaceLayout->setMargin(5);
    rationalSurfaceLayout->setSpacing(10);

    showRationalSurfaces = new QCheckBox(tr("Detect Rational Surfaces"), rationalSurfaceGroup);
    connect(showRationalSurfaces, SIGNAL(toggled(bool)),
            this, SLOT(showRationalSurfacesChanged(bool)));
    rationalSurfaceLayout->addWidget(showRationalSurfaces, 0, 0);

    rationalSurfaceMaxIterationsLabel =
      new QLabel(tr("Maximum iterations"), rationalSurfaceGroup);
    rationalSurfaceMaxIterations = new QSpinBox(rationalSurfaceGroup);
    rationalSurfaceMaxIterations->setKeyboardTracking(false);
    rationalSurfaceMaxIterations->setRange(0, 100);
    connect(rationalSurfaceMaxIterations, SIGNAL(valueChanged(int)),
            this, SLOT(rationalSurfaceMaxIterationsChanged(int)));
    rationalSurfaceLayout->addWidget( rationalSurfaceMaxIterationsLabel, 0, 2);
    rationalSurfaceLayout->addWidget( rationalSurfaceMaxIterations, 0, 3);

    // Create the O/X Point group box.
    QGroupBox *criticalPointGroup = new QGroupBox(central);
    criticalPointGroup->setTitle(tr("Critical Points"));
    mainLayout->addWidget(criticalPointGroup, 5, 0, 1, 3, Qt::AlignTop);

    QGridLayout *criticalPointLayout = new QGridLayout(criticalPointGroup);
    criticalPointLayout->setMargin(5);
    criticalPointLayout->setSpacing(10);

    showOPoints = new QCheckBox(tr("Detect O Points"), criticalPointGroup);
    connect(showOPoints, SIGNAL(toggled(bool)),
            this, SLOT(showOPointsChanged(bool)));
    criticalPointLayout->addWidget(showOPoints, 0, 0);

    OPointMaxIterationsLabel =
      new QLabel(tr("Maximum iterations"), criticalPointGroup);
    OPointMaxIterations = new QSpinBox(criticalPointGroup);
    OPointMaxIterations->setKeyboardTracking(false);
    OPointMaxIterations->setRange(0, 10);
    connect(OPointMaxIterations, SIGNAL(valueChanged(int)),
            this, SLOT(OPointMaxIterationsChanged(int)));
    criticalPointLayout->addWidget( OPointMaxIterationsLabel, 0, 2);
    criticalPointLayout->addWidget( OPointMaxIterations, 0, 3);


    // Create the O Line analysis group box.
    QGroupBox *OLineAnalysisGroup = new QGroupBox(central);
    OLineAnalysisGroup->setTitle(tr("O-Line Analysis"));
    mainLayout->addWidget(OLineAnalysisGroup, 6, 0, 2, 3, Qt::AlignTop);

    QGridLayout *OLineAnalysisLayout = new QGridLayout(OLineAnalysisGroup);
    OLineAnalysisLayout->setMargin(5);
    OLineAnalysisLayout->setSpacing(10);

    performOLineAnalysis = new QCheckBox(tr("Perform O-Line analysis"),
                                OLineAnalysisGroup);
    connect(performOLineAnalysis, SIGNAL(toggled(bool)),
            this, SLOT(performOLineAnalysisChanged(bool)));
    OLineAnalysisLayout->addWidget(performOLineAnalysis, 0, 0);

    OLineToroidalWindingLabel =
      new QLabel(tr("O-Line toroidal windings"), OLineAnalysisGroup);
    OLineToroidalWinding = new QSpinBox(OLineAnalysisGroup);
    OLineToroidalWinding->setKeyboardTracking(false);
    OLineToroidalWinding->setRange(1, 999);
    connect(OLineToroidalWinding, SIGNAL(valueChanged(int)),
            this, SLOT(OLineToroidalWindingChanged(int)));
    OLineAnalysisLayout->addWidget( OLineToroidalWindingLabel, 0, 2);
    OLineAnalysisLayout->addWidget( OLineToroidalWinding, 0, 3);


    OLineAxisFileDialogButton = new QPushButton(central);
    OLineAxisFileDialogButton->setText("O-Line Axis Point File");
    connect(OLineAxisFileDialogButton, SIGNAL(clicked()),
            this, SLOT(OLineAxisFileDialogButtonClicked()));
    OLineAnalysisLayout->addWidget(OLineAxisFileDialogButton, 2, 0);


    OLineAxisFileName = new QLineEdit(central);
    connect(OLineAxisFileName, SIGNAL(returnPressed()),
            this, SLOT(OLineAxisFileNameProcessText()));
    OLineAnalysisLayout->addWidget(OLineAxisFileName, 2, 1, 1, 3);

    // Create the options group box.
    QGroupBox *analysisOptionsGroup = new QGroupBox(central);
    analysisOptionsGroup->setTitle(tr("Options"));
    mainLayout->addWidget(analysisOptionsGroup, 9, 0, 1, 3, Qt::AlignTop);

    QGridLayout *analysisOptionsLayout = new QGridLayout(analysisOptionsGroup);
    analysisOptionsLayout->setMargin(5);
    analysisOptionsLayout->setSpacing(10);

    showChaotic =
      new QCheckBox(tr("Show chaotic fieldlines as points"), analysisOptionsGroup);
    connect(showChaotic, SIGNAL(toggled(bool)),
            this, SLOT(showChaoticChanged(bool)));
    analysisOptionsLayout->addWidget(showChaotic, 0, 0, 1, 2);

    showIslands = new QCheckBox(tr("Show islands only"), analysisOptionsGroup);
    connect(showIslands, SIGNAL(toggled(bool)),
            this, SLOT(showIslandsChanged(bool)));
    analysisOptionsLayout->addWidget(showIslands, 0, 2);


    summaryFlag = new QCheckBox(tr("Summary"), analysisOptionsGroup);
    connect(summaryFlag, SIGNAL(toggled(bool)),
            this, SLOT(summaryFlagChanged(bool)));
    analysisOptionsLayout->addWidget(summaryFlag, 1, 0);

    verboseFlag = new QCheckBox(tr("Verbose"), analysisOptionsGroup);
    connect(verboseFlag, SIGNAL(toggled(bool)),
            this, SLOT(verboseFlagChanged(bool)));
    analysisOptionsLayout->addWidget(verboseFlag, 1, 1);

    show1DPlots = new QCheckBox(tr("Show distance/ridgeline plots"), analysisOptionsGroup);
    connect(show1DPlots, SIGNAL(toggled(bool)),
            this, SLOT(show1DPlotsChanged(bool)));
    analysisOptionsLayout->addWidget(show1DPlots, 1, 2);
}

// ****************************************************************************
// Method: QvisPoincareWindow::CreateAdvancedTab
//
// Purpose:
//   Populates the appearance tab.
//
// Programmer: Dave Pugmire
// Creation:   Tue Dec 29 14:37:53 EST 2009
//
// Modifications:
//
// ****************************************************************************

void
QvisPoincareWindow::CreateAppearanceTab(QWidget *pageAppearance)
{
    QGridLayout *mainLayout = new QGridLayout(pageAppearance);
    mainLayout->setMargin(5);
    mainLayout->setSpacing(10);

    // Create the data group box.
    QGroupBox *dataGroup = new QGroupBox(central);
    dataGroup->setTitle(tr("Data"));
    mainLayout->addWidget(dataGroup, 0, 0);

    QGridLayout *dataLayout = new QGridLayout(dataGroup);
    dataLayout->setMargin(5);
    dataLayout->setSpacing(10);
    dataLayout->setColumnStretch(2, 10);


    dataValueLabel = new QLabel(tr("Data value:"), dataGroup);
    dataLayout->addWidget(dataValueLabel, 0, 0);

    dataValueCombo = new QComboBox(dataGroup);
    dataValueCombo->addItem(tr("None"));
    dataValueCombo->addItem(tr("Safety Factor Q"));
    dataValueCombo->addItem(tr("Safety Factor P"));
    dataValueCombo->addItem(tr("Safety Factor Q if Q==P"));
    dataValueCombo->addItem(tr("Safety Factor P if P!=Q"));
    dataValueCombo->addItem(tr("Toroidal Windings"));
    dataValueCombo->addItem(tr("Poloidal Windings Q"));
    dataValueCombo->addItem(tr("Poloidal Windings P"));
    dataValueCombo->addItem(tr("Fieldline Order"));
    dataValueCombo->addItem(tr("Point Order"));
    dataValueCombo->addItem(tr("Plane Order"));
    dataValueCombo->addItem(tr("Winding Group Order"));
    dataValueCombo->addItem(tr("Winding Point Order"));
    dataValueCombo->addItem(tr("Winding Point Order Modulo"));
    connect(dataValueCombo, SIGNAL(activated(int)),
           this, SLOT(dataValueChanged(int)));
    dataLayout->addWidget(dataValueCombo, 0, 1);

    dataLayout->addWidget(new QLabel(tr("   "), dataGroup), 0, 2);

    // Create the display group box.
    QGroupBox *displayGroup = new QGroupBox(central);
    displayGroup->setTitle(tr("Display"));
    mainLayout->addWidget(displayGroup, 1, 0);

    QGridLayout *displayLayout = new QGridLayout(displayGroup);
    displayLayout->setMargin(5);
    displayLayout->setSpacing(10);

    meshTypeLabel = new QLabel(tr("Mesh type:"), displayGroup);
    displayLayout->addWidget(meshTypeLabel, 0, 0);

    meshTypeCombo = new QComboBox(displayGroup);
    meshTypeCombo->addItem(tr("Curves"));
    meshTypeCombo->addItem(tr("Surfaces"));
    connect(meshTypeCombo, SIGNAL(activated(int)),
           this, SLOT(meshTypeChanged(int)));
    displayLayout->addWidget(meshTypeCombo, 0, 1);

    showLines = new QCheckBox(tr("Show Lines"), displayGroup);
    connect(showLines, SIGNAL(toggled(bool)),
            this, SLOT(showLinesChanged(bool)));
    displayLayout->addWidget(showLines, 0, 2);

    showPoints = new QCheckBox(tr("Show Points"), displayGroup);
    connect(showPoints, SIGNAL(toggled(bool)), this,
            SLOT(showPointsChanged(bool)));
    displayLayout->addWidget(showPoints, 0, 3);

    numberPlanesLabel = new QLabel(tr("Number of planes"), displayGroup);
    displayLayout->addWidget(numberPlanesLabel, 1, 0);
    numberPlanes = new QSpinBox(displayGroup);
    numberPlanes->setKeyboardTracking(false);
    numberPlanes->setMinimum(1);
    numberPlanes->setMaximum(250);
    connect(numberPlanes, SIGNAL(valueChanged(int)),
            this, SLOT(numberPlanesChanged(int)));
    displayLayout->addWidget(numberPlanes, 1, 1);

    singlePlaneLabel = new QLabel(tr("Base plane (degrees)"), displayGroup);
    displayLayout->addWidget(singlePlaneLabel, 1, 2);
    singlePlane = new QLineEdit(displayGroup);
    connect(singlePlane, SIGNAL(returnPressed()),
            this, SLOT(singlePlaneProcessText()));
    displayLayout->addWidget(singlePlane, 1, 3);

    // Create the overlaps group box.
    QGroupBox *overlapsGroup = new QGroupBox(central);
    overlapsGroup->setTitle(tr("Overlaps"));
    mainLayout->addWidget(overlapsGroup, 3, 0);

    QGridLayout *overlapsLayout = new QGridLayout(overlapsGroup);
    overlapsLayout->setMargin(5);
    overlapsLayout->setSpacing(10);

    overlapsLabel = new QLabel(tr("Overlaping curve sections"), central);
    overlapsLayout->addWidget(overlapsLabel, 0, 0, Qt::AlignTop);
    overlaps = new QWidget(central);
    overlapsButtonGroup= new QButtonGroup(overlaps);
    QHBoxLayout *overlapTypeLayout = new QHBoxLayout(overlaps);
    overlapTypeLayout->setMargin(0);
    overlapTypeLayout->setSpacing(10);
    QRadioButton *overlapTypeRaw =
      new QRadioButton(tr("Raw"), overlaps);
    overlapsButtonGroup->addButton(overlapTypeRaw,0);
    overlapTypeLayout->addWidget(overlapTypeRaw);
    QRadioButton *overlapTypeRemove =
      new QRadioButton(tr("Remove"), overlaps);
    overlapsButtonGroup->addButton(overlapTypeRemove,1);
    overlapTypeLayout->addWidget(overlapTypeRemove);
    QRadioButton *overlapTypeMerge =
      new QRadioButton(tr("Merge"), overlaps);
    overlapsButtonGroup->addButton(overlapTypeMerge,2);
    overlapTypeLayout->addWidget(overlapTypeMerge);
    QRadioButton *overlapTypeSmooth =
      new QRadioButton(tr("Smooth"), overlaps);
    overlapsButtonGroup->addButton(overlapTypeSmooth,3);
    overlapTypeLayout->addWidget(overlapTypeSmooth);
    connect(overlapsButtonGroup, SIGNAL(buttonClicked(int)),
            this, SLOT(overlapsChanged(int)));
    overlapsLayout->addWidget(overlaps, 0, 1, Qt::AlignTop);


    // Streamlines/Pathline Group.
    QGroupBox *icGrp = new QGroupBox(pageAppearance);
    icGrp->setTitle(tr("Streamlines vs Pathlines"));
    mainLayout->addWidget(icGrp, 4, 0);

    QGridLayout *icGrpLayout = new QGridLayout(icGrp);
    icGrpLayout->setSpacing(10);
    icGrpLayout->setColumnStretch(1,10);

    icButtonGroup = new QButtonGroup(icGrp);
    QRadioButton *streamlineButton = new QRadioButton(tr("Streamline\n    Compute trajectories in an (instantaneous) snapshot of the vector field.\n    Uses and loads vector data from only the current time slice."), icGrp);
    QRadioButton *pathlineButton = new QRadioButton(tr("Pathline    \n    Compute trajectories in the time-varying vector field.\n    Uses and loads vector data from all relevant time slices"), icGrp);
    streamlineButton->setChecked(true);
    icButtonGroup->addButton(streamlineButton, 0);
    icButtonGroup->addButton(pathlineButton, 1);
    icGrpLayout->addWidget(streamlineButton, 1, 0);
    icGrpLayout->addWidget(pathlineButton, 2, 0);
    connect(icButtonGroup, SIGNAL(buttonClicked(int)), this, SLOT(icButtonGroupChanged(int)));

    // Pathline Options
    QGroupBox *pathlineOptionsGrp = new QGroupBox(icGrp);
    pathlineOptionsGrp->setTitle(tr("Pathline Options"));
    icGrpLayout->addWidget(pathlineOptionsGrp, 3, 0);

    QGridLayout *pathlineOptionsGrpLayout = new QGridLayout(pathlineOptionsGrp);
    pathlineOptionsGrpLayout->setSpacing(10);
    pathlineOptionsGrpLayout->setColumnStretch(1,10);

    pathlineOverrideStartingTimeFlag = new QCheckBox(tr("Override Starting Time"), pathlineOptionsGrp);
    connect(pathlineOverrideStartingTimeFlag, SIGNAL(toggled(bool)),
            this, SLOT(pathlineOverrideStartingTimeFlagChanged(bool)));
    pathlineOptionsGrpLayout->addWidget(pathlineOverrideStartingTimeFlag, 1, 0);

    QLabel *pathlineOverrideStartingTimeLabel = new QLabel(tr("Time"), pathlineOptionsGrp);
    pathlineOverrideStartingTimeLabel->setAlignment(Qt::AlignRight | Qt::AlignCenter);
    pathlineOptionsGrpLayout->addWidget(pathlineOverrideStartingTimeLabel, 1, 1);
    pathlineOverrideStartingTime = new QLineEdit(pathlineOptionsGrp);
    connect(pathlineOverrideStartingTime, SIGNAL(returnPressed()),
            this, SLOT(pathlineOverrideStartingTimeProcessText()));
    pathlineOptionsGrpLayout->addWidget(pathlineOverrideStartingTime, 1, 2);

    QLabel *pathlinePeriodLabel = new QLabel(tr("Period"), pathlineOptionsGrp);
    pathlinePeriodLabel->setAlignment(Qt::AlignRight | Qt::AlignCenter);
    pathlineOptionsGrpLayout->addWidget(pathlinePeriodLabel, 1, 3);
    pathlinePeriod = new QLineEdit(pathlineOptionsGrp);
    connect(pathlinePeriod, SIGNAL(returnPressed()),
            this, SLOT(pathlinePeriodProcessText()));
    pathlineOptionsGrpLayout->addWidget(pathlinePeriod, 1, 4);

    QGroupBox *cmfeOptionsGrp = new QGroupBox(pathlineOptionsGrp);
    cmfeOptionsGrp->setTitle(tr("How to perform interpolation over time"));
    pathlineOptionsGrpLayout->addWidget(cmfeOptionsGrp, 2, 0, 2, 5);

    QGridLayout *cmfeOptionsGrpLayout = new QGridLayout(cmfeOptionsGrp);
    cmfeOptionsGrpLayout->setSpacing(10);
    cmfeOptionsGrpLayout->setColumnStretch(1,10);

    pathlineCMFEButtonGroup = new QButtonGroup(cmfeOptionsGrp);
    QRadioButton *connButton = new QRadioButton(tr("Mesh is static over time (fast, but special purpose)"), cmfeOptionsGrp);
    QRadioButton *posButton = new QRadioButton(tr("Mesh changes over time (slow, but robust)"), cmfeOptionsGrp);
    posButton->setChecked(true);
    pathlineCMFEButtonGroup->addButton(connButton, 0);
    pathlineCMFEButtonGroup->addButton(posButton, 1);
    cmfeOptionsGrpLayout->addWidget(connButton, 2, 0, 1, 5);
    cmfeOptionsGrpLayout->addWidget(posButton, 3, 0, 1, 5);
    connect(pathlineCMFEButtonGroup, SIGNAL(buttonClicked(int)), this, SLOT(pathlineCMFEButtonGroupChanged(int)));
}


// ****************************************************************************
// Method: QvisPoincareWindow::CreateAdvancedTab
//
// Purpose:
//   Populates the advanced tab.
//
// Programmer: Dave Pugmire
// Creation:   Tue Dec 29 14:37:53 EST 2009
//
// Modifications:
//
// ****************************************************************************

void
QvisPoincareWindow::CreateAdvancedTab(QWidget *pageAdvanced)
{
    QGridLayout *mainLayout = new QGridLayout(pageAdvanced);
    mainLayout->setMargin(5);
    mainLayout->setSpacing(5);

    QGroupBox *algoGrp = new QGroupBox(pageAdvanced);
    algoGrp->setTitle(tr("Parallel integration options"));
    //mainLayout->addWidget(algoGrp, 0, 0, 1, 4);
    mainLayout->addWidget(algoGrp, 0, 0);

    // Algorithm group.
    QGridLayout *algoGLayout = new QGridLayout(algoGrp);
    algoGLayout->setSpacing(10);
    algoGLayout->setColumnStretch(1,10);

    parallelAlgoLabel = new QLabel(tr("Parallelization"), algoGrp);
    parallelAlgo = new QComboBox(algoGrp);
    parallelAlgo->addItem(tr("Parallelize over curves"));
    parallelAlgo->addItem(tr("Parallelize over domains"));
    parallelAlgo->addItem(tr("Parallelize over curves and Domains"));
    parallelAlgo->addItem(tr("Have VisIt select the best algorithm"));
    connect(parallelAlgo, SIGNAL(activated(int)),
            this, SLOT(parallelAlgorithmChanged(int)));
    algoGLayout->addWidget( parallelAlgoLabel, 1,0);
    algoGLayout->addWidget( parallelAlgo, 1,1);

    maxSLCountLabel = new QLabel(tr("Communication threshold"), algoGrp);
    maxSLCount = new QSpinBox(algoGrp);
    maxSLCount->setKeyboardTracking(false);
    maxSLCount->setMinimum(1);
    maxSLCount->setMaximum(100000);
    connect(maxSLCount, SIGNAL(valueChanged(int)),
            this, SLOT(maxSLCountChanged(int)));
    algoGLayout->addWidget( maxSLCountLabel, 2,0);
    algoGLayout->addWidget( maxSLCount,2,1);

    maxDomainCacheLabel = new QLabel(tr("Domain cache size"), algoGrp);
    maxDomainCache = new QSpinBox(algoGrp);
    maxDomainCache->setKeyboardTracking(false);
    maxDomainCache->setMinimum(1);
    maxDomainCache->setMaximum(100000);
    connect(maxDomainCache, SIGNAL(valueChanged(int)),
            this, SLOT(maxDomainCacheChanged(int)));
    algoGLayout->addWidget( maxDomainCacheLabel, 3,0);
    algoGLayout->addWidget( maxDomainCache, 3,1);

    workGroupSizeLabel = new QLabel(tr("Work group size"), algoGrp);
    workGroupSize = new QSpinBox(algoGrp);
    workGroupSize->setKeyboardTracking(false);
    workGroupSize->setMinimum(2);
    workGroupSize->setMaximum(1000000);
    connect(workGroupSize, SIGNAL(valueChanged(int)),
            this, SLOT(workGroupSizeChanged(int)));
    algoGLayout->addWidget( workGroupSizeLabel, 4,0);
    algoGLayout->addWidget( workGroupSize, 4,1);

    // Warnings group.
    QGroupBox *warningsGrp = new QGroupBox(pageAdvanced);
    warningsGrp->setTitle(tr("Warnings"));
    //mainLayout->addWidget(warningsGrp, 2, 0, 1, 4);
    mainLayout->addWidget(warningsGrp, 1, 0);

    QGridLayout *warningsGLayout = new QGridLayout(warningsGrp);
    warningsGLayout->setSpacing(10);
    warningsGLayout->setColumnStretch(1,10);

    issueWarningForMaxSteps = new QCheckBox(central);
    connect(issueWarningForMaxSteps, SIGNAL(toggled(bool)),
            this, SLOT(issueWarningForMaxStepsChanged(bool)));
    warningsGLayout->addWidget(issueWarningForMaxSteps, 0, 0);
    QLabel *maxStepsLabel = new QLabel(tr("Issue warning when the maximum number of steps is reached."), warningsGrp);
    warningsGLayout->addWidget(maxStepsLabel, 0, 1, 1, 2);

    issueWarningForStepsize = new QCheckBox(central);
    connect(issueWarningForStepsize, SIGNAL(toggled(bool)),
            this, SLOT(issueWarningForStepsizeChanged(bool)));
    warningsGLayout->addWidget(issueWarningForStepsize, 1, 0);
    QLabel *stepsizeLabel = new QLabel(tr("Issue warning when a step size underflow is detected."), warningsGrp);
    warningsGLayout->addWidget(stepsizeLabel, 1, 1, 1, 2);

    issueWarningForStiffness = new QCheckBox(central);
    connect(issueWarningForStiffness, SIGNAL(toggled(bool)),
            this, SLOT(issueWarningForStiffnessChanged(bool)));
    warningsGLayout->addWidget(issueWarningForStiffness, 2, 0);
    QLabel *stiffnessLabel = new QLabel(tr("Issue warning when a stiffness condition is detected."), warningsGrp);
    warningsGLayout->addWidget(stiffnessLabel, 2, 1, 1, 2);
    QLabel *stiffnessDescLabel1 = new QLabel(tr("(Stiffness refers to one vector component being so much "), warningsGrp);
    warningsGLayout->addWidget(stiffnessDescLabel1, 3, 1, 1, 2);
    QLabel *stiffnessDescLabel2 = new QLabel(tr("larger than another that tolerances can't be met.)"), warningsGrp);
    warningsGLayout->addWidget(stiffnessDescLabel2, 4, 1, 1, 2);

    issueWarningForCriticalPoints = new QCheckBox(central);
    connect(issueWarningForCriticalPoints, SIGNAL(toggled(bool)),
            this, SLOT(issueWarningForCriticalPointsChanged(bool)));
    warningsGLayout->addWidget(issueWarningForCriticalPoints, 5, 0);
    QLabel *critPointLabel = new QLabel(tr("Issue warning when a curve doesn't terminate at a critical point."), warningsGrp);
    warningsGLayout->addWidget(critPointLabel, 5, 1, 1, 2);
    QLabel *critPointDescLabel = new QLabel(tr("(I.e. the curve circles around the critical point without stopping.)"), warningsGrp);
    warningsGLayout->addWidget(critPointDescLabel, 6, 1, 1, 2);
    criticalPointThresholdLabel = new QLabel(tr("Speed cutoff for critical points"), warningsGrp);
    criticalPointThresholdLabel->setAlignment(Qt::AlignRight | Qt::AlignCenter);
    warningsGLayout->addWidget(criticalPointThresholdLabel, 7, 1);
    criticalPointThreshold = new QLineEdit(warningsGrp);
    criticalPointThreshold->setAlignment(Qt::AlignLeft);
    connect(criticalPointThreshold, SIGNAL(returnPressed()),
            this, SLOT(criticalPointThresholdProcessText()));
    warningsGLayout->addWidget(criticalPointThreshold, 7, 2);
}


// ****************************************************************************
// Method: QvisPoincareWindow::UpdateWindow
//
// Purpose:
//   Updates the widgets in the window when the subject changes.
//
// Note:       Autogenerated by xml2window.
//
// Programmer: xml2window
// Creation:   omitted
//
// Modifications:
//   Kathleen Biagas, Thu Apr 9 07:19:54 MST 2015
//   Use helper function DoubleToQString for consistency in formatting across
//   all windows.
//
// ****************************************************************************

void
QvisPoincareWindow::UpdateWindow(bool doAll)
{
    for(int i = 0; i < atts->NumAttributes(); ++i)
    {
        if(!doAll)
        {
            if(!atts->IsSelected(i))
            {
                continue;
            }
        }

        switch(i)
        {
          case PoincareAttributes::ID_minPunctures:
            minPunctures->blockSignals(true);
            minPunctures->setValue(atts->GetMinPunctures());
            minPunctures->blockSignals(false);
            break;
          case PoincareAttributes::ID_maxPunctures:
            maxPunctures->blockSignals(true);
            maxPunctures->setValue(atts->GetMaxPunctures());
            maxPunctures->blockSignals(false);
            break;
          case PoincareAttributes::ID_puncturePlotType:
            puncturePlotTypeButtonGroup->blockSignals(true);
            if(puncturePlotTypeButtonGroup->button((int)atts->GetPuncturePlotType()) != 0)
                puncturePlotTypeButtonGroup->button((int)atts->GetPuncturePlotType())->setChecked(true);
            puncturePlotTypeButtonGroup->blockSignals(false);

            if( (int) atts->GetPuncturePlotType() == PoincareAttributes::Single)
            {
              puncturePeriodTolerance->setEnabled(false);
              puncturePeriodToleranceLabel->setEnabled(false);
              maxSteps->setEnabled(false);
              maxStepsLabel->setEnabled(false);
              limitMaxTime->setEnabled(false);
              maxTime->setEnabled(false);
            }
            else
            {
              puncturePeriodTolerance->setEnabled(true);
              puncturePeriodToleranceLabel->setEnabled(true);
              maxSteps->setEnabled(true);
              maxStepsLabel->setEnabled(true);
              limitMaxTime->setEnabled(true);
              maxTime->setEnabled(atts->GetTerminateByTime());
            }

            break;
          case PoincareAttributes::ID_puncturePeriodTolerance:
            puncturePeriodTolerance->setText(DoubleToQString(atts->GetPuncturePeriodTolerance()));
            break;
          case PoincareAttributes::ID_maxSteps:
            maxSteps->setText(IntToQString(atts->GetMaxSteps()));
            break;
          case PoincareAttributes::ID_terminateByTime:
            limitMaxTime->blockSignals(true);
            limitMaxTime->setChecked(atts->GetTerminateByTime());
            limitMaxTime->blockSignals(false);
            maxTime->setEnabled(atts->GetTerminateByTime());
            break;
          case PoincareAttributes::ID_termTime:
            maxTime->setText(DoubleToQString(atts->GetTermTime()));
            break;
          case PoincareAttributes::ID_puncturePlane:
            puncturePlaneButtonGroup->blockSignals(true);
            if(puncturePlaneButtonGroup->button((int)atts->GetPuncturePlane()) != 0)
                puncturePlaneButtonGroup->button((int)atts->GetPuncturePlane())->setChecked(true);
            puncturePlaneButtonGroup->blockSignals(false);
            break;
          case PoincareAttributes::ID_sourceType:
            if (atts->GetSourceType() == PoincareAttributes::SpecifiedPoint)
            {
                pointSource->setEnabled(true);
                pointSourceLabel->setEnabled(true);

                pointSource->show();
                pointSourceLabel->show();
            }
            else
            {
                pointSource->setEnabled(false);
                pointSourceLabel->setEnabled(false);

                pointSource->hide();
                pointSourceLabel->hide();
            }
            if (atts->GetSourceType() == PoincareAttributes::PointList)
            {
                pointList->setEnabled(true);
                pointListDelPoint->setEnabled(true);
                pointListDelAllPoints->setEnabled(true);
                pointListAddPoint->setEnabled(true);
                pointListReadPoints->setEnabled(true);

                pointList->show();
                pointListDelPoint->show();
                pointListDelAllPoints->show();
                pointListAddPoint->show();
                pointListReadPoints->show();
            }
            else
            {
                pointList->setEnabled(false);
                pointListDelPoint->setEnabled(false);
                pointListDelAllPoints->setEnabled(false);
                pointListAddPoint->setEnabled(false);
                pointListReadPoints->setEnabled(false);

                pointList->hide();
                pointListDelPoint->hide();
                pointListDelAllPoints->hide();
                pointListAddPoint->hide();
                pointListReadPoints->hide();
            }

            if (atts->GetSourceType() == PoincareAttributes::SpecifiedLine)
            {
                lineStart->setEnabled(true);
                lineStartLabel->setEnabled(true);

                lineStartLabel->show();
                lineStart->show();

                lineEnd->setEnabled(true);
                lineEndLabel->setEnabled(true);

                lineEndLabel->show();
                lineEnd->show();

                pointDensity->setEnabled(true);
                pointDensityLabel->setEnabled(true);

                pointDensityLabel->show();
                pointDensity->show();
            }
            else
            {
                lineStart->setEnabled(false);
                lineStartLabel->setEnabled(false);

                lineStartLabel->hide();
                lineStart->hide();

                lineEnd->setEnabled(false);
                lineEndLabel->setEnabled(false);

                lineEndLabel->hide();
                lineEnd->hide();

                pointDensityLabel->setEnabled(false);
                pointDensity->setEnabled(false);

                pointDensityLabel->hide();
                pointDensity->hide();
            }

            sourceTypeCombo->blockSignals(true);
            sourceTypeCombo->setCurrentIndex((int)atts->GetSourceType());
            sourceTypeCombo->blockSignals(false);
            break;
          case PoincareAttributes::ID_velocitySource:
            velocitySource->setText(DoublesToQString(atts->GetVelocitySource(), 3));
            break;
          case PoincareAttributes::ID_pointSource:
            pointSource->setText(DoublesToQString(atts->GetPointSource(), 3));
            break;
          case PoincareAttributes::ID_pointList:
          {
              std::vector<double> points = atts->GetPointList();

              pointList->clear();
              for (size_t i = 0; i < points.size(); i+= 3)
              {
                  char tmp[256];
                  sprintf(tmp, "%lf %lf %lf",
                          points[i], points[i+1], points[i+2]);
                  QString str = tmp;
                  QListWidgetItem *item = new QListWidgetItem(str, pointList);
                  item->setFlags(item->flags() | Qt::ItemIsEditable);
                  pointList->setCurrentItem(item);
              }

              break;
          }
          case PoincareAttributes::ID_lineStart:
            lineStart->setText(DoublesToQString(atts->GetLineStart(), 3));
            break;
          case PoincareAttributes::ID_lineEnd:
            lineEnd->setText(DoublesToQString(atts->GetLineEnd(), 3));
            break;
          case PoincareAttributes::ID_pointDensity:
            pointDensity->blockSignals(true);
            pointDensity->setValue(atts->GetPointDensity());
            pointDensity->blockSignals(false);
            break;

        case PoincareAttributes::ID_fieldType:
            // Update lots of widget visibility and enabled states.
            UpdateFieldAttributes();

            fieldType->blockSignals(true);
            fieldType->setCurrentIndex(atts->GetFieldType());
            fieldType->blockSignals(false);

            integrationType->blockSignals(true);
            if (atts->GetFieldType() == PoincareAttributes::M3DC12DField)
            {
              atts->SetIntegrationType(PoincareAttributes::M3DC12DIntegrator);
              integrationType->setCurrentIndex(PoincareAttributes::M3DC12DIntegrator);
              UpdateIntegrationAttributes();
            }
            else if (atts->GetFieldType() == PoincareAttributes::NIMRODField)
            {
              atts->SetIntegrationType(PoincareAttributes::AdamsBashforth);
              integrationType->setCurrentIndex(PoincareAttributes::AdamsBashforth);
              UpdateIntegrationAttributes();
            }
            else if (atts->GetIntegrationType() == PoincareAttributes::M3DC12DIntegrator)
            {
              atts->SetIntegrationType(PoincareAttributes::DormandPrince);
              integrationType->setCurrentIndex(PoincareAttributes::DormandPrince);
              UpdateIntegrationAttributes();
            }
            integrationType->blockSignals(false);

            break;
        // case PoincareAttributes::ID_forceNodeCenteredData:
        //     forceNodal->blockSignals(true);
        //     forceNodal->setChecked(atts->GetForceNodeCenteredData());
        //     forceNodal->blockSignals(false);
        //     break;

        case PoincareAttributes::ID_fieldConstant:
            fieldConstant->setText(DoubleToQString(atts->GetFieldConstant()));
            break;
        case PoincareAttributes::ID_integrationType:
            // Update lots of widget visibility and enabled states.
            UpdateIntegrationAttributes();

            integrationType->blockSignals(true);
            integrationType->setCurrentIndex(atts->GetIntegrationType());
            integrationType->blockSignals(false);

            fieldType->blockSignals(true);
            if (atts->GetIntegrationType() == PoincareAttributes::M3DC12DIntegrator)
            {
              atts->SetFieldType(PoincareAttributes::M3DC12DField);
              fieldType->setCurrentIndex(PoincareAttributes::M3DC12DField);
              UpdateFieldAttributes();
            }
            else if (atts->GetFieldType() == PoincareAttributes::M3DC12DField)
            {
              atts->SetFieldType(PoincareAttributes::Default);
              fieldType->setCurrentIndex(PoincareAttributes::Default);
              UpdateFieldAttributes();
            }
            fieldType->blockSignals(false);

            break;
        case PoincareAttributes::ID_maxStepLength:
            maxStepLength->setText(DoubleToQString(atts->GetMaxStepLength()));
            break;
        case PoincareAttributes::ID_relTol:
            relTol->setText(DoubleToQString(atts->GetRelTol()));
            break;
        case PoincareAttributes::ID_absTolSizeType:
            absTolSizeType->blockSignals(true);
            absTolSizeType->setCurrentIndex((int) atts->GetAbsTolSizeType());
            absTolSizeType->blockSignals(false);
            if (atts->GetAbsTolSizeType() == PoincareAttributes::FractionOfBBox)
            {
                absTol->setText(DoubleToQString(atts->GetAbsTolBBox()));
            }
            if (atts->GetAbsTolSizeType() == PoincareAttributes::Absolute)
            {
                absTol->setText(DoubleToQString(atts->GetAbsTolAbsolute()));
            }
            break;
        case PoincareAttributes::ID_absTolBBox:
            if (atts->GetAbsTolSizeType() == PoincareAttributes::FractionOfBBox)
            {
                absTol->setText(DoubleToQString(atts->GetAbsTolBBox()));
            }
            break;
        case PoincareAttributes::ID_absTolAbsolute:
            if (atts->GetAbsTolSizeType() == PoincareAttributes::Absolute)
            {
                absTol->setText(DoubleToQString(atts->GetAbsTolAbsolute()));
            }
            break;
//          case PoincareAttributes::ID_coordinateSystem:
//          coordinateButtonGroup->blockSignals(true);
//             coordinateButtonGroup->button(atts->GetCoordinateSystem())->setChecked(true);;
//             coordinateButtonGroup->blockSignals(false);
//            break;
          case PoincareAttributes::ID_analysis:
            analysisButtonGroup->blockSignals(true);
            analysisButtonGroup->button((int)atts->GetAnalysis())->setChecked(true);
            if(atts->GetAnalysis() == 0)
            {
              showLines->setChecked(false);
              showPoints->setChecked(true);

              atts->SetShowLines(false);
              atts->SetShowPoints(false);
            }

            analysisButtonGroup->blockSignals(false);
            break;
          case PoincareAttributes::ID_maximumToroidalWinding:
            maximumToroidalWinding->blockSignals(true);
            maximumToroidalWinding->setValue(atts->GetMaximumToroidalWinding());
            maximumToroidalWinding->blockSignals(false);
            break;
          case PoincareAttributes::ID_overrideToroidalWinding:
            overrideToroidalWinding->blockSignals(true);
            overrideToroidalWinding->setValue(atts->GetOverrideToroidalWinding());
            overrideToroidalWinding->blockSignals(false);
            break;
          case PoincareAttributes::ID_overridePoloidalWinding:
            overridePoloidalWinding->blockSignals(true);
            overridePoloidalWinding->setValue(atts->GetOverridePoloidalWinding());
            overridePoloidalWinding->blockSignals(false);
            break;
          case PoincareAttributes::ID_windingPairConfidence:
            windingPairConfidence->setText(DoubleToQString(atts->GetWindingPairConfidence()));
            break;

          case PoincareAttributes::ID_rationalSurfaceFactor:
            rationalSurfaceFactor->setText(DoubleToQString(atts->GetRationalSurfaceFactor()));
            break;
          case PoincareAttributes::ID_overlaps:
            overlapsButtonGroup->blockSignals(true);
            if(overlapsButtonGroup->button((int)atts->GetOverlaps()) != 0)
                overlapsButtonGroup->button((int)atts->GetOverlaps())->setChecked(true);
            overlapsButtonGroup->blockSignals(false);
            break;
          case PoincareAttributes::ID_meshType:
            UpdateMeshTypeAttributes();
            meshTypeCombo->blockSignals(true);
            meshTypeCombo->setCurrentIndex((int) atts->GetMeshType());
            meshTypeCombo->blockSignals(false);

            showPoints->setEnabled(atts->GetMeshType() == 0 );
            showLines ->setEnabled(atts->GetMeshType() == 0 );

            break;
          case PoincareAttributes::ID_numberPlanes:
            // Disconnect the slot before setting the toggle and
            // reconnect it after. This prevents multiple updates.
            disconnect(numberPlanes, SIGNAL(valueChanged(int)),
                       this, SLOT(numberPlanesChanged(int)));
            numberPlanes->setValue(atts->GetNumberPlanes());
            singlePlaneLabel->setEnabled( atts->GetNumberPlanes() == 1 );
            singlePlane->setEnabled( atts->GetNumberPlanes() == 1 );
            connect(numberPlanes, SIGNAL(valueChanged(int)),
                    this, SLOT(numberPlanesChanged(int)));
            break;
          case PoincareAttributes::ID_singlePlane:
            singlePlane->setText(DoubleToQString(atts->GetSinglePlane()));
            break;

           case PoincareAttributes::ID_dataValue:
            dataValueCombo->blockSignals(true);
            dataValueCombo->setCurrentIndex((int)atts->GetDataValue());
            dataValueCombo->blockSignals(false);
            break;
          case PoincareAttributes::ID_showRationalSurfaces:
            showRationalSurfaces->blockSignals(true);
            showRationalSurfaces->setChecked(atts->GetShowRationalSurfaces());
            showRationalSurfaces->blockSignals(false);
            break;
          case PoincareAttributes::ID_RationalSurfaceMaxIterations:
            rationalSurfaceMaxIterations->blockSignals(true);
            rationalSurfaceMaxIterations->setValue(atts->GetRationalSurfaceMaxIterations());
            rationalSurfaceMaxIterations->blockSignals(false);
            break;
          case PoincareAttributes::ID_showOPoints:
            showOPoints->blockSignals(true);
            showOPoints->setChecked(atts->GetShowOPoints());
            showOPoints->blockSignals(false);
            break;
          case PoincareAttributes::ID_OPointMaxIterations:
            OPointMaxIterations->blockSignals(true);
            OPointMaxIterations->setValue(atts->GetOPointMaxIterations());
            OPointMaxIterations->blockSignals(false);
            break;
          case PoincareAttributes::ID_performOLineAnalysis:
            performOLineAnalysis->blockSignals(true);
            performOLineAnalysis->setChecked(atts->GetPerformOLineAnalysis());
            performOLineAnalysis->blockSignals(false);
            break;
          case PoincareAttributes::ID_OLineToroidalWinding:
            OLineToroidalWinding->blockSignals(true);
            OLineToroidalWinding->setValue(atts->GetOLineToroidalWinding());
            OLineToroidalWinding->blockSignals(false);
            break;
          case PoincareAttributes::ID_OLineAxisFileName:
            OLineAxisFileName->setText(QString(atts->GetOLineAxisFileName().c_str()));
            break;
          case PoincareAttributes::ID_showIslands:
            showIslands->blockSignals(true);
            showIslands->setChecked(atts->GetShowIslands());
            showIslands->blockSignals(false);
            break;
          case PoincareAttributes::ID_showChaotic:
            showChaotic->blockSignals(true);
            showChaotic->setChecked(atts->GetShowChaotic());
            showChaotic->blockSignals(false);
            break;
          case PoincareAttributes::ID_showLines:
            showLines->blockSignals(true);

            if(atts->GetAnalysis() == 0)
            {
              showLines->setChecked(false);
              atts->SetShowLines(false);
            }

            showLines->setChecked(atts->GetShowLines());
            showLines->blockSignals(false);
            break;
        case PoincareAttributes::ID_showPoints:
            showPoints->blockSignals(true);
            showPoints->setChecked(atts->GetShowPoints());
            showPoints->blockSignals(false);
            break;
          case PoincareAttributes::ID_show1DPlots:
            show1DPlots->blockSignals(true);
            show1DPlots->setChecked(atts->GetShow1DPlots());
            show1DPlots->blockSignals(false);
            break;
          case PoincareAttributes::ID_SummaryFlag:
            summaryFlag->blockSignals(true);
            summaryFlag->setChecked(atts->GetSummaryFlag());
            summaryFlag->blockSignals(false);
            break;
          case PoincareAttributes::ID_verboseFlag:
            verboseFlag->blockSignals(true);
            verboseFlag->setChecked(atts->GetVerboseFlag());
            verboseFlag->blockSignals(false);
            break;
        case PoincareAttributes::ID_parallelizationAlgorithmType:
            UpdateAlgorithmAttributes();

            parallelAlgo->blockSignals(true);
            parallelAlgo->setCurrentIndex(atts->GetParallelizationAlgorithmType());
            parallelAlgo->blockSignals(false);
            break;
        case PoincareAttributes::ID_maxProcessCount:
            maxSLCount->blockSignals(true);
            maxSLCount->setValue(atts->GetMaxProcessCount());
            maxSLCount->blockSignals(false);
            break;
        case PoincareAttributes::ID_maxDomainCacheSize:
            maxDomainCache->blockSignals(true);
            maxDomainCache->setValue(atts->GetMaxDomainCacheSize());
            maxDomainCache->blockSignals(false);
            break;
        case PoincareAttributes::ID_workGroupSize:
            workGroupSize->blockSignals(true);
            workGroupSize->setValue(atts->GetWorkGroupSize());
            workGroupSize->blockSignals(false);
            break;

        case PoincareAttributes::ID_pathlines:
            icButtonGroup->blockSignals(true);
            icButtonGroup->button(atts->GetPathlines()?1:0)->setChecked(true);
            pathlineOverrideStartingTimeFlag->setEnabled(atts->GetPathlines());
            if( pathlineOverrideStartingTimeFlag->isChecked() && ! icButtonGroup->button(1)->isChecked() )
                pathlineOverrideStartingTimeFlag->setChecked(false);
            pathlineOverrideStartingTime->setEnabled(atts->GetPathlines() && atts->GetPathlinesOverrideStartingTimeFlag());
            pathlinePeriod->setEnabled(atts->GetPathlines());
            pathlineCMFEButtonGroup->button(0)->setEnabled(atts->GetPathlines());
            pathlineCMFEButtonGroup->button(1)->setEnabled(atts->GetPathlines());
            icButtonGroup->blockSignals(false);
            break;
        case PoincareAttributes::ID_pathlinesOverrideStartingTimeFlag:
            pathlineOverrideStartingTimeFlag->blockSignals(true);
            pathlineOverrideStartingTimeFlag->setChecked(atts->GetPathlinesOverrideStartingTimeFlag());
            pathlineOverrideStartingTime->setEnabled(atts->GetPathlines() && atts->GetPathlinesOverrideStartingTimeFlag());
            pathlineOverrideStartingTimeFlag->blockSignals(false);
            break;
        case PoincareAttributes::ID_pathlinesOverrideStartingTime:
            pathlineOverrideStartingTime->setText(DoubleToQString(atts->GetPathlinesOverrideStartingTime()));
            break;
        case PoincareAttributes::ID_pathlinesPeriod:
            pathlinePeriod->setText(DoubleToQString(atts->GetPathlinesPeriod()));
            break;
        case PoincareAttributes::ID_pathlinesCMFE:
            pathlineCMFEButtonGroup->blockSignals(true);
            pathlineCMFEButtonGroup->button(atts->GetPathlinesCMFE())->setChecked(true);
            pathlineCMFEButtonGroup->blockSignals(false);
            break;

        case PoincareAttributes::ID_issueTerminationWarnings:
            issueWarningForMaxSteps->blockSignals(true);
            issueWarningForMaxSteps->setChecked(atts->GetIssueTerminationWarnings());
            issueWarningForMaxSteps->blockSignals(false);
            break;

        case PoincareAttributes::ID_issueCriticalPointsWarnings:
            issueWarningForCriticalPoints->blockSignals(true);
            issueWarningForCriticalPoints->setChecked(atts->GetIssueCriticalPointsWarnings());
            criticalPointThreshold->setEnabled(atts->GetIssueCriticalPointsWarnings());
            criticalPointThresholdLabel->setEnabled(atts->GetIssueCriticalPointsWarnings());
            issueWarningForCriticalPoints->blockSignals(false);
            break;

        case PoincareAttributes::ID_issueStepsizeWarnings:
            issueWarningForStepsize->blockSignals(true);
            issueWarningForStepsize->setChecked(atts->GetIssueStepsizeWarnings());
            issueWarningForStepsize->blockSignals(false);
            break;
        case PoincareAttributes::ID_issueStiffnessWarnings:
            issueWarningForStiffness->blockSignals(true);
            issueWarningForStiffness->setChecked(atts->GetIssueStiffnessWarnings());
            issueWarningForStiffness->blockSignals(false);
            break;
        case PoincareAttributes::ID_criticalPointThreshold:
            criticalPointThreshold->setText(DoubleToQString(atts->GetCriticalPointThreshold()));
            break;
        }
    }
}


// ****************************************************************************
// Method: QvisPoincareWindow::GetCurrentValues
//
// Purpose:
//   Gets values from certain widgets and stores them in the subject.
//
// Note:       Autogenerated by xml2window.
//
// Programmer: xml2window
// Creation:   omitted
//
// Modifications:
//   Kathleen Biagas, Wed Jun  8 17:10:30 PDT 2016
//   Ensure values are retrieved from spin boxes.
//
// ****************************************************************************

void
QvisPoincareWindow::GetCurrentValues(int which_widget)
{
    bool doAll = (which_widget == -1);

    // Do velocitySource
    if(which_widget == PoincareAttributes::ID_velocitySource || doAll)
    {
        double val[3];
        if(LineEditGetDoubles(velocitySource, val, 3))
            atts->SetVelocitySource(val);
        else
        {
            ResettingError(tr("Velocity Source"),
                DoublesToQString(atts->GetVelocitySource(),3));
            atts->SetVelocitySource(atts->GetVelocitySource());
        }
    }

    // Do pointSource
    if(which_widget == PoincareAttributes::ID_pointSource || doAll)
    {
        double val[3];
        if(LineEditGetDoubles(pointSource, val, 3))
            atts->SetPointSource(val);
        else
        {
            ResettingError(tr("Point Source"),
                DoublesToQString(atts->GetPointSource(),3));
            atts->SetPointSource(atts->GetPointSource());
        }
    }

    // Do pointList
    if(which_widget == PoincareAttributes::ID_pointList || doAll)
    {
        std::vector<double> points;
        double x,y,z;
        for (int i = 0; i < pointList->count(); i++)
        {
            QListWidgetItem *item = pointList->item(i);
            if (item)
            {
                std::string str = item->text().toLatin1().data();
                sscanf(str.c_str(), "%lf %lf %lf", &x, &y, &z);
                points.push_back(x);
                points.push_back(y);
                points.push_back(z);
            }
        }
        atts->SetPointList(points);
    }

    // Do lineStart
    if(which_widget == PoincareAttributes::ID_lineStart || doAll)
    {
        double val[3];
        if(LineEditGetDoubles(lineStart, val, 3))
            atts->SetLineStart(val);
        else
        {
            ResettingError(tr("Line Point Start"),
                DoublesToQString(atts->GetLineStart(),3));
            atts->SetLineStart(atts->GetLineStart());
        }
    }

    // Do lineEnd
    if(which_widget == PoincareAttributes::ID_lineEnd || doAll)
    {
        double val[3];
        if(LineEditGetDoubles(lineEnd, val, 3))
            atts->SetLineEnd(val);
        else
        {
            ResettingError(tr("Line Point End"),
                DoublesToQString(atts->GetLineEnd(),3));
            atts->SetLineEnd(atts->GetLineEnd());
        }
    }

    // Do fieldConstant
    if(which_widget == PoincareAttributes::ID_fieldConstant || doAll)
    {
        double val;
        if(LineEditGetDouble(fieldConstant, val))
            atts->SetFieldConstant(val);
        else
        {
            ResettingError(tr("field constant"),
                DoubleToQString(atts->GetFieldConstant()));
            atts->SetFieldConstant(atts->GetFieldConstant());
        }
    }

    // Do maxStepLength
    if(which_widget == PoincareAttributes::ID_maxStepLength || doAll)
    {
        double val;
        if(LineEditGetDouble(maxStepLength, val))
            atts->SetMaxStepLength(val);
        else
        {
            ResettingError(tr("Maximum step length"),
                DoubleToQString(atts->GetMaxStepLength()));
            atts->SetMaxStepLength(atts->GetMaxStepLength());
        }
    }

    // Do relTol
    if(which_widget == PoincareAttributes::ID_relTol || doAll)
    {
        double val;
        if(LineEditGetDouble(relTol, val))
            atts->SetRelTol(val);
        else
        {
            ResettingError(tr("Relative tolerance"),
                DoubleToQString(atts->GetRelTol()));
            atts->SetRelTol(atts->GetRelTol());
        }
    }

    // Do absTol
    if ((which_widget == PoincareAttributes::ID_absTolBBox || doAll)
        && atts->GetAbsTolSizeType() == PoincareAttributes::FractionOfBBox)
    {
        double val;
        if(LineEditGetDouble(absTol, val))
            atts->SetAbsTolBBox(val);
        else
        {
            ResettingError(tr("absolute tolerance"),
                DoubleToQString(atts->GetAbsTolBBox()));
                atts->SetAbsTolBBox(atts->GetAbsTolBBox());
        }
    }
    if ((which_widget == PoincareAttributes::ID_absTolAbsolute || doAll)
        && atts->GetAbsTolSizeType() == PoincareAttributes::Absolute)
    {
        double val;
        if(LineEditGetDouble(absTol, val))
            atts->SetAbsTolAbsolute(val);
        else
        {
            ResettingError(tr("absolute tolerance"),
                DoubleToQString(atts->GetAbsTolAbsolute()));
                atts->SetAbsTolAbsolute(atts->GetAbsTolAbsolute());
        }
    }

    // Do puncture period tolerance
    if(which_widget == PoincareAttributes::ID_puncturePeriodTolerance || doAll)
    {
        double val;
        if(LineEditGetDouble(puncturePeriodTolerance, val))
            atts->SetPuncturePeriodTolerance(val);
        else
        {
            ResettingError(tr("Puncture Period Tolerance"),
                DoubleToQString(atts->GetPuncturePeriodTolerance()));
            atts->SetPuncturePeriodTolerance(atts->GetPuncturePeriodTolerance());
        }
    }

    // Do termination
    if(which_widget == PoincareAttributes::ID_maxSteps || doAll)
    {
        int val;
        if(LineEditGetInt(maxSteps, val))
            atts->SetMaxSteps(val);
        else
        {
            ResettingError(tr("maxsteps"),
                IntToQString(atts->GetMaxSteps()));
            atts->SetMaxSteps(atts->GetMaxSteps());
        }
    }
    if(which_widget == PoincareAttributes::ID_termTime || doAll)
    {
        double val;
        if(LineEditGetDouble(maxTime, val))
            atts->SetTermTime(val);
        else
        {
            ResettingError(tr("maxtime"),
                DoubleToQString(atts->GetTermTime()));
            atts->SetTermTime(atts->GetTermTime());
        }
    }

    // Do windingPairConfidence
    if(which_widget == PoincareAttributes::ID_windingPairConfidence || doAll)
    {
        double val;
        if(LineEditGetDouble(windingPairConfidence, val))
            atts->SetWindingPairConfidence(val);
        else
        {
            ResettingError(tr("windingPairConfidence"),
                DoubleToQString(atts->GetWindingPairConfidence()));
            atts->SetWindingPairConfidence(atts->GetWindingPairConfidence());
        }
    }

    // Do rationalSurfaceFactor
    if(which_widget == PoincareAttributes::ID_rationalSurfaceFactor || doAll)
    {
        double val;
        if(LineEditGetDouble(rationalSurfaceFactor, val))
            atts->SetRationalSurfaceFactor(val);
        else
        {
            ResettingError(tr("rationalSurfaceFactor"),
                DoubleToQString(atts->GetRationalSurfaceFactor()));
            atts->SetRationalSurfaceFactor(atts->GetRationalSurfaceFactor());
        }
    }

    // Do singlePlane
    if(which_widget == PoincareAttributes::ID_singlePlane || doAll)
    {
        double val;
        if(LineEditGetDouble(singlePlane, val))
            atts->SetSinglePlane(val);
        else
        {
            ResettingError(tr("singlePlane"),
                IntToQString((int)(atts->GetSinglePlane())));
            atts->SetSinglePlane(atts->GetSinglePlane());
        }
    }

    // Do O-Line Axis File Name
    if(which_widget == PoincareAttributes::ID_OLineAxisFileName || doAll)
    {
      if( atts->GetPerformOLineAnalysis() )
      {
        QString val = OLineAxisFileName->text();
        if( val.length() )
            atts->SetOLineAxisFileName(val.toStdString());
        else
        {
            ResettingError(tr("OLineAxisFileName"),
                           QString(atts->GetOLineAxisFileName().c_str()));
            atts->SetOLineAxisFileName(atts->GetOLineAxisFileName());
        }
      }
    }

    if(which_widget == PoincareAttributes::ID_pathlinesOverrideStartingTime || doAll)
    {
        double val;
        if(LineEditGetDouble(pathlineOverrideStartingTime, val))
            atts->SetPathlinesOverrideStartingTime(val);
        else
        {
            ResettingError(tr("Pathlines Override Starting Time"),
                DoubleToQString(atts->GetPathlinesOverrideStartingTime()));
            atts->SetPathlinesOverrideStartingTime(atts->GetPathlinesOverrideStartingTime());
        }
    }
    if(which_widget == PoincareAttributes::ID_pathlinesPeriod || doAll)
    {
        double val;
        if(LineEditGetDouble(pathlinePeriod, val))
            atts->SetPathlinesPeriod(val);
        else
        {
            ResettingError(tr("Pathlines Period"),
                DoubleToQString(atts->GetPathlinesPeriod()));
            atts->SetPathlinesPeriod(atts->GetPathlinesPeriod());
        }
    }

    // maxProcessCount
    if (which_widget == PoincareAttributes::ID_maxProcessCount || doAll)
    {
        // This can only be an integer, so no error checking is needed.
        int val = maxSLCount->value();
        if (val >= 1)
            atts->SetMaxProcessCount(val);
    }

    // workGroupSize
    if (which_widget == PoincareAttributes::ID_workGroupSize || doAll)
    {
        // This can only be an integer, so no error checking is needed.
        int val = workGroupSize->value();
        if (val >= 2)
            atts->SetWorkGroupSize(val);
    }

    // criticalPointThreshold
    if(which_widget == PoincareAttributes::ID_criticalPointThreshold || doAll)
    {
        double val;
        if(LineEditGetDouble(criticalPointThreshold, val))
            atts->SetCriticalPointThreshold(val);
        else
        {
            ResettingError(tr("Speed cutoff for critical points"),
                DoubleToQString(atts->GetCriticalPointThreshold()));
            atts->SetCriticalPointThreshold(atts->GetCriticalPointThreshold());
        }
    }

    if (which_widget == PoincareAttributes::ID_pointDensity || doAll)
    {
        if (pointDensity->value() != atts->GetPointDensity())
            atts->SetPointDensity(pointDensity->value());
    }
    if (which_widget == PoincareAttributes::ID_minPunctures || doAll)
    {
        if (minPunctures->value() != atts->GetMinPunctures())
            atts->SetMinPunctures(minPunctures->value());
    }
    if (which_widget == PoincareAttributes::ID_maxPunctures || doAll)
    {
        if (maxPunctures->value() != atts->GetMaxPunctures())
            atts->SetMaxPunctures(maxPunctures->value());
    }
    if (which_widget == PoincareAttributes::ID_maximumToroidalWinding || doAll)
    {
        if (maximumToroidalWinding->value() != atts->GetMaximumToroidalWinding())
            atts->SetMaximumToroidalWinding(maximumToroidalWinding->value());
    }
    if (which_widget == PoincareAttributes::ID_overrideToroidalWinding || doAll)
    {
        if (overrideToroidalWinding->value() != atts->GetOverrideToroidalWinding())
            atts->SetOverrideToroidalWinding(overrideToroidalWinding->value());
    }
    if (which_widget == PoincareAttributes::ID_overridePoloidalWinding || doAll)
    {
        if (overridePoloidalWinding->value() != atts->GetOverridePoloidalWinding())
            atts->SetOverridePoloidalWinding(overridePoloidalWinding->value());
    }
    if (which_widget == PoincareAttributes::ID_RationalSurfaceMaxIterations || doAll)
    {
        if (rationalSurfaceMaxIterations->value() != atts->GetRationalSurfaceMaxIterations())
            atts->SetRationalSurfaceMaxIterations(rationalSurfaceMaxIterations->value());
    }
    if (which_widget == PoincareAttributes::ID_OPointMaxIterations || doAll)
    {
        if (OPointMaxIterations->value() != atts->GetOPointMaxIterations())
            atts->SetOPointMaxIterations(OPointMaxIterations->value());
    }
    if (which_widget == PoincareAttributes::ID_OLineToroidalWinding || doAll)
    {
        if (OLineToroidalWinding->value() != atts->GetOLineToroidalWinding())
            atts->SetOLineToroidalWinding(OLineToroidalWinding->value());
    }
    if (which_widget == PoincareAttributes::ID_numberPlanes || doAll)
    {
        if (numberPlanes->value() != atts->GetNumberPlanes())
            atts->SetNumberPlanes(numberPlanes->value());
    }
    if (which_widget == PoincareAttributes::ID_maxDomainCacheSize || doAll)
    {
        if (maxDomainCache->value() != atts->GetMaxDomainCacheSize())
            atts->SetMaxDomainCacheSize(maxDomainCache->value());
    }

}


// ****************************************************************************
// Method: QvisPoincareWindow::UpdateMeshTypeAttributes
//
// Purpose:
//   Updates the widgets for the various integration types.
//
// Programmer: Dave Pugmire
// Creation:   Fri Aug 1 16:41:38 EDT 2008
//
// ****************************************************************************

void
QvisPoincareWindow::UpdateMeshTypeAttributes()
{
    //Turn off everything.
    singlePlaneLabel->hide();
    singlePlane->hide();

    switch( atts->GetMeshType() )
    {
    case PoincareAttributes::Curves:
        singlePlaneLabel->show();
        singlePlane->show();
        break;
    case PoincareAttributes::Surfaces:
        break;
    }
}


// ****************************************************************************
// Method: QvisPoincareWindow::UpdateFieldAttributes
//
// Purpose:
//   Updates the widgets for the various field types.
//
// Programmer: Dave Pugmire
// Creation:   Thu Jul 31 14:41:00 EDT 2008
//
// ****************************************************************************

void
QvisPoincareWindow::UpdateFieldAttributes()
{
    switch( atts->GetFieldType() )
    {
    case PoincareAttributes::M3DC12DField:
      if( atts->GetIntegrationType() ==
          PoincareAttributes::M3DC12DIntegrator )
        TurnOn(fieldConstant, fieldConstantLabel);
      else
        TurnOff(fieldConstant, fieldConstantLabel);

      TurnOff(velocitySource, velocitySourceLabel);

      break;

    case PoincareAttributes::FlashField:
      TurnOn(fieldConstant, fieldConstantLabel);
      TurnOn(velocitySource, velocitySourceLabel);
      break;

    case PoincareAttributes::NIMRODField:
    default:
      TurnOff(fieldConstant, fieldConstantLabel);
      TurnOff(velocitySource, velocitySourceLabel);

      break;
    }
}


// ****************************************************************************
// Method: QvisPoincareWindow::UpdateIntegrationAttributes
//
// Purpose:
//   Updates the widgets for the various integration types.
//
// Programmer: Dave Pugmire
// Creation:   Thu Jul 31 14:41:00 EDT 2008
//
// Modifications:
//
//   Dave Pugmire, Fri Aug 8 16:27:03 EDT 2008
//   Change the step label text based on the integration method.
//
//   Hank Childs, Wed Sep 29 20:22:36 PDT 2010
//   Add support for limiting the maximum time step.
//
// ****************************************************************************

void
QvisPoincareWindow::UpdateIntegrationAttributes()
{
    //Turn off everything.
    maxStepLength->hide();
    maxStepLengthLabel->hide();
    limitMaxTimeStep->hide();
    maxTimeStep->hide();
    relTol->hide();
    relTolLabel->hide();
    absTol->hide();
    absTolLabel->hide();
    absTolSizeType->hide();

    switch( atts->GetIntegrationType() )
    {
    case PoincareAttributes::Euler:
    case PoincareAttributes::Leapfrog:
    case PoincareAttributes::RK4:
        maxStepLength->show();
        maxStepLengthLabel->show();
      break;

    case PoincareAttributes::DormandPrince:
        limitMaxTimeStep->show();
        maxTimeStep->show();
        relTol->show();
        relTolLabel->show();
        absTol->show();
        absTolLabel->show();
        absTolSizeType->show();
        break;

    case PoincareAttributes::AdamsBashforth:
    case PoincareAttributes::M3DC12DIntegrator:
        maxStepLength->show();
        maxStepLengthLabel->show();
        absTol->show();
        absTolLabel->show();
        absTolSizeType->show();
        break;

    default:
        break;
    }
}


// ****************************************************************************
// Method: QvisPoincareWindow::UpdateAlgorithmAttributes
//
// Purpose:
//   Updates the widgets for the various integration types.
//
// Programmer: Dave Pugmire
// Creation:   Fri Aug 1 16:41:38 EDT 2008
//
// ****************************************************************************

void
QvisPoincareWindow::UpdateAlgorithmAttributes()
{
    //Turn off everything.
    maxDomainCacheLabel->hide();
    maxDomainCache->hide();
    maxSLCountLabel->hide();
    maxSLCount->hide();
    workGroupSizeLabel->hide();
    workGroupSize->hide();

    switch( atts->GetParallelizationAlgorithmType() )
    {
      case PoincareAttributes::LoadOnDemand:
        maxDomainCacheLabel->show();
        maxDomainCache->show();
        break;
    case PoincareAttributes::ParallelStaticDomains:
        maxSLCountLabel->show();
        maxSLCount->show();
        break;

    case PoincareAttributes::MasterSlave:
        maxDomainCacheLabel->show();
        maxDomainCache->show();
        maxSLCountLabel->show();
        maxSLCount->show();
        workGroupSizeLabel->show();
        workGroupSize->show();
        break;
    default: ///TODO: handle VisItSelects option
        break;
    }
}


void
QvisPoincareWindow::sourceTypeChanged(int val)
{
    if(val != atts->GetSourceType())
    {
        atts->SetSourceType(PoincareAttributes::SourceType(val));
        Apply();
    }
}


void
QvisPoincareWindow::velocitySourceProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_velocitySource);
    Apply();
}


void
QvisPoincareWindow::pointSourceProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_pointSource);
    Apply();
}


void
QvisPoincareWindow::pointListProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_pointList);
    Apply();
}


void
QvisPoincareWindow::pointListDoubleClicked(QListWidgetItem *item)
{
    item->setFlags(item->flags() | Qt::ItemIsEditable);
}

void
QvisPoincareWindow::pointListClicked(QListWidgetItem *item)
{
}

void
QvisPoincareWindow::textChanged(const QString &currentText)
{
}

void
QvisPoincareWindow::addPoint()
{
    QListWidgetItem *item = new QListWidgetItem("0 0 0", pointList);
    item->setFlags(item->flags() | Qt::ItemIsEditable);
    pointList->setCurrentItem(item);
}

void
QvisPoincareWindow::deletePoint()
{
    if (!pointList->selectedItems().empty())
    {
        qDeleteAll(pointList->selectedItems());
    }
}

void
QvisPoincareWindow::deletePoints()
{
    pointList->clear();
}

void
QvisPoincareWindow::readPoints()
{
    QString res = QFileDialog::getOpenFileName(NULL, tr("Open text file"), ".");
    std::string filename = res.toLatin1().data();

    if (filename == "")
        return;
    std::ifstream f;
    f.open(filename.c_str());
    while (f.good())
    {
        char tmp[256];
        f.getline(tmp, 256);
        if (strlen(tmp) == 0)
            break;

        float x, y, z;
        int n = sscanf(tmp, "%f %f %f", &x, &y, &z);
        if (n != 3)
            n = sscanf(tmp, "%f, %f, %f", &x, &y, &z);
        if (n == 2)
        {
            z = 0.0;
            n = 3;
        }
        if (n == 3)
        {
            char vals[256];
            sprintf(vals, "%f %f %f", x,y,z);
            QListWidgetItem *item = new QListWidgetItem(vals, pointList);
            item->setFlags(item->flags() | Qt::ItemIsEditable);
            pointList->setCurrentItem(item);
        }
    }

    f.close();
}


void
QvisPoincareWindow::lineStartProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_lineStart);
    Apply();
}


void
QvisPoincareWindow::lineEndProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_lineEnd);
    Apply();
}


void
QvisPoincareWindow::pointDensityChanged(int val)
{
    atts->SetPointDensity(val);
    Apply();
}


void
QvisPoincareWindow::minPuncturesChanged(int val)
{
    atts->SetMinPunctures(val);
    Apply();
}


void
QvisPoincareWindow::maxPuncturesChanged(int val)
{
    atts->SetMaxPunctures(val);
    Apply();
}


void
QvisPoincareWindow::puncturePlotTypeChanged(int val)
{
    if(val != atts->GetPuncturePlotType())
    {
        atts->SetPuncturePlotType(PoincareAttributes::PuncturePlotType(val));
        Apply();
    }
}


void
QvisPoincareWindow::puncturePeriodToleranceProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_puncturePeriodTolerance);
    Apply();
}


void
QvisPoincareWindow::maxStepsProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_maxSteps);
    Apply();
}


void
QvisPoincareWindow::limitMaxTimeChanged(bool val)
{
    if(val != atts->GetTerminateByTime())
    {
        atts->SetTerminateByTime(val);
        Apply();
    }
}


void
QvisPoincareWindow::maxTimeProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_termTime);
    Apply();
}


void
QvisPoincareWindow::puncturePlaneChanged(int val)
{
    if(val != atts->GetPuncturePlane())
    {
        atts->SetPuncturePlane(PoincareAttributes::PuncturePlaneType(val));
        Apply();
    }
}


void
QvisPoincareWindow::fieldTypeChanged(int val)
 {
    if(val != atts->GetFieldType())
    {
        atts->SetFieldType(PoincareAttributes::FieldType(val));
        Apply();
    }
}


void
QvisPoincareWindow::fieldConstantProccessText()
{
    GetCurrentValues(PoincareAttributes::ID_fieldConstant);
    Apply();
}


void
QvisPoincareWindow::integrationTypeChanged(int val)
{
    if(val != atts->GetIntegrationType())
    {
        atts->SetIntegrationType(PoincareAttributes::IntegrationType(val));
        Apply();
    }
}


void
QvisPoincareWindow::maxStepLengthProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_maxStepLength);
    Apply();
}


void
QvisPoincareWindow::limitMaxTimeStepChanged(bool val)
{
    atts->SetLimitMaximumTimestep(val);
    Apply();
}


void
QvisPoincareWindow::maxTimeStepProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_maxTimeStep);
    Apply();
}


void
QvisPoincareWindow::relTolProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_relTol);
    Apply();
}


void
QvisPoincareWindow::absTolProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_absTolBBox);
    GetCurrentValues(PoincareAttributes::ID_absTolAbsolute);
    Apply();
}


void
QvisPoincareWindow::absTolSizeTypeChanged(int val)
{
    atts->SetAbsTolSizeType((PoincareAttributes::SizeType) val);
    Apply();
}


// void
// QvisPoincareWindow::coordinateButtonGroupChanged(int val)
// {
//     switch( val )
//     {
//         case 0:
//        atts->SetCoordinateSystem(PoincareAttributes::Cartesian);
//        break;
//         case 1:
//        atts->SetCoordinateSystem(PoincareAttributes::Cylindrical);
//        break;
//     }
//     Apply();
// }

void
QvisPoincareWindow::analysisChanged(int val)
{
    if(val != atts->GetAnalysis())
    {
        atts->SetAnalysis(PoincareAttributes::AnalysisType(val));
        Apply();
    }
}


void
QvisPoincareWindow::maximumToroidalWindingChanged(int val)
{
    atts->SetMaximumToroidalWinding(val);
    Apply();
}


void
QvisPoincareWindow::overrideToroidalWindingChanged(int val)
{
    atts->SetOverrideToroidalWinding(val);
    Apply();
}


void
QvisPoincareWindow::overridePoloidalWindingChanged(int val)
{
    atts->SetOverridePoloidalWinding(val);
    Apply();
}


void
QvisPoincareWindow::windingPairConfidenceProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_windingPairConfidence);
    Apply();
}


void
QvisPoincareWindow::rationalSurfaceFactorProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_rationalSurfaceFactor);
    Apply();
}


void
QvisPoincareWindow::overlapsChanged(int val)
{
    if(val != atts->GetOverlaps())
    {
        atts->SetOverlaps(PoincareAttributes::OverlapType(val));
        Apply();
    }
}


void
QvisPoincareWindow::meshTypeChanged(int val)
{
    if(val != atts->GetMeshType())
    {
        atts->SetMeshType(PoincareAttributes::ShowMeshType(val));
        Apply();
    }
}


void
QvisPoincareWindow::numberPlanesChanged(int val)
{
    atts->SetNumberPlanes(val);
    Apply();
}


void
QvisPoincareWindow::singlePlaneProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_singlePlane);
    Apply();
}


void
QvisPoincareWindow::dataValueChanged(int val)
{
    if(val != atts->GetDataValue())
    {
        atts->SetDataValue(PoincareAttributes::DataValue(val));
        atts->SetColorType(PoincareAttributes::ColoringMethod(val?1:0));
        Apply();
    }
}


void
QvisPoincareWindow::showRationalSurfacesChanged(bool val)
{
    atts->SetShowRationalSurfaces(val);
    Apply();
}


void
QvisPoincareWindow::rationalSurfaceMaxIterationsChanged(int val)
{
    atts->SetRationalSurfaceMaxIterations(val);
    Apply();
}


void
QvisPoincareWindow::showOPointsChanged(bool val)
{
    atts->SetShowOPoints(val);
    Apply();
}


void
QvisPoincareWindow::OPointMaxIterationsChanged(int val)
{
    atts->SetOPointMaxIterations(val);
    Apply();
}


void
QvisPoincareWindow::performOLineAnalysisChanged(bool val)
{
    atts->SetPerformOLineAnalysis(val);
    Apply();
}


void
QvisPoincareWindow::OLineToroidalWindingChanged(int val)
{
    atts->SetOLineToroidalWinding(val);
    Apply();
}


void
QvisPoincareWindow::OLineAxisFileNameProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_OLineAxisFileName);
    Apply();
}

void
QvisPoincareWindow::OLineAxisFileDialogButtonClicked()
{
    QString fileName =
        QFileDialog::getOpenFileName(this, tr("Open O-Line Axis Point File"),
                                 "",
                                 tr(""));

    OLineAxisFileName->setText(fileName);

    GetCurrentValues(PoincareAttributes::ID_OLineAxisFileName);
    Apply();
}


void
QvisPoincareWindow::showIslandsChanged(bool val)
{
    atts->SetShowIslands(val);
    Apply();
}


void
QvisPoincareWindow::showChaoticChanged(bool val)
{
    atts->SetShowChaotic(val);
    Apply();
}


void
QvisPoincareWindow::showLinesChanged(bool val)
{
    atts->SetShowLines(val);
    Apply();
}


void
QvisPoincareWindow::show1DPlotsChanged(bool val)
{
    atts->SetShow1DPlots(val);
    Apply();
}


void
QvisPoincareWindow::showPointsChanged(bool val)
{
    atts->SetShowPoints(val);
    Apply();
}


void
QvisPoincareWindow::summaryFlagChanged(bool val)
{
    atts->SetSummaryFlag(val);
    Apply();
}


void
QvisPoincareWindow::verboseFlagChanged(bool val)
{
    atts->SetVerboseFlag(val);
    Apply();
}


// void
// QvisPoincareWindow::forceNodalChanged(bool val)
// {
//     atts->SetForceNodeCenteredData(val);
//     Apply();
// }


void
QvisPoincareWindow::parallelAlgorithmChanged(int val)
{
    if(val != atts->GetParallelizationAlgorithmType())
    {
        atts->SetParallelizationAlgorithmType(PoincareAttributes::ParallelizationAlgorithmType(val));
        Apply();
    }
}

void
QvisPoincareWindow::maxSLCountChanged(int val)
{
    atts->SetMaxProcessCount(val);
    Apply();
}

void
QvisPoincareWindow::maxDomainCacheChanged(int val)
{
    atts->SetMaxDomainCacheSize(val);
    Apply();
}

void
QvisPoincareWindow::workGroupSizeChanged(int val)
{
    atts->SetWorkGroupSize(val);
    Apply();
}

void
QvisPoincareWindow::icButtonGroupChanged(int val)
{
    switch( val )
    {
        case 0: // Streamline
            atts->SetPathlines(false);
            break;
        case 1: // Pathline
            atts->SetPathlines(true);
            break;
    }
    Apply();
}

void
QvisPoincareWindow::pathlineOverrideStartingTimeFlagChanged(bool val)
{
    atts->SetPathlinesOverrideStartingTimeFlag(val);
    Apply();
}

void
QvisPoincareWindow::pathlineOverrideStartingTimeProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_pathlinesOverrideStartingTime);
    Apply();
}

void
QvisPoincareWindow::pathlinePeriodProcessText()
{
    GetCurrentValues(PoincareAttributes::ID_pathlinesPeriod);
    Apply();
}

void
QvisPoincareWindow::pathlineCMFEButtonGroupChanged(int val)
{
    atts->SetPathlinesCMFE((PoincareAttributes::PathlinesCMFE)val);
    Apply();
}

void
QvisPoincareWindow::issueWarningForMaxStepsChanged(bool val)
{
    atts->SetIssueTerminationWarnings(val);
    Apply();
}

void
QvisPoincareWindow::issueWarningForStepsizeChanged(bool val)
{
    atts->SetIssueStepsizeWarnings(val);
    Apply();
}

void
QvisPoincareWindow::issueWarningForStiffnessChanged(bool val)
{
    atts->SetIssueStiffnessWarnings(val);
    Apply();
}

void
QvisPoincareWindow::issueWarningForCriticalPointsChanged(bool val)
{
    atts->SetIssueCriticalPointsWarnings(val);
    Apply();
}

void
QvisPoincareWindow::criticalPointThresholdProcessText(void)
{
    GetCurrentValues(PoincareAttributes::ID_criticalPointThreshold);
    Apply();
}


static void
TurnOn(QWidget *w0, QWidget *w1)
{
    if (w0)
    {
        w0->setEnabled(true);
        w0->show();
    }
    if (w1)
    {
        w1->setEnabled(true);
        w1->show();
    }
}

static void
TurnOff(QWidget *w0, QWidget *w1)
{
    if (w0)
    {
        w0->setEnabled(false);
        w0->hide();
    }
    if (w1)
    {
        w1->setEnabled(false);
        w1->hide();
    }
}
