// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

#include "QvisTimeQueryOptionsWidget.h"

#include <QGroupBox>
#include <QLabel>
#include <QPushButton>
#include <QSpinBox>
#include <QVBoxLayout>
#include <QButtonGroup>
#include <QRadioButton>
#include <QGroupBox>


#include <DatabaseCorrelation.h>
#include <DatabaseCorrelationList.h>
#include <Plot.h>
#include <PlotList.h>
#include <ViewerState.h>
#include <WindowInformation.h>

// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::QvisTimeQueryOptionsWidget
//
// Purpose: 
//   Constructor
//
// Programmer: Kathleen Biagas 
// Creation:   August 1, 2011 
//
// Modifications:
//
//   Alister Maguire, Wed Oct 16 08:41:57 MST 2019
//   Added dataOriginAvail and includeDataOrigin argument.
//
//   Alister Maguire, Mon Mar  9 12:03:51 PDT 2020
//   Removed dataOriginAvail and includeDataOrigin.
//
// ****************************************************************************

QvisTimeQueryOptionsWidget::QvisTimeQueryOptionsWidget(QWidget *parent)
    : QGroupBox(parent), GUIBase()
{
    maxTime = 99;

    setCheckable(true);
    setChecked(false);
    setTitle(tr("Do time curve"));
    CreateWindowContents();
}


// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::QvisTimeQueryOptionsWidget
//
// Purpose: 
//   Constructor
//
// Programmer: Kathleen Biagas 
// Creation:   August 1, 2011 
//
// Modifications:
//
//   Alister Maguire, Wed Oct 16 08:41:57 MST 2019
//   Added dataOriginAvail and includeDataOrigin argument.
//
//   Alister Maguire, Mon Mar  9 12:03:51 PDT 2020
//   Removed dataOriginAvail and includeDataOrigin.
//
// ****************************************************************************

QvisTimeQueryOptionsWidget::QvisTimeQueryOptionsWidget(const QString & title, 
    QWidget *parent)
    : QGroupBox(title, parent), GUIBase()
{
    maxTime = 99;

    setCheckable(true);
    setChecked(false);
    setTitle(title);
    CreateWindowContents();
}


// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::CreateWindowContents
//
// Purpose: 
//   Creates the widgets this window uses.
//
// Programmer: Kathleen Biagas 
// Creation:   August 1, 2011 
//
// Modifications:
//
//   Alister Maguire, Wed Oct 16 08:41:57 MST 2019
//   Added the dataOrigin section to determine whether to use
//   actual or original data.
//
//   Alister Maguire, Mon Mar  9 12:03:51 PDT 2020
//   Removed dataOrigin section.
//
// ****************************************************************************

void
QvisTimeQueryOptionsWidget::CreateWindowContents()
{
    QVBoxLayout *topLayout = new QVBoxLayout(this);    

    //
    // Message
    //
    QHBoxLayout *msgLayout = new QHBoxLayout();
    msgLabel = new QLabel(tr("Start and end are time steps,\nnot cycles or times."));
    msgLabel->setWordWrap(true);
    msgLabel->setAlignment(Qt::AlignHCenter);
    msgLayout->addWidget(msgLabel);

    topLayout->addLayout(msgLayout);

    QGridLayout *gLayout = new QGridLayout();
    topLayout->addLayout(gLayout);

    //
    // StartTime 
    //
    startLabel = new QLabel(tr("Starting timestep"));
    gLayout->addWidget(startLabel, 0, 0);

    startTime = new QSpinBox();
    startTime->setKeyboardTracking(false);
    startTime->setValue(0);
    startTime->setMinimum(0);
    startTime->setButtonSymbols(QAbstractSpinBox::PlusMinus);
    gLayout->addWidget(startTime, 0, 1);

    //
    // EndTime 
    //
    endLabel   = new QLabel(tr("Ending timestep"));
    gLayout->addWidget(endLabel, 1, 0);
    endTime = new QSpinBox();
    endTime->setKeyboardTracking(false);
    endTime->setValue(0);
    endTime->setMinimum(0);
    endTime->setButtonSymbols(QAbstractSpinBox::PlusMinus);
    gLayout->addWidget(endTime, 1, 1);

    //
    // Stride 
    //
    strideLabel = new QLabel(tr("Stride"));
    gLayout->addWidget(strideLabel, 2, 0);
    stride = new QSpinBox();
    stride->setKeyboardTracking(false);
    stride->setMinimum(1);
    stride->setValue(1);
    stride->setButtonSymbols(QAbstractSpinBox::PlusMinus);
    gLayout->addWidget(stride, 2, 1);
}


// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::~QvisTimeQueryOptionsWidget
//
// Purpose: 
//   Destructor
//
// Programmer: Kathleen Biagas 
// Creation:   August 1, 2011
//
// Modifications:
//   
// ****************************************************************************

QvisTimeQueryOptionsWidget::~QvisTimeQueryOptionsWidget()
{
    ;
}


// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::setCheckable
//
// Purpose: 
//   This is a Qt slot function that sets the checkable state of the widget.
//
// Arguments:
//   val : The new checkable state.
//
// Programmer: Kathleen Biagas 
// Creation:   August 9, 2011
//
// Modifications:
//   
// ****************************************************************************

void
QvisTimeQueryOptionsWidget::setCheckable(bool val)
{
    QGroupBox::setCheckable(val);
    if (val)
    {
        setTitle(tr("Do Time Query"));
        setChecked(false);
    }
    else 
    {
        setTitle(tr("Time Query Options"));
    }
}


// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::setEnabled
//
// Purpose: 
//   This is a Qt slot function that sets the enabled state of the widget.
//   Also updates the Max for the widgets.
//
// Arguments:
//   val : The new enabled state.
//
// Programmer: Kathleen Biagas 
// Creation:   August 9, 2011
//
// Modifications:
//   Kathleen Biagas, Wed Jan 25 15:53:01 MST 2012
//   Make enabled state also depend upon number of time states available.
//   
// ****************************************************************************

void
QvisTimeQueryOptionsWidget::setEnabled(bool val)
{
    QGroupBox::setEnabled(val);
}


// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::show
//
// Purpose: 
//   This is a Qt slot function that shows the widget.
//   Also updates the Max for the widgets.
//
// Programmer: Kathleen Biagas 
// Creation:   August 9, 2011
//
// Modifications:
//   
// ****************************************************************************

void
QvisTimeQueryOptionsWidget::show()
{
    QGroupBox::show();
}


// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::SetMax
//
// Purpose: 
//   Sets the max value for the widgets.
//
// Arguments:
//   val : The new max.
//
// Programmer: Kathleen Biagas 
// Creation:   August 9, 2011
//
// Modifications:
//   
// ****************************************************************************

void
QvisTimeQueryOptionsWidget::SetMax(const int val)
{
    if (val > 0)
    {
        maxTime = val;
        startTime->setMaximum(maxTime-1);
        endTime->setMaximum(maxTime);
        endTime->setValue(maxTime);
        if (maxTime > 1)
            stride->setMaximum(maxTime);
        else 
            stride->setMaximum(1);
    }
}


// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::GetTimeQueryOptions
//
// Purpose: 
//   Gets values from the widgets and stores them in the MapNode.
//
// Programmer: Kathleen Biagas 
// Creation:   August 1, 2011 
//
// Modifications:
//
//   Alister Maguire, Wed Oct 16 08:41:57 MST 2019
//   Set the use_actual_data flag.
//
//   Alister Maguire, Mon Mar  9 12:03:51 PDT 2020
//   Removed use_actual_data flag. This logic is now handled elsewhere.
//   
// ****************************************************************************

bool
QvisTimeQueryOptionsWidget::GetTimeQueryOptions(MapNode &options)
{
    QString temp;
    options["start_time"] = startTime->value();
    options["end_time"] = endTime->value();
    options["stride"] = stride->value();
    return true;
}

// ****************************************************************************
// Method: QvisTimeQueryOptionsWidget::GetTimeQueryOptions
//
// Purpose: 
//   Updates enabledState and maxTime based on currently selected plot.
//
// Programmer: Kathleen Biagas 
// Creation:   April 11, 2012
//
// Modifications:
//   
// ****************************************************************************

void
QvisTimeQueryOptionsWidget::UpdateState(PlotList *plotList)
{
    if (!plotList)
        return;
    int nStates = 1;
    int selectedPlot = plotList->FirstSelectedIndex();
    if (selectedPlot >= 0)
    {
        const std::string &dbName = 
            plotList->GetPlots(selectedPlot).GetDatabaseName();

        DatabaseCorrelationList *correlations = 
            GetViewerState()->GetDatabaseCorrelationList();

        DatabaseCorrelation *c = correlations->FindCorrelation(dbName);
        if (c)
            nStates = c->GetNumStates();
    }
    SetMax(nStates -1 ); 
    setEnabled(plotList->GetNumPlots() > 0  && nStates > 1);
}

