// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

#ifndef GENERATOR_BASE_H
#define GENERATOR_BASE_H
#include <QTextStream>
#include <AttributeBase.h>
#include <stdlib.h>
#include <sys/types.h>

#if !defined(_WIN32)
#include <sys/time.h>
#else
#include <time.h>
#endif
#include <visit-config.h>
#ifdef TIME_WITH_SYS_TIME
#include <time.h>
#endif

#if defined(_WIN32)
#include <windows.h>
#endif

// ****************************************************************************
// Class: GeneratorBase
//
// Purpose:
//   Base class for field generator classes, which contains code that will
//   be needed by subclasses to generate code for their fields.
//
// Notes:
//
// Programmer: Brad Whitlock
// Creation:   Wed Mar 5 11:45:12 PDT 2008
//
// Modifications:
//    Jeremy Meredith, Thu Aug  7 14:55:48 EDT 2008
//    Use const char * for string literals.
//
//    Mark C. Miller, Wed Aug 26 10:58:41 PDT 2009
//    Added custom base class for dervied state objects.
// ****************************************************************************

class GeneratorBase : public AttributeBase
{
public:
    QString generatorName;
public:
    GeneratorBase(const QString &n, const QString &p, const QString &f,
                      const QString &e, const QString &ei, const QString &gName,
                      const QString &bc = "AttributeSubject")
        : AttributeBase(n,p,f,e,ei,bc), generatorName(gName)
    {
    }

    virtual ~GeneratorBase()
    {
    }

    bool HasFunction(const QString &f) const
    {
        return AttributeBase::HasFunction(f, generatorName);
    }
    void PrintFunction(QTextStream &out, const QString &f) const
    {
        AttributeBase::PrintFunction(out, f, generatorName);
    }
    void DeleteFunction(QTextStream &out, const QString &f)
    {
        AttributeBase::DeleteFunction(out, f, generatorName);
    }

    bool HasCode(const QString &cName, int part) const
    {
        return AttributeBase::HasCode(cName, part, generatorName);
    }
    void PrintCode(QTextStream &out, const QString &cName, int part) const
    {
        AttributeBase::PrintCode(out, cName, part, generatorName);
    }

    static QString
    CurrentTime()
    {
        const char *tstr[] = {"PDT", "PST"};
        char s1[10], s2[10], s3[10], tmpbuf[200];
        time_t t;
        char *c = NULL;
        int h,m,s,y;
        t = time(NULL);
        c = asctime(localtime(&t));
        // Read the hour.
        sscanf(c, "%s %s %s %d:%d:%d %d", s1, s2, s3, &h, &m, &s, &y);
        // Reformat the string a little.
        sprintf(tmpbuf, "%s %s %s %02d:%02d:%02d %s %d",
                s1, s2, s3, h, m, s, tstr[h > 12], y);

        return QString(tmpbuf);
    }

    void
    WriteMethodComment(QTextStream &out, const QString &className,
                       const QString &methodName, const QString &purposeString) const
    {
        out << "// ****************************************************************************" << endl;
        out << "// Method: " << className << "::" << methodName << endl;
        out << "//" << endl;
        out << "// Purpose:" << endl;
        if (!purposeString.isEmpty())
            out << "//   " << purposeString << endl;
        out << "//" << endl;
        out << "// Note:       Autogenerated by " << generatorName << "." << endl;
        out << "//" << endl;
        out << "// Programmer: " << generatorName << endl;
        out << "// Creation:   omitted" << endl;
        out << "//" << endl;
        out << "// Modifications:" << endl;
        out << "//" << endl;
        out << "// ****************************************************************************" << endl;
        out << endl;
    }

    void
    WriteClassComment(QTextStream &h, const QString &purposeString) const
    {
        h << "// ****************************************************************************" << endl;
        h << "// Class: " << name << endl;
        h << "//" << endl;
        h << "// Purpose:" << endl;
        if (!purposeString.isEmpty())
            h << "//    " << purposeString << endl;
        h << "//" << endl;
        h << "// Notes:      Autogenerated by " << generatorName << "." << endl;
        h << "//" << endl;
        h << "// Programmer: " << generatorName << endl;
        h << "// Creation:   omitted" << endl;
        h << "//" << endl;
        h << "// Modifications:" << endl;
        h << "//" << endl;
        h << "// ****************************************************************************" << endl;
        h << endl;
    }

    void
    WriteClassComment(QTextStream &h, const QString &cName, const QString &purposeString) const
    {
        h << "// ****************************************************************************" << endl;
        h << "// Class: " << cName << endl;
        h << "//" << endl;
        h << "// Purpose:" << endl;
        if (!purposeString.isEmpty())
            h << "//    " << purposeString << endl;
        h << "//" << endl;
        h << "// Notes:      Autogenerated by " << generatorName << "." << endl;
        h << "//" << endl;
        h << "// Programmer: " << generatorName << endl;
        h << "// Creation:   omitted" << endl;
        h << "//" << endl;
        h << "// Modifications:" << endl;
        h << "//" << endl;
        h << "// ****************************************************************************" << endl;
        h << endl;
    }
};

#endif
