// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

#include <CGNSPluginInfo.h>

#include <climits>

#include <avtCGNSFileFormat.h>
#include <avtMTMDFileFormatInterface.h>
#include <avtMTSDFileFormatInterface.h>
#include <avtGenericDatabase.h>
#include <DebugStream.h>

// ****************************************************************************
//  Method: CGNSCommonPluginInfo constructor
//
//  Programmer: Eric Brugger
//  Creation:   Fri Feb 28 13:40:33 PST 2020
//
//  Modifications:
//
// ****************************************************************************

CGNSCommonPluginInfo::CGNSCommonPluginInfo() : CommonDatabasePluginInfo(), CGNSGeneralPluginInfo()
{
    // Assume MD by default.
    dbType = DB_TYPE_MTMD;
}

// ****************************************************************************
//  Method:  CGNSCommonPluginInfo::GetDatabaseType
//
//  Purpose:
//    Returns the type of a CGNS database.
//
//  Programmer:  generated by xml2info
//  Creation:    omitted
//
// ****************************************************************************
DatabaseType
CGNSCommonPluginInfo::GetDatabaseType()
{
    return dbType;
}

// ****************************************************************************
//  Method: CGNSCommonPluginInfo::SetupDatabase
//
//  Purpose:
//      Sets up a CGNS database.
//
//  Arguments:
//      list    A list of file names.
//      nList   The number of timesteps in list.
//      nBlocks The number of blocks in the list.
//
//  Returns:    A CGNS database from list.
//
//  Programmer: generated by xml2info
//  Creation:   omitted
//
// ****************************************************************************
avtDatabase *
CGNSCommonPluginInfo::SetupDatabase(const char *const *list,
                                   int nList, int nBlock)
{
    avtDatabase *db = NULL;

    //
    // If the number of blocks is 1, see if the names follow the
    // conventions for a multi-file multi-block file.
    //
    if (nBlock == 1)
    {
        bool multiblockFile = true;
        int  globalNBlocks = -1;
        for (int f = 0 ; multiblockFile && f < nList; f++)
        {
            //
            // Eliminate the path information to get just the filename.
            //
            const char *file = strrchr(list[f], VISIT_SLASH_CHAR);
            if (file == NULL) file = list[f];

            //
            // Determine if the filename matches "basename.cgns.nblocks.iblock"
            // where basename is an any string, cgns is the string "cgns",
            // nblocks is the number of blocks, and iblock is the block
            // index. Furthermore, nblocks must evenly divide the number of
            // files in the list and iblock must be monotonically increasing
            // with no gaps in the numbering.
            //
            const char *str = strstr(file, "cgns");
            if (str == NULL)
            {
                multiblockFile = false;
                break;
            }
            if (str[4] != '.')
            {
                multiblockFile = false;
                break;
            }
            char *str2;
            long int nBlocks = strtol(&str[5], &str2, 10);
            if (nBlocks == 0 || nBlocks == LONG_MAX ||
                nBlocks == LONG_MIN || nBlocks < 0 ||
                nBlocks > nList ||
                nList % nBlocks != 0 || str2[0] != '.')
            {
                multiblockFile = false;
                break;
            }
            if (f == 0) globalNBlocks = nBlocks;
            if (nBlocks != globalNBlocks)
            {
                multiblockFile = false;
                break;
            }
            char *str3;
            long int iBlock = strtol(&str2[1], &str3, 10);
            if (iBlock == LONG_MAX || iBlock == LONG_MIN ||
                iBlock < 0 || iBlock >= nBlocks ||
                iBlock != f % nBlocks || str3[0] != '\0')
            {
                multiblockFile = false;
                break;
            }
        }
        if (multiblockFile)
            nBlock = globalNBlocks;
    }

    // ignore any nBlocks past 1
    int nTimestepGroups = nList / nBlock;
    if (nBlock > 1)
    {
        dbType = DB_TYPE_MTSD;

        avtMTSDFileFormat ***ffl = new avtMTSDFileFormat**[nTimestepGroups];
        for (int i = 0 ; i < nTimestepGroups ; i++)
        {
            ffl[i] = new avtMTSDFileFormat*[nBlock];
            for (int j = 0 ; j < nBlock ; j++)
            {
                 ffl[i][j] = new avtCGNS_MTSDFileFormat(list[i*nBlock+j]);
            }
        }
        avtMTSDFileFormatInterface *inter =
            new avtMTSDFileFormatInterface(ffl, nTimestepGroups, nBlock);
        db = new avtGenericDatabase(inter);
        debug5 << "CGNSCommonPluginInfo::SetupDatabase MTSD" << endl;
    }
    else
    {
        dbType = DB_TYPE_MTMD;

        avtMTMDFileFormat **ffl = new avtMTMDFileFormat*[nTimestepGroups];
        for (int i = 0 ; i < nTimestepGroups ; i++)
        {
            ffl[i] = new avtCGNS_MTMDFileFormat(list[i]);
        }
        avtMTMDFileFormatInterface *inter =
            new avtMTMDFileFormatInterface(ffl, nTimestepGroups);
        db = new avtGenericDatabase(inter);
        debug5 << "CGNSCommonPluginInfo::SetupDatabase MTMD" << endl;
    }

    return db;
}
