// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

// ************************************************************************* //
//                      avtBinarySubtractExpression.C                        //
// ************************************************************************* //

#include <avtBinarySubtractExpression.h>

#include <vtkDataArray.h>
#include <vtkDataArray.h>

#include <ExpressionException.h>


// ****************************************************************************
//  Method: avtBinarySubtractExpression constructor
//
//  Purpose:
//      Defines the constructor.  Note: this should not be inlined in the
//      header because it causes problems for certain compilers.
//
//  Programmer: Hank Childs
//  Creation:   February 5, 2004
//
// ****************************************************************************

avtBinarySubtractExpression::avtBinarySubtractExpression()
{
    ;
}


// ****************************************************************************
//  Method: avtBinarySubtractExpression destructor
//
//  Purpose:
//      Defines the destructor.  Note: this should not be inlined in the header
//      because it causes problems for certain compilers.
//
//  Programmer: Hank Childs
//  Creation:   February 5, 2004
//
// ****************************************************************************

avtBinarySubtractExpression::~avtBinarySubtractExpression()
{
    ;
}


// ****************************************************************************
//  Method: avtBinarySubtractExpression::DoOperation
//
//  Purpose:
//      Differences two arrays and puts the results into a third array.
//
//  Arguments:
//      in1           The first input data array.
//      in2           The second input data array.
//      out           The output data array.
//      ncomponents   The number of components ('1' for scalar, '2' or '3' for
//                    vectors, etc.)
//      ntuples       The number of tuples (ie 'npoints' or 'ncells')
//
//  Programmer: Sean Ahern          <Header added by Hank Childs>
//  Creation:   November 18, 2002   <Header creation date>
//
//  Modifications:
//
//    Hank Childs, Mon Nov 18 07:35:07 PST 2002
//    Added support for vectors and arbitrary data types.
//
//    Hank Childs, Thu Aug 14 11:40:23 PDT 2003
//    Added support for mixing scalars and vectors.
//
//    Hank Childs, Mon Jan 14 17:58:58 PST 2008
//    Add support for singleton constants.
//
// ****************************************************************************
 
void
avtBinarySubtractExpression::DoOperation(vtkDataArray *in1, vtkDataArray *in2,
                                     vtkDataArray *out, int ncomponents,
                                     int ntuples)
{
    bool var1IsSingleton = (in1->GetNumberOfTuples() == 1);
    bool var2IsSingleton = (in2->GetNumberOfTuples() == 1);
    int in1ncomps = in1->GetNumberOfComponents();
    int in2ncomps = in2->GetNumberOfComponents();
    if (in1ncomps == in2ncomps)
    {
        for (int i = 0 ; i < ntuples ; i++)
        {
            vtkIdType tup1 = (var1IsSingleton ? 0 : i);
            vtkIdType tup2 = (var2IsSingleton ? 0 : i);
            for (int j = 0 ; j < in1ncomps ; j++)
            {
                double val1 = in1->GetComponent(tup1, j);
                double val2 = in2->GetComponent(tup2, j);
                out->SetComponent(i, j, val1 - val2);
            }
        }
    }
    else if (in1ncomps > 1 && in2ncomps == 1)
    {
        for (int i = 0 ; i < ntuples ; i++)
        {
            vtkIdType tup1 = (var1IsSingleton ? 0 : i);
            vtkIdType tup2 = (var2IsSingleton ? 0 : i);
            double val2 = in2->GetTuple1(tup2);
            for (int j = 0 ; j < in1ncomps ; j++)
            {
                double val1 = in1->GetComponent(tup1, j);
                out->SetComponent(i, j, val1 - val2);
            }
        }
    }
    else if (in1ncomps == 1 && in2ncomps > 1)
    {
        for (int i = 0 ; i < ntuples ; i++)
        {
            vtkIdType tup1 = (var1IsSingleton ? 0 : i);
            vtkIdType tup2 = (var2IsSingleton ? 0 : i);
            double val1 = in1->GetTuple1(tup1);
            for (int j = 0 ; j < in2ncomps ; j++)
            {
                double val2 = in2->GetComponent(tup2, j);
                out->SetComponent(i, j, val1 - val2);
            }
        }
    }
    else
    {
        EXCEPTION2(ExpressionException, outputVariableName, 
                   "Don't know how to subtract vectors "
                   "of differing dimensions.");
    }
}


// ****************************************************************************
//  Method:  avtBinarySubtractExpression::GetVariableType
//
//  Purpose:
//    Better support for array variables.  We actually return the output
//    type to be the same as the input type (assuming they are the same,
//    or if one has 1 compoent).
//
//  Note:
//    This can't yet be pulled up into avtBinaryMathExpression because
//    mult and div do non-elementwise operations.  If that changes,
//    we should pull this up.
//
//  Arguments:
//    none
//
//  Programmer:  Jeremy Meredith
//  Creation:    March 18, 2009
//
// ****************************************************************************
avtVarType
avtBinarySubtractExpression::GetVariableType()
{
    avtDataAttributes &atts = GetInput()->GetInfo().GetAttributes();
    if (varnames.size() != 2)
        return AVT_UNKNOWN_TYPE;

    if (!atts.ValidVariable(varnames[0]) ||
        !atts.ValidVariable(varnames[1]))
        return AVT_UNKNOWN_TYPE;

    int ncomp1 = atts.GetVariableDimension(varnames[0]);
    int ncomp2 = atts.GetVariableDimension(varnames[1]);
    avtVarType type1 = atts.GetVariableType(varnames[0]);
    avtVarType type2 = atts.GetVariableType(varnames[1]);

    if (type1 == type2)
    {
        return type1;
    }
    else if (ncomp1 == 1)
    {
        return type2;
    }
    else if (ncomp2 == 1)
    {
        return type1;
    }
    else
    {
        return AVT_UNKNOWN_TYPE;
    }
}
