/*****************************************************************************
*
* Copyright (c) 2000 - 2018, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-442911
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                            avtWPPImageFileFormat.C                           //
// ************************************************************************* //

#include <avtWPPImageFileFormat.h>

#include <string>

#include <vtkDoubleArray.h>
#include <vtkFloatArray.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnstructuredGrid.h>

#include <avtDatabaseMetaData.h>

#include <DBOptionsAttributes.h>
#include <Expression.h>

#include <InvalidVariableException.h>
#include <InvalidDBTypeException.h>

#include <DebugStream.h>

#include <fcntl.h>
#include <snprintf.h>

using namespace std;
#ifdef WIN32
#include <io.h>

#define CLOSE   ::_close
#define OPEN    ::_open
#define LSEEK   ::_lseek
#define READ    ::_read

#else

#include <unistd.h>

#define O_BINARY 0
#define CLOSE   ::close
#define OPEN    ::open
#define LSEEK   ::lseek
#define READ    ::read

#endif



// ****************************************************************************
//  Method: avtWPPImageFileFormat constructor
//
//  Programmer: bjorn -- generated by xml2avt
//  Creation:   Fri Jan 13 16:02:12 PST 2012
//
// ****************************************************************************

avtWPPImageFileFormat::avtWPPImageFileFormat(const char *filename)
    : avtSTMDFileFormat(&filename, 1)
{
    m_filename    = filename;
    m_initialized = false;

    // INITIALIZE DATA MEMBERS
}


// ****************************************************************************
//  Method: avtWPPImageFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: bjorn -- generated by xml2avt
//  Creation:   Fri Jan 13 16:02:12 PST 2012
//
// ****************************************************************************

void
avtWPPImageFileFormat::FreeUpResources(void)
{
}


// ****************************************************************************
//  Method: avtWPPImageFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: bjorn -- generated by xml2avt
//  Creation:   Fri Jan 13 16:02:12 PST 2012
//
// ****************************************************************************

void
avtWPPImageFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md)
{
    debug5 << "Populatedatabase begins" << endl;

    if( !m_initialized )
        Initialize();

    // Mesh information
    avtMeshMetaData *mmd = new avtMeshMetaData;

    mmd->name = "Cartesian slice";

    mmd->spatialDimension = 3;
    mmd->topologicalDimension = 3;
    mmd->numBlocks = m_nblocks;
    //   mmd->hasUnits = true;
    //   mmd->units = "meters";
    mmd->meshType = AVT_RECTILINEAR_MESH;

    debug5 << "before b-loop " << endl;

    mmd->blockNames.resize(m_nblocks);
    for( int b=0 ; b < m_nblocks ; b++ )
    {
        debug5 << "Block = " << b << endl;
        debug5 << "grid size = " << m_gridsize[b] << endl;
        debug5 << "no of pts = " << m_ni[b] << " " << m_nj[b] << " " 
               << m_nk[b] << endl;
        char buf[50];
        SNPRINTF(buf,50,"h=%f",m_gridsize[b]);
        string bname= buf;
        mmd->blockNames[b] = bname;
    }
    md->Add(mmd);
    debug5 << "after b-loop " << endl;

    // Scalar variable information
    avtScalarMetaData *smd1 = new avtScalarMetaData;
    smd1->name = m_mode;
    // copy the mesh name from above
    smd1->meshName = mmd->name;
    smd1->centering = AVT_NODECENT;
    smd1->hasUnits = false;
    md->Add(smd1);
    debug5 << "Populatedatabase done " << endl;
}


// ****************************************************************************
//  Method: avtWPPImageFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      domain      The index of the domain.  If there are NDomains, this
//                  value is guaranteed to be between 0 and NDomains-1,
//                  regardless of block origin.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: bjorn -- generated by xml2avt
//  Creation:   Fri Jan 13 16:02:12 PST 2012
//
// ****************************************************************************

vtkDataSet *
avtWPPImageFileFormat::GetMesh(int domain, const char *meshname)
{
    debug5 << "constructing a Cartesian grid" << endl;

    if( !m_initialized )
        Initialize();

 
    vtkFloatArray *coords[3]= {0,0,0};
    coords[0] = vtkFloatArray::New();
    coords[0]->SetNumberOfTuples(m_ni[domain]);
    coords[1] = vtkFloatArray::New();
    coords[1]->SetNumberOfTuples(m_nj[domain]);
    coords[2] = vtkFloatArray::New();
    coords[2]->SetNumberOfTuples(m_nk[domain]);
    float* x = (float *)coords[0]->GetVoidPointer(0);
    float* y = (float *)coords[1]->GetVoidPointer(0);
    float* z = (float *)coords[2]->GetVoidPointer(0);
    for( int i=0 ; i < m_ni[domain] ; i++ )
        x[i] = i*m_gridsize[domain]+m_xmin[domain];
    for( int i=0 ; i < m_nj[domain] ; i++ )
        y[i] = i*m_gridsize[domain]+m_ymin[domain];
    for( int i=0 ; i < m_nk[domain] ; i++ )
        z[i] = i*m_gridsize[domain]+m_zmin[domain];
    vtkRectilinearGrid *rgrid= vtkRectilinearGrid::New();
    int dims[3];
    dims[0] = m_ni[domain];
    dims[1] = m_nj[domain];
    dims[2] = m_nk[domain];
    rgrid->SetDimensions(dims);
    rgrid->SetXCoordinates(coords[0]);
    rgrid->SetYCoordinates(coords[1]);
    rgrid->SetZCoordinates(coords[2]);
    coords[0]->Delete();
    coords[1]->Delete();
    coords[2]->Delete();
    debug5 << "done get mesh " << endl;
    return rgrid;
}


// ****************************************************************************
//  Method: avtWPPImageFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      domain     The index of the domain.  If there are NDomains, this
//                 value is guaranteed to be between 0 and NDomains-1,
//                 regardless of block origin.
//      varname    The name of the variable requested.
//
//  Programmer: bjorn -- generated by xml2avt
//  Creation:   Fri Jan 13 16:02:12 PST 2012
//
//  Modifications:
//    Kathleen Biagas, Tue Sep 4 10:18:27 MST 2012
//    Create vtkDoubleArray when m_prec calls for it.
//
// ****************************************************************************

vtkDataArray *
avtWPPImageFileFormat::GetVar(int domain, const char *varname)
{
    debug5 << "avtWPPImageFileFormat::GetVar" << endl;
    if( !m_initialized )
        Initialize();

    int fd = OPEN(m_filename.c_str(), O_RDONLY | O_BINARY );
    char errmsg[500];
    if( fd == -1 )
    {
        SNPRINTF(errmsg,500,"Error opening file %s",m_filename.c_str());
        EXCEPTION1( InvalidDBTypeException, errmsg );
    }
    off_t nr = LSEEK(fd,m_offset[domain],SEEK_CUR);
    if( nr != m_offset[domain] )
    {
        CLOSE(fd);
        SNPRINTF(errmsg,500,"Error accessing array in %s",m_filename.c_str());
        EXCEPTION1( InvalidDBTypeException, errmsg );
    }
    size_t npts = ((size_t) m_ni[domain])*m_nj[domain]*m_nk[domain];
    vtkDataArray *arr = NULL;
    if( m_prec == 4 )
    {
        arr = vtkFloatArray::New();
        arr->SetNumberOfTuples(npts);
        float* data = (float *)arr->GetVoidPointer(0);
#ifdef WIN32
        nr = _read(fd, data, (unsigned int)(sizeof(float)*npts));
#else
        nr = READ(fd,data,sizeof(float)*npts);
#endif
        if( (size_t)nr != sizeof(float)*npts )
        {
            CLOSE(fd);
            SNPRINTF(errmsg,500,"Error reading array in %s", 
                     m_filename.c_str());
            EXCEPTION1( InvalidDBTypeException, errmsg );
        }
    }
    else
    {
        arr = vtkDoubleArray::New();
        arr->SetNumberOfTuples(npts);
        double* data = (double *)arr->GetVoidPointer(0);
#ifdef WIN32
        nr = _read(fd, data, (unsigned int)(sizeof(double)*npts));
#else
        nr = READ(fd,data,sizeof(double)*npts);
#endif
        if( (size_t)nr != sizeof(double)*npts )
        {
            CLOSE(fd); 
            SNPRINTF(errmsg,500,"Error reading array in %s", 
                     m_filename.c_str());
            EXCEPTION1( InvalidDBTypeException, errmsg );
        }
    }
    CLOSE(fd);
    debug5 << "done get var " << endl;
    return arr;
}


// ****************************************************************************
//  Method: avtWPPImageFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      domain     The index of the domain.  If there are NDomains, this
//                 value is guaranteed to be between 0 and NDomains-1,
//                 regardless of block origin.
//      varname    The name of the variable requested.
//
//  Programmer: bjorn -- generated by xml2avt
//  Creation:   Fri Jan 13 16:02:12 PST 2012
//
// ****************************************************************************

vtkDataArray *
avtWPPImageFileFormat::GetVectorVar(int domain, const char *varname)
{
    // There are no vector variables 
    return 0;
}

// ****************************************************************************
//  Method: avtvolimageFileFormat::Initialize
//
//  Purpose:
//      Read the file.
//
//
// ****************************************************************************

void avtWPPImageFileFormat::Initialize()
{
    char errmsg[500];
    debug5 << "Initialize begins" << endl;

    size_t k=m_filename.rfind('.');

    if( k != string::npos ) 
        m_mode = m_filename.substr(k+1);
    else
        m_mode = "(no dot in filename)";
    if( !(m_mode == "ux" || m_mode == "uy" || m_mode == "uz" || 
          m_mode == "rho" || m_mode == "lambda" || m_mode == "mu" || 
          m_mode == "p" || m_mode == "s" || m_mode == "div" || 
          m_mode == "curl" || m_mode == "veldiv" || m_mode == "velcurl" ||
          m_mode == "lat" || m_mode == "lon" || m_mode == "hvelmax" || 
          m_mode == "vvelmax" || m_mode == "topo" || m_mode == "grid" || 
          m_mode == "uxerr" || m_mode == "uyerr" || m_mode == "uzerr" || 
          m_mode == "fx" || m_mode == "fy" || m_mode == "fz" ||
          m_mode == "velmag" || m_mode == "qs" || m_mode == "qp" || 
          m_mode == "hvel" ) ) 
    {
        SNPRINTF(errmsg,500,"Error: Unknown volimage mode %s" , 
                 m_mode.c_str() );
        EXCEPTION1( InvalidDBTypeException, errmsg );
    }
    debug5 << "mode = " << m_mode << endl;

    size_t find3D = m_filename.find("3D");
    if (find3D != string::npos && find3D == k-2)
    {
        // We have a '3D.mode' type file -- this is for volimage
        SNPRINTF(errmsg,500,"Error: WPPImage reader does not read 3D volimage"
                " files -- use the volimage reader instead. mode -- %s", m_mode.c_str());
        debug1 << errmsg << endl; 
        EXCEPTION1( InvalidDBTypeException, errmsg );
    }
    else
    { 
        debug5 << "Not a volimage file, continuing..." << endl; 
    }
      
    int fd = OPEN( m_filename.c_str(), O_RDONLY|O_BINARY );
    if( fd == -1 )
    {
        SNPRINTF(errmsg,500,"Error in WPPImage opening file %s",
                 m_filename.c_str());
        EXCEPTION1( InvalidDBTypeException, errmsg );
    }
    debug5 << "file opened " << endl;
    size_t nr = READ(fd,&m_prec,sizeof(int) );
    if( nr != sizeof(int) )
    {
        CLOSE(fd); // closing the solution file
        SNPRINTF(errmsg,500,"Error reading precision in %s",
                 m_filename.c_str());
        EXCEPTION1( InvalidDBTypeException, errmsg );
    }
    if( (m_prec != 4) && (m_prec != 8 ) )
    {
        CLOSE(fd); // closing the solution file
        SNPRINTF(errmsg,500,"Error, precision is %i, should be 4 or 8\n",
                 m_prec);
        EXCEPTION1( InvalidDBTypeException, errmsg );
    }
    nr = READ(fd,&m_nblocks,sizeof(int) );
    if( nr != sizeof(int) )
    {
        CLOSE(fd); // closing the solution file
        SNPRINTF(errmsg,500,"Error reading nblocks in %s",m_filename.c_str());
        EXCEPTION1( InvalidDBTypeException, errmsg );
    }
    debug5 << "prec = " << m_prec << " " << " nblocks = " << m_nblocks << endl;

    // was there a grid file?
    m_CartGrid = true;
    debug5 << "No grid file, assuming Cartesian grid" << endl;

    double plane_value = 0;
    int slice_plane = 2;
    const char *ptr = 0;
    if((ptr = strstr(m_filename.c_str(), "x=")) != NULL)
       slice_plane = 0;
    else if((ptr = strstr(m_filename.c_str(), "y=")) != NULL)
       slice_plane = 1;
    else if((ptr = strstr(m_filename.c_str(), "z=")) != NULL)
       slice_plane = 2;
    if(ptr != 0)
       sscanf(ptr+2, "%lf", &plane_value );

    m_gridsize.resize(m_nblocks);
    m_xmin.resize(m_nblocks);
    m_ymin.resize(m_nblocks);
    m_zmin.resize(m_nblocks);
    m_ni.resize(m_nblocks);
    m_nj.resize(m_nblocks);
    m_nk.resize(m_nblocks);
    m_offset.resize(m_nblocks);

    size_t header_offset = 
        2*sizeof(int)+m_nblocks*(sizeof(double)+4*sizeof(int));
    int dims[4];
    debug5 << "reading block headers " << endl;
    for( int b=0 ; b < m_nblocks ; b++ )
    {
        debug5 << "b = " << b << endl;
        nr = READ(fd,&m_gridsize[b],sizeof(double));
        if( nr != sizeof(double) )
        {
            CLOSE(fd); // closing the solution file
            SNPRINTF(errmsg,500,"Error reading gridsizes in %s",
                     m_filename.c_str());
            EXCEPTION1( InvalidDBTypeException, errmsg );
        }
        nr = READ(fd,dims,sizeof(int)*4);
        if( nr != sizeof(int)*4 )
        {
            CLOSE(fd); // closing the solution file
            SNPRINTF(errmsg,500,"Error reading dimensions in %s",
                     m_filename.c_str());
            EXCEPTION1( InvalidDBTypeException, errmsg );
        }
        if( slice_plane == 0 )
        {
            m_ni[b] = 1;
            m_nj[b] = dims[1]-dims[0]+1;
            m_nk[b] = dims[3]-dims[2]+1;
            m_xmin[b] = plane_value;
            m_ymin[b] = (dims[0]-1)*m_gridsize[b];
            m_zmin[b] = (dims[2]-1)*m_gridsize[b];
        }
        else if( slice_plane == 1 )
        {
            m_ni[b] = dims[1]-dims[0]+1;
            m_nj[b] = 1;
            m_nk[b] = dims[3]-dims[2]+1;
            m_xmin[b] = (dims[0]-1)*m_gridsize[b];
            m_ymin[b] = plane_value;
            m_zmin[b] = (dims[2]-1)*m_gridsize[b];
        }
        else
        {
            m_ni[b] = dims[1]-dims[0]+1;
            m_nj[b] = dims[3]-dims[2]+1;
            m_nk[b] = 1;
            m_xmin[b] = (dims[0]-1)*m_gridsize[b];
            m_ymin[b] = (dims[2]-1)*m_gridsize[b];
            m_zmin[b] = plane_value;
        }
        debug5 << "b = " << b << " dims[0] = " << dims[0] << " dims[1] = " 
               << dims[1] << " dims[2] = " << dims[2] << " dims[3] = " 
               << dims[3] << endl;
        debug5 << "x_min = " << m_xmin[b] << " y_min = " << m_ymin[b] 
               << " z_min = " << m_zmin[b] << endl;
    }
    m_offset[0] = (off_t)header_offset;
    int datasize;
    if( m_prec == 4 )
        datasize = sizeof(float);
    else
        datasize = sizeof(double);

    // calculate offsets for accessing data
    for( int b=1 ; b < m_nblocks ; b++ )
        m_offset[b] = m_offset[b-1] + ((size_t) m_ni[b-1])*m_nj[b-1]*m_nk[b-1]*datasize;

    // z-offset for x- or y- slices
    if( slice_plane == 0 || slice_plane == 1 )
       for( int b=m_nblocks-1 ; b > 0 ; b-- )
           m_zmin[b-1] = m_zmin[b] + (m_nk[b]-1)*m_gridsize[b];

    CLOSE(fd); // closing the solution file

    m_initialized = true;
    debug5 << "Initialize done " << endl;
}
