/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

#include <visit-config.h>
#include <snprintf.h>

#include <QSocketNotifier>

#include <ViewerServerManager.h>
#include <Connection.h>
#include <HostProfileList.h>
#include <MachineProfile.h>
#include <LauncherProxy.h>
#include <RemoteProxyBase.h>
#include <LostConnectionException.h>
#include <CouldNotConnectException.h>
#include <CancelledConnectException.h>
#include <ViewerConnectionProgressDialog.h>
#include <ViewerPasswordWindow.h>
#include <ViewerProperties.h>
#include <ViewerSubject.h>
#include <ViewerRemoteProcessChooser.h>

#include <DebugStream.h>
#include <avtCallback.h>
#include <Utility.h>

//
// Global variables.
//
extern ViewerSubject *viewerSubject;

//
// Static members.
//
HostProfileList *ViewerServerManager::clientAtts = 0;
int ViewerServerManager::debugLevel = 0;
bool ViewerServerManager::bufferDebug = false;
std::string ViewerServerManager::localHost("localhost");
stringVector ViewerServerManager::arguments;
ViewerServerManager::LauncherMap ViewerServerManager::launchers;
void * ViewerServerManager::cbData[2] = {0,0};
bool ViewerServerManager::sshTunnelingForcedOn = false;

// ****************************************************************************
// Method: ViewerServerManager::ViewerServerManager
//
// Purpose: 
//   This is the constructor for the ViewerServerManager class.
//
// Programmer: Brad Whitlock
// Creation:   Fri May 3 16:23:16 PST 2002
//
// Modifications:
//   Brad Whitlock, Mon Feb 12 17:47:27 PST 2007
//   Added ViewerBase base class.
//
// ****************************************************************************

ViewerServerManager::ViewerServerManager() : ViewerBase(0)
{
}

// ****************************************************************************
// Method: ViewerServerManager::ViewerServerManager
//
// Purpose: 
//   This is the destructor for the ViewerServerManager class.
//
// Programmer: Brad Whitlock
// Creation:   Fri May 3 16:23:16 PST 2002
//
// Modifications:
//   
// ****************************************************************************

ViewerServerManager::~ViewerServerManager()
{
}

// ****************************************************************************
// Method: ViewerServerManager::GetClientAtts
//
// Purpose: 
//   Returns a pointer to the host profile list.
//
// Returns:    A pointer to the host profile list.
//
// Programmer: Brad Whitlock
// Creation:   Fri May 3 16:12:56 PST 2002
//
// Modifications:
//   
// ****************************************************************************

HostProfileList *
ViewerServerManager::GetClientAtts()
{
    //
    // If the client attributes haven't been allocated then do so.
    //
    if (clientAtts == 0)
    {
        clientAtts = new HostProfileList;
    }

    return clientAtts;
}

// ****************************************************************************
// Method: ViewerServerManager::SetDebugLevel
//
// Purpose: 
//   Sets the debug level that is passed to the engines.
//
// Arguments:
//   level : The debug level.
//
// Programmer: Brad Whitlock
// Creation:   Mon Nov 27 17:29:01 PST 2000
//
// Modifications:
//   
//    Mark C. Miller, Tue Apr 21 14:24:18 PDT 2009
//    Added bufferDebug to control buffering of debug logs.
// ****************************************************************************

void
ViewerServerManager::SetDebugLevel(int level, bool doBuf)
{
    debugLevel = level;
    bufferDebug = doBuf;
}

// ****************************************************************************
// Method: ViewerFileServer::SetArguments
//
// Purpose: 
//   Sets the arguements passed to the mdservers that get launched.
//
// Arguments:
//   args : the arguments
//
// Programmer: Jeremy Meredith
// Creation:   April 19, 2001
//
// Modifications:
//   
// ****************************************************************************

void
ViewerServerManager::SetArguments(const stringVector &arg)
{
    arguments = arg;
}

// ****************************************************************************
// Method: ViewerServerManager::SetLocalHost
//
// Purpose: 
//   Sets the localHost name used internally to determine which engines are
//   active.
//
// Arguments:
//   hostName : The name of the localhost machine.
//
// Programmer: Brad Whitlock
// Creation:   Mon Sep 24 14:00:59 PST 2001
//
// Modifications:
//   
// ****************************************************************************

void
ViewerServerManager::SetLocalHost(const std::string &hostName)
{
    localHost = hostName;
}

// ****************************************************************************
// Method: ViewerServerManager::HostIsLocalHost
//
// Purpose: 
//   Determines if the hostname is localhost.
//
// Arguments:
//   hostName : The hostname to consider.
//
// Programmer: Brad Whitlock
// Creation:   Wed May 7 10:45:42 PDT 2003
//
// Modifications:
//   
// ****************************************************************************

bool
ViewerServerManager::HostIsLocalHost(const std::string &hostName)
{
    return hostName == localHost || hostName == "localhost";
}

// ****************************************************************************
// Method: ViewerServerManager::RealHostName
//
// Purpose: 
//   Compares the hostName to "localhost" and if they are equal then the
//   real localhost name is returned.
//
// Arguments:
//   hostName : The hostname to check.
//
// Programmer: Brad Whitlock
// Creation:   Mon Sep 24 14:42:04 PST 2001
//
// Modifications:
//   
// ****************************************************************************

const char *
ViewerServerManager::RealHostName(const char *hostName) const
{
    const char *retval = hostName;
    if(strcmp(hostName, "localhost") == 0)
        retval = localHost.c_str();
    return retval;
}

// ****************************************************************************
// Method: ViewerServerManager::AddProfileArguments
//
// Purpose: 
//   This method finds a host profile that matches the specified hostname
//   and adds the profile arguments to the component proxy.
//
// Arguments:
//   component : The proxy to which we're adding arguments.
//   host      : The host where the component will be run.
//
// Programmer: Brad Whitlock
// Creation:   Tue May 6 14:00:21 PST 2003
//
// Modifications:
//   Eric Brugger, Wed Dec  3 08:27:47 PST 2003
//   I removed the default timeout passed to the mdserver.
//
//   Brad Whitlock, Thu Aug 5 10:28:33 PDT 2004
//   I made it call RemoteProxyBase::AddProfileArguments.
//
//   Jeremy Meredith, Thu Feb 18 15:25:27 EST 2010
//   Split HostProfile int MachineProfile and LaunchProfile.
//
// ****************************************************************************

void
ViewerServerManager::AddProfileArguments(RemoteProxyBase *component,
    const std::string &host)
{
    //
    // Check for a host profile for the hostName. If one exists, add
    // any arguments to the command line for the engine proxy.
    //
    const MachineProfile *profile =
         clientAtts->GetMachineProfileForHost(host);
    if(profile != 0)
        component->AddProfileArguments(*profile, false);
}

// ****************************************************************************
// Method: ViewerServerManager::GetClientMachineNameOptions
//
// Purpose: 
//   This method finds a host profile that matches the specified hostname
//   and returns the client host name options.
//
// Arguments:
//   host          : The host where the component will be run.
//   chd           : The type of client host name determination to use.
//   clientHostName: The manual host name, if manual determination is requested
//
// Programmer: Jeremy Meredith
// Creation:   October  9, 2003
//
// Modifications:
//    Jeremy Meredith, Thu Feb 18 15:25:27 EST 2010
//    Split HostProfile int MachineProfile and LaunchProfile.
//   
// ****************************************************************************

void
ViewerServerManager::GetClientMachineNameOptions(const std::string &host,
                                 MachineProfile::ClientHostDetermination &chd,
                                 std::string &clientHostName)
{
    //
    // Check for a host profile for the hostName. If one exists, 
    // return the client host name options.
    //
    const MachineProfile *profile =
         clientAtts->GetMachineProfileForHost(host);
    if(profile != 0)
    {
        chd = profile->GetClientHostDetermination();
        clientHostName = profile->GetManualClientHostName();
    }
    else
    {
        chd = MachineProfile::MachineName;
        clientHostName = "";
    }
}

// ****************************************************************************
// Method: ViewerServerManager::GetClientSSHPortOptions
//
// Purpose: 
//   This method finds a host profile that matches the specified hostname
//   and returns the ssh port options.
//
// Arguments:
//   host          : The host where the component will be run.
//   manualSSHPort : True if a manual ssh port was specified
//   sshPort       : The manual ssh port
//
// Programmer: Jeremy Meredith
// Creation:   October  9, 2003
//
// Modifications:
//    Jeremy Meredith, Thu Feb 18 15:25:27 EST 2010
//    Split HostProfile int MachineProfile and LaunchProfile.
//   
// ****************************************************************************

void
ViewerServerManager::GetSSHPortOptions(const std::string &host,
                                       bool &manualSSHPort,
                                       int &sshPort)
{
    //
    // Check for a host profile for the hostName. If one exists, 
    // return the ssh port options.
    //
    const MachineProfile *profile =
         clientAtts->GetMachineProfileForHost(host);
    if(profile != 0)
    {
        manualSSHPort = profile->GetSshPortSpecified();
        sshPort = profile->GetSshPort();
    }
    else
    {
        manualSSHPort = false;
        sshPort = -1;
    }
}

// ****************************************************************************
// Method: ViewerServerManager::GetSSHTunnelOptions
//
// Purpose: 
//   This method finds a host profile that matches the specified hostname
//   and returns the ssh tunneling options.
//
// Arguments:
//   host          : The host where the component will be run.
//   tunnelSSH     : True if we need to attempt tunneling
//
// Programmer: Jeremy Meredith
// Creation:   May 22, 2007
//
// Modifications:
//    Thomas R. Treadway, Mon Oct  8 13:27:42 PDT 2007
//    Backing out SSH tunneling on Panther (MacOS X 10.3)
//   
//    Jeremy Meredith, Wed Dec  3 16:48:35 EST 2008
//    Allowed commandline override forcing-on of SSH tunneling.
//
//    Jeremy Meredith, Thu Feb 18 15:25:27 EST 2010
//    Split HostProfile int MachineProfile and LaunchProfile.
//
// ****************************************************************************

void
ViewerServerManager::GetSSHTunnelOptions(const std::string &host,
                                         bool &tunnelSSH)
{
    //
    // Check for a host profile for the hostName. If one exists, 
    // return the ssh tunneling options.
    //
#if defined(PANTHERHACK)
// Broken on Panther
    tunnelSSH = false;
#else
    if (sshTunnelingForcedOn)
    {
        tunnelSSH = true;
    }
    else
    {
        const MachineProfile *profile =
            clientAtts->GetMachineProfileForHost(host);
        if(profile != 0)
        {
            tunnelSSH = profile->GetTunnelSSH();
        }
        else
        {
            tunnelSSH = false;
        }
    }
#endif
}

// ****************************************************************************
//  Method: ViewerServerManager::ShouldShareBatchJob
//
//  Purpose: 
//    This method finds a host profile that matches the specified hostname
//    and checks if the MDServer and Engine should share a single batch
//    job.  (If so, then the launcher should start in batch.)
//
//  Arguments:
//    host      : The host where the component will be run.
//
//  Programmer: Jeremy Meredith
//  Creation:   June 17, 2003
//
//  Modifications:
//    Jeremy Meredith, Thu Feb 18 15:25:27 EST 2010
//    Split HostProfile int MachineProfile and LaunchProfile.
//   
// ****************************************************************************

bool
ViewerServerManager::ShouldShareBatchJob(const std::string &host)
{
    //
    // Check for a host profile for the hostName. If one exists, check it.
    //
    const MachineProfile *profile =
         clientAtts->GetMachineProfileForHost(host);
    if (profile != 0)
    {
        return profile->GetShareOneBatchJob();
    }
    else
    {
        return false;
    }
}

// ****************************************************************************
// Method: ViewerServerManager::AddArguments
//
// Purpose: 
//   Adds standard arguments to the remote component.
//
// Arguments:
//   component : The proxy to which we're adding arguments.
//   args      : The arguments to add to the component.
//
// Programmer: Brad Whitlock
// Creation:   Tue May 6 14:01:33 PST 2003
//
// Modifications:
//   
//    Mark C. Miller, Tue Apr 21 14:24:18 PDT 2009
//    Added bufferDebug to control buffering of debug logs.
// ****************************************************************************

void
ViewerServerManager::AddArguments(RemoteProxyBase *component,
    const stringVector &args)
{
    // Add arguments to the mdserver.
    if(debugLevel > 0)
    {
        char temp[10];
        if (bufferDebug)
            SNPRINTF(temp, 10, "%db", debugLevel);
        else
            SNPRINTF(temp, 10, "%d", debugLevel);
        component->AddArgument("-debug");
        component->AddArgument(temp);
    }

    //
    // Add arguments stored in ViewerServerManager.
    //
    int i;
    for (i = 0; i < arguments.size(); ++i)
         component->AddArgument(arguments[i].c_str());

    //
    // Add any other arguments given to us by the caller
    //
    for (i = 0; i < args.size(); ++i)
         component->AddArgument(args[i].c_str());
}

// ****************************************************************************
// Method: ViewerServerManager::SetupConnectionProgressWindow
//
// Purpose: 
//   Creates a connection progress dialog that is hooked up to the component
//   that we're launching. This lets us see how things are launched and lets
//   us cancel the launch if we want.
//
// Arguments:
//   component : The component to launch.
//   host      : The host where the component will be run.
//
// Programmer: Brad Whitlock
// Creation:   Tue May 6 14:02:41 PST 2003
//
// Modifications:
//   Brad Whitlock, Mon May 19 17:40:40 PST 2003
//   I made the timeout for showing the window be zero if the component
//   being launched is remote or parallel.
//
//   Brad Whitlock, Wed Oct 8 14:49:45 PST 2003
//   I made the timeout be zero on MacOS X. This can be undone later when
//   VisIt launches faster there.
//
//   Brad Whitlock, Tue May 27 15:54:41 PDT 2008
//   VisIt is fast on Intel Macs so make the timeout be non-zero.
//
//   Brad Whitlock, Tue Apr 14 11:31:48 PDT 2009
//   Use ViewerProperties.
//
// ****************************************************************************

ViewerConnectionProgressDialog *
ViewerServerManager::SetupConnectionProgressWindow(RemoteProxyBase *component, 
    const std::string &host)
{
    ViewerConnectionProgressDialog *dialog = 0;
#ifdef HAVE_THREADS
    //
    // Set the engine proxy's progress callback.
    //
    if(!GetViewerProperties()->GetNowin())
    {
        int timeout = (component->Parallel() || !HostIsLocalHost(host)) ? 0 : 4000;

        // Create a new connection dialog.
        dialog = new ViewerConnectionProgressDialog(
            component->GetComponentName().c_str(),
            host.c_str(), component->Parallel(), timeout);
        cbData[0] = (void *)viewerSubject;
        cbData[1] = (void *)dialog;
        component->SetProgressCallback(ViewerSubject::LaunchProgressCB,
                                       cbData);

        // Register the dialog with the password window so we can set
        // the dialog's timeout to zero if we have to prompt for a
        // password.
        ViewerPasswordWindow::SetConnectionProgressDialog(dialog);
    }
#endif

    return dialog;
}

// ****************************************************************************
// Method: ViewerServerManager::CloseLaunchers
//
// Purpose: 
//   Close all of the launcher programs.
//
// Programmer: Brad Whitlock
// Creation:   Wed May 7 14:00:16 PST 2003
//
// Modifications:
//   Brad Whitlock, Wed Nov 21 15:00:45 PST 2007
//   Changed map storage type.
//
// ****************************************************************************

void
ViewerServerManager::CloseLaunchers()
{
    LauncherMap::iterator pos;
    for(pos = launchers.begin(); pos != launchers.end(); ++pos)
    {
        pos->second.launcher->Close();
        delete pos->second.launcher;
        pos->second.launcher = 0;
        delete pos->second.notifier;
        pos->second.notifier = 0;
    }
}

// ****************************************************************************
// Method: ViewerServerManager::SendKeepAlivesToLaunchers
//
// Purpose: 
//   Sends keep alive signals to the component launchers.
//
// Programmer: Brad Whitlock
// Creation:   Fri Mar 12 12:02:25 PDT 2004
//
// Modifications:
//   Brad Whitlock, Wed Nov 21 15:01:35 PST 2007
//   Changed map storage type.
//
//   Mark C. Miller, Wed Jun 17 14:27:08 PDT 2009
//   Replaced CATCHALL(...) with CATCHALL.
// ****************************************************************************

void
ViewerServerManager::SendKeepAlivesToLaunchers()
{
    LauncherMap::iterator pos;
    for(pos = launchers.begin(); pos != launchers.end();)
    {
        TRY
        {
            debug2 << "Sending keep alive signal to launcher on "
                   << pos->first.c_str() << endl;
            pos->second.launcher->SendKeepAlive();
            ++pos;
        }
        CATCHALL
        {
            debug2 << "Could not send keep alive signal to launcher on "
                   << pos->first.c_str() << " so that launcher will be closed."
                   << endl;
            delete pos->second.launcher;
            pos->second.launcher = 0;
            delete pos->second.notifier;
            pos->second.notifier = 0;
            launchers.erase(pos++);
        }
        ENDTRY
    }
}

// ****************************************************************************
// Method: ViewerServerManager::StartLauncher
//
// Purpose: 
//   This method starts a launcher process on the specified host.
//
// Arguments:
//   host : The host where we want to run the launcher.
//
// Programmer: Brad Whitlock
// Creation:   Tue May 6 13:59:41 PST 2003
//
// Modifications:
//   Brad Whitlock, Tue Jun 10 14:21:38 PST 2003
//   I made it use the visitPath if it is a valid value.
//
//   Jeremy Meredith, Thu Jun 26 10:51:14 PDT 2003
//   Added ability for launcher to start a parallel batch job
//   for itself, the mdserver, and the engine if they are told
//   to share one batch job in the host profile.
//
//   Jeremy Meredith, Thu Oct  9 14:03:11 PDT 2003
//   Added ability to manually specify a client host name or to have it
//   parsed from the SSH_CLIENT (or related) environment variables.  Added
//   ability to specify an SSH port.
//
//   Brad Whitlock, Thu Aug 5 10:52:40 PDT 2004
//   I made it get its profile from the chooser if possible.
//
//   Jeremy Meredith, Thu May 24 10:17:57 EDT 2007
//   Since SSH tunneling is only useful when launching the VCL, we
//   check its actual value and use pass it along.
//
//   Brad Whitlock, Wed Nov 21 14:39:37 PST 2007
//   Added support for reading console output from VCL's children.
//
//   Jeremy Meredith, Thu Feb 18 15:25:27 EST 2010
//   Split HostProfile int MachineProfile and LaunchProfile.
//
// ****************************************************************************

void
ViewerServerManager::StartLauncher(const std::string &host,
     const std::string &visitPath, ViewerConnectionProgressDialog *dialog)
{
    if(launchers.find(host) == launchers.end())
    {
        MachineProfile profile;
        const MachineProfile *mp = clientAtts->GetMachineProfileForHost(host);
        bool shouldShareBatchJob = false;
        if (mp != 0)
        {
            shouldShareBatchJob = mp->GetShareOneBatchJob();
            profile = *mp;
        }

        if (shouldShareBatchJob)
        {
            ViewerRemoteProcessChooser *chooser =
                ViewerRemoteProcessChooser::Instance();

            chooser->ClearCache(host);

            if (! chooser->SelectProfile(clientAtts, host, false, profile))
                return;
        }

        // Create a new launcher proxy and add the right arguments to it.
        LauncherProxy *newLauncher = new LauncherProxy;
        stringVector args;
        if(visitPath.size() > 0)
        {
            args.push_back("-dir");
            args.push_back(visitPath);
        }
        AddArguments(newLauncher, args);
        newLauncher->AddProfileArguments(profile, shouldShareBatchJob);

        TRY
        {
            // If we're reusing the dialog, set the launcher's progress
            // callback function so that the window pops up.
            if(dialog)
            {
                cbData[0] = (void *)viewerSubject;
                cbData[1] = (void *)dialog;
                newLauncher->SetProgressCallback(
                    ViewerSubject::LaunchProgressCB, cbData);
                dialog->setIgnoreHide(true);
            }

            // Get the client machine name options
            MachineProfile::ClientHostDetermination chd;
            std::string clientHostName;
            GetClientMachineNameOptions(host, chd, clientHostName);

            // Get the ssh port options
            bool manualSSHPort;
            int  sshPort;
            GetSSHPortOptions(host, manualSSHPort, sshPort);

            bool useTunneling;
            GetSSHTunnelOptions(host, useTunneling);

            //
            // Launch the VisIt component launcher on the specified host.
            //
            newLauncher->Create(host, chd, clientHostName,
                                manualSSHPort, sshPort, useTunneling);
            launchers[host].launcher = newLauncher;

            // Create a socket notifier for the launcher's data socket so
            // we can read remote console output as it is forwarded.
            launchers[host].notifier = new ViewerConnectionPrinter(
                 newLauncher->GetWriteConnection(1));

            // Set the dialog's information back to the previous values.
            if(dialog)
            {
                dialog->setIgnoreHide(false);
            }
        }
        CATCH(VisItException)
        {
            delete newLauncher;
            RETHROW;
        }
        ENDTRY
    }
}

// ****************************************************************************
// Method: ViewerServerManager::OpenWithLauncher
//
// Purpose: 
//   This method is a callback function that lets a component proxy be launched
//   via a launcher program that we run on a remote machine. This lets us
//   spawn multiple processes remotely without needing multiple passwords.
//
// Arguments:
//   remoteHost : The name of the computer where we want the component to run.
//   args       : The command line to run on the remote machine.
//   data       : Optional callback data.
//
// Programmer: Brad Whitlock
// Creation:   Tue May 6 14:04:41 PST 2003
//
// Modifications:
//   Brad Whitlock, Tue Jun 10 14:22:01 PST 2003
//   I made it extract the path to VisIt from the arguments.
//
//   Brad Whitlock, Thu Mar 9 10:35:40 PDT 2006
//   I made it throw a CancelledConnect exception in the event that we're
//   launching on a machine that shares mdserver and engine batch jobs. That
//   way, if the launch was cancelled, we don't crash!
//
//   Brad Whitlock, Wed Nov 21 14:41:54 PST 2007
//   Changed map storage type.
//
// ****************************************************************************

void
ViewerServerManager::OpenWithLauncher(
    const std::string &remoteHost, 
    const stringVector &args, void *data)
{
    bool retry = false;
    int  numAttempts = 0;
    bool launched = false;
    bool cancelled = false;

    do
    {
        TRY
        {
            // We use the data argument to pass in a pointer to the connection
            // progress window.
            ViewerConnectionProgressDialog *dialog =
                (ViewerConnectionProgressDialog *)data;

            // Search the args list and see if we've supplied the path to
            // the visit executable.
            std::string visitPath;
            for(int i = 0; i < args.size(); ++i)
            {
                if(args[i] == "-dir" && (i+1) < args.size())
                {
                    visitPath = args[i+1];
                    ++i;
                }
            }

            // Try to start a launcher on remoteHost.
            StartLauncher(remoteHost, visitPath, dialog);

            // Try to make the launcher launch the process.
            if(launchers.find(remoteHost) == launchers.end())
                cancelled = true;
            else
            {
                launchers[remoteHost].launcher->LaunchProcess(args);
                // Indicate success.
                launched = true;
            }

            retry = false;
        }
        CATCH(LostConnectionException)
        {
            // We lost the connection to the launcher program so we need
            // to delete its proxy and remove it from the launchers map
            // so the next time we go through this loop, we relaunch it.
            delete launchers[remoteHost].launcher;
            launchers[remoteHost].launcher = 0;
            delete launchers[remoteHost].notifier;
            launchers[remoteHost].notifier = 0;
            LauncherMap::iterator pos = launchers.find(remoteHost);
            launchers.erase(pos);

            retry = true;
            ++numAttempts;
        }
        // All other VisItExceptions are thrown out of this routine so they
        // can be handled by the managers.
        ENDTRY
    } while(retry && numAttempts < 2);

    if(cancelled)
    {
        EXCEPTION0(CancelledConnectException);
    }

    if(!launched)
    {
        EXCEPTION0(CouldNotConnectException);
    }
}

// ****************************************************************************
//  Method: ViewerServerManager::SimConnectThroughLauncher 
//
//  Purpose:
//    Connect to a simulation using the launcher.  This is similar to 
//    OpenWithLauncher, but it connects to a running simulation instead
//    of starting a new engine process.
//
//  Arguments:
//   remoteHost : The name of the computer where we want the component to run.
//   args       : The command line to run on the remote machine.
//   data       : Optional callback data.
//
//  Programmer:  Jeremy Meredith
//  Creation:    March 30, 2004
//
//  Modifications:
//    Brad Whitlock, Wed Mar 31 10:20:33 PDT 2004
//    Fixed code so it builds on the SGI.
//
//    Jeremy Meredith, Wed May 11 09:04:52 PDT 2005
//    Added security key to simulation connection.
//
//    Brad Whitlock, Thu Mar 9 10:35:40 PDT 2006
//    I made it throw a CancelledConnect exception in the event that we're
//    launching on a machine that shares mdserver and engine batch jobs. That
//    way, if the launch was cancelled, we don't crash! I also added support
//    for a connection progress dialog when launching VCL. A connection
//    progress dialog for aborting the connection to the sim will take quite
//    a bit more work.
//
//    Brad Whitlock, Wed Nov 21 14:43:04 PST 2007
//    Changed the map storage type.
//
// ****************************************************************************

void
ViewerServerManager::SimConnectThroughLauncher(const std::string &remoteHost, 
                                               const stringVector &args,
                                               void *data)
{
    bool retry = false;
    int  numAttempts = 0;
    bool launched = false;
    bool cancelled = false;

    do
    {
        TRY
        {
            // We use the data argument to pass in a pointer to the connection
            // progress window.
            typedef struct {
                string h; int p; string k;
                ViewerConnectionProgressDialog *d;
                bool tunnel;} SimData;
            SimData *simData = (SimData*)data;

            // Search the args list and see if we've supplied the path to
            // the visit executeable.
            std::string visitPath;
            for(int i = 0; i < args.size(); ++i)
            {
                if(args[i] == "-dir" && (i+1) < args.size())
                {
                    visitPath = args[i+1];
                    ++i;
                }
            }

            // Try to start a launcher on remoteHost.
            StartLauncher(remoteHost, visitPath, simData->d);

            // Try to make the launcher launch the process.
            if(launchers.find(remoteHost) == launchers.end())
                cancelled = true;
            else
            {
                // If we're doing SSH tunneling, change the arguments here.
                stringVector args2(args);
                if(simData->tunnel)
                    ConvertArgsToTunneledValues(GetPortTunnelMap(remoteHost), args2);

                launchers[remoteHost].launcher->ConnectSimulation(args2,
                    simData->h, simData->p, simData->k);

                // Indicate success.
                launched = true;
            }

            retry = false;
        }
        CATCH(LostConnectionException)
        {
            // We lost the connection to the launcher program so we need
            // to delete its proxy and remove it from the launchers map
            // so the next time we go through this loop, we relaunch it.
            delete launchers[remoteHost].launcher;
            launchers[remoteHost].launcher = 0;
            delete launchers[remoteHost].notifier;
            launchers[remoteHost].notifier = 0;
            LauncherMap::iterator pos = launchers.find(remoteHost);
            launchers.erase(pos);

            retry = true;
            ++numAttempts;
        }
        // All other VisItExceptions are thrown out of this routine so they
        // can be handled by the managers.
        ENDTRY
    } while(retry && numAttempts < 2);

    if(cancelled)
    {
        EXCEPTION0(CancelledConnectException);
    }

    if(!launched)
    {
        EXCEPTION0(CouldNotConnectException);
    }
}



// ****************************************************************************
//  Method:  ViewerServerManager::GetPortTunnelMap
//
//  Purpose:
//    Retrieve the SSH tunneling local-to-remote port map for the
//    appropriate host.
//
//  Arguments:
//    host       the host for which we need the ssh port tunnel map
//
//  Programmer:  Jeremy Meredith
//  Creation:    May 24, 2007
//
//  Modifications:
//    Thomas R. Treadway, Mon Oct  8 13:27:42 PDT 2007
//    Backing out SSH tunneling on Panther (MacOS X 10.3)
//
//    Brad Whitlock, Wed Nov 21 14:43:24 PST 2007
//    Changed the map storage type.
//
// ****************************************************************************
#if defined(PANTHERHACK)
// Broken on Panther
#else
std::map<int,int>
ViewerServerManager::GetPortTunnelMap(const std::string &host)
{
    std::map<int,int> ret;
    if (launchers.count(host))
        ret = launchers[host].launcher->GetPortTunnelMap();
    return ret;
}
#endif

//
// ViewerConnectionPrinter class. It's really a minor class so it's in here.
//

// ****************************************************************************
// Method: ViewerConnectionPrinter::ViewerConnectionPrinter
//
// Purpose: 
//   Constructor.
//
// Arguments:
//   conn : The connection associated with the socket.
//   name : The name of the object.
//
// Programmer: Brad Whitlock
// Creation:   Wed Nov 21 15:22:42 PST 2007
//
// Modifications:
//   
// ****************************************************************************

ViewerConnectionPrinter::ViewerConnectionPrinter(Connection *c) : 
    QSocketNotifier(c->GetDescriptor(), QSocketNotifier::Read, 0)
{
    conn = c;
    connect(this, SIGNAL(activated(int)),
            this, SLOT(HandleRead(int)));
}

// ****************************************************************************
// Method: ViewerConnectionPrinter::~ViewerConnectionPrinter
//
// Purpose: 
//   Destructor.
//
// Programmer: Brad Whitlock
// Creation:   Wed Nov 21 15:23:38 PST 2007
//
// Modifications:
//   
// ****************************************************************************

ViewerConnectionPrinter::~ViewerConnectionPrinter()
{
}

// ****************************************************************************
// Method: ViewerConnectionPrinter::HandleRead
//
// Purpose: 
//   This is a Qt slot function that is called when the socket has data. The
//   data gets read and printed to the console.
//
// Programmer: Brad Whitlock
// Creation:   Wed Nov 21 15:23:52 PST 2007
//
// Modifications:
//   Brad Whitlock, Fri Jan 9 15:13:01 PST 2009
//   Catch the rest of the possible exceptions.
//
//   Mark C. Miller, Wed Jun 17 17:46:18 PDT 2009
//   Replaced CATCHALL(...) with CATCHALL
// ****************************************************************************

void
ViewerConnectionPrinter::HandleRead(int)
{
    TRY
    {
        // Fill up the connection from the socket.
        conn->Fill();

        // Print the output that we read to stdout.
        while(conn->Size() > 0)
        {
            unsigned char c;
            conn->Read(&c);
            fputc((int)c, stdout);
        }
        fflush(stdout);
    }
    CATCH(LostConnectionException)
    {
        debug1 << "Lost connection in ViewerConnectionPrinter::HandleRead" << endl;
    }
    CATCHALL
    {
        ; // nothing
    }
    ENDTRY
}

// ****************************************************************************
//  Method:  ViewerServerManager::ForceSSHTunnelingForAllConnections
//
//  Purpose:
//    Force SSH tunnelling for all connections.  This allows a more
//    convenient way to initiate SSH tunneling on the command line, as
//    this will override values in the host profiles.
//
//  Arguments:
//    
//
//  Programmer:  Jeremy Meredith
//  Creation:    December  3, 2008
//
// ****************************************************************************

void 
ViewerServerManager::ForceSSHTunnelingForAllConnections()
{
    sshTunnelingForcedOn = true;
}
