/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                      avtOpenGL3DTextureVolumeRenderer.C                   //
// ************************************************************************* //

#include "avtOpenGL3DTextureVolumeRenderer.h"

#include <vtkDataArray.h>
#include <vtkDataSet.h>
#include <vtkRectilinearGrid.h>
#include <vtkCamera.h>
#include <vtkMath.h>
#include <vtkMatrix4x4.h>
#include <vtkRenderer.h>
#include <vtkRenderWindow.h>
#include <BoundingBoxContourer.h>
#include <VolumeAttributes.h>
#include <avtViewInfo.h>
#include <avtCallback.h>
#include <LightList.h>
#include <DebugStream.h>

#include <float.h>

#ifndef VTK_IMPLEMENT_MESA_CXX
  // Include GLEW.
  #include <visit-config.h>
  #include <avtGLEWInitializer.h>

  #if defined(__APPLE__) && (defined(VTK_USE_CARBON) || defined(VTK_USE_COCOA))
    #include <OpenGL/gl.h>
  #else
    #if defined(_WIN32)
       #include <windows.h>
       // On Windows, we have to access glTexImage3D as an OpenGL extension.
       // In case texture3D extension is NOT available.
       static PFNGLTEXIMAGE3DEXTPROC glTexImage3D_ptr = 0;
    #endif
    #include <GL/gl.h>
  #endif
#endif

#ifndef MAX
#define MAX(a,b) ((a) > (b) ? (a) : (b))
#endif
#ifndef MIN
#define MIN(a,b) ((a) < (b) ? (a) : (b))
#endif

// ****************************************************************************
//  Method: avtOpenGL3DTextureVolumeRenderer::avtOpenGL3DTextureVolumeRenderer
//
//  Purpose:
//    Initialize the texture memory pointer and the OpenGL texture ID.
//
//  Programmer:  Jeremy Meredith
//  Creation:    September 30, 2003
//
//  Modifications:
//    Brad Whitlock, Fri Aug 20 16:52:18 PST 2004
//    Added support for looking up glTexImage3D extension on Windows.
//
//    Brad Whitlock, Fri Sep 15 12:50:14 PDT 2006
//    I removed the code to get the glTexImage3D extension and initialized 
//    the GLEW library.
//
//    Tom Fogal, Sat Jul 25 19:45:26 MDT 2009
//    Use new GLEW initialization wrapper.
//
// ****************************************************************************

avtOpenGL3DTextureVolumeRenderer::avtOpenGL3DTextureVolumeRenderer()
{
    volumetex = NULL;
    volumetexId = 0;

#ifndef VTK_IMPLEMENT_MESA_CXX
    avt::glew::initialize();
#endif
}


// ****************************************************************************
//  Method: avtOpenGL3DTextureVolumeRenderer::~avtOpenGL3DTextureVolumeRenderer
//
//  Purpose:
//    Destructor.  Free volumetex.
//
//  Programmer:  Jeremy Meredith
//  Creation:    October  1, 2003
//
//    Thomas R. Treadway, Tue Feb  6 17:04:03 PST 2007
//    The gcc-4.x compiler no longer just warns about automatic type conversion.
//
// ****************************************************************************
avtOpenGL3DTextureVolumeRenderer::~avtOpenGL3DTextureVolumeRenderer()
{
    delete[] volumetex;
    volumetex = NULL;

    // Assumes context is current!  This is the job of the container class.
    if (volumetexId != 0)
    {
        glDeleteTextures(1, (GLuint*)&volumetexId);
        volumetexId = 0;
    }
}


// ****************************************************************************
//  Method:  avtOpenGL3DTextureVolumeRenderer::Render
//
//  Purpose:
//    Render one image using a 3D texture and view-aligned slice planes.
//
//  Arguments:
//    grid      : the data set to render
//    data,opac : the color/opacity variables
//    view      : the viewing information
//    atts      : the current volume plot attributes
//    vmin/max/size : the min/max/range of the color variable
//    omin/max/size : the min/max/range of the opacity variable
//    gx/gy/gz      : the gradient of the opacity variable
//    gm            : the gradient magnitude, un-normalized
//    gm_max        : the max gradient magnitude over the data set
//
//  Programmer:  Jeremy Meredith
//  Creation:    October  1, 2003
//
//  Modifications:
//    Jeremy Meredith, Fri Oct 10 12:39:38 PDT 2003
//    Added check for the (new) HAVE_GL_TEX_IMAGE_3D flag.
//    Also, unconditionally enable it if avtOpenGL3DTextureVolumeRenderer
//    has been redefined, because this means we're doing Mesa, and Mesa
//    always supports 3D texturing.
//
//    Jeremy Meredith, Fri Oct 10 16:29:31 PDT 2003
//    Made it work with Mesa again.
//
//    Hank Childs, Tue May 11 15:24:45 PDT 2004
//    Turn off blending so transparent surfaces can work afterwards.
//
//    Eric Brugger, Tue Jul 27 11:52:38 PDT 2004
//    Add several casts and change a double constant to a float constant
//    to fix compile errors.
//
//    Brad Whitlock, Thu Aug 12 14:38:43 PST 2004
//    I removed the ifdef that prevented the code from building on Windows
//    and moved it into visit-config.h. I also added Windows-specific code.
//
//    Brad Whitlock, Fri Sep 15 12:52:53 PDT 2006
//    I made it use GLEW for getting the texture3D function.
//
//    Kathleen Bonnell, Wed Jan  3 15:51:59 PST 2007 
//    I made it revert (for Windows) to glTexImage3D_ptr if the
//    texture3D extension is not available.
//
//    Hank Childs, Tue Feb  6 15:41:58 PST 2007
//    Give an error that the user can see if 3D texturing is not available.
//
//    Thomas R. Treadway, Tue Feb  6 17:04:03 PST 2007
//    The gcc-4.x compiler no longer just warns about automatic type conversion.
//
//    Gunther H. Weber, Mon May 21 13:46:15 PDT 2007
//    Fixed 3D texture detection problem for Mac OS by checking for
//    GL_VERSION_1_2 before using GLEW.
//
//    Brad Whitlock, Thu Jan 10 14:47:34 PST 2008
//    Added reducedDetail argument.
//
//    Brad Whitlock, Wed Apr 22 12:06:12 PDT 2009
//    I changed the interface.
//
//    Jeremy Meredith, Tue Jan  5 15:49:43 EST 2010
//    Added ability to reduce amount of lighting for low-gradient-mag areas.
//    This was already enabled, to some degree, by default, but it's now both
//    optional and configurable.
//
// ****************************************************************************

void
avtOpenGL3DTextureVolumeRenderer::Render(
    const avtVolumeRendererImplementation::RenderProperties &props,
    const avtVolumeRendererImplementation::VolumeData &volume)
{
    static bool haveIssuedWarning = false;
#ifndef VTK_IMPLEMENT_MESA_CXX
    // OpenGL mode
#if !defined(GL_VERSION_1_2)
#ifdef HAVE_LIBGLEW
    // If we have GLEW then we're in the OpenGL version and we should
    // be sure that the extension exists on the display.
    if(glew_initialized && !GLEW_EXT_texture3D)
    {
#ifdef _WIN32
        // On Windows, glTexImage3D is an OpenGL extension. We have to look
        // up the function pointer.
        if(glTexImage3D_ptr == 0)
            glTexImage3D_ptr = (PFNGLTEXIMAGE3DEXTPROC)wglGetProcAddress("glTexImage3D");
        if(glTexImage3D_ptr == 0)
        {
            debug1 << "The glTexImage3D function was not located." << endl;
            if (!haveIssuedWarning)
            {
                avtCallback::IssueWarning("3D textured volume rendering is not "
                           "available, because the OpenGL functions cannot be "
                           "located.");
                haveIssuedWarning = true;
            }
            return;
        }
#else
        debug1 << "avtOpenGL3DTextureVolumeRenderer::Render: "
                  "returning because there is no texture3D extension."
               << endl;
        if (!haveIssuedWarning)
        {
            avtCallback::IssueWarning("3D textured volume rendering is not "
                       "available, because the 3D texturing extensions can "
                       "not be located.");
            haveIssuedWarning = true;
        }
        return;
#endif
    }
#endif
#endif
#endif

    // Get the transfer function
    int ncolors=256;
    int nopacities=256;
    unsigned char rgba[256*4];
    props.atts.GetTransferFunction(rgba);
    
    // Get the dimensions
    int dims[3];
    volume.grid->GetDimensions(dims);

    int nx=dims[0];
    int ny=dims[1];
    int nz=dims[2];

    // Find the smallest power of two in each dimension 
    // that will accomodate this data set
    int newnx = MAX(int(pow(2.0,1+int(log(double(nx-1))/log(2.0)))),1);
    int newny = MAX(int(pow(2.0,1+int(log(double(ny-1))/log(2.0)))),1);
    int newnz = MAX(int(pow(2.0,1+int(log(double(nz-1))/log(2.0)))),1);

    // Get the new lighting parameters
    LightList lights = avtCallback::GetCurrentLightList();

    // Determine if we need to invalidate the old texture
    if (volumetex && (props.atts != oldAtts || lights != oldLights))
    {
        glDeleteTextures(1, (GLuint*)&volumetexId);
        volumetexId = 0;
        delete[] volumetex;
        volumetex = NULL;
    }
    oldAtts = props.atts;
    oldLights = lights;


    //
    // Extract the lighting information from the actual light list
    //

    float light[4] = {0,0,1, 0};
    float ambient = 0.0;

    // Find an ambient light
    int i;
    for (i=0; i<lights.NumLights(); i++)
    {
        const LightAttributes &l = lights.GetLight(i);
        if (l.GetEnabledFlag() && l.GetType()==LightAttributes::Ambient)
        {
            // Take it's overall brightness
            double rgba[4];
            l.GetColor().GetRgba(rgba);
            ambient = l.GetBrightness() * (rgba[0]+rgba[1]+rgba[2])/3.;
            break;
        }
    }

    // Find a directional (object or camera) light
    for (i=0; i<lights.NumLights(); i++)
    {
        const LightAttributes &l = lights.GetLight(i);
        if (l.GetEnabledFlag() && l.GetType()!=LightAttributes::Ambient)
        {
            // Take it's direction
            const double *dir = l.GetDirection();
            light[0] = dir[0];
            light[1] = dir[1];
            light[2] = dir[2];
            break;
        }
    }

    // If we want to transform the light so it is attached to the camera:
    //I->MultiplyPoint(light, light);


    //
    // Create the 3D texture if we need to
    //

    if (!volumetex)
    {
        int nels=newnx*newny*newnz;
        volumetex = new unsigned char[4*nels];
        int outindex = -1;
        for (int k=0; k<newnz; k++)
        {
            for (int j=0; j<newny; j++)
            {
                for (int i=0; i<newnx; i++)
                {
                    int ijk[3]={i,j,k};
                    outindex++;
                    volumetex[outindex*4 + 0] = 0;
                    volumetex[outindex*4 + 1] = 0;
                    volumetex[outindex*4 + 2] = 0;
                    volumetex[outindex*4 + 3] = 0;

                    if (i>=nx || j>=ny || k>=nz)
                    {
                        // out of bounds data
                        continue;
                    }

                    int index = volume.grid->ComputePointId(ijk);

                    float  v = volume.data.data->GetTuple1(index);
                    float  o = volume.opacity.data->GetTuple1(index);

                    // drop empty ones
                    if (v < -1e+37)
                        continue;

                    // normalize the value
                    v = (v < volume.data.min ? volume.data.min : v);
                    v = (v > volume.data.max ? volume.data.max : v);
                    v = (v-volume.data.min)/volume.data.size;
                    o = (o < volume.opacity.min ? volume.opacity.min : o);
                    o = (o > volume.opacity.max ? volume.opacity.max : o);
                    o = (o-volume.opacity.min)/volume.opacity.size;

                    // opactity map
                    float opacity;
                    opacity = float(rgba[int(o*(nopacities-1))*4 + 3])*
                        props.atts.GetOpacityAttenuation()/256.;
                    opacity = MAX(0,MIN(1,opacity));

                    // drop transparent splats 
                    //if (opacity < .0001)
                    //    continue;

                    // do shading
                    float brightness;
                    const bool shading = props.atts.GetLightingFlag();
                    if (shading)
                    {
                        // Get the gradient
                        float gi = volume.gx[index];
                        float gj = volume.gy[index];
                        float gk = volume.gz[index];

                        // Amount of shading should be somewhat proportional
                        // to the magnitude of the gradient
                        float gm = 1.0;
                        if (props.atts.GetLowGradientLightingReduction() !=
                                                      VolumeAttributes::Off)
                        {
                           double lp = 1.0;
                           switch (props.atts.GetLowGradientLightingReduction())
                           {
                             case VolumeAttributes::Lowest:  lp = 1./16.;break;
                             case VolumeAttributes::Lower:   lp = 1./8.; break;
                             case VolumeAttributes::Low:     lp = 1./4.; break;
                             case VolumeAttributes::Medium:  lp = 1./2.; break;
                             case VolumeAttributes::High:    lp = 1.;    break;
                             case VolumeAttributes::Higher:  lp = 2.;    break;
                             case VolumeAttributes::Highest: lp = 4.;    break;
                             default: break;
                           }
                           if (props.atts.GetLowGradientLightingClampFlag())
                           {
                               gm = volume.gm[index] /
                                 props.atts.GetLowGradientLightingClampValue();
                           }
                           else
                           {
                               gm = volume.gmn[index];
                           }
                           gm = pow((double)gm, lp);
                        }
                        if (gm < 0)
                            gm = 0;
                        if (gm > 1)
                            gm = 1;

                        // Get the base lit brightness based on the 
                        // light direction and the gradient
                        float grad[3] = {gi,gj,gk};
                        float lightdir[3] = {light[0],light[1],light[2]};
                        brightness = vtkMath::Dot(grad,lightdir);
                        if (brightness<0) brightness *= -1;

                        // Modulate by the gradient magnitude
                        brightness = (1.0 - gm)*1.0 + gm*brightness;

                        // Modulate by the amount of ambient lighting
                        brightness = (1.0 - ambient)*brightness + ambient;
                    }
                    else
                    {
                        // No shading ata ll
                        brightness=1;
                    }
                
                    // Determine the actual color and opacity now
                    int colorindex = int(ncolors * v);
                    if (colorindex < 0)
                        colorindex = 0;
                    if (colorindex >= ncolors)
                        colorindex =  ncolors-1;
                    int colorindex4 = colorindex*4;
                    float scaledbrightness = brightness/255.;
                    float r,g,b;
                    r = float(rgba[colorindex4 + 0])*scaledbrightness;
                    g = float(rgba[colorindex4 + 1])*scaledbrightness;
                    b = float(rgba[colorindex4 + 2])*scaledbrightness;

                    volumetex[outindex*4 + 0] = (unsigned char)(r*255);
                    volumetex[outindex*4 + 1] = (unsigned char)(g*255);
                    volumetex[outindex*4 + 2] = (unsigned char)(b*255);
                    volumetex[outindex*4 + 3] = (unsigned char)(opacity*255);
                }
            }
        }

        // Create the texture
        //glPixelStorei(GL_UNPACK_ALIGNMENT,1);
        glGenTextures(1, (GLuint*)&volumetexId);
        glBindTexture(GL_TEXTURE_3D, volumetexId);

#ifndef VTK_IMPLEMENT_MESA_CXX
        // OpenGL mode
#ifdef GL_VERSION_1_2
        // OpenGL supports glTexImage3D.
        glTexImage3D(GL_TEXTURE_3D, 0, GL_RGBA, newnx, newny, newnz,
                     0, GL_RGBA, GL_UNSIGNED_BYTE, volumetex);
#elif defined(HAVE_LIBGLEW)
        if (GLEW_EXT_texture3D)
        {
            // glTexImage3D via GLEW.
            glTexImage3DEXT(GL_TEXTURE_3D, 0, GL_RGBA, newnx, newny, newnz,
                            0, GL_RGBA, GL_UNSIGNED_BYTE, volumetex);
        }
#ifdef _WIN32
        else if (glTexImage3D_ptr != 0)
        {
            glTexImage3D_ptr(GL_TEXTURE_3D, 0, GL_RGBA, newnx, newny, newnz,
                             0, GL_RGBA, GL_UNSIGNED_BYTE, volumetex);
        }
#endif
#endif
#else
        // Mesa mode
        glTexImage3D(GL_TEXTURE_3D, 0, GL_RGBA, newnx, newny, newnz,
                     0, GL_RGBA, GL_UNSIGNED_BYTE, volumetex);
#endif
    }

    //
    // We have the texture; now draw with it
    //

    // Set up OpenGL parameters
    glDisable(GL_LIGHTING);
    glBindTexture(GL_TEXTURE_3D, volumetexId);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glEnable(GL_TEXTURE_3D);
    //glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    bool alreadyBlending = glIsEnabled(GL_BLEND);
    if (!alreadyBlending)
        glEnable(GL_BLEND);
    glDepthMask(false);

    // Set up camera parameters
    vtkCamera *camera = vtkCamera::New();
    props.view.SetCameraFromView(camera);
    vtkMatrix4x4 *cameraMatrix = camera->GetViewTransformMatrix();

    //
    // Contour the bounding box at a user specified number of depths
    //

    BoundingBoxContourer bbox;

    // Extract the depth values at the corners of our bounding box
    int mapStructuredToUnstructured[8] = {0, 1, 3, 2, 4, 5, 7, 6};
    int bbox_index = 0;
    float minz =  FLT_MAX;
    float maxz = -FLT_MAX;
    for (int k=0; k<nz; k+=nz-1)
    {
        for (int j=0; j<ny; j+=ny-1)
        {
            for (int i=0; i<nx; i+=nx-1)
            {
                int ijk[] = {i,j,k};
                double worldpt[4] = {0,0,0,1};
                volume.grid->GetPoint(volume.grid->ComputePointId(ijk), worldpt);

                // Get the world space coordinates

                // The contourer expects an unstructured hex
                int pt_index = mapStructuredToUnstructured[bbox_index];

                bbox.x[pt_index] = worldpt[0];
                bbox.y[pt_index] = worldpt[1];
                bbox.z[pt_index] = worldpt[2];

                // Get the texture coordinates
                bbox.r[pt_index] = float(i) / float(newnx + 0) + (0.5 / float(newnx));
                bbox.s[pt_index] = float(j) / float(newny + 0) + (0.5 / float(newny));
                bbox.t[pt_index] = float(k) / float(newnz + 0) + (0.5 / float(newnz));

                // Get the camera space coordinates
                double viewpt[4];
                cameraMatrix->MultiplyPoint(worldpt, viewpt);
                float dist = viewpt[2];

                bbox.v[pt_index] = dist;

                if (dist < minz)
                    minz = dist;
                if (dist > maxz)
                    maxz = dist;

                bbox_index++;

                if (nx == 1)
                    break;
            }

            if (ny == 1)
                break;
        }

        if (nz == 1)
            break;
    }

    // Determine the depth values we need
    int pt0 = mapStructuredToUnstructured[0];
    int ptn = mapStructuredToUnstructured[7];
    float dx = bbox.x[pt0] - bbox.x[ptn];
    float dy = bbox.y[pt0] - bbox.y[ptn];
    float dz = bbox.z[pt0] - bbox.z[ptn];
    float dist = sqrt(dx*dx + dy*dy + dz*dz);

    // Do the actual contouring 
    glBegin(GL_TRIANGLES);
    glColor4f(1.,1.,1.,1.);
    int ns = props.atts.GetNum3DSlices();
    if (ns < 1)
        ns = 1;
    if (ns > 500)
        ns = 500;

    float tr[15], ts[15], tt[15];
    float vx[15], vy[15], vz[15];
    int   ntriangles;
    for (int z=0; z<ns; z++)
    {
        float value = (float(z)+0.5)/float(ns);

        bbox.ContourTriangles(minz + dist*value,
                              ntriangles, tr, ts, tt, vx, vy, vz);

        for (int t=0; t<ntriangles*3; t++)
        {
            glTexCoord3f(tr[t], ts[t], tt[t]);
            glVertex3f(vx[t], vy[t], vz[t]);
        }
    }
    glEnd();

    // Set some GL parameters back to their expected values and free memory
    glDepthMask(true);
    glDisable(GL_TEXTURE_3D);
    if (!alreadyBlending)
        glDisable(GL_BLEND);
    glEnable(GL_LIGHTING);
    volume.opacity.data->Delete();
    volume.data.data->Delete();
    camera->Delete();
}


#if 0
// ****************************************************************************
//  Method:  avtOpenGL3DTextureVolumeRenderer::Render
//
//  Purpose:
//    Render one image using axis-aligned slices.  This code is
//    disabled for now, but kept as a reference if we want to include
//    a 2D texturing volume renderer
//
//  Arguments:
//    grid      : the data set to render
//    data,opac : the color/opacity variables
//    view      : the viewing information
//    atts      : the current volume plot attributes
//    vmin/max/size : the min/max/range of the color variable
//    omin/max/size : the min/max/range of the opacity variable
//    gx/gy/gz      : the gradient of the opacity variable
//    gmn           : the gradient magnitude, normalized to the max grad mag
//
//  Programmer:  Jeremy Meredith
//  Creation:    October  1, 2003
//
//  Modifications:
//
//    Hank Childs, Tue May 11 15:24:45 PDT 2004
//    Turn off blending so transparent surfaces can work afterwards.
//
//    Thomas R. Treadway, Tue Feb  6 17:04:03 PST 2007
//    The gcc-4.x compiler no longer just warns about automatic type conversion.
//
// ****************************************************************************

void
avtOpenGL3DTextureVolumeRenderer::Render(vtkRectilinearGrid *grid,
                                         vtkDataArray *data,
                                         vtkDataArray *opac,
                                         const avtViewInfo &view,
                                         const VolumeAttributes &atts,
                                         float vmin, float vmax, float vsize,
                                         float omin, float omax, float osize,
                                         float *gx, float *gy, float *gz,
                                         float *gmn)
{
    int dims[3];
    grid->GetDimensions(dims);

    // get attribute parameters
    int ncolors=256;
    int nopacities=256;
    unsigned char rgba[256*4];
    atts.GetTransferFunction(rgba);
    
    int nx=dims[0];
    int ny=dims[1];
    int nz=dims[2];

    int newnx = MAX(int(pow(2,1+int(log(nx-1)/log(2)))),1);
    int newny = MAX(int(pow(2,1+int(log(ny-1)/log(2)))),1);
    int newnz = MAX(int(pow(2,1+int(log(nz-1)/log(2)))),1);

    LightList lights = avtCallback::GetCurrentLightList();

    // Create the 3D texture
    if (volumetex && 
        (atts != oldAtts ||
         lights != oldLights))
    {
        glDeleteTextures(1, (GLuint*)&volumetexId);
        volumetexId = 0;
        delete[] volumetex;
        volumetex = NULL;
    }
    oldAtts = atts;
    oldLights = lights;

    float light[4] = {0,0,1, 0};
    float ambient = 0.0;

    // Find an ambient light
    for (int i=0; i<lights.NumLights(); i++)
    {
        const LightAttributes &l = lights.GetLight(i);
        if (l.GetEnabledFlag() && l.GetType()==LightAttributes::Ambient)
        {
            double rgba[4];
            l.GetColor().GetRgba(rgba);
            ambient = l.GetBrightness() * (rgba[0]+rgba[1]+rgba[2])/3.;
            break;
        }
    }

    // Find a directional (object or camera) light
    for (int i=0; i<lights.NumLights(); i++)
    {
        const LightAttributes &l = lights.GetLight(i);
        if (l.GetEnabledFlag() && l.GetType()!=LightAttributes::Ambient)
        {
            const double *dir = l.GetDirection();
            light[0] = dir[0];
            light[1] = dir[1];
            light[2] = dir[2];
            break;
        }
    }

    // If we want to transform the light so it is attached to the camera:
    //I->MultiplyPoint(light, light);

    if (!volumetex)
    {
        int nels=newnx*newny*newnz;
        volumetex = new unsigned char[4*nels];
        int outindex = -1;
        for (int k=0; k<newnz; k++)
        {
            for (int j=0; j<newny; j++)
            {
                for (int i=0; i<newnx; i++)
                {
                    int ijk[3]={i,j,k};
                    outindex++;
                    volumetex[outindex*4 + 0] = 0;
                    volumetex[outindex*4 + 1] = 0;
                    volumetex[outindex*4 + 2] = 0;
                    volumetex[outindex*4 + 3] = 0;

                    if (i>=nx || j>=ny || k>=nz)
                    {
                        // out of bounds
                        continue;
                    }

                    int index = grid->ComputePointId(ijk);

                    float  v = data->GetTuple1(index);
                    float  o = opac->GetTuple1(index);

                    // drop empty ones
                    if (v < -1e+37)
                        continue;

                    // normalize the value
                    v = (v < vmin ? vmin : v);
                    v = (v > vmax ? vmax : v);
                    v = (v-vmin)/vsize;
                    o = (o < omin ? omin : o);
                    o = (o > omax ? omax : o);
                    o = (o-omin)/osize;

                    // opactity map:
                    float opacity;
                    opacity = float(rgba[int(o*(nopacities-1))*4 + 3])*
                        atts.GetOpacityAttenuation()/256.;
                    opacity = MAX(0,MIN(1,opacity));

                    // drop transparent splats 
                    //if (opacity < .0001)
                    //    continue;

                    // do shading
                    float brightness;
                    const bool shading = atts.GetLightingFlag();
                    if (shading)
                    {
                        float gi = gx[index];
                        float gj = gy[index];
                        float gk = gz[index];

                        // Amount of shading should be somewhat proportional
                        // to the magnitude of the gradient
                        float gm = pow(gmn[index], 0.25);

                        // Get the base lit brightness 
                        float grad[3] = {gi,gj,gk};
                        float lightdir[3] = {light[0],light[1],light[2]};
                        brightness = vtkMath::Dot(grad,lightdir);
                        if (brightness<0) brightness *= -1;

                        // Modulate by the gradient magnitude
                        brightness = (1.0 - gm)*1.0 + gm*brightness;

                        // Modulate by the amount of ambient lighting
                        brightness = (1.0 - ambient)*brightness + ambient;
                    }
                    else
                    {
                        brightness=1;
                    }
                
                    float r,g,b;

                    int colorindex = int(ncolors * v);
                    if (colorindex < 0)
                        colorindex = 0;
                    if (colorindex >= ncolors)
                        colorindex =  ncolors-1;
                    int colorindex4 = colorindex*4;
                    float scaledbrightness = brightness/255.;
                    r = float(rgba[colorindex4 + 0])*scaledbrightness;
                    g = float(rgba[colorindex4 + 1])*scaledbrightness;
                    b = float(rgba[colorindex4 + 2])*scaledbrightness;

                    volumetex[outindex*4 + 0] = (unsigned char)(r*255);
                    volumetex[outindex*4 + 1] = (unsigned char)(g*255);
                    volumetex[outindex*4 + 2] = (unsigned char)(b*255);
                    volumetex[outindex*4 + 3] = (unsigned char)(opacity*255);
                }
            }
        }

        // Create the texture
        //glPixelStorei(GL_UNPACK_ALIGNMENT,1);
        glGenTextures(1, (GLuint*)&volumetexId);
        glBindTexture(GL_TEXTURE_3D, volumetexId);
        glTexImage3D(GL_TEXTURE_3D, 0, GL_RGBA, newnx, newny, newnz,
                     0, GL_RGBA, GL_UNSIGNED_BYTE, volumetex);
    }

    // Set up OpenGL parameters
    glDisable(GL_LIGHTING);
    glBindTexture(GL_TEXTURE_3D, volumetexId);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_WRAP_R, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_WRAP_S, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_WRAP_T, GL_CLAMP);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_3D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glEnable(GL_TEXTURE_3D);
    //glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    bool alreadyBlending = glIsEnabled(GL_BLEND);
    if (!alreadyBlending)
        glEnable(GL_BLEND);

    // set up parameters
    vtkCamera *camera = vtkCamera::New();
    view.SetCameraFromView(camera);
    vtkMatrix4x4 *cameraMatrix = camera->GetViewTransformMatrix();
    vtkMatrix4x4 *I = vtkMatrix4x4::New();
    I->DeepCopy(cameraMatrix);
    I->Invert();

    // Set traversal order across and within each dimension
    int  svx,svy,svz;
    int  i,j,k;
    int  imin,imax;
    int  jmin,jmax;
    int  kmin,kmax;
    int *c1,*c2,*c3;
    int *c1min,*c2min,*c3min;
    int *c1max,*c2max,*c3max;
    int *c1s,*c2s,*c3s;

    float vx[4] = {-1,0,0, 0};
    float vy[4] = {0,-1,0, 0};
    float vz[4] = {0,0,-1, 0};

    cameraMatrix->MultiplyPoint(vx,vx);
    cameraMatrix->MultiplyPoint(vy,vy);
    cameraMatrix->MultiplyPoint(vz,vz);

    float view_dir[4] = {0,0,1, 0};

    float vxm = vtkMath::Dot(vx,view_dir);
    float vym = vtkMath::Dot(vy,view_dir);
    float vzm = vtkMath::Dot(vz,view_dir);
    float avx = fabs(vxm);
    float avy = fabs(vym);
    float avz = fabs(vzm);
    svx = (vxm>0 ? -1 : 1);
    svy = (vym>0 ? -1 : 1);
    svz = (vzm>0 ? -1 : 1);

    imin = (svx>0) ? 0 : dims[0]-1;
    jmin = (svy>0) ? 0 : dims[1]-1;
    kmin = (svz>0) ? 0 : dims[2]-1;
    imax = (svx<0) ? -1 : dims[0];
    jmax = (svy<0) ? -1 : dims[1];
    kmax = (svz<0) ? -1 : dims[2];

    float tci, tcj, tck;
    float *tc1, *tc2, *tc3;
    float tcimax = float(nx)/float(newnx);
    float tcjmax = float(ny)/float(newny);
    float tckmax = float(nz)/float(newnz);
    float *tc1max, *tc2max, *tc3max;

    if (avx>=avy && avx>=avz)
    {
        c1    = &i;
        c1min = &imin;
        c1max = &imax;
        c1s   = &svx;
        tc1   = &tci;
        tc1max= &tcimax;
        if (avy>avz)
        {
            c2    = &j;
            c2min = &jmin;
            c2max = &jmax;
            c2s   = &svy;
            tc2   = &tcj;
            tc2max= &tcjmax;
            c3    = &k;
            c3min = &kmin;
            c3max = &kmax;
            c3s   = &svz;
            tc3   = &tck;
            tc3max= &tckmax;
        }
        else
        {
            c3    = &j;
            c3min = &jmin;
            c3max = &jmax;
            c3s   = &svy;
            tc3   = &tcj;
            tc3max= &tcjmax;
            c2    = &k;
            c2min = &kmin;
            c2max = &kmax;
            c2s   = &svz;
            tc2   = &tck;
            tc2max= &tckmax;
        }
    }
    else if (avy>=avx && avy>=avz)
    {
        c1    = &j;
        c1min = &jmin;
        c1max = &jmax;
        c1s   = &svy;
        tc1   = &tcj;
        tc1max= &tcjmax;
        if (avx>avz)
        {
            c2    = &i;
            c2min = &imin;
            c2max = &imax;
            c2s   = &svx;
            tc2   = &tci;
            tc2max= &tcimax;
            c3    = &k;
            c3min = &kmin;
            c3max = &kmax;
            c3s   = &svz;
            tc3   = &tck;
            tc3max= &tckmax;
        }
        else
        {
            c3    = &i;
            c3min = &imin;
            c3max = &imax;
            c3s   = &svx;
            tc3   = &tci;
            tc3max= &tcimax;
            c2    = &k;
            c2min = &kmin;
            c2max = &kmax;
            c2s   = &svz;
            tc2   = &tck;
            tc2max= &tckmax;
        }
    }
    else if (avz>=avx && avz>=avy)
    {
        c1    = &k;
        c1min = &kmin;
        c1max = &kmax;
        c1s   = &svz;
        tc1   = &tck;
        tc1max= &tckmax;
        if (avx>avy)
        {
            c2    = &i;
            c2min = &imin;
            c2max = &imax;
            c2s   = &svx;
            tc2   = &tci;
            tc2max= &tcimax;
            c3    = &j;
            c3min = &jmin;
            c3max = &jmax;
            c3s   = &svy;
            tc3   = &tcj;
            tc3max= &tcjmax;
        }
        else
        {
            c3    = &i;
            c3min = &imin;
            c3max = &imax;
            c3s   = &svx;
            tc3   = &tci;
            tc3max= &tcimax;
            c2    = &j;
            c2min = &jmin;
            c2max = &jmax;
            c2s   = &svy;
            tc2   = &tcj;
            tc2max= &tcjmax;
        }
    }

    glDepthMask(false);

    glBegin(GL_QUADS);
    glColor4f(1.,1.,1.,1.);
    for (*c1 = *c1min; *c1 != *c1max; *c1 += *c1s)
    {
        *tc1 = *tc1max * float(*c1 - *c1min) / float(*c1max - *c1min);
        if (*c1s < 0)
            *tc1 = *tc1max  - *tc1;

        *c2 = *c2min;
        *c3 = *c3min;
        int ijk1[]={i,j,k};
        float p1[3];
        grid->GetPoint(grid->ComputePointId(ijk1),p1);

        *c2 = *c2max-*c2s;
        *c3 = *c3min;
        int ijk2[]={i,j,k};
        float p2[3];
        grid->GetPoint(grid->ComputePointId(ijk2),p2);

        *c2 = *c2max-*c2s;
        *c3 = *c3max-*c3s;
        int ijk3[]={i,j,k};
        float p3[3];
        grid->GetPoint(grid->ComputePointId(ijk3),p3);

        *c2 = *c2min;
        *c3 = *c3max-*c3s;
        int ijk4[]={i,j,k};
        float p4[3];
        grid->GetPoint(grid->ComputePointId(ijk4),p4);

        *tc2 = (*c2s > 0) ? 0 : *tc2max; 
        *tc3 = (*c3s > 0) ? 0 : *tc3max; 
        glTexCoord3f(tci, tcj, tck);
        glVertex3fv(p1);

        *tc2 = (*c2s > 0) ? *tc2max : 0; 
        *tc3 = (*c3s > 0) ? 0 : *tc3max; 
        glTexCoord3f(tci, tcj, tck);
        glVertex3fv(p2);

        *tc2 = (*c2s > 0) ? *tc2max : 0; 
        *tc3 = (*c3s > 0) ? *tc3max : 0; 
        glTexCoord3f(tci, tcj, tck);
        glVertex3fv(p3);

        *tc2 = (*c2s > 0) ? 0 : *tc2max; 
        *tc3 = (*c3s > 0) ? *tc3max : 0; 
        glTexCoord3f(tci, tcj, tck);
        glVertex3fv(p4);
    }

    glEnd();

    glDepthMask(true);

    glDisable(GL_TEXTURE_3D);
    if (!alreadyBlending)
        glDisable(GL_BLEND);
    glEnable(GL_LIGHTING);
    opac->Delete();
    data->Delete();
    camera->Delete();
    I->Delete();
}
#endif

