/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                             avtPoincarePlot.C                             //
// ************************************************************************* //

#include <avtPoincarePlot.h>

#include <avtPoincareFilter.h>
#include <avtLookupTable.h>
#include <avtVariableLegend.h>
#include <avtVariableMapper.h>
#include <avtPoincareFilter.h>
#include <InvalidLimitsException.h>
#include <vtkPlane.h>

// ****************************************************************************
//  Method: avtPoincarePlot constructor
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
//  Modifications:
//
//   Dave Pugmire, Wed May 27 15:03:42 EDT 2009
//   Set color LUT.
//
// ****************************************************************************

avtPoincarePlot::avtPoincarePlot()
{
#ifdef ENGINE
    poincareFilter = new avtPoincareFilter;
#endif
    avtLUT = new avtLookupTable; 
    varMapper = new avtVariableMapper;
    varMapper->SetLookupTable(avtLUT->GetLookupTable());

    varLegend = new avtVariableLegend;
    varLegend->SetTitle("Poincare");
    varLegend->SetLookupTable(avtLUT->GetLookupTable());
    //
    // This is to allow the legend to be reference counted so the behavior can
    // still access it after the plot is deleted.  The legend cannot be
    // reference counted all of the time since we need to know that it is a 
    // VariableLegend.
    //
    varLegendRefPtr = varLegend;
}


// ****************************************************************************
//  Method: avtPoincarePlot destructor
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
// ****************************************************************************

avtPoincarePlot::~avtPoincarePlot()
{
#ifdef ENGINE
    if (poincareFilter != NULL)
    {
        delete poincareFilter;
        poincareFilter = NULL;
    }
#endif

    if (varMapper != NULL)
    {
        delete varMapper;
        varMapper = NULL;
    }
    if (avtLUT != NULL)
    {
        delete avtLUT;
        avtLUT = NULL;
    }

    //
    // Do not delete the varLegend since it is being held by varLegendRefPtr.
    //

}


// ****************************************************************************
//  Method:  avtPoincarePlot::Create
//
//  Purpose:
//    Call the constructor.
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
// ****************************************************************************

avtPlot*
avtPoincarePlot::Create()
{
    return new avtPoincarePlot;
}


// ****************************************************************************
//  Method: avtPoincarePlot::GetMapper
//
//  Purpose:
//      Gets a mapper for this plot, it is actually a variable mapper.
//
//  Returns:    The variable mapper typed as its base class mapper.
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
// ****************************************************************************

avtMapper *
avtPoincarePlot::GetMapper(void)
{
    return varMapper;
}


// ****************************************************************************
//  Method: avtPoincarePlot::ApplyOperators
//
//  Purpose:
//      Applies the operators associated with a Poincare plot.  
//      The output from this method is a query-able object.
//
//  Arguments:
//      input   The input data object.
//
//  Returns:    The data object after the Poincare plot has been applied.
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
// ****************************************************************************

avtDataObject_p
avtPoincarePlot::ApplyOperators(avtDataObject_p input)
{
#ifdef ENGINE

    poincareFilter->SetInput(input);

    avtDataObject_p dob = poincareFilter->GetOutput();
    return dob;
#else
    return input;
#endif
}


// ****************************************************************************
//  Method: avtPoincarePlot::ApplyRenderingTransformation
//
//  Purpose:
//      Applies the rendering transformation associated with a Poincare plot.  
//
//  Arguments:
//      input   The input data object.
//
//  Returns:    The data object after the Poincare plot has been applied.
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
// ****************************************************************************

avtDataObject_p
avtPoincarePlot::ApplyRenderingTransformation(avtDataObject_p input)
{
    return input;
}


// ****************************************************************************
//  Method: avtPoincarePlot::CustomizeBehavior
//
//  Purpose:
//      Customizes the behavior as appropriate for a Poincare plot.  This includes
//      behavior like shifting towards or away from the screen.
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
// ****************************************************************************

void
avtPoincarePlot::CustomizeBehavior(void)
{
    SetLegendRanges();
    behavior->SetLegend(varLegendRefPtr);
    behavior->SetShiftFactor(0.3);
}


// ****************************************************************************
//  Method: avtPoincarePlot::CustomizeMapper
//
//  Purpose:
//      A hook from the base class that allows the plot to change its mapper
//      based on the dataset input. 
//
//  Arguments:
//      doi     The data object information.
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
// ****************************************************************************

void
avtPoincarePlot::CustomizeMapper(avtDataObjectInformation &doi)
{
}

// ****************************************************************************
//  Method: avtPoincarePlot::EnhanceSpecification
//
//  Purpose:
//      Modifies the contract to tell it we want the "colorVar" to be the 
//      primary variable for the pipeline.  If we don't do that, the primary
//      variable will be some vector variable and it will confuse our mapper.
//      The only reason that this works is that the poincare filter 
//      understands the colorVar trick and produces that variable.
//
//  Programmer: Dave Pugmire
//  Creation:   Fri Nov  7 13:22:17 EST 2008
//
// ****************************************************************************

avtContract_p
avtPoincarePlot::EnhanceSpecification(avtContract_p in_contract)
{
    avtDataRequest_p in_dr = in_contract->GetDataRequest();
    const char *var = in_dr->GetVariable();
    avtDataRequest_p out_dr = new avtDataRequest(in_dr, "colorVar");
    out_dr->AddSecondaryVariable(var);
    out_dr->SetOriginalVariable(var);
    avtContract_p out_contract = new avtContract(in_contract, out_dr);
    return out_contract;
}

// ****************************************************************************
//  Method: avtPoincarePlot::SetAtts
//
//  Purpose:
//      Sets the atts for the Poincare plot.
//
//  Arguments:
//      atts    The attributes for this Poincare plot.
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
//  Modifications:
//
//    Dave Pugmire, Wed Feb 25 09:52:11 EST 2009
//    Add terminate by steps, add AdamsBashforth solver, Allen Sanderson's new code.
//
//    Jeremy Meredith, Wed Apr  8 16:48:05 EDT 2009
//    Initial steps to unification with streamline attributes.
//
//    Dave Pugmire, Fri Apr 17 11:32:40 EDT 2009
//    Load LUT or load single color, depending on attrs.
//
//    Dave Pugmire, Tue Apr 28 09:26:06 EDT 2009
//    GUI reorganization.
//
//    Dave Pugmire, Wed May 27 15:03:42 EDT 2009
//    Call ChangesRequireRecalculation. Fix color table display.
//
//    Dave Pugmire, Tue Aug 11 10:33:05 EDT 2009
//    Add number of intersections termination criterion
//    
// ****************************************************************************

void
avtPoincarePlot::SetAtts(const AttributeGroup *a)
{
    const PoincareAttributes *newAtts = (const PoincareAttributes *)a;
    needsRecalculation =
        atts.ChangesRequireRecalculation(*(const PoincareAttributes*)newAtts);
    
    atts = *newAtts;

#ifdef ENGINE

    // Streamline specific attributes (avtStreamlineFilter).

    // Make the number of punctures 2x because the analysis uses only
    // the punctures in the same direction as the plane.
    poincareFilter->SetTermination(STREAMLINE_TERMINATE_INTERSECTIONS,
                                   2*atts.GetMinPunctures());

    poincareFilter->SetMaxPunctures(2*atts.GetMaxPunctures());
    
    vtkPlane *intPlane = vtkPlane::New();
    intPlane->SetOrigin( 0,0,0 ); 
    intPlane->SetNormal( 0,1,0 ); 
    poincareFilter->SetIntersectionObject(intPlane);    
    intPlane->Delete();
    
    poincareFilter->SetStreamlineDirection(0);

    switch (atts.GetSourceType())
    {
      case PoincareAttributes::SpecifiedPoint:
        poincareFilter->SetSourceType(STREAMLINE_SOURCE_POINT);
        poincareFilter->SetPointSource(atts.GetPointSource());
        break;

      case PoincareAttributes::SpecifiedLine:
        if( atts.GetPointDensity() > 1 )
        {
          poincareFilter->SetSourceType(STREAMLINE_SOURCE_LINE);
          poincareFilter->SetLineSource(atts.GetLineStart(), atts.GetLineEnd());
          poincareFilter->SetPointDensity(atts.GetPointDensity()-1);
        }
        else
        {
          double pt[3];

          pt[0] = (atts.GetLineStart()[0] + atts.GetLineEnd()[0]) / 2;
          pt[1] = (atts.GetLineStart()[1] + atts.GetLineEnd()[1]) / 2;
          pt[2] = (atts.GetLineStart()[2] + atts.GetLineEnd()[2]) / 2;

          poincareFilter->SetSourceType(STREAMLINE_SOURCE_POINT);
          poincareFilter->SetPointSource(pt);
        }

        break;
    }

    // Set the streamline attributes.
    poincareFilter->SetIntegrationType(atts.GetIntegrationType());

    poincareFilter->SetStreamlineAlgorithm(STREAMLINE_PARALLEL_STATIC_DOMAINS,
                                           10, 3, 1);
    poincareFilter->SetMaxStepLength(atts.GetMaxStepLength());
    poincareFilter->SetTolerances(atts.GetRelTol(),atts.GetAbsTol());


    poincareFilter->SetStreamlineAlgorithm(atts.GetStreamlineAlgorithmType(), 
                                           atts.GetMaxStreamlineProcessCount(),
                                           atts.GetMaxDomainCacheSize(),
                                           atts.GetWorkGroupSize());

    // Poincare specific attributes.
    poincareFilter->SetMaxToroidalWinding( atts.GetMaxToroidalWinding() );
    poincareFilter->SetOverrideToroidalWinding( atts.GetOverrideToroidalWinding() );
    poincareFilter->SetHitRate( atts.GetHitRate() );
    poincareFilter->SetOverlaps( atts.GetOverlaps() );
    poincareFilter->SetAdjustPlane( atts.GetAdjustPlane() );


    poincareFilter->SetShowCurves( atts.GetMeshType() == 0 );

    vector < double > planes;
    unsigned int nplanes = atts.GetNumberPlanes();

    // Offset of M_PI/2.0 gives a Y normal but whether the
    // intersection is on the +X or -X side depends on the direction
    // of the fieldline.

    if( nplanes == 1 )
      planes.push_back( atts.GetSinglePlane() / 360.0 * 2.0 * M_PI + M_PI/2.0);
    else
      for( unsigned int i=0; i<nplanes; i++ )
        planes.push_back(2.0 * M_PI * (double) i / (double) nplanes + M_PI/2.0);

    poincareFilter->SetClipPlanes( planes );

    poincareFilter->SetDisplayMethod(STREAMLINE_DISPLAY_LINES);
    poincareFilter->SetColoringMethod(STREAMLINE_COLOR_SOLID);

    poincareFilter->SetDataValue( atts.GetDataValue() );

    poincareFilter->SetShowOPoints( atts.GetShowOPoints() );
    poincareFilter->SetShowChaotic( atts.GetShowChaotic() );
    poincareFilter->SetShowIslands( atts.GetShowIslands() );
    poincareFilter->SetShowLines(atts.GetShowLines());
    poincareFilter->SetShowPoints(atts.GetShowPoints());
    poincareFilter->SetShowRidgelines(atts.GetShowRidgelines());
    poincareFilter->SetVerboseFlag( atts.GetVerboseFlag() );
#endif

    if(atts.GetColorType() == PoincareAttributes::ColorByColorTable)
    {
        if (atts.GetColorTableName() == "Default")
            avtLUT->SetColorTable(NULL, true);
        else
            avtLUT->SetColorTable(atts.GetColorTableName().c_str(), true);
    }
    else
    {
        avtLUT->SetLUTColors(atts.GetSingleColor().GetColor(), 1);
    }
    
    //SetLighting(atts.GetLightingFlag());
    varMapper->TurnLightingOn();
    varMapper->SetSpecularIsInappropriate(false);
    SetLegend(atts.GetLegendFlag());
}

// ****************************************************************************
//  Method: avtPoincarePlot::SetLegend
//
//  Purpose:
//    Turns the legend on or off.
//
//  Arguments:
//    legendOn  : true if the legend should be turned on, false otherwise.
//
//  Programmer: Dave Pugmire
//  Creation:   Fri Nov  7 13:29:26 EST 2008
//
// ****************************************************************************

void
avtPoincarePlot::SetLegend(bool legendOn)
{
    if (legendOn)
    {
        // Set scaling.
        varLegend->LegendOn();
        varLegend->SetLookupTable(avtLUT->GetLookupTable());
        varLegend->SetScaling();
        varMapper->SetLookupTable(avtLUT->GetLookupTable());

        //
        //  Retrieve the actual range of the data
        //
        varMapper->SetMin(0.);
        varMapper->SetMaxOff();
        varMapper->SetLimitsMode(0);
        SetLegendRanges();
    }
    else
        varLegend->LegendOff();
}

// ****************************************************************************
// Method: avtPoincarePlot::SetLegendRanges
//
// Purpose: 
//   Sets the range to use for the legend.
//
// Programmer: Dave Pugmire
// Creation:   Fri Nov  7 13:31:08 EST 2008
//
// Modifications:
//
// Dave Pugmire, Fri Apr 24 15:47:02 EDT 2009
// Add min/max variable ranges.
//
// ****************************************************************************

void
avtPoincarePlot::SetLegendRanges()
{
    double min, max;
    varMapper->GetVarRange(min, max);

    if (atts.GetMinFlag())
        min = atts.GetMin();
    if (atts.GetMaxFlag())
        max = atts.GetMax();

    if (atts.GetMinFlag() && atts.GetMaxFlag() && min >= max)
    {
        EXCEPTION1(InvalidLimitsException, false); 
    }
    varMapper->SetMin(min);
    varMapper->SetMax(max);

    //
    // Set the range for the legend's text and colors.
    //
    varLegend->SetVarRange(min, max);
    varLegend->SetRange(min, max);
}
