/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                             avtPoincareFilter.C                           //
// ************************************************************************* //

#include <avtPoincareFilter.h>
#include <avtSLAlgorithm.h>
#include <avtStreamlineWrapper.h>

#include <vtkDataSet.h>
#include <vtkSlicer.h>
#include <vtkPolyData.h>
#include <vtkTubeFilter.h>
#include <vtkPolyLine.h>
#include <vtkQuad.h>
#include <vtkUnstructuredGrid.h>
#include <vtkCleanPolyData.h>

#include <vtkCellArray.h>
#include <vtkCellData.h>
#include <vtkPointData.h>
#include <vtkAppendPolyData.h>
#include <vtkSphereSource.h>
#include <vtkFloatArray.h>
#include <avtDatasetExaminer.h>
#include <avtExtents.h>

#include <utility>

//#define STRAIGHTLINE_SKELETON 1

#ifdef STRAIGHTLINE_SKELETON
#include "skelet.h"
#endif

using namespace std;

#define SIGN(x) ((x) < 0.0 ? -1 : 1)


static const int DATA_OriginalValue = 0;
static const int DATA_InputOrder = 1;
static const int DATA_PointIndex = 2;
static const int DATA_Plane = 3;
static const int DATA_WindingOrder = 4;
static const int DATA_WindingPointOrder = 5;
static const int DATA_WindingPointOrderModulo = 6;
static const int DATA_ToroidalWindings = 7;
static const int DATA_PoloidalWindings = 8;
static const int DATA_SafetyFactor = 9;
static const int DATA_Confidence = 10;
static const int DATA_RidgelineVariance = 11;
static const int DATA_Solid = 12;

// ****************************************************************************
//  Method: avtPoincareFilter constructor
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
//  Modifications:
//
//    Dave Pugmire, Wed Feb 25 09:52:11 EST 2009
//    Add terminate by steps, add AdamsBashforth solver,
//    Allen Sanderson's new code.
//
//    Dave Pugmire, Fri Apr 17 11:32:40 EDT 2009
//    Add variables for dataValue var.
//
//    Dave Pugmire, Tue Apr 28 09:26:06 EDT 2009
//    Changed color to dataValue
//
//    Dave Pugmire, Wed May 27 15:03:42 EDT 2009
//    Initialize streamlinePts.
//
//    Dave Pugmire, Tue Aug 18 09:10:49 EDT 2009
//    Add ability to restart streamline integration.
//
// ****************************************************************************

avtPoincareFilter::avtPoincareFilter() :
    maxToroidalWinding( 30 ),
    override( 0 ),
    hitrate( 0.90 ),
    adjust_plane(-1),
    overlaps(1),

    is_curvemesh(1),
    dataValue(DATA_SafetyFactor),

    showOPoints( false ),
    showIslands( false ),
    showLines( true ),
    showPoints( false ),
    verboseFlag( true )
{
    planes.resize(1);
    planes[0] = 0;
    streamlines.resize(0);
}


// ****************************************************************************
//  Method: avtPoincareFilter destructor
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
//  Modifications:
//
//    Dave Pugmire, Wed May 27 15:03:42 EDT 2009
//    Zero out streaminePts.
//
// ****************************************************************************

avtPoincareFilter::~avtPoincareFilter()
{
    streamlines.resize(0);
}

static vtkPolyData *
CreateSphere(float val, double rad, double pt[3])
{
    // Create the sphere polydata.
    vtkSphereSource *sphere = vtkSphereSource::New();
    sphere->SetCenter(pt[0], pt[1], pt[2]);
    sphere->SetRadius(rad);
    sphere->SetLatLongTessellation(1);
    sphere->SetPhiResolution(8);
    sphere->SetThetaResolution(8);
    vtkPolyData *sphereData = sphere->GetOutput();
    sphereData->Update();

    // Set the sphere's scalar to val.
    vtkFloatArray *arr = vtkFloatArray::New();
    int npts = sphereData->GetNumberOfPoints();
    arr->SetNumberOfTuples(npts);
    for (int i = 0; i < npts; ++i)
        arr->SetTuple1(i, val);
    arr->SetName("colorVar");
    sphereData->GetPointData()->SetScalars(arr);
    arr->Delete();

    sphereData->Register(NULL);
    sphere->Delete();

    return sphereData;
}


// ****************************************************************************
//  Method: avtPoincareFilter::PreExecute
//
//  Purpose:
//      Get the current spatial extents if necessary.
//
//  Programmer: Dave Pugmire
//  Creation:   Fri Nov  7 13:01:47 EST 2008
//
//  Modifications:
//
//
// ****************************************************************************

void
avtPoincareFilter::PreExecute(void)
{
    avtStreamlineFilter::PreExecute();
}


// ****************************************************************************
//  Method: avtPoincareFilter::PostExecute
//
//  Purpose:
//      Get the current spatial extents if necessary.
//
//  Programmer: Dave Pugmire
//  Creation:   Fri Nov  7 13:01:47 EST 2008
//
//  Modifications:
//
//
// ****************************************************************************

void
avtPoincareFilter::PostExecute(void)
{
    avtStreamlineFilter::PostExecute();
    
    double range[2];
    avtDataset_p ds = GetTypedOutput();
    avtDatasetExaminer::GetDataExtents(ds, range, "colorVar");

    avtExtents *e;
    e = GetOutput()->GetInfo().GetAttributes().GetCumulativeTrueDataExtents();
    e->Merge(range);
    e = GetOutput()->GetInfo().GetAttributes().GetCumulativeCurrentDataExtents();
    e->Merge(range);
}

// ****************************************************************************
//  Method: avtPoincareFilter::CreateStreamlineOutput
//
//  Purpose:
//      Create the VTK poly data output from the streamline.
//
//  Programmer: Dave Pugmire
//  Creation:   Tue Dec  23 12:51:29 EST 2008
//
//  Modifications:
//
//    Dave Pugmire (for Allen Sanderson), Wed Feb 25 09:52:11 EST 2009
//    Add terminate by steps, add AdamsBashforth solver, Allen Sanderson's new code.
//
//    Dave Pugmire, Wed May 27 15:03:42 EDT 2009
//    Moved GetStreamlinePoints to this method.
//
//    Dave Pugmire, Tue Aug 18 09:10:49 EDT 2009
//    Add ability to restart streamline integration.
//
// ****************************************************************************

void
avtPoincareFilter::CreateStreamlineOutput(vector<avtStreamlineWrapper *> &sls)
{
    streamlines.resize(sls.size());
    
    for ( int i=0; i<sls.size(); ++i )
    {
        // Use the streamline id so that the point are in the order that
        // the seeds were generated.
        unsigned int j = sls[i]->id;

        streamlines[j].slSeg = sls[i];
        avtStreamline::iterator siter = sls[i]->sl->begin();
        streamlines[j].streamlinePts.resize(0);
        
        while (siter != sls[i]->sl->end())
        {
            avtVector pt;
            pt.x = (*siter)->front()[0];
            pt.y = (*siter)->front()[1];
            pt.z = (*siter)->front()[2];
            
            streamlines[j].streamlinePts.push_back(pt);
            
            ++siter;
        }
        //cerr<<"CreateStreamlineOutput: "<<streamlines[i].slSeg->id<<" pts= "<<streamlines[i].streamlinePts.size()<<" term= "<<streamlines[i].slSeg->termination<<endl;
    }
}

// ****************************************************************************
//  Method: avtPoincareFilter::ModifyContract
//
//  Purpose:
//      
//
//  Programmer: Dave Pugmire
//  Creation:   Fri Nov  7 13:15:45 EST 2008
//
//  Modifications:
//
//
// ****************************************************************************

avtContract_p
avtPoincareFilter::ModifyContract(avtContract_p in_contract)
{
//     avtDataRequest_p in_dr = in_contract->GetDataRequest();
//     avtDataRequest_p out_dr = NULL;

//     if ( integrationType == STREAMLINE_INTEGRATE_M3D_C1_INTEGRATOR )
//     {
//         out_dr = new avtDataRequest(in_dr,in_dr->GetOriginalVariable());

//         // Add in the other fields that the M3D Interpolation needs
//         // for doing their Newton's Metod.

//         // Assume the user has selected B as the primary variable.
//      // Which is ignored.

//         // Single variables stored as attributes on the header
//         out_dr->AddSecondaryVariable("hidden/header/linear");  // /linear
//         out_dr->AddSecondaryVariable("hidden/header/ntor");    // /ntor
        
//         out_dr->AddSecondaryVariable("hidden/header/bzero");    // /bzero
//         out_dr->AddSecondaryVariable("hidden/header/rzero");    // /rzero
        

//         // The mesh - N elememts x 7
//         out_dr->AddSecondaryVariable("hidden/elements"); // /time_000/mesh/elements

//         // Variables on the mesh - N elements x 20
//         out_dr->AddSecondaryVariable("hidden/equilibrium/f");  // /equilibrium/fields/f
//         out_dr->AddSecondaryVariable("hidden/equilibrium/psi");// /equilibrium/fields/psi

//         out_dr->AddSecondaryVariable("hidden/f");      // /time_XXX/fields/f
//         out_dr->AddSecondaryVariable("hidden/f_i");    // /time_XXX/fields/f_i
//         out_dr->AddSecondaryVariable("hidden/psi");    // /time_XXX/fields/psi
//         out_dr->AddSecondaryVariable("hidden/psi_i");  // /time_XXX/fields/psi_i
//     }
    
//     avtContract_p out_contract;
//     if ( *out_dr )
//         out_contract = new avtContract(in_contract, out_dr);
//     else
//         out_contract = new avtContract(in_contract);

//     return avtStreamlineFilter::ModifyContract(out_contract);
  return avtStreamlineFilter::ModifyContract(in_contract);
}

// ****************************************************************************
//  Method: avtPoincareFilter::Execute
//
//  Purpose:
//      Calculate poincare points.
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
//  Modifications:
//    Dave Pugmire (for Allen Sanderson), Wed Feb 25 09:52:11 EST 2009
//    Add terminate by steps, add AdamsBashforth solver, Allen Sanderson's new code.
//
//    Dave Pugmire, Wed May 27 15:03:42 EDT 2009
//    Re-organization. GetStreamlinePoints removed.
//
//    Dave Pugmire, Tue Aug 18 09:10:49 EDT 2009
//    Add ability to restart streamline integration.
//
// ****************************************************************************

void
avtPoincareFilter::Execute()
{
    FLlib.verboseFlag = verboseFlag;

    avtStreamlineFilter::Execute();

    avtDataTree *dt = CreatePoincareOutput();

    SetOutputDataTree(dt);
}

// ****************************************************************************
//  Method: avtPoincareFilter::ContinueExecute
//
//  Purpose:
//      See if execution needs to continue.
//
//  Programmer: Dave Pugmire
//  Creation:   Mon Aug 17 08:30:06 EDT 2009
//
//  Modifications:
//
// ****************************************************************************

bool
avtPoincareFilter::ContinueExecute()
{
    debug5 << "Continue execute " << endl;

    vector<avtStreamlineWrapper *> sls;
    
    GetTerminatedStreamlines(sls);
    CreateStreamlineOutput(sls);

    if (! ClassifyStreamlines())
    {
        //Modify termination criteria.
        return true;
    }
    
    //Analysis complete, no need to continue.
    return false;
}


// ****************************************************************************
//  Method: avtPoincareFilter::UpdateDataObjectInfo
//
//  Purpose:
//      Allows the filter to change its output's data object information, which
//      is a description of the data object.
//
//  Programmer: Dave Pugmire -- generated by xml2avt
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
//  Modifications:
//    Jeremy Meredith, Wed Apr  8 13:23:10 EDT 2009
//    Set topological dimension and normals request appropriately.
//
// ****************************************************************************

void
avtPoincareFilter::UpdateDataObjectInfo(void)
{ 
    avtStreamlineFilter::UpdateDataObjectInfo();

    avtDataAttributes &atts = GetOutput()->GetInfo().GetAttributes();
    avtDataValidity   &val  = GetOutput()->GetInfo().GetValidity();
    if (is_curvemesh)
    {
        atts.SetTopologicalDimension(1);
        val.SetNormalsAreInappropriate(true);
    }
    else
    {
        atts.SetTopologicalDimension(2);
        val.SetNormalsAreInappropriate(false);
    }

    if (! atts.ValidVariable("colorVar"))
    {
        atts.AddVariable("colorVar");
        atts.SetActiveVariable("colorVar");
        atts.SetVariableDimension(1);
    }
}

// ****************************************************************************
//  Method: avtPoincareFilter::ClassifyStreamlines
//
//  Purpose:
//      Classify the streamlines (toroidal/poloidal winding).
//
//  Arguments:
//
//  Returns:      void
//
//  Programmer: Dave Pugmire
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
//  Modifications:
//    Dave Pugmire (for Allen Sanderson), Wed Feb 25 09:52:11 EST 2009
//    Add terminate by steps, add AdamsBashforth solver, Allen Sanderson's new code.
//
//    Dave Pugmire, Tue Aug 18 09:10:49 EDT 2009
//    Add ability to restart streamline integration.
//
// ****************************************************************************

bool
avtPoincareFilter::ClassifyStreamlines()
{
    debug5 << "Classifying Streaming " << endl;

    bool analysisComplete = true;

    poincareClassification.resize(streamlines.size());
    
    for ( int i = 0; i < streamlines.size(); i++ )
    {
        FieldlineInfo fi =
          FLlib.fieldlineProperties( streamlines[i].streamlinePts,
                                     override,
                                     maxToroidalWinding,
                                     hitrate );

        // Double the number of point needed because here are
        // punctures but the streamlines works off of plane
        // intersections which there are two for each puncture.
        fi.nPuncturesNeeded *= 2;

        poincareClassification[i] = fi;

        if( fi.nPuncturesNeeded > maxPunctures )
          fi.nPuncturesNeeded = maxPunctures;

        // Check to see if there are enough points for the analysis.
        if( fi.nPuncturesNeeded != 0 &&
            fi.nPuncturesNeeded > streamlines[i].slSeg->termination )
        {
          analysisComplete = false;

          streamlines[i].slSeg->termination = fi.nPuncturesNeeded;
          streamlines[i].slSeg->terminated = false;
        }
        else
          streamlines[i].slSeg->terminated = true;

        double safetyFactor;
        
        if ( fi.poloidalWinding > 0 )
            safetyFactor =
              (double) fi.toroidalWinding / (double) fi.poloidalWinding;
        else
            safetyFactor = 0;

        if(verboseFlag )
          cerr <<"Classify Streamline: "<<i<<" id= "<<streamlines[i].slSeg->id
               << "  ptCnt = " << streamlines[i].streamlinePts.size()
               << "  type = " << fi.type
               << "  toroidal/poloidal windings = " <<  fi.toroidalWinding
               << "/" << fi.poloidalWinding
               << "  (" << safetyFactor << ")"
               << "  windingGroupOffset = " << fi.windingGroupOffset
               << "  islands = " << fi.islands
               << "  nodes = " << fi.nnodes
               << "  confidence = " << fi.confidence
               << "  ridgelinePeriod = " << fi.ridgelinePeriod
               << "  complete " << (fi.complete ? "Yes " : "No ")
               << (streamlines[i].slSeg->terminated ? 0 : streamlines[i].slSeg->termination )
               << endl << endl;
    }

    debug5 << "Classifying Streaming "
           << (analysisComplete ? "Analysis completed" : "Analysis was not complete")
           << endl;

    return analysisComplete;
}

// ****************************************************************************
//  Method: avtPoincareFilter::misc crap
//
//  Purpose:
//      Create poincare output
//
//  Arguments:
//
//  Returns:      Poincare segments
//
//  Programmer: Allen Sanderson
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
// ****************************************************************************
template< class T > int
pairsortfirst( const pair < T, T > s0, const pair < T, T > s1 )
{
  return (s0.first > s1.first );
}

template< class T > int
pairsortsecond( const pair < T, T > s0, const pair < T, T > s1 )
{
  return (s0.second > s1.second );
}

void realDFTamp( vector< double > &g, vector< double > &G )
{
  unsigned int N = g.size();

  G.resize(N/2);

  for(unsigned int i=0; i<N/2; i++)
  {
    double freq = double(i) / double(N);

    double GRe = 0;
    double GIm = 0;

    for( unsigned int j=0; j<N; j++)
    {
      double a = -2.0 * M_PI * double(j) * freq;
//    if(inverse) a *= -1.0;
      double ca = cos(a);
      double sa = sin(a);
      
      GRe += g[j] * ca; // - in[x][1] * sa;
      GIm += g[j] * sa; // + in[x][1] * ca;
    }

    G[i] = sqrt(GRe*GRe + GIm*GIm);
  }
}


// ****************************************************************************
//  Method: avtPoincareFilter::CreatePoincareOutput
//
//  Purpose:
//      Create poincare output
//
//  Arguments:
//
//  Returns:      Poincare segments
//
//  Programmer: Dave Pugmire
//  Creation:   Tue Oct 7 09:02:52 PDT 2008
//
//  Modifications:
//    Dave Pugmire (for Allen Sanderson), Wed Feb 25 09:52:11 EST 2009
//    Add terminate by steps, add AdamsBashforth solver, Allen Sanderson's new code.
//
//    Dave Pugmire, Fri Apr 17 11:32:40 EDT 2009
//    Add variables for dataValue var.
//
//    Dave Pugmire, Tue Apr 28 09:26:06 EDT 2009
//    Changed color to dataValue
//
//    Dave Pugmire, Wed May 27 15:03:42 EDT 2009
//    Replaced cerr/cout with debug5.
//
// ****************************************************************************
avtDataTree *
avtPoincareFilter::CreatePoincareOutput()
{
    debug5 << "Creating output " << endl;

    avtDataTree *dt = new avtDataTree();
    
    for ( int i=0; i<streamlines.size(); ++i )
    {
        FieldlineType type           = poincareClassification[i].type;
        unsigned int toroidalWinding = poincareClassification[i].toroidalWinding;
        unsigned int poloidalWinding = poincareClassification[i].poloidalWinding;
        unsigned int islands         = poincareClassification[i].islands;
        unsigned int windingGroupOffset            = poincareClassification[i].windingGroupOffset;
        unsigned int nnodes          = (unsigned int) poincareClassification[i].nnodes;
        double confidence            = poincareClassification[i].confidence;
        bool   complete              = poincareClassification[i].complete;

        unsigned int ridgelinePeriod = (unsigned int) poincareClassification[i].ridgelinePeriod;
        double ridgelineVariance     = poincareClassification[i].ridgelineVariance;

        bool completeIslands = true;

        // If toroidal winding is zero, skip it.
        if( type == CHAOTIC )
        {
          if( showChaotic )
            toroidalWinding = 1;
          else
            continue;
        }
        else if( type == UNKNOWN ) 
        {
            if( verboseFlag ) 
              cerr << i << " SKIPPING UNKNOWN " << endl;
            
            std::pair< unsigned int, unsigned int > topo( 0, 0 );
            //      topology.push_back(topo);
            
            continue;
        }
        else if( toroidalWinding == 0 ) 
        {
            if( verboseFlag ) 
              cerr << i << " SKIPPING TOROIDALWINDING OF 0" << endl;
            
            std::pair< unsigned int, unsigned int > topo( 0, 0 );
            //      topology.push_back(topo);
            
            continue;
        }
        
        // Get the direction of the streamline toroidalWinding.
        Point lastPt = streamlines[i].streamlinePts[0];
        Point currPt = streamlines[i].streamlinePts[1];
        
        bool CCWstreamline = (atan2( lastPt.y, lastPt.x ) <
                              atan2( currPt.y, currPt.x ));
        
        double lastDist, currDist;

        // Put all of the points into the bins for each plane.
        std::vector< std::vector< std::vector < Point > > > puncturePts;
        
        puncturePts.resize( planes.size() );
        
        unsigned int startIndex = 0;
        
        for( unsigned int p=0; p<planes.size(); ++p ) 
        {
            Vector planeN;
            Vector planePt(0,0,0);
            
            // Go through the planes in the same direction as the streamline.
            if( CCWstreamline )
            {
                planeN = Vector( cos(planes[p]),
                                 sin(planes[p]),
                                 0 );
            }
            else
            {
                planeN = Vector( cos(planes[planes.size()-1-p]),
                                 sin(planes[planes.size()-1-p]),
                                 0 );
            }
            
            // Set up the plane equation.
            double plane[4];
            
            plane[0] = planeN.x;
            plane[1] = planeN.y;
            plane[2] = planeN.z;
            plane[3] = planePt.dot(planeN);
            
            puncturePts[p].resize( toroidalWinding );
            int bin = 0;
            
            // So to get the winding groups consistant start examining
            // the streamline in the same place for each plane.
            currPt = streamlines[i].streamlinePts[startIndex];
            currDist = planeN.dot( currPt ) - plane[3];
            
            for( unsigned int j=startIndex+1; j<streamlines[i].streamlinePts.size(); ++j )
            {
                lastPt = currPt;
                currPt = Vector(streamlines[i].streamlinePts[j]);
                
                lastDist = currDist;
                currDist = Dot( planeN, currPt ) - plane[3];
                
                // First look at only points that intersect the plane.
                if( SIGN(lastDist) != SIGN(currDist) ) 
                {
                    Vector dir(currPt-lastPt);
                    
                    double dot = Dot(planeN, dir);
                    
                    // If the segment is in the same direction as the plane then
                    // find where it intersects the plane.
                    if( dot > 0.0 )
                    {
                        // In order to get the winding groups
                        // consistant start examining the streamline
                        // in the same place for each plane so store
                        // the index of the first puncture point.
                        if( startIndex == 0 )
                            startIndex = j - 1;
                        
                        Vector w = lastPt - planePt;
                        
                        double t = -Dot(planeN, w ) / dot;
                        
                        Point point = Point(lastPt + dir * t);
                        
                        puncturePts[p][bin].push_back( point );
                        
                        bin = (bin + 1) % toroidalWinding;
                    }
                }
            }
        }
        

        unsigned int npts = 0;

        // Set up the plane equation as the base analysis takes place in
        // the X-Z plane.
        Vector planeN( 0, 1, 0 );
        Vector planePt(0,0,0);
        
        // Set up the plane equation.
        double plane[4];
        
        // Get the centroid of the puncture points.
        Vector centroid[4];

        for( unsigned int cc=0; cc<4; ++cc)
        {
          if( cc == 0 )
            planeN = Vector( 1, 0, 0 );
          else if( cc == 1 )
            planeN = Vector( -1, 0, 0 );
          else if( cc == 2 )
            planeN = Vector( 0, -1, 0 );
          else if( cc == 3 )
            planeN = Vector( 0, 1, 0 );

          plane[0] = planeN.x;
          plane[1] = planeN.y;
          plane[2] = planeN.z;
          plane[3] = Dot( planePt, planeN );
          
          currPt = streamlines[i].streamlinePts[0];
          currDist = Dot(planeN, currPt) - plane[3];
          
          centroid[cc] = Vector(0,0,0);
          npts = 0;

          for( unsigned int j=1; j<streamlines[i].streamlinePts.size(); ++j)
          {
            lastPt = currPt;
            currPt = streamlines[i].streamlinePts[j];
            
            // Poloidal plane distances.
            lastDist = currDist;
            currDist = Dot( planeN, currPt ) - plane[3];
            
            // First look at only points that intersect the poloidal plane.
            if( SIGN(lastDist) != SIGN(currDist) ) 
            {
              Vector dir(currPt-lastPt);
              
              double dot = Dot(planeN, dir);
              
              // If the segment is in the same direction as the poloidal plane
              // then find where it intersects the plane.
              if( dot > 0.0 )
              {
                Vector w = (Vector) lastPt - planePt;
                
                double t = -Dot(planeN, w ) / dot;
                
                Point point = Point(lastPt + dir * t);
                
                centroid[cc] += (Vector) point;
                ++npts;
              }
            }
          }

          centroid[cc] /= (float) npts;
        }


        Vector normalZ0 =
          Cross(centroid[2] - centroid[0], centroid[3] - centroid[0]);
        Vector normalZ1 =
          Cross(centroid[3] - centroid[1], centroid[2] - centroid[1]);

        normalZ0.normalize();
        normalZ1.normalize();

        Vector normalZ = (normalZ0 + normalZ1) / 2.0;

        // Set up the Z plane equation. Because the centroid of the puncture
        // plot is not on the z axis use it as the normal for the z plane.
        planeN = normalZ;

        // Get the ridgeline points. There is one point between each
        // poloidal plane puncture.
        
        // Set up the plane equation.
        plane[0] = planeN.x;
        plane[1] = planeN.y;
        plane[2] = planeN.z;
        plane[3] = planePt.dot(planeN);
            
        std::vector < Point > ridgelinePts;

        bool haveFirstIntersection = false;
        double maxZ = 0;

        // So to get the winding groups consistant start examining the
        // streamline in the same place for each plane.
        currPt = streamlines[i].streamlinePts[0];
        currDist = planeN.dot( currPt ) - plane[3];
            
        for( unsigned int j=startIndex+1;
             j<streamlines[i].streamlinePts.size();
             ++j )
        {
          lastPt = currPt;
          currPt = Vector(streamlines[i].streamlinePts[j]);
          
          lastDist = currDist;
          currDist = Dot( planeN, currPt ) - plane[3];
          
          // First look at only points that intersect the plane.
          if( SIGN(lastDist) != SIGN(currDist) ) 
          {
            Vector dir(currPt-lastPt);
            
            double dot = Dot(planeN, dir);
            
            // If the segment is in the same direction as the plane then
            // record the max Z value.
            if( dot > 0.0 )
            {
              if( haveFirstIntersection )
                ridgelinePts.push_back( Point( (float) ridgelinePts.size()/50.0,
                                               0,
                                               maxZ) );
              else
                haveFirstIntersection = true;

              maxZ = 0;
            }
          }

          if( maxZ < currPt.z )
            maxZ = currPt.z;
        }
        
//      // Get the rest of the info only from the phi = zero plane.
//         unsigned int p;
        
//         if( CCWstreamline )
//             p = 0;
//         else
//             p = planes.size()-1;
        
//         // Get the centroid of each toroidal winding group and all
//         // puncture points.

//      Vector globalCentroid(0,0,0);
//      unsigned int npts = 0;

//         std::vector< Vector > localCentroids;
//         std::vector< Vector > localSeparatrices[2];
        
//         localCentroids.resize(toroidalWinding);
//         localSeparatrices[0].resize(toroidalWinding);
//         localSeparatrices[1].resize(toroidalWinding);
        
//         for( unsigned int j=0; j<toroidalWinding; ++j ) 
//         {
//             localCentroids[j] = Vector(0,0,0);
            
//             for( unsigned int k=0; k<puncturePts[p][j].size(); ++k ) 
//                 localCentroids[j] += (Vector) puncturePts[p][j][k];
            
//             if( puncturePts[p][j].size() ) 
//             {
//                 localCentroids[j] /= (double) puncturePts[p][j].size();
//          }
//         }


//         // Get the direction of the points within a group.
//         //    Vector v0 = (Vector) puncturePts[p][0][0] - globalCentroid;
//         //    Vector v1 = (Vector) puncturePts[p][0][1] - globalCentroid;
        
//         //    bool groupCCW = (FLlib.ccw( v0, v1 ) == 1);
//         //    cerr << 0.0<< "  " << groupCCW << endl;
        
//         if( type == ISLAND_CHAIN ) 
//         {
//             for( unsigned int j=0; j<toroidalWinding; ++j ) 
//             {
//                 unsigned int startIndex;
//                 unsigned int middleIndex;
//                 unsigned int stopIndex;
//                 unsigned int nodes;
                
//                 Vector localCentroid;
                
//                 unsigned int turns =
//                     FLlib.islandProperties( puncturePts[p][j], localCentroid,
//                                             startIndex, middleIndex, stopIndex, nodes );
                
//                 //      cerr << "Island " << i  << "   "
//                 //           << "Turns " << turns  << "   "
//                 //           << "nodes " << nodes  << "   "
//                 //           << "Indexes "
//                 //           << startIndex  << "  "
//                 //           << middleIndex << "  "
//                 //           << stopIndex   << endl;
                
                
//              if( turns < 3 )
//                completeIslands = false;
                
//                 if( turns >= 2 ) 
//                 {
//                     //        localSeparatrices[0][j] = (Vector) puncturePts[p][j][startIndex];
//                     //        localSeparatrices[1][j] = (Vector) puncturePts[p][j][middleIndex];
//                 }
                
//                 if( turns == 3 ) 
//                 {
//                     unsigned int index0 = (middleIndex - startIndex ) / 2;
//                     unsigned int index1 = (  stopIndex - middleIndex) / 2;
                    
//                     //      cerr << "Indexes mid " << nodes << " nodes "
//                     //           << "  " << ( startIndex + index0)%nodes 
//                     //           << "  " << (middleIndex - index0)%nodes
//                     //           << "  " << (middleIndex + index1)%nodes
//                     //           << "  " << (  stopIndex - index1)%nodes << endl;
                    
//                     localCentroids[j] =
//                         ( (Vector) puncturePts[p][j][( startIndex + index0)%nodes] + 
//                           (Vector) puncturePts[p][j][(middleIndex - index0)%nodes] +
                          
//                           (Vector) puncturePts[p][j][(middleIndex + index1)%nodes] + 
//                           (Vector) puncturePts[p][j][(  stopIndex - index1)%nodes] ) / 4.0;
//                 }
//                 else if( turns == 2 ) 
//                 {
//                     unsigned int index0 = (middleIndex - startIndex ) / 2;
                    
//                     //      cerr << "Indexes mid " << nodes << " nodes "
//                     //           << "  " << ( startIndex + index0)%nodes 
//                     //           << "  " << (middleIndex - index0)%nodes
//                     //           << "  " << (middleIndex + index1)%nodes
//                     //           << "  " << (  stopIndex - index1)%nodes << endl;
                    
//                     localCentroids[j] =
//                         ( (Vector) puncturePts[p][j][( startIndex + index0)%nodes] + 
//                           (Vector) puncturePts[p][j][(middleIndex - index0)%nodes] ) / 2.0;
                    
//                 } 
//                 else if( turns == 1 ) 
//                 {
//                     unsigned int index0 = (stopIndex - startIndex ) / 2;
                    
//                     //      cerr << "Indexes mid " << nodes << " nodes "
//                     //           << "  " << ( startIndex + index0)%nodes 
//                     //           << "  " << (middleIndex - index0)%nodes
//                     //           << "  " << (middleIndex + index1)%nodes
//                     //           << "  " << (  stopIndex - index1)%nodes << endl;
                    
//                     localCentroids[j] =
//                         ( (Vector) puncturePts[p][j][(startIndex + index0)%nodes] + 
//                           (Vector) puncturePts[p][j][( stopIndex - index0)%nodes] ) / 2.0;
//                 }
                
//                 //      // Get the principal axes of the island.
//                 //      Vector localCentroid(0,0,0);
                
//                 //      for( unsigned int k=0; k<puncturePts[p][j].size(); ++k )
//                 //        localCentroid += (Vector) puncturePts[p][j][k];
                
//                 //      localCentroid /= (float) puncturePts[p][j].size();
                
//                 //      float Ixx = 0.0;
//                 //      float Ixz = 0.0;
//                 //      float Izz = 0.0;
                
//                 //      double maxDist = 0;
                
//                 //      for( unsigned int k=0; k<puncturePts[p][j].size(); k++ ) {
                
//                 //        Vector vec = (Vector) puncturePts[p][j][k] - localCentroid;
                
//                 //        if( maxDist < vec.length() )
//                 //          maxDist = vec.length();
                
//                 //        Ixx += vec.z()*vec.z();
//                 //        Ixz -= vec.x()*vec.z();
//                 //        Izz += vec.x()*vec.x();
//                 //      }
                
//                 //      float alpha = atan( 2.0 * Ixz / (Ixx - Izz) ) / 2.0;
                
//                 // //       cerr << "PRINCIPAL AXES " << alpha * 180.0 / M_PI << "    "
//                 // //      << Ixx + Ixz * sin(alpha       )/cos(alpha       ) << "    "
//                 // //      << Izz + Ixz * cos(alpha+M_PI/2)/sin(alpha+M_PI/2) << endl;
                
//                 //      if( Ixx + Ixz * sin(alpha       )/cos(alpha       ) >
//                 //          Izz + Ixz * cos(alpha+M_PI/2)/sin(alpha+M_PI/2) )
//                 //        localCentroid += Vector(  cos(alpha), 0, sin(alpha) ) * maxDist;
//                 //      else
//                 //        localCentroid += Vector( -sin(alpha), 0, cos(alpha) ) * maxDist;
                
//                 //      localCentroids[j] = localCentroid;
                
//             }
//         }  // if( type == ISLAND_CHAIN )
        
        for( unsigned int p=0; p<planes.size(); p++ ) 
        {
            if( type == CHAOTIC )
              nnodes = puncturePts[p][0].size();

            else if( type != RATIONAL )
            {
                if( overlaps == 1 || overlaps == 3 )
                    FLlib.removeOverlap( puncturePts[p], nnodes,
                                         toroidalWinding, poloidalWinding,
                                         windingGroupOffset, islands );
                if( overlaps == 2 )
                    FLlib.mergeOverlap( puncturePts[p], nnodes,
                                        toroidalWinding, poloidalWinding,
                                        windingGroupOffset, islands );
                else if( overlaps == 3 )
                    FLlib.smoothCurve( puncturePts[p], nnodes,
                                       toroidalWinding, poloidalWinding,
                                       windingGroupOffset, islands );
            }
            
            bool VALID = true;
            
            // Sanity check
            for( unsigned int j=0; j<toroidalWinding; ++j ) 
            {
                if( nnodes > puncturePts[p][j].size() )
                    nnodes = puncturePts[p][j].size();
                
                if( puncturePts[p][j].size() < 1 ) 
                {
                    if( verboseFlag ) 
                      cerr << "Clean up check failed - Plane " << p
                           << " bin  " << j
                           << " number of points " << puncturePts[p][j].size()
                           << endl;
                    
                    VALID = false;
                    
//                    return NULL;
                }
                
                //      cerr << "Surface " << i
                //           << " plane " << p
                //           << " bin " << j
                //           << " base number of nodes " << nnodes
                //           << " number of points " << puncturePts[p][j].size()
                //           << endl;
            }
        }
        
        if( verboseFlag ) 
        {
          cerr << "Surface " << i << " is a "
               << toroidalWinding << ":" << poloidalWinding << " ("
               << (double) toroidalWinding / (double) poloidalWinding << ")  ";

          if( type == RATIONAL )
            cerr << "rational surface  ";

          else if( type == FLUX_SURFACE )
            cerr << "flux surface  ";

          else if( type == ISLAND_CHAIN )
            cerr << "island chain that contains "
                 << islands << " islands  ";

          else if( type == CHAOTIC )
            cerr << "chaotic  ";

          cerr << "with " << nnodes << " nodes"
               << (complete ? " (Complete)  " : "  ")
               << "confidence " << confidence
               << endl;

          if( type == ISLAND_CHAIN && islands != toroidalWinding ) 
            cerr << "WARNING - The island count does not match the toroidalWinding count" << endl;
        }
        
        // Record the topology.
        //std::pair< unsigned int, unsigned int >
        //  topo( toroidalWinding, poloidalWinding );
        
        //    topology.push_back(topo);

        if( !showIslands || (showIslands && islands) ) 
        {
            double color_value = 0;
            
            if( dataValue == DATA_OriginalValue ||
                dataValue == DATA_InputOrder )
                color_value = i;
            else if( dataValue == DATA_ToroidalWindings )
                color_value = toroidalWinding;
            else if( dataValue == DATA_PoloidalWindings )
                color_value = poloidalWinding;
            else if( dataValue == DATA_SafetyFactor )
                color_value = (double) toroidalWinding / (double) poloidalWinding;
            else if( dataValue == DATA_Confidence )
              color_value = confidence;
            else if( dataValue == DATA_RidgelineVariance )
              color_value = ridgelineVariance;
                  
            // Currently the surface mesh is a structquad so set the dims - it
            // really should be and unstructured surface so multiple surface
            // can be generated.
            if( is_curvemesh ) 
            {
              if( type == RATIONAL ) 
                loadCurve( dt, puncturePts, nnodes, islands, windingGroupOffset,
                           dataValue, color_value );
              else if( type == CHAOTIC )
              {
                bool tmpLines  = showLines;
                bool tmpPoints = showPoints;

                showLines  = false;
                showPoints = true;

                loadCurve( dt, puncturePts, nnodes, DATA_SafetyFactor, 0 );

                showLines  = tmpLines;
                showPoints = tmpPoints;
              }
              else
                loadCurve( dt, puncturePts, nnodes, dataValue, color_value );

              if( type == ISLAND_CHAIN && showOPoints )
                findIslandCenter( dt, puncturePts, dataValue, color_value );
            }
            else
            {
              loadSurface( dt, puncturePts, nnodes, islands, windingGroupOffset,
                           dataValue, color_value );
            }

            if( showRidgelines )
              loadPoints( dt, ridgelinePts, ridgelinePeriod,
                          nnodes, islands, poloidalWinding,
                          dataValue, color_value, true );

//     double best_period = 0;
//     double best_amp = 0;


//     vector< unsigned int > periodList;

//     for(unsigned int p=2; p<ridgelinePts.size()/2; p++)
//     {
//       unsigned int N = p * (ridgelinePts.size() / p);

//       if( find( periodList.begin(), periodList.end(), N/2 ) != periodList.end())
//      continue;
//       else
//      periodList.push_back( N/2 );

//       vector< pair < double, double > > dft;
//       dft.resize( N/2 );

//       vector< pair < double, double > > g, G;

//       g.resize( N/2 );
//       G.resize( N/2 );

//       double local_best_period = 0;
//       double local_best_amp = 0;

//       for(unsigned int f=1; f<N/2; f++)
//       {
//      double freq = double(f) / double(N);
        
//      double GRe = 0;
//      double GIm = 0;

//      for( unsigned int cc=0; cc<N; cc++)
//      {
//        double a = -2.0 * M_PI * double(cc) * freq;
// //        if(inverse) a *= -1.0;
//        double ca = cos(a);
//        double sa = sin(a);
          
//        GRe += ridgelinePts[cc].z * ca; // - in[x][1] * sa;
//        GIm += ridgelinePts[cc].z * sa; // + in[x][1] * ca;
//      }

//      G[f].first  = GRe;
//      G[f].second = GIm;

//      double amp = sqrt(GRe*GRe + GIm*GIm);

//      dft[f] = pair< double, double >(freq, amp);

//      if( local_best_amp < amp )
//      {
//        local_best_amp = amp;
//        local_best_period = 1.0 / freq;
//      }
//       }

//       if( best_amp < local_best_amp )
//       {
//      best_amp = local_best_amp;
//      best_period = local_best_period;
//       }
      
//       cerr << "local  "
//         << p << "  "
//         << N << "  "
//         << local_best_period << "  " << local_best_amp << endl;
      
//       continue;
      
//       for( unsigned int cc=1; cc<N/2; ++cc )
//       {
//      double period = ((int) (10000.0 / dft[cc].first) / 10000.0);
        
//      double amp_1 = dft[cc-1].second;
//      double amp   = dft[cc  ].second;
//      double amp1  = dft[cc+1].second;
        
// //       cerr << period << "  "
// //      << dft[cc].first  << "  "
// //      << dft[cc].second << "  "
// //      << ((amp_1<amp && amp>amp1) ? "******" : "")
// //      << endl;
//       }
//     }
    
//     cerr << "best overall  "
//       << best_period << "  " << best_amp << endl;

    ///////////////////////////
  
//     best_period = 0;
//     best_amp = 0;

//     unsigned int N = ridgelinePts.size();

//     vector< double > height;
//     vector< double > Gamp;
//     vector< double > gamp;

//     height.resize( N );

//     for( unsigned int cc=0; cc<N; cc++)
//       height[cc] = ridgelinePts[cc].z;

//     // DFT of the original height signal
//     realDFTamp( height, Gamp );

// //     ridgelinePts.resize( N / 2 );
// //     ridgelinePts[0].z = 0;
// //     for(unsigned int f=1; f<ridgelinePts.size(); f++)
// //       ridgelinePts[f].z = Gamp[f] / 100.0;

//     // Throw awway the DC component and compute the log of the amplitude.
//     gamp.resize( Gamp.size() - 1 );

//     for( unsigned int cc=1; cc<Gamp.size(); cc++)
//       gamp[cc-1] = log2( Gamp[cc] );

//     // DFT of the amplitude signal
//     realDFTamp( gamp, Gamp );

//     bool foundFirstMin = false;
//     double last_amp = 1.0e15;

//     for(unsigned int cc=1; cc<Gamp.size(); cc++)
//     {
//       cerr << cc << "  " << Gamp[cc];

//       if( 1<cc && cc<Gamp.size()-1 &&
//        Gamp[cc-1] < Gamp[cc] && Gamp[cc] > Gamp[cc+1] )
//      cerr << "**" << endl;
//       else
//      cerr << endl;

//       if( !foundFirstMin && last_amp < Gamp[cc] )
//      foundFirstMin = true;

//       if( foundFirstMin && best_amp < Gamp[cc] )
//       {
//        best_amp = Gamp[cc];
//        best_period = cc;
//       }

//       last_amp = Gamp[cc];
//     }

//     cerr << __LINE__ << endl;

//     cerr << "best  " << best_period << "  " << best_amp << endl;

 //    ridgelinePts.resize( Gamp.size() - 1 );
//     for(unsigned int f=1; f<Gamp.size(); f++)
//       ridgelinePts[f-1].z = Gamp[f] / 100.0;


//     loadPoints( dt, ridgelinePts, ridgelinePts.size(),
//              nnodes, islands, poloidalWinding,
//              dataValue, color_value, true );


        }
    }
    
    if( verboseFlag ) 
        cerr << endl << endl << "count " << streamlines.size() << endl << endl;

    debug5 << "Finished creating output " << endl;
    
    return dt;
}

// ****************************************************************************
//  Method: avtPoincareFilter::loadCurve
//
//  Purpose: This method is for rational surfaces.
//           Creates a curve from the puncture points. Each curve
//           represents one toroidal winding group. 
//
//  Arguments:
//
//  Returns:      void
//
//  Programmer: Allen Sanderson
//  Creation:   Wed Feb 25 09:52:11 EST 2009
//
//  Modifications:
//
// ****************************************************************************

void
avtPoincareFilter::loadCurve( avtDataTree *dt,
                              vector< vector < vector < Point > > > &nodes,
                              unsigned int nnodes,
                              unsigned int islands,
                              unsigned int offset,
                              unsigned int color,
                              double color_value ) 
{
    vtkAppendPolyData *append = vtkAppendPolyData::New();
    
    unsigned int nplanes = nodes.size();
    unsigned int toroidalWindings = nodes[0].size();

    if( toroidalWindings == 1 )
      return;

    // If an island then only points.
    if( showLines && islands == 0 )
    {
      // Loop through each plane
      for( unsigned int p=0; p<nplanes; ++p ) 
      {
        if( color == DATA_Plane )
            color_value = p;
        
        //Create groups that represent the toroidial groups.
        vtkPoints *points = vtkPoints::New();
        vtkCellArray *cells = vtkCellArray::New();
        vtkFloatArray *scalars = vtkFloatArray::New();
            
        cells->InsertNextCell(toroidalWindings+1);
        scalars->Allocate    (toroidalWindings+1);
            
        // Loop through each toroidial group taking just the first
        // point from each group.
        for( unsigned int jj=0; jj<=toroidalWindings*offset; jj+=offset ) 
        {
            unsigned int j = jj % toroidalWindings;

            if( color == DATA_WindingOrder )
                color_value = j;
            
            // Use the first point in each toroidial group
            unsigned int i=0;

            points->InsertPoint(j,
                                nodes[p][j][i].x,
                                nodes[p][j][i].y,
                                nodes[p][j][i].z);

            cells->InsertCellPoint(j);

            if( color == DATA_PointIndex )
              color_value = (i*toroidalWindings+j)*nplanes + p;
            else if( color == DATA_WindingPointOrder )
              color_value =  i;
            else if( color == DATA_WindingPointOrderModulo )
              color_value =  i % nnodes;
                
            scalars->InsertTuple1(j, color_value);
        }
         
        // Create a new VTK polyline.
        vtkPolyData *pd = vtkPolyData::New();
        pd->SetPoints(points);
        pd->SetLines(cells);
        scalars->SetName("colorVar");
        pd->GetPointData()->SetScalars(scalars);
        
        append->AddInput(pd);
        
        points->Delete();
        cells->Delete();
        scalars->Delete();
      }   
    }

    if (showPoints)
    {
        for( unsigned int p=0; p<nplanes; ++p ) 
        {
            if( color == DATA_Plane )
                color_value = p;
            
            // Loop through each toroidial group
            for( unsigned int j=0; j<toroidalWindings; ++j ) 
            {
                if( color == DATA_WindingOrder )
                    color_value = j;
                
                // Loop through each point in toroidial group
                for( unsigned int i=0; i<nodes[p][j].size(); ++i )
                {      
                    double pt[3] =
                      { nodes[p][j][i].x, nodes[p][j][i].y, nodes[p][j][i].z };
                    
                    double rad = 0.0025;
                    
                    if( color == DATA_PointIndex )
                      color_value = (i*toroidalWindings+j)*nplanes + p;
                    else if( color == DATA_WindingPointOrder )
                      color_value = i;
                    else if( color == DATA_WindingPointOrderModulo )
                      color_value =  i % nnodes;
                    
                    vtkPolyData *ball = CreateSphere(color_value, rad, pt);
                    
                    append->AddInput(ball);
                    ball->Delete();
                }
            }
        }
    }
    
    if (showPoints)
    {
        for( unsigned int p=0; p<nplanes; ++p ) 
        {
            if( color == DATA_Plane )
                color_value = p;
            
            // Loop through each toroidial group
            for( unsigned int j=0; j<toroidalWindings; ++j ) 
            {
                if( color == DATA_WindingOrder )
                    color_value = j;
                
                // Loop through each point in toroidial group
                for( unsigned int i=0; i<nodes[p][j].size(); ++i )
                {      
                    double pt[3] =
                      { nodes[p][j][i].x, nodes[p][j][i].y, nodes[p][j][i].z };
                    
                    double rad = 0.0025;
                    
                    if( color == DATA_PointIndex )
                      color_value = (i*toroidalWindings+j)*nplanes + p;
                    else if( color == DATA_WindingPointOrder )
                      color_value = i;
                    else if( color == DATA_WindingPointOrderModulo )
                      color_value = i % nnodes;
                    
                    vtkPolyData *ball = CreateSphere(color_value, rad, pt);
                    
                    append->AddInput(ball);
                    ball->Delete();
                }
            }
        }
    }
    
    if (0 && showPoints)
    {
        for( unsigned int p=0; p<nplanes; ++p ) 
        {
            if( color == DATA_Plane )
                color_value = p;
            
            // Loop through each toroidial group
            for( unsigned int j=0; j<toroidalWindings; ++j ) 
            {
                if( color == DATA_WindingOrder )
                    color_value = j;
                
                //Create groups that represent the toroidial groups.
                vtkPoints *points = vtkPoints::New();
                vtkCellArray *cells = vtkCellArray::New();
                vtkFloatArray *scalars = vtkFloatArray::New();

                scalars->Allocate( nodes[p][j].size() );
        
                // Loop through each point in toroidial group
                for( unsigned int i=0; i<nodes[p][j].size(); ++i )
                {      
                    points->InsertNextPoint(nodes[p][j][i].x,
                                            nodes[p][j][i].y,
                                            nodes[p][j][i].z );
                    
                    cells->InsertNextCell(1, (vtkIdType*) &i);
                    
                    if( color == DATA_PointIndex )
                      color_value = (i*toroidalWindings+j)*nplanes + p;
                    else if( color == DATA_WindingPointOrder )
                      color_value = i;
                    else if( color == DATA_WindingPointOrderModulo )
                      color_value =  i % nnodes;
                    
                    scalars->InsertTuple1(i, color_value);
                }

                // Create a new VTK point clouds.
                vtkPolyData *pd = vtkPolyData::New();
                pd->SetPoints(points);
                pd->SetVerts(cells);
                scalars->SetName("colorVar");
                pd->GetPointData()->SetScalars(scalars);
    
                append->AddInput(pd);
    
                points->Delete();
                cells->Delete();
                scalars->Delete();  
            }
        }
    }
    
    append->Update();
    vtkPolyData *outPD = append->GetOutput();
    outPD->Register(NULL);
    outPD->SetSource(NULL);
    append->Delete();
    
    dt->Merge( new avtDataTree(outPD, 0) );
}

// ****************************************************************************
//  Method: avtPoincareFilter::loadCurve
//
//  Purpose: Creates a curve from the puncture points. Each curve
//           represents one toroidal winding group.
//
//  Arguments:
//
//  Returns:      void
//
//  Programmer: Allen Sanderson
//  Creation:   Wed Feb 25 09:52:11 EST 2009
//
//  Modifications:
//
// ****************************************************************************

void
avtPoincareFilter::loadCurve( avtDataTree *dt,
                              vector< vector < vector < Point > > > &nodes,
                              unsigned int nnodes,
                              unsigned int color,
                              double color_value ) 
{
    vtkAppendPolyData *append = vtkAppendPolyData::New();
    
    unsigned int nplanes = nodes.size();
    unsigned int toroidalWindings = nodes[0].size();

    if (showLines)
    {
      if( color == DATA_WindingPointOrderModulo )
      {
        // Loop through each plane
        for( unsigned int p=0; p<nplanes; ++p ) 
        {
          if( color == DATA_Plane )
            color_value = p;
          
          // Loop through each toroidial group
          for( unsigned int j=0; j<toroidalWindings; ++j ) 
          {
            for( unsigned int n=0; n<nnodes; ++n ) 
            {
              //Create groups that represent the toroidial groups.
              vtkPoints *points = vtkPoints::New();
              vtkCellArray *cells = vtkCellArray::New();
              vtkFloatArray *scalars = vtkFloatArray::New();

              unsigned int npts = ceil((nodes[p][j].size()-n)/ (float) nnodes);
            
              cells->InsertNextCell(npts);
              scalars->Allocate    (npts);
            
              if( color == DATA_WindingOrder )
                color_value = j;

              unsigned int cc = 0;

              // Loop through each point in toroidial group
              for( unsigned int i=n; i<nodes[p][j].size(); i+=nnodes ) 
              {
                points->InsertPoint(cc,
                                    nodes[p][j][i].x,
                                    nodes[p][j][i].y,
                                    nodes[p][j][i].z);

                cells->InsertCellPoint(cc);

                if( color == DATA_PointIndex )
                  color_value = (i*toroidalWindings+j)*nplanes + p;
                else if( color == DATA_WindingPointOrder )
                  color_value =  i;
                else if( color == DATA_WindingPointOrderModulo )
                  color_value =  i % nnodes;
                
                scalars->InsertTuple1(cc++, color_value);
              }

              // Create a new VTK polyline.
              vtkPolyData *pd = vtkPolyData::New();
              pd->SetPoints(points);
              pd->SetLines(cells);
              scalars->SetName("colorVar");
              pd->GetPointData()->SetScalars(scalars);
            
              append->AddInput(pd);
            
              points->Delete();
              cells->Delete();
              scalars->Delete();       
            }
          }
        }
      }

      else
      {
        // Loop through each plane
        for( unsigned int p=0; p<nplanes; ++p ) 
        {
            if( color == DATA_Plane )
              color_value = p;
        
            // Loop through each toroidial group
            for( unsigned int j=0; j<toroidalWindings; ++j ) 
            {
                //Create groups that represent the toroidial groups.
              vtkPoints *points = vtkPoints::New();
              vtkCellArray *cells = vtkCellArray::New();
              vtkFloatArray *scalars = vtkFloatArray::New();
            
              cells->InsertNextCell(nodes[p][j].size());
              scalars->Allocate    (nodes[p][j].size());
            
              if( color == DATA_WindingOrder )
                color_value = j;
            
              // Loop through each point in toroidial group
              for( unsigned int i=0; i<nodes[p][j].size(); ++i ) 
              {
                  points->InsertPoint(i,
                                      nodes[p][j][i].x,
                                      nodes[p][j][i].y,
                                      nodes[p][j][i].z);

                  cells->InsertCellPoint(i);

                  if( color == DATA_PointIndex )
                    color_value = (i*toroidalWindings+j)*nplanes + p;
                  else if( color == DATA_WindingPointOrder )
                    color_value =  i;
                  else if( color == DATA_WindingPointOrderModulo )
                    color_value =  i % nnodes;
                
                  scalars->InsertTuple1(i, color_value);
              }
            
              // Create a new VTK polyline.
              vtkPolyData *pd = vtkPolyData::New();
              pd->SetPoints(points);
              pd->SetLines(cells);
              scalars->SetName("colorVar");
              pd->GetPointData()->SetScalars(scalars);
            
              append->AddInput(pd);
            
              points->Delete();
              cells->Delete();
              scalars->Delete();
            }
        }
      }
    }
    
    if (showPoints)
    {
        for( unsigned int p=0; p<nplanes; ++p ) 
        {
            if( color == DATA_Plane )
                color_value = p;
            
            // Loop through each toroidial group
            for( unsigned int j=0; j<toroidalWindings; ++j ) 
            {
                if( color == DATA_WindingOrder )
                    color_value = j;

                // Loop through each point in toroidial group
                for( unsigned int i=0; i<nodes[p][j].size(); ++i )
                {      
                    double pt[3] =
                      { nodes[p][j][i].x, nodes[p][j][i].y, nodes[p][j][i].z };
                    
                    double rad = 0.0025;
                    
                    if( color == DATA_PointIndex )
                      color_value = (i*toroidalWindings+j)*nplanes + p;
                    else if( color == DATA_WindingPointOrder )
                      color_value = i;
                    else if( color == DATA_WindingPointOrderModulo )
                      color_value = i % nnodes;
                    
                    vtkPolyData *ball = CreateSphere(color_value, rad, pt);

                    append->AddInput(ball);
                    ball->Delete();
                }
            }
        }
    }
    
    if (0 && showPoints)
    {
        for( unsigned int p=0; p<nplanes; ++p ) 
        {
            if( color == DATA_Plane )
                color_value = p;
            
            // Loop through each toroidial group
            for( unsigned int j=0; j<toroidalWindings; ++j ) 
            {
                if( color == DATA_WindingOrder )
                    color_value = j;

                //Create groups that represent the toroidial groups.
                vtkPoints *points = vtkPoints::New();
                vtkCellArray *cells = vtkCellArray::New();
                vtkFloatArray *scalars = vtkFloatArray::New();

                scalars->Allocate( nodes[p][j].size() );
        
                // Loop through each point in toroidial group
                for( unsigned int i=0; i<nodes[p][j].size(); ++i )
                {      
                    points->InsertNextPoint(nodes[p][j][i].x,
                                            nodes[p][j][i].y,
                                            nodes[p][j][i].z );
                  
                    cells->InsertNextCell(1, (vtkIdType*) &i);
                  
                    if( color == DATA_PointIndex )
                      color_value = (i*toroidalWindings+j)*nplanes + p;
                    else if( color == DATA_WindingPointOrder )
                      color_value =  i;
                    else if( color == DATA_WindingPointOrderModulo )
                      color_value =  i % nnodes;
                  
                    scalars->InsertTuple1(i, color_value);
                }

                // Create a new VTK point clouds.
                vtkPolyData *pd = vtkPolyData::New();
                pd->SetPoints(points);
                pd->SetVerts(cells);
                scalars->SetName("colorVar");
                pd->GetPointData()->SetScalars(scalars);
    
                append->AddInput(pd);
    
                points->Delete();
                cells->Delete();
                scalars->Delete();  
            }
        } 
    }
    
    append->Update();
    vtkPolyData *outPD = append->GetOutput();
    outPD->Register(NULL);
    outPD->SetSource(NULL);
    append->Delete();
    
    dt->Merge( new avtDataTree(outPD, 0) );
}


// ****************************************************************************
//  Method: avtPoincareFilter::findIslandCenter
//
//  Purpose: Creates a curve from the puncture points. Each curve represent one
//           toroidal winding group.
//
//  Arguments:
//
//  Returns:      void
//
//  Programmer: Allen Sanderson
//  Creation:   Wed Feb 25 09:52:11 EST 2009
//
//  Modifications:
//
// ****************************************************************************

void
avtPoincareFilter::findIslandCenter( avtDataTree *dt,
                                     vector< vector < vector < Point > > > &nodes,
                                     unsigned int color,
                                     double color_value ) 
{
#ifdef STRAIGHTLINE_SKELETON
    vtkAppendPolyData *append = vtkAppendPolyData::New();
    
    unsigned int nplanes = nodes.size();
    unsigned int toroidalWindings = nodes[0].size();

    
    for( unsigned int p=0; p<nplanes; ++p ) 
    {
      if( color == DATA_Plane )
        color_value = p;
      
      if( nodes[p][0][0].y < -1.0e-8 || 1.0e-8 < nodes[p][0][0].y )
        continue;

      // Loop through each toroidial group
      for( unsigned int j=0; j<toroidalWindings; ++j ) 
      {
        cerr << "Island " << j << "  ";

        if( color == DATA_WindingOrder )
          color_value = j;

        vector< Point > points;

        bool selfIntersect = false;

        // Loop through each point in toroidial group
        for( unsigned int i=0; i<nodes[p][j].size()-1; i+=3 ) 
        {
          // Check for self intersections
          for( unsigned int k=i+2; k<nodes[p][j].size()-2; ++k ) 
          {
            int result;

            if( result = FLlib.intersect( nodes[p][j][i], nodes[p][j][i+1],
                                          nodes[p][j][k], nodes[p][j][k+1] ) )
            {
              cerr << "Island " << j << " self intersects  "
                   << i << "  " << k << "  " << result << endl;

              // Self intersection found so skip this island.
              selfIntersect = true;
              break;
            }
          }

          // Self intersection found so skip this island.
          if( selfIntersect )
            break;
         
          // Points for the convex hull check
          points.push_back( Point( nodes[p][j][i].x,
                                   nodes[p][j][i].y,
                                   nodes[p][j][i].z) );
        }
        
        if( selfIntersect )
          continue;

        // Get the points in clockwise direction
        unsigned int nhulls;
        int direction = -1;
        bool reversed;

        cerr << " hull check ";

        FLlib.hullCheck( points, nodes[p][j].size()-1,
                         direction, nhulls, reversed );

        Skeleton::PointVector pointVec;

        // Get the points from the hull check that are now in
        // a clockwise ??? direction.
        for( unsigned int i=0; i<nodes[p][j].size()-1; ++i ) 
        {
           pointVec.push_back( Skeleton::Point( nodes[p][j][i].x,
                                                nodes[p][j][i].z ) );
        }
        
        if( reversed )
          reverse( pointVec.begin(), pointVec.end() );

//      for( unsigned int i=0; i<nodes[p][j].size()-1; ++i ) 
//      {
//        double pt[3] = { pointVec[i].x, 0, pointVec[i].y };
                  
//        double rad = 0.0025;
                  
//        vtkPolyData *ball = CreateSphere(i, rad, pt);
                  
//        append->AddInput(ball);
//        ball->Delete();
//      }

        cerr << " Skeleton check ";

        Skeleton::Skeleton s (Skeleton::makeSkeleton (pointVec));
                                
        // Delete all of the hull points.
        list<Skeleton::SkeletonLine>::iterator SL = s.begin();
        list<Skeleton::SkeletonLine>::iterator deleteSL;

        // Remove all of the points on the boundary while getting
        // the cord length of the remains interior segments.

        double cordLength = 0;

        map< int, int > indexCount;
        map<int, int>::iterator ic;

        cerr << " cordLength ";

        while ( SL != s.end() )
        {
          if( (*SL).lower.vertex->ID < pointVec.size() ||
              (*SL).higher.vertex->ID < pointVec.size() ) 
          {
            // Boundary point so delete it.
            deleteSL = SL;
            ++SL;
            s.erase( deleteSL );
          }
          else
          {
            // Running cord length.
            cordLength += sqrt( ((*SL).higher.vertex->point.x-
                                 (*SL).lower.vertex->point.x) *
                                ((*SL).higher.vertex->point.x-
                                 (*SL).lower.vertex->point.x) +

                                 ((*SL).higher.vertex->point.y -
                                  (*SL).lower.vertex->point.y) * 
                                 ((*SL).higher.vertex->point.y -
                                  (*SL).lower.vertex->point.y) );


            // Count the number of times each end point is used

            // Lower end point
            int index = (*SL).lower.vertex->ID;

            ic = indexCount.find( index );

            if( ic == indexCount.end() )
              indexCount.insert( pair<int, int>( index,1) );
            else (*ic).second++;

            // Higher end point
            index = (*SL).higher.vertex->ID;

            ic = indexCount.find( index );

            if( ic == indexCount.end() )
              indexCount.insert( pair<int, int>( index,1) );
            else (*ic).second++;

            ++SL;
          }
        }                   
        

        int startIndex, endIndex;
        int cc = 0;
        ic = indexCount.begin();

        while( ic != indexCount.end() )
        {
          // Find the end points that are used but once. 
          // There should only be two.
          if( (*ic).second == 1 )
          {
            if( cc == 1 )
              startIndex = (*ic).first;
            else if( cc == 0 )
              endIndex = (*ic).first;

            ++cc;
          }

          if( (*ic).second > 2 )
          {
            cerr << "double segment ??? " << (*ic).first << endl;

            SL = s.begin();

            while( SL != s.end() &&
                   (*SL).lower.vertex->ID != (*ic).first &&
                   (*SL).higher.vertex->ID != (*ic).first )
              SL++;
                    
            // Have an index so process
            if( SL != s.end() )
            {

              // Remove double segments;
              if( (*SL).lower.vertex->ID == (*ic).first &&
                  (*indexCount.find( (*SL).higher.vertex->ID )).second > 2 )
              {
                cerr << "removing double segment ??? " << (*ic).first << "  "
                     << (*indexCount.find( (*SL).higher.vertex->ID )).first
                     << endl;
                
                (*ic).second--;
                (*indexCount.find( (*SL).higher.vertex->ID )).second--;
                
                s.erase( SL );
              }
                  
              // Remove double segments;
              if( (*SL).higher.vertex->ID == (*ic).first &&
                  
                  (*indexCount.find( (*SL).lower.vertex->ID )).second > 2 )
              {
                cerr << "removing double segment ??? " << (*ic).first << "  "
                     << (*indexCount.find( (*SL).lower.vertex->ID )).first
                     << endl;
                
                
                (*ic).second--;
                (*indexCount.find( (*SL).lower.vertex->ID )).second--;
                
                s.erase( SL );
              }
            }                
          }

          ++ic;
        }

        double lastpt[3];

        if( cc < 2 )
        {
          cerr << "Not enough start points " << cc << endl;
          continue;
        }

        else if( cc > 2 )
        {
          cerr << "Too many start points " << cc << endl;
          continue;
        }

        cerr << "Island " << j << " New skeleton "
             << "start index " << startIndex
             << "  end index " << endIndex
             << endl;
        

        double currentCord = 0;

        int nextIndex = startIndex;

        unsigned int nlines = 0;

        // Loop through all of the lines which are described
        // as set of paired points.
        while ( s.begin() != s.end() )
        {
          // Index of the first point.
          SL = s.begin();

          while( SL != s.end() &&
                 (*SL).lower.vertex->ID != nextIndex &&
                 (*SL).higher.vertex->ID != nextIndex )
            SL++;
                    
          // Have an index so process
          if( SL != s.end() )
          {
//          cerr << "Island " << j
//               << " Line segment " << nlines++
//               << " index " << nextIndex;

            int lastIndex = nextIndex;

            // Index of the leading point.
            if( (*SL).lower.vertex->ID == nextIndex )
              nextIndex = (*SL).higher.vertex->ID;
            else // if( (*SL).higher.vertex->ID == startIndex )
              nextIndex = (*SL).lower.vertex->ID;
            
//          cerr << " index " << nextIndex
//               << endl;

            double localCord = sqrt( ((*SL).higher.vertex->point.x-
                                      (*SL).lower.vertex->point.x) *
                                     ((*SL).higher.vertex->point.x-
                                      (*SL).lower.vertex->point.x) +
                                     
                                     ((*SL).higher.vertex->point.y -
                                      (*SL).lower.vertex->point.y) * 
                                     ((*SL).higher.vertex->point.y -
                                      (*SL).lower.vertex->point.y) );

            // Check to see if the segment spans the mid cord.
            if( currentCord < cordLength/2 &&
                cordLength/2.0 <= currentCord + localCord )
            {
              double t = (cordLength/2.0 - currentCord) / localCord;

              // Already updated the nextIndex so use the lastIndex
              // for figuring out the closest to the cordlength center.
              if( (*SL).lower.vertex->ID == lastIndex )
              {
                lastpt[0] = (*SL).lower.vertex->point.x + t *
                  ((*SL).higher.vertex->point.x-(*SL).lower.vertex->point.x);
                lastpt[1] = 0;
                lastpt[2] = (*SL).lower.vertex->point.y + t *
                  ((*SL).higher.vertex->point.y-(*SL).lower.vertex->point.y);
              }

              else //if( (*SL).higher.vertex->ID == lastIndex )
              {
                lastpt[0] = (*SL).higher.vertex->point.x + t *
                  ((*SL).lower.vertex->point.x-(*SL).higher.vertex->point.x);
                lastpt[1] = 0;
                lastpt[2] = (*SL).higher.vertex->point.y + t *
                  ((*SL).lower.vertex->point.y-(*SL).higher.vertex->point.y);
              }
            }

            // Update the running cord length so that the mid cord is
            // not calculated again.
            currentCord += localCord;

            if( 0 ) // Points
            {
                double pt[3] = { (*SL).lower.vertex->point.x,
                                 0,
                                 (*SL).lower.vertex->point.y };

                double rad = 0.0025;
                      
                vtkPolyData *ball = CreateSphere(nlines, rad, pt);
                      
                append->AddInput(ball);
                ball->Delete();
            }

            if( 0 ) // Lines
            {
                vtkPoints *points = vtkPoints::New();
                vtkCellArray *cells = vtkCellArray::New();
                vtkFloatArray *scalars = vtkFloatArray::New();

                cells->InsertNextCell( 2 );
                scalars->Allocate    ( 2 );

                points->InsertPoint(0,
                                    (*SL).lower.vertex->point.x,
                                    0,
                                    (*SL).lower.vertex->point.y);

                cells->InsertCellPoint(0);
                scalars->InsertTuple1(0, nlines);

                points->InsertPoint(1,
                                    (*SL).higher.vertex->point.x,
                                    0,
                                    (*SL).higher.vertex->point.y);


                cells->InsertCellPoint(1);
                scalars->InsertTuple1(1, nlines);

                // Create a new VTK polyline.
                vtkPolyData *pd = vtkPolyData::New();
                pd->SetPoints(points);
                pd->SetLines(cells);
                scalars->SetName("colorVar");
                pd->GetPointData()->SetScalars(scalars);
        
                append->AddInput(pd);
            }

            // Remove this point from the list so that it is
            // not touched again.
            deleteSL = SL;
            ++SL;
            s.erase( deleteSL );
          }
          else
          {
            if( nextIndex != endIndex )
            {
              cerr << "Did not find end index  "
                   << nextIndex << "  " <<  endIndex
                   << endl;
              
              break;
            }
          }
        }
        
        cerr << "O Point "
             << lastpt[0] << "  " << lastpt[1] << "  " << lastpt[2]
             << endl;
        
        double rad = 0.005;
        
        vtkPolyData *ball = CreateSphere(nlines, rad, lastpt);
        
        append->AddInput(ball);
        ball->Delete();
      }
      
      // Do only one plane
      break;
    }
    
    append->Update();
    vtkPolyData *outPD = append->GetOutput();
    outPD->Register(NULL);
    outPD->SetSource(NULL);
    append->Delete();
    
    dt->Merge( new avtDataTree(outPD, 0) );
#endif
}


// ****************************************************************************
//  Method: avtPoincareFilter::loadSurface
//
//  Purpose: Creates a surface from a series the puncture points. The
//           surface is sweep from each toroidal winding group around
//           each plane in the torus. Each is connected together to
//           form a circular cross section.
//
//  Arguments:
//
//  Returns:      void
//
//  Programmer: Allen Sanderson
//  Creation:   Wed Feb 25 09:52:11 EST 2009
//
//  Modifications:
//
// ****************************************************************************

void
avtPoincareFilter::loadSurface( avtDataTree *dt,
                                vector< vector < vector < Point > > > &nodes,
                                unsigned int nnodes,
                                unsigned int islands,
                                unsigned int offset,
                                unsigned int color,
                                double color_value) 
{
    unsigned int nplanes = nodes.size();
    unsigned int toroidalWindings = nodes[0].size();
    
    int dims[2];
    
    // Add one to the first dimension to create a closed cylinder. Add
    // one to the second dimension to form a torus.
    dims[0] = nnodes + 1;
    dims[1] = nplanes * toroidalWindings + 1;
    
    // Create an unstructured quad for the island surface.
    vtkUnstructuredGrid *grid = vtkUnstructuredGrid::New();
    vtkQuad *quad = vtkQuad::New();
    vtkPoints *points = vtkPoints::New();
    vtkFloatArray *scalars = vtkFloatArray::New();
    
    points->SetNumberOfPoints(dims[0]*dims[1]);
    scalars->Allocate(dims[0]*dims[1]);
    
    float *points_ptr = (float *) points->GetVoidPointer(0);
    
    // Determine if the winding group order matches the point
    // ordering. This is only needed when building surfaces.
    Vector v0 = nodes[0][   0][1] - nodes[0][0][0];
    Vector v1 = nodes[0][offset][0] - nodes[0][0][0];

    bool flip;

    if( Dot( v0, v1 ) < 0 )
      flip = true;
    else
      flip = false;

    // Loop through each toroidial group
    for( unsigned int j=0; j<toroidalWindings; ++j )
    {
        if( color == DATA_WindingOrder )
            color_value = j;

        // Loop through each plane.
        for( unsigned int p=0; p<nplanes; ++p ) 
        {
            // Normally each toroidial winding group can be displayed
            // in the order received. Except for the last plane where
            // it needs to be adjusted by one group. That is if the
            // streamline started in the "correct" place. This is not
            // always the case so it may be necessary to adjust the
            // toroidal winding group location by one.
            unsigned int k;
            
            if( p == adjust_plane )
            {
                k = (j-1 + toroidalWindings) % toroidalWindings;
            }
            else
            {
                k = j;
            }
            
            unsigned int jj = nplanes * j + p;
            
            if( color == DATA_Plane )
                color_value = jj;
            
            // Loop through each point in toroidial group.
            for(unsigned int i=0; i<nnodes; ++i )
            {
                unsigned int n1 = jj * dims[0] + i;

                points_ptr[n1*3+0] = nodes[p][k][i].x;
                points_ptr[n1*3+1] = nodes[p][k][i].y;
                points_ptr[n1*3+2] = nodes[p][k][i].z;

                if( color == DATA_PointIndex )
                    color_value = (i*toroidalWindings+j)*nplanes + p;
                else if( color == DATA_WindingPointOrder )
                    color_value =  i;
                else if( color == DATA_WindingPointOrderModulo )
                  color_value =  i % nnodes;
                
                scalars->InsertTuple1(n1, color_value);

                // Create the quad.
                quad->GetPointIds()->SetId( 0,   jj    * dims[0] + i );
                quad->GetPointIds()->SetId( 1,  (jj+1) * dims[0] + i );
                quad->GetPointIds()->SetId( 2,  (jj+1) * dims[0] + i + 1);
                quad->GetPointIds()->SetId( 3,   jj    * dims[0] + i + 1);
                
                grid->InsertNextCell( quad->GetCellType(),
                                      quad->GetPointIds() );                
            }

            // For a surface add in the first point from the adjacent
            // toroidial group. Otherwise for an island add in the
            // first point from the current toroidal group.
            if( !islands )
            {
              if( flip )
                k = (k-offset+toroidalWindings) % toroidalWindings;
              else
                k = (k+offset) % toroidalWindings;
            }

            unsigned int i = nnodes;

            unsigned int n1 = jj * dims[0] + i;
            
            points_ptr[n1*3+0] = nodes[p][k][0].x;
            points_ptr[n1*3+1] = nodes[p][k][0].y;
            points_ptr[n1*3+2] = nodes[p][k][0].z;
            
            if( color == DATA_PointIndex )
              color_value = (i*toroidalWindings+j)*nplanes + p;
            else if( color == DATA_WindingPointOrder )
              color_value =  i;
            else if( color == DATA_WindingPointOrderModulo )
              color_value =  i % nnodes;
            
            scalars->InsertTuple1(n1, color_value);
        }
    }
    
    // Add in the first toroidal group from the first plane to complete
    // the torus.
    unsigned int j = 0;
    
    if( color == DATA_WindingOrder )
        color_value = j;
    
    // Add in the first toroidal group from the first plane to complete
    // the torus.
    unsigned int p = 0;
    
    // Normally each toroidial group can be displayed in the order
    // received. Except for the last plane where it needs to be
    // adjusted by one group. That is if the streamline started in
    // the "correct" place. This is not always the case so it may be
    // necessary to adjust the winding group location by one.
    unsigned int k;
    
    if( p == adjust_plane )
    {
        k = (j-1 + toroidalWindings) % toroidalWindings;
    }
    else
    {
        k = j;
    }
    
    unsigned int jj = nplanes * toroidalWindings;
    
    if( color == DATA_Plane )
        color_value = jj;
    
    // Loop through each point in toroidial group.
    for(unsigned int i=0; i<nnodes; ++i )
    {
      // Normally each point in a toroidial group can be displayed in
      // the order received. Except when dealing with 1:1 surfaces for
      // the last plane where it needs to be adjusted by one
      // location. That is if the streamline started in the "correct"
      // place. This is not always the case so it may be necessary to
      // adjust the point ordering by one.
      unsigned int ii;

      if( p == adjust_plane && toroidalWindings == 1) 
        ii = (i-1 + nnodes) % nnodes;
      else
        ii = i;

        unsigned int n1 = jj * dims[0] + ii;
        
        points_ptr[n1*3+0] = nodes[p][k][i].x;
        points_ptr[n1*3+1] = nodes[p][k][i].y;
        points_ptr[n1*3+2] = nodes[p][k][i].z;

        if( color == DATA_PointIndex )
            color_value = (i*toroidalWindings+j)*nplanes + p;
        else if( color == DATA_WindingPointOrder )
          color_value =  i;
        else if( color == DATA_WindingPointOrderModulo )
          color_value =  i % nnodes;
        
        scalars->InsertTuple1(n1, color_value);
    }

    // For a surface add in the first point from the adjacent
    // toroidial group. Otherwise for an island add in the
    // first point from the current toroidal group.
    if( !islands )
    {
      if( flip )
        k = (k-offset+toroidalWindings) % toroidalWindings;
      else
        k = (k+offset) % toroidalWindings;
    }

    unsigned int i = nnodes;
    unsigned int n1 = jj * dims[0] + i;
    
    points_ptr[n1*3+0] = nodes[p][k][0].x;
    points_ptr[n1*3+1] = nodes[p][k][0].y;
    points_ptr[n1*3+2] = nodes[p][k][0].z;
    
    if( color == DATA_PointIndex )
      color_value = (i*toroidalWindings+j)*nplanes + p;
    else if( color == DATA_WindingPointOrder )
      color_value =  i;
    else if( color == DATA_WindingPointOrderModulo )
      color_value =  i % nnodes;
    
    scalars->InsertTuple1(n1, color_value);


    // Stuff the points and scalars into the VTK unstructure grid.
    grid->SetPoints(points);
    scalars->SetName("colorVar");
    grid->GetPointData()->SetScalars(scalars);
    
    dt->Merge( new avtDataTree(grid, 0) );
    
    quad->Delete();
    points->Delete();
    scalars->Delete();
}


void
avtPoincareFilter::loadPoints( avtDataTree *dt,
                               vector < Point  > &nodes,
                               unsigned int period,
                               unsigned int nnodes,
                               unsigned int islands,
                               unsigned int poloidalWindings,
                               unsigned int color,
                               double color_value,
                               bool ptFlag )
{
  if( period <= 1 )
    period = nodes.size();

  unsigned int colorMax = 0;

  vtkAppendPolyData *append = vtkAppendPolyData::New();

  if( islands )
    poloidalWindings *= nnodes;
  
  if (showLines)
  {
    //Create groups that represent the toroidial groups.
    vtkPoints *points;
    vtkCellArray *cells;
    vtkFloatArray *scalars;
    
    unsigned int cc = 0;
  
    // Loop through each point in poloidal group
    for( unsigned int i=0; i<nodes.size(); ++i )
    {      
      if( i % period == 0 )
      {
        //Create groups that represent the toroidial groups.
        points = vtkPoints::New();
        cells = vtkCellArray::New();
        scalars = vtkFloatArray::New();

        unsigned int npts = period < (nodes.size()-i) ?
          period : (nodes.size()-i);
      
        cells->InsertNextCell( npts );
        scalars->Allocate    ( npts );
      
        cc = 0;
      }

      if( ptFlag )
        points->InsertPoint(cc,
                            (float) (i % period) / 50.0,
                            nodes[i].y,
                            nodes[i].z);
      else
        points->InsertPoint(cc, nodes[i].x, nodes[i].y, nodes[i].z);
    
      cells->InsertCellPoint(cc);

      if( color == DATA_PointIndex )
        color_value = i;
      else if( color == DATA_WindingOrder )
        color_value = i / poloidalWindings;
      else if( color == DATA_WindingPointOrder )
        color_value = i % poloidalWindings;
      else if( color == DATA_WindingPointOrderModulo )
        color_value =  (i % poloidalWindings) % nnodes;
          
      scalars->InsertTuple1(cc, color_value);
        
      ++cc;
            
      if( i % period == 0 )
      {
        // Create a new VTK polyline.
        vtkPolyData *pd = vtkPolyData::New();
        pd->SetPoints(points);
        pd->SetLines(cells);
        scalars->SetName("colorVar");
        pd->GetPointData()->SetScalars(scalars);
        
        append->AddInput(pd);
        
        points->Delete();
        cells->Delete();
        scalars->Delete();       
      }
    }
  }

  if (0 & showPoints)
  {
    // Loop through each poloidal group
    // Loop through each point in poloidial group
    for( unsigned int i=0; i<nodes.size(); ++i )
    {
      double pt[3] = { nodes[i].x, nodes[i].y, nodes[i].z };
      
      if( ptFlag )
        pt[0] = (float) (i % period) / 50.0;
      else
        pt[0] = nodes[i].x;
          
      double rad = 0.0025;

      if( color == DATA_PointIndex )
        color_value = i;
      else if( color == DATA_WindingOrder )
        color_value = i / poloidalWindings;
      else if( color == DATA_WindingPointOrder )
        color_value = i % poloidalWindings;
      else if( color == DATA_WindingPointOrderModulo )
        color_value =  (i % poloidalWindings) % nnodes;

      if( colorMax < color_value )
        colorMax = color_value;
      
      vtkPolyData *ball = CreateSphere(color_value, rad, pt);
      
      append->AddInput(ball);
      ball->Delete();
    }
  }

  if (showPoints)
  {
    //Create groups that represent the toroidial groups.
    vtkPoints *points = vtkPoints::New();
    vtkCellArray *cells = vtkCellArray::New();
    vtkFloatArray *scalars = vtkFloatArray::New();

    scalars->Allocate( nodes.size() );

    // Loop through each poloidal group
    // Loop through each point in poloidial group
    for( unsigned int i=0; i<nodes.size(); ++i )
    {      
      if( ptFlag )
        points->InsertNextPoint( (float) (i % period) / 50.0,
                                 nodes[i].y,
                                 nodes[i].z);
      else
        points->InsertNextPoint(nodes[i].x, nodes[i].y, nodes[i].z);

      cells->InsertNextCell(1, (vtkIdType*) &i);
      
      if( color == DATA_PointIndex )
        color_value = i;
      else if( color == DATA_WindingOrder )
        color_value = i / poloidalWindings;
      else if( color == DATA_WindingPointOrder )
        color_value = i % poloidalWindings;
      else if( color == DATA_WindingPointOrderModulo )
        color_value =  (i % poloidalWindings) % nnodes;

      if( colorMax < color_value )
        colorMax = color_value;

      scalars->InsertTuple1(i, color_value);
    }
    
    // Create a new VTK point clouds.
    vtkPolyData *pd = vtkPolyData::New();
    pd->SetPoints(points);
    pd->SetVerts(cells);
    scalars->SetName("colorVar");
    pd->GetPointData()->SetScalars(scalars);
    
    append->AddInput(pd);
    
    points->Delete();
    cells->Delete();
    scalars->Delete();       
  }

  append->Update();
  vtkPolyData *outPD = append->GetOutput();
  outPD->Register(NULL);
  outPD->SetSource(NULL);
  append->Delete();
  
  dt->Merge( new avtDataTree(outPD, 0) );
}
