/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//  File: avtTraceHistoryFilter.C
// ************************************************************************* //

#include <avtTraceHistoryFilter.h>

#include <vtkCellType.h>
#include <vtkUnstructuredGrid.h>

#include <vtkVisItUtility.h>

#include <snprintf.h>
#include <DBOptionsAttributes.h>
#include <DebugStream.h>
#include <DatabasePluginManager.h>
#include <DatabasePluginInfo.h>
#include <Expression.h>
#include <ParsingExprList.h>

#include <avtSourceFromAVTDataset.h>
#include <avtExpressionEvaluatorFilter.h>
#include <avtDisplaceFilter.h>
#include <avtDatabaseWriter.h>
#include <avtDatabaseMetaData.h>


// ****************************************************************************
//  Method: avtTraceHistoryFilter constructor
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Sun Apr 8 17:04:24 PST 2007
//
// ****************************************************************************

avtTraceHistoryFilter::avtTraceHistoryFilter()
{
}


// ****************************************************************************
//  Method: avtTraceHistoryFilter destructor
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Sun Apr 8 17:04:24 PST 2007
//
//  Modifications:
//
// ****************************************************************************

avtTraceHistoryFilter::~avtTraceHistoryFilter()
{
}


// ****************************************************************************
//  Method:  avtTraceHistoryFilter::Create
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Sun Apr 8 17:04:24 PST 2007
//
// ****************************************************************************

avtFilter *
avtTraceHistoryFilter::Create()
{
    return new avtTraceHistoryFilter();
}


// ****************************************************************************
//  Method:      avtTraceHistoryFilter::SetAtts
//
//  Purpose:
//      Sets the state of the filter based on the attribute object.
//
//  Arguments:
//      a        The attributes to use.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Sun Apr 8 17:04:24 PST 2007
//
// ****************************************************************************

void
avtTraceHistoryFilter::SetAtts(const AttributeGroup *a)
{
    atts = *(const TraceHistoryAttributes*)a;
}


// ****************************************************************************
//  Method: avtTraceHistoryFilter::Equivalent
//
//  Purpose:
//      Returns true if creating a new avtTraceHistoryFilter with the given
//      parameters would result in an equivalent avtTraceHistoryFilter.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Sun Apr 8 17:04:24 PST 2007
//
// ****************************************************************************

bool
avtTraceHistoryFilter::Equivalent(const AttributeGroup *a)
{
    return (atts == *(TraceHistoryAttributes*)a);
}


// ****************************************************************************
//  Method: avtTraceHistoryFilter::Execute
//
//  Purpose:
//      Sends the specified input and output through the TraceHistory filter.
//
//  Arguments:
//      in_ds      The input dataset.
//      <unused>   The domain number.
//      <unused>   The label.
//
//  Returns:       The output dataset.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Sun Apr 8 17:04:24 PST 2007
//
// ****************************************************************************

void
avtTraceHistoryFilter::Execute(void)
{
    //
    // Write out the connectivity.  "-1" is the hint to do that.
    //
    OutputTime(GetTypedInput(), -1);

    //
    // Pull out just the points.
    //
    avtDataset_p ds = InitializeDataset();

    //
    // Put the fields from the current time step on the points and then
    // output those fields.
    //
    PerformIteration(0, ds, false);
    OutputTime(ds, 0);

    //
    // Now iterate through the time slices, displacing and evaluating.
    //
    for (int i = 0 ; i < atts.GetNumiter() ; i++)
    {
        ds->SetSource(GetInput()->GetSource());
        PerformIteration(i, ds, true);
        OutputTime(ds, i+1);
    }

    // The operator just passes the data through.
    GetDataTree() = GetInputDataTree();
}


// ****************************************************************************
//  Method: avtTraceHistoryFilter::AdditionalPipelineFilters
//
//  Purpose:
//      Does a best guess at the number of substages, in an attempt to keep
//      the progress as accurate.
//
//  Programmer: Hank Childs
//  Creation:   April 8, 2007
//
// ****************************************************************************

int
avtTraceHistoryFilter::AdditionalPipelineFilters(void)
{
    return atts.GetNumiter()*(5+atts.GetVars().size());
}


// ****************************************************************************
//  Method: avtTraceHistoryFilter::InitializeDataset
//
//  Purpose:
//      Takes the input data set and extracts its points.  Also dumps out a 
//      "root"-file and the connectivity information.
//
//  Programmer: Hank Childs
//  Creation:   April 8, 2007
//
// ****************************************************************************

avtDataset_p
avtTraceHistoryFilter::InitializeDataset(void)
{
    int  i, j;

    avtDataTree_p inTree = GetInputDataTree();
    int nLeaves;
    vtkDataSet **ds = inTree->GetAllLeaves(nLeaves);
    vtkDataSet **ugrids = new vtkDataSet*[nLeaves];
    for (i = 0 ; i < nLeaves ; i++)
    {
        vtkPoints *pts = vtkVisItUtility::GetPoints(ds[i]);

        vtkUnstructuredGrid *ugrid = vtkUnstructuredGrid::New();
        ugrid->SetPoints(pts);
        int npts = pts->GetNumberOfPoints();
        ugrid->Allocate(2*npts);
        vtkIdType onevertex[1];
        for (j = 0 ; j < npts ; j++)
        {
             onevertex[0] = j;
             ugrid->InsertNextCell(VTK_VERTEX, 1, onevertex);
        }
        ugrids[i] = ugrid;

        pts->Delete();
    }

    avtDataTree_p newTree = new avtDataTree(nLeaves, ugrids, -1);
    for (i = 0 ; i < nLeaves ; i++)
        ugrids[i]->Delete();
    delete [] ugrids;

    avtDataset_p rv = new avtDataset(GetTypedInput(), newTree);
    avtDataAttributes &atts = rv->GetInfo().GetAttributes();
    int nVars = atts.GetNumberOfVariables();
    while (nVars > 0)
    {
        atts.RemoveVariable(atts.GetVariableName(0));
        nVars = atts.GetNumberOfVariables();
    }

    return rv;
}


// ****************************************************************************
//  Method: avtTraceHistoryFilter::OutputTime
//
//  Purpose:
//      Outputs a time slice.
//
//  Programmer: Hank Childs
//  Creation:   May 22, 2007
//
//  Modifications:
//    Brad Whitlock, Mon Aug 18 09:32:03 PDT 2008
//    Made it directly include the ExtrudedVol writer instead of accessing
//    it via the plugin managers since those are now off limits. The AVT pipeline
//    probably should not be aware of plugins anyway.
//
// ****************************************************************************

#include <avtExtrudedVolWriter.h>

void
avtTraceHistoryFilter::OutputTime(avtDataset_p ds, int time)
{
    DBOptionsAttributes *opts = new DBOptionsAttributes();
    opts->SetInt("Time", time);
    opts->SetInt("nTimes", atts.GetNumiter());

    avtExtrudedVolWriter *wrtr = new avtExtrudedVolWriter(opts);

    avtDataObject_p dob;
    CopyTo(dob, ds);
    wrtr->SetInput(dob);
    char filename[1024];
    strcpy(filename, atts.GetOutput().c_str());
    avtDatabaseMetaData md;
    avtMeshMetaData mmd;
    mmd.name = "mesh";
    md.AddMeshes(mmd);
    std::vector<std::string> vars;
    avtVectorMetaData vmd;
    vmd.name = atts.GetDisplacement();
    vmd.meshName = mmd.name;
    md.AddVectors(vmd);

    //vars.push_back(vmd.name);
    wrtr->Write(filename, &md, vars, false);
    delete wrtr;
    delete opts;
}


// ****************************************************************************
//  Method: avtTraceHistoryFilter::AddExpression
//
//  Purpose:
//      Adds an expression to the list of expressions used by the expression
//      evaluator filter.
//
//  Programmer: Hank Childs
//  Creation:   May 22, 2007
//
// ****************************************************************************

void
AddExpression(ExpressionList &new_list, std::string &var, int time)
{
    char exp_name[1024];
    SNPRINTF(exp_name, 1024, "AVT_TRACE_HIST_%s", var.c_str());
    char exp_defn[1024];
    const char *meshname = var.c_str(); // The expression language will let
                                        // you put in a variable for the
                                        // meshname since it can use that to 
                                        // find the meshname.
    meshname = "quadmesh";
    SNPRINTF(exp_defn, 1024, "pos_cmfe(<[%d]i:%s>, %s, 0.)", time, 
                                                var.c_str(), meshname);

    Expression exp;
    exp.SetName(exp_name);
    exp.SetDefinition(exp_defn);
    exp.SetType(Expression::Unknown);
    new_list.AddExpressions(exp);
}


// ****************************************************************************
//  Method: avtTraceHistoryFilter::PerformIteration
//
//  Purpose:
//      Move the points forward by the displacement vector, evaluate the fields
//      at their position, and then output them.
//
//  Programmer: Hank Childs
//  Creation:   May 22, 2007
//
// ****************************************************************************

void
avtTraceHistoryFilter::PerformIteration(int time, avtDataset_p &ds, bool doDisp)
{
    int   i;

    // Store off the original expression list.
    ParsingExprList *pel = ParsingExprList::Instance();
    ExpressionList orig_list = *(pel->GetList());

    avtContract_p spec = GetGeneralContract();
    avtDataRequest_p dataRequest = spec->GetDataRequest();
    int timestep = dataRequest->GetTimestep();

    int newTimeIndex = timestep-time;
    if (newTimeIndex < 0)
    {
        EXCEPTION1(VisItException, "You asked to trace back before the first"
                      " dump.");
    }

    ExpressionList new_list = *(pel->GetList());
    std::vector<std::string> vars = atts.GetVars();
    for (i = 0 ; i < vars.size () ; i++)
    {
        AddExpression(new_list, vars[i], newTimeIndex);
    }
    if (doDisp)
        AddExpression(new_list, atts.GetDisplacement(), newTimeIndex);
    *(pel->GetList()) = new_list;

    avtSourceFromAVTDataset termsrc(ds);
    avtDataObject_p data = termsrc.GetOutput();

    avtExpressionEvaluatorFilter eef;
    eef.SetInput(data);
    
    avtDataRequest_p dataRequest2 = new avtDataRequest(dataRequest);
    for (i = 0 ; i < vars.size() ; i++)
    {
        char exp_name[1024];
        SNPRINTF(exp_name, 1024, "AVT_TRACE_HIST_%s", vars[i].c_str());
        dataRequest2->AddSecondaryVariable(exp_name);
    }

    if (doDisp)
    {
        char disp_name[1024];
        SNPRINTF(disp_name, 1024, "AVT_TRACE_HIST_%s", 
                                               atts.GetDisplacement().c_str());

        avtDisplaceFilter disp;
        disp.SetInput(eef.GetOutput());

        disp.SetVariable(disp_name);
        disp.SetFactor(1.);
        dataRequest2->AddSecondaryVariable(disp_name);

        avtContract_p spec2 = new avtContract(spec, 
                                                                       dataRequest2);
        disp.Update(spec2);
        ds = disp.GetTypedOutput();
    }
    else
    {
        avtContract_p spec2 = new avtContract(spec, 
                                                                       dataRequest2);
        eef.Update(spec2);
        ds = eef.GetTypedOutput();
    }

    // Restore the original expression list ... i.e. undo the temporary 
    // expressions we put in.
    *(pel->GetList()) = orig_list;
}


