/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//  File: avtPersistentParticlesFilter.C
// ************************************************************************* //

#include <avtPersistentParticlesFilter.h>
#include <ImproperUseException.h>
#include <vtkDataSet.h>
#include <vtkUnstructuredGrid.h>
#include <vtkPointData.h>
#include <vtkCellType.h>
#include <vtkIdList.h>
#include <vtkPoints.h>
#include <vtkFloatArray.h>
#include <vtkCellArray.h>
#include <vtkCellData.h>

//Use of named selection
#include <avtCallback.h>
#include <avtContract.h>
#include <avtParallel.h>
#include <avtIdentifierSelection.h>


// ****************************************************************************
//  Method: avtPersistentParticlesFilter constructor
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri Jan 25 11:02:55 PDT 2008
//
// ****************************************************************************

avtPersistentParticlesFilter::avtPersistentParticlesFilter()
{
    particlePathData = NULL;
    haveData = true;
}


// ****************************************************************************
//  Method: avtPersistentParticlesFilter destructor
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri Jan 25 11:02:55 PDT 2008
//
//  Modifications:
//
// ****************************************************************************

avtPersistentParticlesFilter::~avtPersistentParticlesFilter()
{
      if( particlePathData ){
            particlePathData->Delete();
            particlePathData = NULL;
      }
      particlePaths.clear();
}


// ****************************************************************************
//  Method:  avtPersistentParticlesFilter::Create
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri Jan 25 11:02:55 PDT 2008
//
// ****************************************************************************

avtFilter *
avtPersistentParticlesFilter::Create()
{
    return new avtPersistentParticlesFilter();
}


// ****************************************************************************
//  Method:      avtPersistentParticlesFilter::SetAtts
//
//  Purpose:
//      Sets the state of the filter based on the attribute object.
//
//  Arguments:
//      a        The attributes to use.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri Jan 25 11:02:55 PDT 2008
//
// ****************************************************************************

void
avtPersistentParticlesFilter::SetAtts(const AttributeGroup *a)
{
    atts = *(const PersistentParticlesAttributes*)a;
    SetTimeLoop(atts.GetStartIndex(), atts.GetStopIndex(), atts.GetStride());
}


// ****************************************************************************
//  Method: avtPersistentParticlesFilter::Equivalent
//
//  Purpose:
//      Returns true if creating a new avtPersistentParticlesFilter with the given
//      parameters would result in an equivalent avtPersistentParticlesFilter.
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri Jan 25 11:02:55 PDT 2008
//
// ****************************************************************************

bool
avtPersistentParticlesFilter::Equivalent(const AttributeGroup *a)
{
    return (atts == *(PersistentParticlesAttributes*)a);
}

// ****************************************************************************
//  Method: avtPersistentParticlesFilter::ExamineContact
//
//  Purpose: 
//    Examine the contract to get the current state of the time slider. The time slider 
//    state is needed in case that the start and end time are relative to the time slider. 
//
//  Programmer: Oliver Ruebel
//  Creation:   May 07, 2009
//
//    Oliver Ruebel, Thu May 11 10:50
//
// ****************************************************************************
void
avtPersistentParticlesFilter::ExamineContract(avtContract_p in_contract)
{
    //Call the examine contract function of the super classes first
    avtPluginFilter::ExamineContract(in_contract);
    avtExecuteThenTimeLoopFilter::ExamineContract(in_contract);
    //Save the current timestep
    activeTimeStep = in_contract->GetDataRequest()->GetTimestep();
}

// ****************************************************************************
//  Method: avtPersistentParticlesFilter::Execute
//
//  Purpose: 
//    Defines what it means for this filter to "Execute". This is where the
//    actual iteration over time happens. This functions is overwritten here
//    to allow the dynamic setting of start and end-time of the iteration.
//    The iteration over time is performed using avtExecuteThenTimeLoopFilter::Execute(void)
//
//  Programmer: Oliver Ruebel
//  Creation:   May 07, 2009
//
//
//  Modifications:
//  
//    Hank Childs, Thu Jan  7 16:14:55 PST 2010
//    Add an exception if there are transform operators above stream in the
//    pipeline.
//
// ****************************************************************************

void
avtPersistentParticlesFilter::Execute(void)
{
    if ( (! GetInput()->GetInfo().GetValidity().GetSpatialMetaDataPreserved()) ||
         (! GetInput()->GetInfo().GetValidity().GetZonesPreserved()) ||
         (! GetInput()->GetInfo().GetValidity().GetNodesPreserved()))
    {
        std::string msg("The PersistentParticle operator is implemented "
                        "such that it must be the first operator applied."
                        "  Please try again, moving all operators after "
                        "PersistentParticles in the pipeline.");
        EXCEPTION1(ImproperUseException, msg);
    }

    int numStates = GetInput()->GetInfo().GetAttributes().GetNumStates(); 
    int myStart = atts.GetStartIndex();
    int myStop  = atts.GetStopIndex();

    if( atts.GetStartIndex() < 0 )
    {
      std::string msg(GetType());
      msg = msg + ": Start index/Number of slices must be positive.";
      EXCEPTION1(ImproperUseException, msg);
    }

    if( atts.GetStopIndex() < 0 )
    {
      std::string msg(GetType());
      msg = msg + ": Stop index/Number of slices must be positive.";
      EXCEPTION1(ImproperUseException, msg);
    }

    // If the path type is absolute.
    if( atts.GetStartPathType() == PersistentParticlesAttributes::Absolute )
    {
        myStart = atts.GetStartIndex();
    }

    // If the path type is relative to the current time slider state
    else //if( atts.GetStartPathType() == PersistentParticlesAttributes::Relative )
    {
        int startOffset = atts.GetStartIndex();
        myStart = activeTimeStep - startOffset;
    }

    // If the path type is absolute.
    if( atts.GetStopPathType() == PersistentParticlesAttributes::Absolute )
    {
         myStop  = atts.GetStopIndex();
    }

    // If the path type is relative to the current time slider state
    else //if( atts.GetStopPathType()  == PersistentParticlesAttributes::Relative )
    {
        int stopOffset = atts.GetStopIndex();
        myStop = activeTimeStep + stopOffset;
    }

    // Check if the times are valid and correct if needed
    if (myStart < 0)
      myStart = 0;
    
    if( myStart >= numStates )
      myStart = numStates-1;
    
    if( myStop < 0 )
      myStop = 0;
    
    if (myStop >= numStates)
      myStop = numStates-1;
    
//     if( myStart == myStop )
//     {
//       std::string msg(GetType());
//       msg = msg + ":  Start and stop indexes are the same. The start time " +
//         "must be smaller than the stop time.";
//       avtCallback::IssueWarning(msg.c_str());
//     }

    //Update the start and end frame 
    SetStartFrame(myStart);
    SetEndFrame(myStop);

    //Iterate over the time series
    avtExecuteThenTimeLoopFilter::Execute();
}


// ****************************************************************************
//  Method: avtPersistentParticlesFilter::InspectPrincipalData
//
//  Purpose:
//      This is where the particles are examined.  This allows us to see
//      which particles have been "thresholded in".
//
//  Programmer: childs -- generated by xml2avt
//  Creation:   Fri Jan 25 11:02:55 PDT 2008
//
// ****************************************************************************

void
avtPersistentParticlesFilter::InspectPrincipalData(void)
{
    // No-op right now
}


// ****************************************************************************
//  Method: avtPersistentParticlesFilter::Iterate
//
//  Purpose:
//      This method is the mechanism where the base class shares the data from
//      the current time slice with the derived type. Here the operator adds
//      the information from the current time slice to the new dataset with
//      the particle paths. 
//
//  Programmer: Hank Childs
//  Creation:   January 25, 2008
//
//  Modifications:
//
//    Oliver Ruebel, Mon Mar 28 2009
//    Clean-up. Removed cout statements used for debugging and added comments
//
//    Kathleen Bonnell, Mon Apr 20 17:49:52 MST 2009
//    Use vtk's SafeDownCast method instead of dynamic_cast.
//
//    Hank Childs, Sat Nov 21 13:29:21 PST 2009
//    Remove unneeded call to GetNumberOfZones (found while adding support 
//    for long longs).
//
//    Oliver Ruebel, Thu Dec. 03 2009
//    Added functionality for replacing data dimensions to allow tracing in
//    arbitrary dimensions
//
//    Oliver Ruebel, Mon Dec. 07 2009
//    Splitted function inot IterateMergeData and IterateTraceData to ease
//    readability
//
// ****************************************************************************
void
avtPersistentParticlesFilter::Iterate(int ts, avtDataTree_p tree)
{
    if( ! atts.GetConnectParticles() ){
            IterateMergeData( ts , tree );
    }else{
            IterateTraceData( ts , tree );
    }
}


// ****************************************************************************
//  Method: avtPersistentParticlesFilter::IterateMergeData
//
//  Purpose:
//      Merge the data of the current time slice with the previous ones.
//
//  Arguments:
//       ts int : The input timestep.
//       tree avtDataTree_p : The input dataset.
//
//  Programmer: Oliver Ruebel
//  Creation:   December 07, 2009
//
//  Modifications:
//
//    Hank Childs, Tue Jan  5 15:32:52 PST 2010
//    Add support for parallel and also for polydata.
//
// ****************************************************************************
void
avtPersistentParticlesFilter::IterateMergeData(int ts, avtDataTree_p tree)
{
    //Merge the datasets but do not connect the particles
    trees.push_back(tree);

    //Define whether any dimensions needs to be replaced
    bool replaceX = (atts.GetTraceVariableX() != "default");
    bool replaceY = (atts.GetTraceVariableY() != "default");
    bool replaceZ = (atts.GetTraceVariableZ() != "default");
    //We are done if data only needs to be merged
    if( !(replaceX || replaceY || replaceZ) ){
        return;
    }
    //If data is only merged (i.e. no paths are computed) but at least one
    //variable needs to be replaced, then replace the variable and return.

    //Get the needed data.
    vtkPointSet*         uGrid     =0;
    vtkPoints*           currPoints=0;
    vtkDataArray*        currXData =0;
    vtkDataArray*        currYData =0;
    vtkDataArray*        currZData =0;

    //Ask for the dataset
    vtkDataSet **dsets;
    int nds;
    dsets = tree->GetAllLeaves(nds);
    int nds2 = nds;
    SumIntAcrossAllProcessors(nds2);
    if (nds2 < 1 || nds > 1)
    {
        EXCEPTION1(ImproperUseException, "Filter expected only one vtkDataSet"
                                         " in avtDataTree");
    }

    if (nds == 0)
    {
        haveData = false;
        return;
    }

    vtkDataSet *currDs = dsets[0];
    uGrid = vtkPointSet::SafeDownCast(currDs);
    if (uGrid == 0)
    {
        EXCEPTION1(ImproperUseException, "avtPersistentParticlesFilter only supports "
                                         "vtkPointSet data");
    }
    //Get the point data from the current timestep
    currPoints = uGrid->GetPoints();
    //Get the data to be used as x variable
    currXData = 0;
    if( replaceX ){
            currXData = uGrid->GetPointData()->GetArray( atts.GetTraceVariableX().c_str() );
            if( currXData == 0 )  EXCEPTION1(ImproperUseException, "X coordinate variable not found.");

    }
    //Get the data to be used as y variable
    currYData = 0;
    if( replaceY  ){
            currYData = uGrid->GetPointData()->GetArray( atts.GetTraceVariableY().c_str() );
            if( currYData == 0 )  EXCEPTION1(ImproperUseException, "Y coordinate variable not found.");
    }
    //Get the data to be used as z variable
    currZData = 0;
    if( atts.GetTraceVariableZ() != "default" ){
            currZData = uGrid->GetPointData()->GetArray( atts.GetTraceVariableZ().c_str() );
            if( currZData == 0 )  EXCEPTION1(ImproperUseException, "Z coordinate variable not found.");
    }

    //Replace the requested data dimensions
    unsigned int nPoints = uGrid->GetNumberOfPoints();
    for (unsigned int i = 0; i < nPoints; ++i) {
       //Get the next point and update its coordinates if necessary
       double* nextPoint = currPoints->GetPoint(i);
       if (replaceX)
           nextPoint[0] = currXData->GetTuple1(i);
       if (replaceY)
           nextPoint[1] = currYData->GetTuple1(i);
       if (replaceZ)
       nextPoint[2] = currZData->GetTuple1(i);
       currPoints->SetPoint(i , nextPoint);
     }
}


// ****************************************************************************
//  Method: avtPersistentParticlesFilter::IterateTraceData
//
//  Purpose:
//      Merge the data of the current time slice with the previous ones.
//      Trace the particles over time and define their path by connecting
//      the corresponding particles.
//
//  Arguments:
//       ts int : The input timestep.
//       tree avtDataTree_p : The input dataset.
//
//  Programmer: Oliver Ruebel
//  Creation:   December 07, 2009
//
//  Modifications:
//
//    Hank Childs, Tue Jan  5 15:32:52 PST 2010
//    Add support for parallel, polydata, and vector data.
//
// ****************************************************************************
void
avtPersistentParticlesFilter::IterateTraceData(int ts, avtDataTree_p tree)
{
    //Define whether any dimensions needs to be replaced
    bool replaceX = (atts.GetTraceVariableX() != "default");
    bool replaceY = (atts.GetTraceVariableY() != "default");
    bool replaceZ = (atts.GetTraceVariableZ() != "default");

    //Get the needed data.
    vtkPointSet*         uGrid     =0;
    vtkPoints*           currPoints=0;
    vtkDataArray*        currWeight=0;
    vtkDataArray*        currXData =0;
    vtkDataArray*        currYData =0;
    vtkDataArray*        currZData =0;
    vtkPointData*        currData  =0;

    //Ask for the dataset
    vtkDataSet **dsets;
    int nds;
    dsets = tree->GetAllLeaves(nds);
    int nds2 = nds;
    SumIntAcrossAllProcessors(nds2);
    if (nds2 < 1 || nds > 1)
    {
        EXCEPTION1(ImproperUseException, "Filter expected only one vtkDataSet"
                                         " in avtDataTree");
    }

    if (nds == 0)
    {
        haveData = false;
        return;
    }

    vtkDataSet *currDs = dsets[0];
    uGrid = vtkPointSet::SafeDownCast(currDs);
    if (uGrid == 0)
    {
        EXCEPTION1(ImproperUseException, "avtPersistentParticlesFilter only supports "
                                         "vtkPointSets data");
    }

    //Get the point data from the current timestep
    currPoints = uGrid->GetPoints();
    //Get the ID variable. Use the mainVariable if default is specified
    currWeight = 0;
    if( atts.GetIndexVariable() != "default" ){
        currWeight = uGrid->GetPointData()->GetArray( atts.GetIndexVariable().c_str() );
    }else{
          currWeight = uGrid->GetPointData()->GetArray( mainVariable.c_str() );
    }
    if (currWeight == 0){
        EXCEPTION1(ImproperUseException, "Index variable not found.");
    }
    //Get the data to be used as x variable
    currXData = 0;
    if( replaceX ){
            currXData = uGrid->GetPointData()->GetArray( atts.GetTraceVariableX().c_str() );
            if( currXData == 0 )  EXCEPTION1(ImproperUseException, "X coordinate variable not found.");

    }
    //Get the data to be used as y variable
    currYData = 0;
    if( replaceY  ){
            currYData = uGrid->GetPointData()->GetArray( atts.GetTraceVariableY().c_str() );
            if( currYData == 0 )  EXCEPTION1(ImproperUseException, "Y coordinate variable not found.");
    }
    //Get the data to be used as z variable
    currZData = 0;
    if( atts.GetTraceVariableZ() != "default" ){
            currZData = uGrid->GetPointData()->GetArray( atts.GetTraceVariableZ().c_str() );
            if( currZData == 0 )  EXCEPTION1(ImproperUseException, "Z coordinate variable not found.");
    }
    //Get the current PointData
    currData   = uGrid->GetPointData();

    //Create a new output dataset if needed
    //If first timestep or if output dataset has not been created yet
    if( ts == atts.GetStartIndex() || !particlePathData ){
          //Delete the current output dataset if necessary
          if( particlePathData ){
            particlePathData->Delete();
            particlePathData = NULL;
          }
          //Create and initalize the new dataset
          particlePathData = vtkUnstructuredGrid::New();
          particlePathData->SetPoints( vtkPoints::New() );
          vtkPointData* allData = uGrid->GetPointData();
          particlePathData->GetPointData()->ShallowCopy(allData);
    }

    //Write the current particles into a map to decide which ones should be traced
    //Check if particle ID is unique, if not then stop tracing of that particle
    int nPoints = uGrid->GetNumberOfPoints();
    std::map<double , bool> trace;
    int numSkipped = 0;
    for( unsigned int i=0 ; i<nPoints ;++i )
    {
              //if particle ID is not already in the map then true, else false
            std::map<double , bool >::iterator fP = trace.find( currWeight->GetTuple1(i) );
            trace[currWeight->GetTuple1(i)] = (fP==trace.end());
            if( fP!=trace.end() ){
                  numSkipped++;
            }
    }

    //Traverse all points
    for( unsigned int i=0 ; i<nPoints ; ++i ) {
          //trace only if particle id is unique
          if( trace[ currWeight->GetTuple1(i)] )
          {
                //Get the current particle path if it is already defined
                std::map<double , int >::iterator lastPoint = particlePaths.find( currWeight->GetTuple1(i) );
                //Get the next point and update its coordinates if necessary
                    double* nextPathPoint = currPoints->GetPoint(i);
                    if( replaceX )
                            nextPathPoint[0] = currXData->GetTuple1(i);
                    if( replaceY )
                            nextPathPoint[1] = currYData->GetTuple1(i);
                    if( replaceZ )
                            nextPathPoint[2] = currZData->GetTuple1(i);
                //Add the point to the map
                    vtkPoints* pathPoints = particlePathData->GetPoints();
                    pathPoints->InsertNextPoint( nextPathPoint );
                particlePathData->SetPoints( pathPoints );

                //The index of the new point
                int  newPointIndex = particlePathData->GetPoints()->GetNumberOfPoints()-1;

                //Copy the pointdata from the input mesh to the output mesh
                vtkPointData* allData = particlePathData->GetPointData();
                for( int j=0 ; j<allData->GetNumberOfArrays() ; j++) {
                      allData->GetArray(j)->InsertTuple(
                      newPointIndex  ,
                              currData->GetArray(j)->GetTuple(i)
                      );
                }

                //add a new line segment
                if( lastPoint != particlePaths.end() ){
                      //define the points of the lines
                      int* pointList = new int[2];
                      pointList[0]   = (*lastPoint).second;
                      pointList[1]   = newPointIndex;
                      //add a new line segment
                      particlePathData->InsertNextCell( VTK_LINE , 2 , pointList );
                      delete[] pointList;
                }

                //Update the map
                particlePaths[ currWeight->GetTuple1(i) ] = newPointIndex;
          }
    }
}

// ****************************************************************************
//  Method: avtPersistentParticlesFilter::Finalize
//
//  Purpose:
//      This method is the mechanism for the base class to tell its derived
//      types that no more time slices are coming and it should put together
//      its final output. This method creates the final output dataset with
//      either all points of the different time slices or the dataset with
//      the particle paths.
//
//  Modifications:
//    Oliver Ruebel, Mo Mar 28 2009
//    Clean-up. Removed cout statements used for debugging and added comments.
//
//
//  Programmer: Hank Childs
//  Creation:   January 25, 2008
//
//  Modifications:
//
//    Hank Childs, Tue Jan  5 15:32:52 PST 2010
//    Add support for the parallel case (where there is no data on some procs).
//
// ****************************************************************************

void
avtPersistentParticlesFilter::Finalize(void)
{
     if (! haveData)
         return;

      //Merge the data but do not connect the traces of particles
      if( ! atts.GetConnectParticles() ){
          avtDataTree_p newTree = new avtDataTree(trees.size(), &(trees[0]));
          SetOutputDataTree(newTree);
          trees.clear();
      }
      else{
            //Find the index of the main variable in the new path datasets
            int index =0;
            vtkPointData* pointData = particlePathData->GetPointData();
            //find the main variable
            for( int i=0 ; i<pointData->GetNumberOfArrays() ; ++i ){
                  if( mainVariable.compare( pointData->GetArray(i)->GetName() ) == 0 ){
                        index =i;
                        break;
                  }
            }
            //Set the according array to be the active array
            pointData->SetScalars( pointData->GetArray(index) );
            avtDataTree_p newTree = new avtDataTree( particlePathData , 0 );
            SetOutputDataTree(newTree);
            particlePaths.clear(); //Clear the map for the next run
            particlePaths = std::map<double , int>();
            particlePathData->Delete();
            particlePathData = NULL;
      }
}


// ****************************************************************************
//  Method: avtPersistentParticlesFilter::ModifyContract
//
//  Purpose:
//      This method makes any necessay modification to the VisIt contract
//      to, e.g., request that the index variable is also loaded if it 
//      is not already part of the contract.
//
//  Modifications:
//    Oliver Ruebel, Mo Mar 28 2009
//    Clean-up. Removed cout statements used for debugging and added comments.
//
// ****************************************************************************

avtContract_p
avtPersistentParticlesFilter::ModifyContract(avtContract_p in_contract)
{
    //Create the output contract
    avtContract_p out_contract = new avtContract(in_contract);

    //add the index variable to the contract if necessay
    if( atts.GetIndexVariable() != "default")
       out_contract->GetDataRequest()->AddSecondaryVariable( atts.GetIndexVariable().c_str() );
    //add the tracing variables to the contract if necessary
    if( atts.GetTraceVariableX() != "default")
       out_contract->GetDataRequest()->AddSecondaryVariable( atts.GetTraceVariableX().c_str() );
    if( atts.GetTraceVariableY() != "default")
       out_contract->GetDataRequest()->AddSecondaryVariable( atts.GetTraceVariableY().c_str() );
    if( atts.GetTraceVariableZ() != "default")
       out_contract->GetDataRequest()->AddSecondaryVariable( atts.GetTraceVariableZ().c_str() );
    mainVariable = string( out_contract->GetDataRequest()->GetVariable());
    SetContract(out_contract);
    return out_contract;
}


// ****************************************************************************
//  Method: avtPersistentParticlesFilter::UpdateDataObjectInfo
//
//  Purpose: Update the information about the data object, in this case the
//           topology and spatial dimensionality.
//
//  Modification:
//  Oliver Ruebel, Thu Dec 12 2009
//  Set spatial dimensions to 3D if necessary. This became necessary because
//  the user now can specify in which data dimensions the tracing should be
//  done, i.e., even if the data is 2D the output traces may be 3D (or 2D).
//
// ****************************************************************************
void
avtPersistentParticlesFilter::UpdateDataObjectInfo(void)
{
    //Define the topology of the output data
    avtDataObject_p           out_data_object = GetOutput();
    avtDataObjectInformation &out_data_info   = out_data_object->GetInfo();
    avtDataAttributes        &out_data_atts   = out_data_info.GetAttributes();
    avtDataValidity &out_data_validity = GetOutput()->GetInfo().GetValidity();
    out_data_atts.SetTopologicalDimension(2); //we have lines as output
    out_data_atts.SetCentering( AVT_NODECENT ); //the output is node centered data

    //In case that we have a 2D dataset but the output data is 3D we also
    //need to update the dimensionality of the data
    if( atts.GetTraceVariableZ() != "default" && out_data_atts.GetSpatialDimension()!=3 ){
         //Update the data validity
           out_data_validity.InvalidateSpatialMetaData();
           out_data_validity.InvalidateZones();
           out_data_validity.SetPointsWereTransformed(true);
           //update the spatial dimensions
           out_data_atts.SetSpatialDimension(3);
           out_data_atts.SetCanUseTransform(false);
           if( out_data_atts.HasInvTransform() ){
                   out_data_atts.SetCanUseInvTransform(false);
           }
    }
}
