/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//  File: avtIndexSelectFilter.C
// ************************************************************************* //

#include <avtIndexSelectFilter.h>

#include <vtkCell.h>
#include <vtkCellData.h>
#include <vtkDataSetRemoveGhostCells.h>
#include <vtkMaskPoints.h>
#include <vtkFloatArray.h>
#include <vtkIntArray.h>
#include <vtkPointData.h>
#include <vtkPolyData.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnsignedIntArray.h>
#include <vtkUnstructuredGrid.h>
#include <vtkVisItExtractGrid.h>
#include <vtkVisItExtractRectilinearGrid.h>
#include <vtkVisItUtility.h>

#include <avtCallback.h>
#include <avtLogicalSelection.h>
#include <avtSILNamespace.h>
#include <avtSILRestrictionTraverser.h>
#include <avtOriginatingSource.h>

#include <DebugStream.h>
#include <ImproperUseException.h>
#include <InvalidSetException.h>
#include <InvalidCategoryException.h>
#include <InvalidVariableException.h>
#include <snprintf.h>


// ****************************************************************************
//  Method: avtIndexSelectFilter constructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
//  Modifications:
//
//    Mark C. Miller, Tue Sep 28 19:57:42 PDT 2004
//    Added data selection id
//
//    Kathleen Bonnell, Tue May 10 11:19:24 PDT 2005 
//    Use VisIt versions of vtkExtractGrid and vtkExtractRectilinearGrid, 
//    they have been modified to correctly handle cell data when VOI is
//    along max boundary. 
//
//    Kathleen Bonnell, Mon Jan 30 15:10:26 PST 2006 
//    Add vtkMaskPoints for a points filter. 
// 
//    Kathleen Bonnell, Thu Jun  7 14:37:49 PDT 2007 
//    Added groupCategory.
// 
//    Kathleen Bonnell, Thu Jun 21 16:31:59 PDT 2007 
//    Added amrLevel, amrMesh.
//
// ****************************************************************************

avtIndexSelectFilter::avtIndexSelectFilter()
{
    curvilinearFilter = vtkVisItExtractGrid::New();
    rectilinearFilter = vtkVisItExtractRectilinearGrid::New();
    pointsFilter = vtkMaskPoints::New();
    pointsFilter->GenerateVerticesOn();
    pointsFilter->RandomModeOff();
    haveIssuedWarning = false;
    selID             = -1;
    amrLevel          = -1;
    amrMesh           = false;
    groupCategory = false;
}


// ****************************************************************************
//  Method: avtIndexSelectFilter destructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
//  Modifications:
//    Kathleen Bonnell, Mon Jan 30 15:10:26 PST 2006 
//    Delete vtkMaskPoints.
//
// ****************************************************************************

avtIndexSelectFilter::~avtIndexSelectFilter()
{
    if (curvilinearFilter != NULL)
    {
        curvilinearFilter->Delete();
        curvilinearFilter = NULL;
    }
    if (rectilinearFilter != NULL)
    {
        rectilinearFilter->Delete();
        rectilinearFilter = NULL;
    }
    if (pointsFilter != NULL)
    {
        pointsFilter->Delete();
        pointsFilter = NULL;
    }
}


// ****************************************************************************
//  Method:  avtIndexSelectFilter::Create
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
// ****************************************************************************

avtFilter *
avtIndexSelectFilter::Create()
{
    return new avtIndexSelectFilter();
}


// ****************************************************************************
//  Method:      avtIndexSelectFilter::SetAtts
//
//  Purpose:
//      Sets the state of the filter based on the attribute object.
//
//  Arguments:
//      a        The attributes to use.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
// ****************************************************************************

void
avtIndexSelectFilter::SetAtts(const AttributeGroup *a)
{
    atts = *(const IndexSelectAttributes*)a;
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::PrepareFilters
//
//  Purpose:
//      Prepares the curvilinear and rectilinear filters for the current
//      dataset.  This can take into account block indices.
//
//  Programmer: Hank Childs
//  Creation:   June 25, 2002
//
//  Modifications:
//    Kathleen Bonnell, Wed Sep  8 09:36:30 PDT 2004
//    Always Set IncludeBoundary to true for filters, so they can handle
//    modulo prolbems (eg sample rate of 3, but dimension is 10).
//
//    Kathleen Bonnell, Wed Jul 20 11:39:34 PDT 2005 
//    Don't subtract 1 from the groupIndices. 
//
//    Kathleen Bonnell, Mon Jan 30 15:10:26 PST 2006 
//    Setup vtkMaskPoints.
//
//    Kathleen Bonnell, Thu Jun  7 14:37:49 PDT 2007 
//    Use groupCategory insead of WhichData == OneGroup.
// 
//    Kathleen Bonnell, Thu Jun 21 16:31:59 PDT 2007 
//    Added int* amri arg.  Read from this arg if AMR mesh in order to
//    determine how to correctly compute min/max.
//
// ****************************************************************************

void
avtIndexSelectFilter::PrepareFilters(int groupIndices[3], int *amri)
{
    //
    // Only adjust the base index if we are index selecting by group.
    //
    int bi[3];
    if (groupCategory)
    {
        bi[0] = groupIndices[0];
        bi[1] = groupIndices[1];
        bi[2] = groupIndices[2];
    }
    else
    {
        bi[0] = 0;
        bi[1] = 0;
        bi[2] = 0;
    }

    int voi[6];

    int minmax[6] = {atts.GetXMin(), atts.GetXMax(), atts.GetYMin(),
                     atts.GetYMax(), atts.GetZMin(), atts.GetZMax()};

    if (amrMesh && groupCategory)
    {
        int OP = amri[3];
        for (int i = 0; i < 3; i++)
        {
            if (amri[i] > 1)
            {
                if (OP == 0)
                {
                    // min
                    minmax[i*2] = minmax[i*2]*amri[i]; 
                    // max
                    minmax[i*2+1] = minmax[i*2+1]*amri[i]; 
                }
                else 
                {
                    // min
                    minmax[i*2] = minmax[i*2]/amri[i]; 
                    // max
                    if (minmax[i*2+1] != -1)
                    minmax[i*2+1] = minmax[i*2+1]/amri[i] + 1; 
                }
            }
        }
    }
  
    voi[0] = (minmax[0] - bi[0] < 0 ? 0 : minmax[0] - bi[0]);
    voi[1] = (minmax[1] < 0 ? 1000000   : minmax[1] - bi[0]);
    if (atts.GetDim() == IndexSelectAttributes::TwoD ||
            atts.GetDim() == IndexSelectAttributes::ThreeD)
    {
        voi[2] = (minmax[2] - bi[1] < 0 ? 0 : minmax[2] - bi[1]);
        voi[3] = (minmax[3] < 0 ? 1000000   : minmax[3] - bi[1]);
    }
    else
    {
        voi[2] = 0;
        voi[3] = 1000000;
    }
    if (atts.GetDim() == IndexSelectAttributes::ThreeD)
    {
        voi[4] = (minmax[4] - bi[2] < 0 ? 0 : minmax[4] - bi[2]);
        voi[5] = (minmax[5] < 0 ? 1000000   : minmax[5] - bi[2]);
    }
    else
    {
        voi[4] = 0;
        voi[5] = 1000000;
    }
    curvilinearFilter->SetVOI(voi);
    rectilinearFilter->SetVOI(voi);
    int sampleRate[3];
    sampleRate[0] = atts.GetXIncr();
    sampleRate[1] = atts.GetYIncr();
    sampleRate[2] = atts.GetZIncr();
    curvilinearFilter->SetSampleRate(sampleRate);
    rectilinearFilter->SetSampleRate(sampleRate);

    pointsFilter->SetOnRatio(sampleRate[0]);
    pointsFilter->SetOffset(voi[0]);
    if (voi[1] != 1000000)
    {
        int maxpts = (voi[1] - voi[0]) / sampleRate[0]; 
        pointsFilter->SetMaximumNumberOfPoints(maxpts);
    }
    else
    {
        pointsFilter->SetMaximumNumberOfPoints(VTK_LARGE_INTEGER);
    }
    curvilinearFilter->SetIncludeBoundary(1);
    rectilinearFilter->SetIncludeBoundary(1);
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::Equivalent
//
//  Purpose:
//      Returns true if creating a new avtIndexSelectFilter with the given
//      parameters would result in an equivalent avtIndexSelectFilter.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
// ****************************************************************************

bool
avtIndexSelectFilter::Equivalent(const AttributeGroup *a)
{
    return (atts == *(IndexSelectAttributes*)a);
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::ExecuteData
//
//  Purpose:
//      Sends the specified input and output through the IndexSelect filter.
//
//  Arguments:
//      in_ds      The input dataset.
//      <unused>   The domain number.
//      <unused>   The label.
//
//  Returns:       The output dataset.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Jun 5 09:09:11 PDT 2002
//
//  Modifications:
//
//    Hank Childs, Wed Jun 19 09:47:01 PDT 2002
//    Fix stupid crash when applying to unstructured grids.
//
//    Hank Childs, Tue Jun 25 18:22:35 PDT 2002
//    Use group indices when calculating indices.
//
//    Hank Childs, Wed Oct  2 10:13:26 PDT 2002
//    Added support for meshes that have already been broken up.
//
//    Hank Childs, Mon Dec  9 08:53:15 PST 2002
//    Account for potentially changing format of avtOriginalCellNumbers.
//
//    Kathleen Bonnell, Fri Dec 13 16:41:12 PST 2002   
//    Use NewInstance instead of MakeObject in order to match vtk's new api. 
//    
//    Jeremy Meredith, Fri Jan 30 17:45:47 PST 2004
//    Added code to preserve dataset type if the input was polydata.
//
//    Hank Childs, Fri Aug 27 15:25:22 PDT 2004
//    Rename ghost data array.
//
//    Mark C. Miller, Tue Sep 28 19:57:42 PDT 2004
//    Added code to bypass the operator if the selection is applied by a
//    plugin
//
//    Kathleen Bonnell, Fri Feb 18 09:41:16 PST 2005 
//    Account for the fact the vtkExtractGrid and vtkRectilinearExtractGrid
//    may return 'empty' datasets, so we want to return a NULL dataset. 
//
//    Kathleen Bonnell, Fri Aug 19 15:45:46 PDT 2005
//    Remove ghost nodes array, ensure ghost zones are skipped in
//    'zones not preserved' case.   Removed retrieval and use of
//    'avtRealDims'.  Its retrieval was incorrect so it was never ever used.
//
//    Kathleen Bonnell, Mon Jan 30 15:10:26 PST 2006 
//    Use vtkMaskPoints for point meshes.
//
//    Kathleen Bonnell, Wed May 17 10:46:58 PDT 2006
//    Remove call to SetSource(NULL) as it now removes information necessary
//    for the dataset. 
//
//    Kathleen Bonnell, Thu Jun  7 14:37:49 PDT 2007 
//    Use groupCategory insead of WhichData == OneGroup.
// 
//    Kathleen Bonnell, Thu Jun 21 16:31:59 PDT 2007 
//    If this is an AMR mesh, retrieve AMR indices and don't remove ghost zones.
//
// ****************************************************************************

vtkDataSet *
avtIndexSelectFilter::ExecuteData(vtkDataSet *in_ds, int dom, std::string)
{
    vtkDataSet *out_ds = NULL;

    int topoDim = GetInput()->GetInfo().GetAttributes().
                  GetTopologicalDimension();

    amrMesh = (GetInput()->GetInfo().GetAttributes().GetMeshType() == 
                AVT_AMR_MESH);

    //
    // If the selection this filter exists to create has already been handled,
    // then we can skip execution
    //
    if (GetInput()->GetInfo().GetAttributes().GetSelectionApplied(selID)) 
    {
        debug1 << "Bypassing IndexSelect operator because database plugin "
                  "claims to have applied the selection already" << endl;
        successfullyExecuted = true;
        return in_ds;
    }
    else if (GetInput()->GetInfo().GetValidity().GetZonesPreserved())
    {
        //
        // We have the normal case -- a structured mesh that we are going to
        // index select.
        //
        vtkDataSet *ds = NULL;

        //
        // All of our indices are incorrect if we leave the ghost zones in -- 
        // we can also have some weird phenomenon where boundaries are missing
        // between blocks.
        //
        vtkDataSetRemoveGhostCells *removeGhostCells = NULL;
        if (!amrMesh &&  
            in_ds->GetCellData()->GetArray("avtGhostZones"))
        {
            removeGhostCells  = vtkDataSetRemoveGhostCells::New();
            removeGhostCells->SetInput(in_ds);
    
            //
            // There is something buggy about the extents when this filter is 
            // used for repeated executions.  Just force the execution now.
            //
            removeGhostCells->Update();
            ds = removeGhostCells->GetOutput()->NewInstance();
            ds->ShallowCopy(removeGhostCells->GetOutput());
        }
        else
        {
            ds = in_ds;
        }
        in_ds->GetPointData()->RemoveArray("avtGhostNodes");
    
        //
        // The indices should reflect the "base_index"'s, so dummy one up if
        // we don't have one.
        //
        vtkDataArray *arr = in_ds->GetFieldData()->GetArray("base_index");
        int ind[3] = { 0, 0, 0 };
        if (arr != NULL)
        {
            vtkIntArray *ar2 = (vtkIntArray *) arr;
            ind[0] = ar2->GetValue(0);
            ind[1] = ar2->GetValue(1);
            ind[2] = ar2->GetValue(2);
        }

        int *amri = NULL;
        if (amrMesh && groupCategory)
        {
            vtkIntArray *amrdims = (vtkIntArray *)
                in_ds->GetFieldData()->GetArray("avtAMRDimensions");
            if (amrdims == NULL)
            {
                if (!haveIssuedWarning)
                {
                    avtCallback::IssueWarning("An internal error occurred and "
                            "the index select operator was not applied.");
                    haveIssuedWarning = true;
                }
                return in_ds;
            }
            else
            {
                amri = (int*)amrdims->GetVoidPointer(0);
            }
        }
        PrepareFilters(ind, amri);
    
        vtkDataSet *rv = NULL;
        int dstype = ds->GetDataObjectType();
        if (dstype == VTK_STRUCTURED_GRID)
        {
            curvilinearFilter->SetInput((vtkStructuredGrid *) ds);
            curvilinearFilter->Update();
            rv = curvilinearFilter->GetOutput();
        }
        else if (dstype == VTK_RECTILINEAR_GRID)
        {
            rectilinearFilter->SetInput((vtkRectilinearGrid *) ds);
            rectilinearFilter->Update();
            rv = rectilinearFilter->GetOutput();
        }
        else if (topoDim == 0 && 
                 (dstype == VTK_POLY_DATA || dstype == VTK_UNSTRUCTURED_GRID))
        {
            pointsFilter->SetInput(ds);
            pointsFilter->Update();
            rv = pointsFilter->GetOutput();
        }
        else
        {
            if (!haveIssuedWarning)
            {
                avtCallback::IssueWarning("The index select operator was "
                            "applied to a non-structured mesh.  It is not "
                            "being applied.");
                haveIssuedWarning = true;
            }
            return in_ds;
        }

        rv->Update();

        if (removeGhostCells != NULL)
        {
            removeGhostCells->Delete();
        }

        if (rv->GetNumberOfPoints() > 0 && rv->GetNumberOfCells() > 0)
        {
            out_ds = (vtkDataSet *) rv->NewInstance();
            out_ds->ShallowCopy(rv);
            ManageMemory(out_ds);
            out_ds->Delete();
        }
    }
    else
    {
        //
        // The dataset has been changed before it got here -- most likely it
        // was material selected.  We should have passed enough clues
        // downstream to figure out what happened.
        //
        vtkUnsignedIntArray *origZones = (vtkUnsignedIntArray *)
                                         in_ds->GetCellData()->
                                         GetArray("avtOriginalCellNumbers");
        if (origZones == NULL)
        {
            if (!haveIssuedWarning)
            {
                avtCallback::IssueWarning("An internal error occurred and the "
                                          "index select operator was not "
                                          "applied.");
                haveIssuedWarning = true;
            }
            return in_ds;
        }

        vtkUnsignedIntArray *dims = (vtkUnsignedIntArray *)
                                            in_ds->GetFieldData()->GetArray(
                                            "avtOriginalStructuredDimensions");
        if (dims == NULL)
        {
            if (!haveIssuedWarning)
            {
                avtCallback::IssueWarning("An internal error occurred and the "
                                          "index select operator was not "
                                          "applied.");
                haveIssuedWarning = true;
            }
            return in_ds;
        }
        int d[3];
        d[0] = dims->GetValue(0);
        d[1] = dims->GetValue(1);
        d[2] = dims->GetValue(2);

        int base[3] = { 0, 0, 0 };
        if (groupCategory)
        {
            vtkDataArray *arr = in_ds->GetFieldData()->GetArray("base_index");
            if (arr != NULL)
            {
                vtkIntArray *ar2 = (vtkIntArray *) arr;
                base[0] += ar2->GetValue(0);
                base[1] += ar2->GetValue(1);
                base[2] += ar2->GetValue(2);
            }
        }
        
        vtkDataArray *ghosts = in_ds->GetCellData()->GetArray("avtGhostZones");
        unsigned char *gz = NULL;
        if (ghosts)
            gz = (unsigned char *)ghosts->GetVoidPointer(0);

        //
        // We should have everything lined up now -- we know what the original
        // indexing of the structured mesh was and what the base index is.
        //
        vtkUnstructuredGrid *out_ug = vtkUnstructuredGrid::New();
        vtkPoints *p1 = vtkVisItUtility::GetPoints(in_ds);
        out_ug->SetPoints(p1);
        p1->Delete();
        out_ug->GetPointData()->PassData(in_ds->GetPointData());
        vtkCellData *out_cd = out_ug->GetCellData();
        vtkCellData *in_cd  = in_ds->GetCellData();
        out_cd->CopyAllocate(in_cd);
        int ncells = in_ds->GetNumberOfCells();
        out_ug->Allocate(ncells);

        int out_cell = 0;
        int xmin = atts.GetXMin();
        int xmax = atts.GetXMax();
        xmax = (xmax < 0 ? 10000000 : xmax);
        int ymin = atts.GetYMin();
        int ymax = atts.GetYMax();
        ymax = (ymax < 0 ? 10000000 : ymax);
        int zmin = atts.GetZMin();
        int zmax = atts.GetZMax();
        zmax = (zmax < 0 ? 10000000 : zmax);
        int ncomps = origZones->GetNumberOfComponents();
        int comp = ncomps-1;
        for (int i = 0 ; i < ncells ; i++)
        {
            int cell_id = (int) origZones->GetComponent(i, comp);
            int x = cell_id % (d[0]-1);
            int y = (cell_id / (d[0]-1)) % (d[1]-1);
            int z = cell_id / ((d[0]-1)*(d[1]-1));
            x += base[0];
            y += base[1];
            z += base[2];
            if (x < xmin || x >= xmax)
            {
                continue;
            }
            if (atts.GetDim() == IndexSelectAttributes::TwoD ||
                atts.GetDim() == IndexSelectAttributes::ThreeD)
            {
                if (y < ymin || y >= ymax)
                    continue;
            }
            if (atts.GetDim() == IndexSelectAttributes::ThreeD)
            {
                if (z < zmin || z >= zmax)
                    continue;
            }
            // only allow AMR ghosts to be part of the output.
            if (gz && !(gz[i] == 0 || gz[i] == 8))
                continue;

            out_cd->CopyData(in_cd, i, out_cell++);
            vtkCell *cell = in_ds->GetCell(i);
            vtkIdList *list = cell->GetPointIds();
            out_ug->InsertNextCell(in_ds->GetCellType(i), list);
        }
        out_ds = out_ug;

        if (out_ds->GetNumberOfCells() <= 0)
        {
            out_ds = NULL;
        }

        //
        // If we had poly data input, we want poly data output.  The VTK filter
        // only returns unstructured grids, so convert that now.
        //
        bool shouldDelete = false;
        if (in_ds->GetDataObjectType() == VTK_POLY_DATA && out_ds != NULL)
        {
            vtkUnstructuredGrid *ugrid = (vtkUnstructuredGrid *) out_ds;
            vtkPolyData *out_pd = vtkPolyData::New();
            out_pd->SetPoints(ugrid->GetPoints());
            out_pd->GetPointData()->ShallowCopy(ugrid->GetPointData());
            out_pd->GetCellData()->ShallowCopy(ugrid->GetCellData());
            int ncells = ugrid->GetNumberOfCells();
            out_pd->Allocate(ncells);
            for (int i = 0 ; i < ncells ; i++)
            {
                int celltype = ugrid->GetCellType(i);
                vtkIdType *pts;
                int npts;
                ugrid->GetCellPoints(i, npts, pts);
                out_pd->InsertNextCell(celltype, npts, pts);
            }
            out_ds = out_pd;
            shouldDelete = true;
        }

        ManageMemory(out_ds);
        out_ug->Delete();
        if (shouldDelete)
            out_ds->Delete();
    }

    successfullyExecuted = true;

    bool wrap[3];
    wrap[0] = atts.GetXWrap();
    wrap[1] = atts.GetYWrap();
    wrap[2] = atts.GetZWrap();

    if( wrap[0] || wrap[1] || wrap[2] )
      return Replicate( out_ds, wrap );
    else
      return out_ds;
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::ModifyContract
//
//  Purpose:
//      Restricts the SIL to the domains requested by the user.
//
//  Programmer: Hank Childs
//  Creation:   June 5, 2002
//
//  Modifications:
//
//    Hank Childs, Sun Jun 16 20:50:53 PDT 2002
//    Add support for non 0-origin blocks.
//
//    Hank Childs, Mon Sep 30 17:23:33 PDT 2002
//    Add support for index selecting after a destructive operation.
//
//    Hank Childs, Mon Dec  2 09:59:56 PST 2002
//    Account for changing interface for SIL restriction.
//
//    Hank Childs, Thu Aug 14 07:44:49 PDT 2003
//    Also request the structured indices if we are specifically told to do
//    interface reconstruction.
//
//    Mark C. Miller, Tue Sep 28 19:57:42 PDT 2004
//    Added code to build a data selection
//
//    Kathleen Bonnell, Tue Nov 16 16:13:08 PST 2004 
//    Gracefully handle domainIndex that is out-of-range, and issue warning. 
//    Also, use domainIndex when determining chunk for trav.GetMaterials, when
//    appropriate.
//
//    Kathleen Bonnell, Thu Jul 21 07:52:49 PDT 2005 
//    For OneGroup selection, determine domains belonging to that group,
//    and restrict by domains to ensure that sets that were previously
//    turned off remains so.  When determining if StructuredIndices are
//    required, ensure checking is done only on domains needed by this filter. 
//
//    Kathleen Bonnell, Thu Aug  4 15:47:59 PDT 2005 
//    Request original node numbers when required. 
//    
//    Jeremy Meredith, Wed Aug 24 13:37:07 PDT 2005
//    Added support for group origin.
//
//    Kathleen Bonnell, Thu Jun  7 14:37:49 PDT 2007 
//    Modified how SIL selection is done, based on new atts. 
// 
//    Kathleen Bonnell, Thu Jun 21 16:31:59 PDT 2007
//    If this is an AMR mesh and category is group, don't restrict the SIL
//    and also request AMR indices.
// 
//    Hank Childs, Fri Oct 26 16:40:57 PDT 2007
//    Correct some SIL handling.  Compact SIL attributes are only defined
//    over a subset of the nodes in the SIL graph; comparing them without
//    re-indexing ... which was being done previously ... led to incorrect
//    results.
//
//    Hank Childs, Wed Jan  9 16:10:33 PST 2008
//    Beef up logic to handle species selection.  We were turning those sets
//    off, which resulted in all zeroes.
//
//    Dave Bremer, Thu Jan 31 17:52:55 PST 2008
//    Small tweak to guard against a case in which the RealMapsOut are 
//    requested from an avtSILSet, but the set goes out of scope and its maps
//    out are deleted before this method is done using them.
//
//    Hank Childs, Mon Dec 14 16:55:10 PST 2009
//    Add support for new SIL interface.
//
// ****************************************************************************

avtContract_p
avtIndexSelectFilter::ModifyContract(avtContract_p spec)
{
    avtContract_p rv = new avtContract(spec);

    amrMesh = GetInput()->GetInfo().GetAttributes().GetMeshType() == AVT_AMR_MESH;
    bool skipSILRestriction = amrMesh && groupCategory;

    if (!atts.GetUseWholeCollection() && !skipSILRestriction) 
    {
        string category = atts.GetCategoryName();
        string subset = atts.GetSubsetName();
        avtSILRestriction_p silr = spec->GetDataRequest()->GetRestriction();
        avtSILRestriction_p old_values = new avtSILRestriction(silr);
        avtSILRestrictionTraverser trav(old_values);
        int collectionID = silr->GetCollectionIndex(category, silr->GetTopSet());
        int setID = silr->GetSetIndex(subset, collectionID);
        if (trav.UsesSetData(setID) == NoneUsed) 
        {
            EXCEPTION1(InvalidSetException, subset.c_str());
        }
        TRY
        {
            // If we've got species info, we need to maintain that.
            // So see which species are on.
            std::vector<int>  species;
            std::vector<bool> setState;
            unsigned int i;
        
            int topset = silr->GetTopSet();
            avtSILSet_p pTopset = silr->GetSILSet(topset);

            const std::vector<int> &mapsOut = pTopset->GetRealMapsOut();
            avtSILCollection_p speciesColl = NULL;
            for (i = 0 ; i < mapsOut.size() ; i++)
            {
                avtSILCollection_p coll = silr->GetSILCollection(mapsOut[i]);
                if (coll->GetRole() == SIL_SPECIES)
                {
                    speciesColl = coll;
                }
            }
            if (*speciesColl != NULL)
            {
                for (i = 0 ; i < speciesColl->GetNumberOfSubsets() ; i++)
                    setState.push_back(trav.UsesData(speciesColl->GetSubset(i)));
            }
            // End logic for seeing which species is on.

            silr = rv->GetDataRequest()->GetRestriction();
            silr->TurnOffAll();
            silr->TurnOnSet(setID);
            // We've just turned on an entire set, but some parts
            // (materials) may have been turned off before, so ensure
            // that remains the case.
            int numSets = silr->GetNumSets();
            for (i = 0; i < numSets ; i++)
            {
                if (setID == i)
                    continue;
                if (trav.UsesSetData(i) == NoneUsed)
                    silr->TurnOffSet(i);
            }

            // Turn sets back on if species are on.
            for (i = 0 ; i < species.size() ; i++)
                if (setState[i])
                    silr->TurnOnSet(species[i]);
        }
        CATCH(InvalidVariableException)
        {
            // If for some reason the GetSetIndex fails.
            RETHROW;
        }
        ENDTRY
    }

    if (amrMesh && groupCategory && amrLevel != -1)
    {
        rv->GetDataRequest()->SetNeedAMRIndices(amrLevel);
    }

    if (!GetInput()->GetInfo().GetValidity().GetZonesPreserved())
    {
        rv->GetDataRequest()->SetNeedStructuredIndices(true);
    }
    else if (rv->GetDataRequest()->
                                       MustDoMaterialInterfaceReconstruction())
    {
        rv->GetDataRequest()->SetNeedStructuredIndices(true);
    }
    else 
    {
        bool needSI = false;
        avtSILRestriction_p silr =rv->GetDataRequest()->GetRestriction();
        avtSILRestrictionTraverser trav(silr);
        if (atts.GetUseWholeCollection())
        {
            intVector chunks;
            trav.GetDomainList(chunks);
            for (int i = 0; i < chunks.size(); i++)
            {
                bool hasMats = false;
                trav.GetMaterials(chunks[i], hasMats);
                needSI |= hasMats;
            }
        }
        else 
        {
            needSI = !trav.UsesAllMaterials();
        }
        if (needSI)
        {
            rv->GetDataRequest()->SetNeedStructuredIndices(true);
        }
    }

    //
    // Indicate this operator's data selection 
    //
    avtLogicalSelection *sel = new avtLogicalSelection;
    switch (atts.GetDim())
    {
        case IndexSelectAttributes::OneD:   sel->SetNDims(1); break;
        case IndexSelectAttributes::TwoD:   sel->SetNDims(2); break;
        case IndexSelectAttributes::ThreeD: sel->SetNDims(3); break;
    }
    int vec[3];
    vec[0] = atts.GetXMin();
    vec[1] = atts.GetYMin();
    vec[2] = atts.GetZMin();
    sel->SetStarts(vec);
    // avtLogicalSelection's stops are inclusive
    // also, we need to deal with using '-1' to mean 'max'
    if (atts.GetXMax() == -1)
        vec[0] = -1;
    else
        vec[0] = atts.GetXMax()-1;
    if (atts.GetYMax() == -1)
        vec[1] = -1;
    else
        vec[1] = atts.GetYMax()-1;
    if (atts.GetZMax() == -1)
        vec[2] = -1;
    else
        vec[2] = atts.GetZMax()-1;
    sel->SetStops(vec);
    vec[0] = atts.GetXIncr();
    vec[1] = atts.GetYIncr();
    vec[2] = atts.GetZIncr();
    sel->SetStrides(vec);
    selID = rv->GetDataRequest()->AddDataSelection(sel);

    if (rv->GetDataRequest()->MayRequireNodes())
        rv->GetDataRequest()->TurnNodeNumbersOn();

    return rv;
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::PreExecute
//
//  Purpose:
//      Called before Execute, which in turn calls ExecuteData.
//
//  Programmer: Hank Childs
//  Creation:   June 29, 2002
//
// ****************************************************************************

void
avtIndexSelectFilter::PreExecute(void)
{
    avtPluginDataTreeIterator::PreExecute();
    successfullyExecuted = false;
    if (!GetInput()->GetInfo().GetValidity().GetZonesPreserved())
    {
        if (atts.GetXIncr()!=1 || atts.GetYIncr()!=1 || atts.GetZIncr()!=1)
        {
            avtCallback::IssueWarning("The data was already modified "
                           "before the index select operator was applied."
                           "  It is only possible to do increments of 1.");
        }
    }
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::PostExecute
//
//  Purpose:
//      Called after Execute (which in turn called ExecuteData).
//
//  Programmer: Hank Childs
//  Creation:   June 29, 2002
//
// ****************************************************************************

void
avtIndexSelectFilter::PostExecute(void)
{
    avtPluginDataTreeIterator::PostExecute();

    if (successfullyExecuted)
    {
        int numMatches = 0;
        switch (atts.GetDim())
        {
          case IndexSelectAttributes::ThreeD:
            if (atts.GetZMin() == atts.GetZMax())
            {
                numMatches++;
            }
    
            // FALLTHRU
    
          case IndexSelectAttributes::TwoD:
            if (atts.GetYMin() == atts.GetYMax())
            {
                numMatches++;
            }
    
            // FALLTHRU
    
          case IndexSelectAttributes::OneD:
            if (atts.GetXMin() == atts.GetXMax())
            {
                numMatches++;
            }
            break;
    
          default:
            EXCEPTION0(ImproperUseException);
        }

        int indim = GetInput()->GetInfo().GetAttributes().
                                                     GetTopologicalDimension();
        int newdim = indim - numMatches;
        newdim = (newdim < 0 ? 0 : newdim);
        GetOutput()->GetInfo().GetAttributes().SetTopologicalDimension(newdim);
    }
}


// ****************************************************************************
//  Method: avtIndexSelectFilter::ReleaseData
//
//  Purpose:
//      Releases the problem size data associated with this filter.
//
//  Programmer: Hank Childs
//  Creation:   September 10, 2002
//
//  Modifications:
//
//    Hank Childs, Fri Mar  4 08:12:25 PST 2005
//    Do not set outputs of filters to NULL, since this will prevent them
//    from re-executing correctly in DLB-mode.
//
//    Hank Childs, Fri Mar 11 07:37:05 PST 2005
//    Fix non-problem size leak introduced with last fix.
//
//    Kathleen Bonnell, Mon Jan 30 15:10:26 PST 2006
//    Handle vtkMaskPoints. 
//
// ****************************************************************************

void
avtIndexSelectFilter::ReleaseData(void)
{
    avtPluginDataTreeIterator::ReleaseData();

    curvilinearFilter->SetInput(NULL);
    vtkStructuredGrid *s = vtkStructuredGrid::New();
    curvilinearFilter->SetOutput(s);
    s->Delete();

    rectilinearFilter->SetInput(NULL);
    vtkRectilinearGrid *r = vtkRectilinearGrid::New();
    rectilinearFilter->SetOutput(r);
    r->Delete();

    pointsFilter->SetInput(NULL);
    vtkPolyData *p = vtkPolyData::New();
    pointsFilter->SetOutput(p);
    p->Delete();
}

// ****************************************************************************
//  Method: avtIndexSelectFilter::UpdateDataObjectInfo
//
//  Purpose:
//    Indicates that original nodes are required for Pick, and that
//    original zones cannot be used with Pick.
//
//  Programmer: Kathleen Bonnell 
//  Creation:   August 4, 2005 
//
//  Modifications:
//    Kathleen Bonnell, Mon May  1 08:57:41 PDT 2006
//    Changed call from OrigNodes to OrigElements, indicating that either
//    nodes or zones are required, or both. 
//
//    Hank Childs, Sat Mar  3 16:28:16 PST 2007
//    Put in about data attributes that we removed ghost data.
//
// ****************************************************************************

void
avtIndexSelectFilter::UpdateDataObjectInfo(void)
{
    //
    // Node Pick returns wrong results on an Index selected plot unless it has 
    // original node numbers.  (The points are not transformed, but their 
    // numbering is probably different.  So set a flag that specifies that
    // they are needed for pick. 
    //

    GetOutput()->GetInfo().GetAttributes().SetOrigElementsRequiredForPick(true);

    //
    // Zone Pick CANNOT use Original zone numbers on an Index selected plot
    // because it may be the case that MANY original zones map to a SINGLE
    // current zone.  So set a flag specifying that the original zones
    // array CANNOT be used with pick.
    //
    GetOutput()->GetInfo().GetAttributes().SetCanUseOrigZones(false);
    GetOutput()->GetInfo().GetAttributes().SetContainsGhostZones(AVT_NO_GHOSTS);
}


// ****************************************************************************
//  Method:  avtIndexSelectFilter::FilterUnderstandsTransformedRectMesh
//
//  Purpose:
//    If this filter returns true, this means that it correctly deals
//    with rectilinear grids having an implied transform set in the
//    data attributes.  It can do this conditionally if desired.
//
//  Arguments:
//    none
//
//  Programmer:  Jeremy Meredith
//  Creation:    February 15, 2007
//
// ****************************************************************************

bool
avtIndexSelectFilter::FilterUnderstandsTransformedRectMesh()
{
    // Index select is based on logical coordinates only.
    return true;
}


// ****************************************************************************
//  Method: avtOnionPeelFilter::VerifyInput
//
//  Purpose:
//    Throw an exception if user-selected domain is out of range. 
//
//  Programmer: Kathleen Bonnell 
//  Creation:   June 7, 2007 
//
//  Modifications:
//    Kathleen Bonnell, Thu Jun 21 16:31:59 PDT 2007
//    Determine amrLevel during set validation.
//
//    Eric Brugger, Wed Dec  3 08:23:58 PST 2008
//    I modified the routine to set groupCatergory to true when "Use Whole
//    Collection" was set, so that it would would index select based on the
//    whole mesh (using base_index) and not on a per block basis.
//
//    Hank Childs, Mon Dec 14 16:55:10 PST 2009
//    Updated for new SIL interface.
//
// ****************************************************************************

void
avtIndexSelectFilter::VerifyInput()
{
    if (atts.GetUseWholeCollection() || atts.GetSubsetName() == "Whole")
    {
        groupCategory = true;
        return;
    }

    std::string category = atts.GetCategoryName();
    std::string subset = atts.GetSubsetName();
    avtSILRestriction_p silr = GetOriginatingSource()->
        GetFullDataRequest()->GetRestriction();

    int setID, collectionID;
    TRY
    {
        collectionID = silr->GetCollectionIndex(category, silr->GetTopSet());
        setID = silr->GetSetIndex(subset, collectionID);
        avtSILCollection_p coll = silr->GetSILCollection(collectionID);

        if (coll->GetRole() != SIL_DOMAIN && coll->GetRole() != SIL_BLOCK)
        {
            //
            //  May occur if user types in a category name.
            //
            EXCEPTION1(InvalidCategoryException, category.c_str()); 
        }

        bool validSet = false;
        int nEls = coll->GetNumberOfSubsets();
        for (int i = 0; i < nEls && !validSet; i++)
        {
            validSet = (setID == coll->GetSubset(i));
            if (validSet && coll->GetRole() == SIL_BLOCK)
                amrLevel = i;
        }

        if (!validSet)
        {
            //
            //  May occur if user types in a set name.
            //
            EXCEPTION2(InvalidSetException, category.c_str(), subset.c_str());
        }

        if (coll->GetRole() == SIL_BLOCK)
        {
            groupCategory = true;
        }
    }
    CATCH(InvalidVariableException)
    {
        //
        //  SIL could not match category name or subset name to an id.
        //
        RETHROW; 
    }
    ENDTRY
}

// ****************************************************************************
//  Method: avtOnionPeelFilter::Replicate
//
//  Purpose:
//    Replicates the first slice to the last slice for wrapping.
//
//  Programmer: Hank Childs/Allen Sanderson
//  Creation:   April 14, 2010
//
//  Modifications:
//
// ****************************************************************************

vtkDataSet *
avtIndexSelectFilter::Replicate(vtkDataSet *in_ds, bool wrap[3] )
{
   int   dims_in[3];
   int   dims_out[3];

   vtkDataSet *out_ds = in_ds;

   // Instantiate the output and copy over the coordinates.
   if (in_ds->GetDataObjectType() == VTK_RECTILINEAR_GRID)
   {
     // Get the original dimensions.
     vtkRectilinearGrid *rgrid = (vtkRectilinearGrid *) out_ds;
     rgrid->GetDimensions(dims_in);

     // Do each dimension individually.
     for( unsigned int d=0; d<3; ++d )
     {
       if( !wrap[d] )
         continue;

       unsigned int d0 = d;
       unsigned int d1 = (d+1)%3;
       unsigned int d2 = (d+2)%3;

       // Learn about the input grid
       int dims[3];
       vtkRectilinearGrid *rgrid = (vtkRectilinearGrid *) out_ds;
       rgrid->GetDimensions(dims);
       dims_out[0] = dims[0] + (d==0 && wrap[0] ? 1 : 0);
       dims_out[1] = dims[1] + (d==1 && wrap[1] ? 1 : 0);
       dims_out[2] = dims[2] + (d==2 && wrap[2] ? 1 : 0);

       // Make a new coord with a new value at the end.
       vtkDataArray *coor = GetCoordinates(rgrid, d0);
       vtkDataArray *coor_new = vtkFloatArray::New();
       coor_new->SetNumberOfTuples(dims_out[d0]);
       for (int i=0; i<dims[d0]; ++i)
           coor_new->SetTuple1(i, coor->GetTuple1(i));

       double lastVal = coor->GetTuple1(dims[d0]-1);
       double prevVal = coor->GetTuple1(dims[d0]-2);
       coor_new->SetTuple1(dims[d0], lastVal + (lastVal-prevVal));

       // Set up the output, including the coordinates
       vtkRectilinearGrid *out_rg = vtkRectilinearGrid::New();
       out_rg->SetDimensions(dims_out);
       SetCoordinates(out_rg, coor_new, d0);
       SetCoordinates(out_rg, GetCoordinates(rgrid, d1), d1);
       SetCoordinates(out_rg, GetCoordinates(rgrid, d2), d2);
       coor_new->Delete();

       out_ds = out_rg;
     }
   }
   else if (in_ds->GetDataObjectType() == VTK_STRUCTURED_GRID)
   {
     // Learn about the input grid
     vtkStructuredGrid *sgrid = (vtkStructuredGrid *) in_ds;
     sgrid->GetDimensions(dims_in);

     dims_out[0] = dims_in[0] + (wrap[0] ? 1 : 0);
     dims_out[1] = dims_in[1] + (wrap[1] ? 1 : 0);
     dims_out[2] = dims_in[2] + (wrap[2] ? 1 : 0);

     vtkPoints *pts = sgrid->GetPoints();
     vtkPoints *new_pts = vtkPoints::New();
     vtkStructuredGrid *out_sg = vtkStructuredGrid::New();
     out_sg->SetDimensions(dims_out);
     out_sg->SetPoints(new_pts);
     new_pts->Delete();
     new_pts->SetNumberOfPoints(dims_out[0]*dims_out[1]*dims_out[2]);

     // Copy the original points over.
     for (int i=0; i<dims_out[0]; ++i)
     {
       int i0 = i % dims_in[0];
       
       for (int j=0; j<dims_out[1]; ++j)
       {
         int j0 = j % dims_in[1];
         
         for (int k=0; k<dims_out[2]; ++k)
         {
           int k0 = k % dims_in[2];
           
           int idx_in  = k0*dims_in[1]*dims_in[0] + j0*dims_in[0] + i0;
           int idx_out = k*dims_out[1]*dims_out[0] + j*dims_out[0] + i;
           
           new_pts->SetPoint(idx_out, pts->GetPoint(idx_in));
         }
       }
     }

     out_ds = out_sg;
   }
   
   // Copy over the point data.
   vtkPointData *inPD  =  in_ds->GetPointData();
   vtkPointData *outPD = out_ds->GetPointData();
   outPD->CopyAllocate(inPD, dims_out[0]*dims_out[1]*dims_out[2]);

   for (int i1=0; i1<dims_out[0]; ++i1)
   {
     int i0 = i1 % dims_in[0];

     for (int j1=0; j1<dims_out[1]; ++j1)
     {
       int j0 = j1 % dims_in[1];
       
       for (int k1=0; k1<dims_out[2]; ++k1)
       {
         int k0 = k1 % dims_in[2];
         
         int idx_in  = k0*dims_in[1]*dims_in[0] + j0*dims_in[0] + i0;
         int idx_out = k1*dims_out[1]*dims_out[0] + j1*dims_out[0] + i1;

         outPD->CopyData(inPD, idx_in, idx_out);
       }
     }
   }

   // Copy over the cell data.
   int xdims = (dims_in[0]-1 < 1 ? 1 : dims_in[0]-1);
   int ydims = (dims_in[1]-1 < 1 ? 1 : dims_in[1]-1);
   int zdims = (dims_in[2]-1 < 1 ? 1 : dims_in[2]-1);

   int xdims_out = (dims_out[0]-1 < 1 ? 1 : dims_out[0]-1);
   int ydims_out = (dims_out[1]-1 < 1 ? 1 : dims_out[1]-1);
   int zdims_out = (dims_out[2]-1 < 1 ? 1 : dims_out[2]-1);

   vtkCellData *outCD = out_ds->GetCellData();
   vtkCellData *inCD = in_ds->GetCellData();
   outCD->CopyAllocate(inCD, xdims_out*ydims_out*zdims_out);

   for (int i1=0; i1<xdims_out; ++i1)
   {
     int i0 = i1 % xdims;

     for (int j1=0; j1<ydims_out; ++j1)
     {
       int j0 = j1 % ydims;

       for (int k1=0; k1<zdims_out; ++k1)
       {
         int k0 = k1 % zdims;

         int idx_in = k0*(ydims*xdims) + j0*xdims + i0;
         int idx_out = k1*(ydims_out*xdims_out) + j1*xdims_out + i1;
         outCD->CopyData(inCD, idx_in, idx_out);
       }
     }
   }
   
   return out_ds;
}

vtkDataArray *avtIndexSelectFilter::GetCoordinates( vtkRectilinearGrid *grid,
                                                    unsigned int coor)
{
  if( coor == 0 )
    return grid->GetXCoordinates();
  else if( coor == 1 )
    return grid->GetYCoordinates();
  else if( coor == 2 )
    return grid->GetZCoordinates();
  else
    return 0;
};


void avtIndexSelectFilter::SetCoordinates( vtkRectilinearGrid *grid,
                                           vtkDataArray *coordinates,
                                           unsigned int coor)
{
  if( coor == 0 )
    grid->SetXCoordinates(coordinates);
  else if( coor == 1 )
    grid->SetYCoordinates(coordinates);
  else if( coor == 2 )
    grid->SetZCoordinates(coordinates);
};
