/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

#ifndef QVIS_HOST_PROFILE_WINDOW_H
#define QVIS_HOST_PROFILE_WINDOW_H
#include <gui_exports.h>
#include <QvisPostableWindowObserver.h>
#include <QMap>

// Forward declrations
class HostProfileList;
class MachineProfile;
class LaunchProfile;
class QPushButton;
class QLabel;
class QLineEdit;
class QSpinBox;
class QGroupBox;
class QLabel;
class QComboBox;
class QTabWidget;
class QListWidget;
class QListWidgetItem;
class QCheckBox;
class QButtonGroup;
class QRadioButton;

// ****************************************************************************
//  Class: QvisHostProfileWindow
//
//  Purpose:
//    This window manages the host profiles that are used to launch
//    remote VisIt engines.
//
//  Notes:      
//
//  Programmer: Brad Whitlock
//  Creation:   Tue Sep 19 18:06:49 PST 2000
//
//  Modifications:
//    Brad Whitlock, Wed May 2 11:41:28 PDT 2001
//    Made it postable by switching the class from which it inherits.
//
//    Jeremy Meredith, Tue Jul 17 17:42:07 PDT 2001
//    Added numNodes, launchMethod, and partitionName, as well as checkboxes
//    for each.  Also reworded "default profile" to be "active profile".
//
//    Jeremy Meredith, Fri Sep 21 14:29:59 PDT 2001
//    Added buttons for forcing static and dynamic load balancing.
//
//    Brad Whitlock, Mon Sep 24 09:22:24 PDT 2001
//    Added the internal Apply, ReplaceLocalHost methods.
//
//    Jeremy Meredith, Thu Feb 14 15:14:32 PST 2002
//    I made hostName a combo box instead of a line edit.
//    I also changed the type of callback for the tab page changed.
//
//    Brad Whitlock, Thu Feb 21 10:16:38 PDT 2002
//    I removed the defaultUserName string.
//
//    Jeremy Meredith, Mon Jul 22 11:38:55 PDT 2002
//    Added bank name.
//
//    Jeremy Meredith, Thu Nov 21 11:28:29 PST 2002
//    Added processProfileNameText.
//
//    Jeremy Meredith, Fri Jan 24 14:43:28 PST 2003
//    Added optional arguments to the parallel launcher.
//
//    Jeremy Meredith, Mon Apr 14 18:25:17 PDT 2003
//    Added host aliases.
//
//    Jeremy Meredith, Mon May 19 12:49:17 PDT 2003
//    Added shareMDServer toggle.
//
//    Jeremy Meredith, Mon Aug 18 13:37:25 PDT 2003
//    Changed processUserNameText to userNameChanged.
//
//    Jeremy Meredith, Thu Oct  9 15:43:39 PDT 2003
//    Added ability to manually specify a client host name or to have it
//    parsed from the SSH_CLIENT (or related) environment variables.  Added
//    ability to specify an SSH port.
//
//    Jeremy Meredith, Thu Sep 15 15:20:55 PDT 2005
//    Added machine file support and the ability to use the VisIt script
//    to set up env variables in parallel.
//
//    Hank Childs, Fri Dec  2 11:01:41 PST 2005
//    Added support for hardware acceleration.
//
//    Eric Brugger, Thu Feb 15 12:10:21 PST 2007
//    Added support for additional sublauncher arguments.
//
//    Jeremy Meredith, Thu May 24 11:05:45 EDT 2007
//    Added support for SSH port tunneling.
//
//    Brad Whitlock, Tue Jun 5 15:53:20 PST 2007
//    Changed all slot functions for QLineEdit so they modify the host profile
//    as text is typed to avoid losing data when editing multiple profiles
//    and not clicking Apply.
//
//    Jeremy Meredith, Thu Jun 28 13:31:38 EDT 2007
//    Have the ssh tunneling option override and disable the client host
//    name determination method.
//
//    Brad Whitlock, Wed Apr  9 10:57:10 PDT 2008
//    QString for caption, shortName.
//
//    Dave Bremer, Wed Apr 16 17:54:14 PDT 2008
//    Added fields for commands to run pre and post the mpi command.
//
//    Cyrus Harrison, Tue Jun 24 11:15:28 PDT 2008
//    Initial Qt4 Port.
//
//    Hank Childs, Thu May  7 19:05:36 PDT 2009
//    Added field for host nickname.
//
//    Jeremy Meredith, Thu Feb 18 15:54:11 EST 2010
//    Overhauled window entirely.  Split HostProfile into
//    MachineProfile and LaunchProfile.
//
// ****************************************************************************

class GUI_API QvisHostProfileWindow : public QvisPostableWindowObserver
{
    Q_OBJECT

public:
    QvisHostProfileWindow(HostProfileList *profiles, 
                          const QString &caption = QString::null,
                          const QString &shortName = QString::null,
                          QvisNotepadArea *notepad = 0);
    virtual ~QvisHostProfileWindow();
    virtual void CreateWindowContents();
private:
    void UpdateWindow(bool doAll);
    void UpdateMachineProfile();
    void UpdateLaunchProfile();
    void UpdateWindowSensitivity();
    bool GetCurrentValues();
    void Apply(bool val = false);
    void ReplaceLocalHost();
private slots:
    void apply();

    void currentHostChanged();
    void currentLaunchChanged();

    void addMachineProfile();
    void delMachineProfile();
    void copyMachineProfile();

    void addLaunchProfile();
    void delLaunchProfile();
    void copyLaunchProfile();
    void makeDefaultLaunchProfile();

    void processProfileNameText(const QString &);
    void processDirectoryText(const QString &);
    void processEngineArgumentsText(const QString &);
    void processPartitionNameText(const QString &);
    void processBankNameText(const QString &);
    void processTimeLimitText(const QString &);
    void processMachinefileText(const QString &);
    void processLaunchArgsText(const QString &);
    void processSublaunchArgsText(const QString &);
    void processSublaunchPreCmdText(const QString &);
    void processSublaunchPostCmdText(const QString &);
    void numProcessorsChanged(int value);
    void timeoutChanged(int value);
    void launchMethodChanged(const QString &method);
    void numNodesChanged(int value);
    void toggleLaunch(bool);
    void toggleNumNodes(bool);
    void togglePartitionName(bool);
    void toggleBankName(bool);
    void toggleTimeLimit(bool);
    void toggleMachinefile(bool);
    void toggleLaunchArgs(bool);
    void toggleSublaunchArgs(bool);
    void toggleSublaunchPreCmd(bool);
    void toggleSublaunchPostCmd(bool);
    void toggleParallel(bool);
    void toggleShareMDServer(bool);
    void toggleUseVisItScriptForEnv(bool);
    void loadBalancingChanged(int);
    void hostNameChanged(const QString &host);
    void hostAliasesChanged(const QString &host);
    void hostNicknameChanged(const QString &host);
    void userNameChanged(const QString &username);
    void toggleSSHPort(bool);
    void sshPortChanged(const QString &port);
    void clientHostNameMethodChanged(int);
    void clientHostNameChanged(const QString &);
    void toggleCanDoHW(bool);
    void preCommandChanged(const QString &);
    void togglePreCommand(bool);
    void postCommandChanged(const QString &);
    void togglePostCommand(bool);
    void toggleTunnelSSH(bool);
private:
    // Main Window
    QListWidget *hostList;
    QPushButton *addHost;
    QPushButton *delHost;
    QPushButton *copyHost;

    QTabWidget *machineTabs;
    QWidget    *machineSettingsGroup;
    QWidget    *launchProfilesGroup;
    void CreateMachineSettingsGroup();
    void CreateLaunchProfilesGroup();

    // Launch Profiles
    QLabel      *profileHeaderLabel;
    QListWidget *profileList;
    QPushButton *addProfile;
    QPushButton *delProfile;
    QPushButton *copyProfile;
    QPushButton *makeDefaultProfile;

    QLabel       *profileNameLabel;
    QLineEdit    *profileName;
    QCheckBox    *parallelCheckBox;

    QTabWidget *profileTabs;
    QWidget    *basicSettingsGroup;
    QWidget    *launchSettingsGroup;
    QWidget    *advancedSettingsGroup;
    QWidget    *hwAccelSettingsGroup;
    void CreateBasicSettingsGroup();
    void CreateLaunchSettingsGroup();
    void CreateAdvancedSettingsGroup();
    void CreateHWAccelSettingsGroup();

    // Launch Basic Settings
    QCheckBox    *launchCheckBox;
    QComboBox    *launchMethod;
    QCheckBox    *partitionCheckBox;
    QLineEdit    *partitionName;
    QLabel       *numProcLabel;
    QSpinBox     *numProcessors;
    QCheckBox    *numNodesCheckBox;
    QSpinBox     *numNodes;
    QCheckBox    *bankCheckBox;
    QLineEdit    *bankName;
    QCheckBox    *timeLimitCheckBox;
    QLineEdit    *timeLimit;
    QCheckBox    *machinefileCheckBox;
    QLineEdit    *machinefile;

    // Launch Advanced Settings
    QCheckBox    *useVisItScriptForEnvCheckBox;
    QLabel       *loadBalancingLabel;
    QComboBox    *loadBalancing;
    QCheckBox    *launchArgsCheckBox;
    QLineEdit    *launchArgs;
    QCheckBox    *sublaunchArgsCheckBox;
    QLineEdit    *sublaunchArgs;
    QCheckBox    *sublaunchPreCmdCheckBox;
    QLineEdit    *sublaunchPreCmd;
    QCheckBox    *sublaunchPostCmdCheckBox;
    QLineEdit    *sublaunchPostCmd;
    QLabel       *timeoutLabel;
    QSpinBox     *timeout;

    // Launch HW Accel Settings
    QLabel       *hwdisclaimer;
    QCheckBox    *canDoHW;
    QLineEdit    *preCommand;
    QCheckBox    *preCommandCheckBox;
    QLineEdit    *postCommand;
    QCheckBox    *postCommandCheckBox;

    // Machine Settings
    QLabel       *hostNameLabel;
    QLineEdit    *hostName;
    QLabel       *hostAliasesLabel;
    QLineEdit    *hostAliases;
    QLabel       *hostNicknameLabel;
    QLineEdit    *hostNickname;
    QLabel       *userNameLabel;
    QLineEdit    *userName;
    QCheckBox    *shareMDServerCheckBox;
    QCheckBox    *tunnelSSH;
    QButtonGroup *clientHostNameMethod;
    QRadioButton *chnMachineName;
    QRadioButton *chnParseFromSSHClient;
    QRadioButton *chnSpecifyManually;
    QLineEdit    *clientHostName;
    QLabel       *clientHostNameMethodLabel;
    QCheckBox    *sshPortCheckBox;
    QLineEdit    *sshPort;
    QLabel       *engineArgumentsLabel;
    QLineEdit    *engineArguments;
    QLabel       *directoryLabel;
    QLineEdit    *directory;

    // other stuff
    int          profileCounter;
    MachineProfile *currentMachine;
    LaunchProfile  *currentLaunch;
};

#endif
