/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                            avtMFIXFileFormat.C                            //
// ************************************************************************* //

#include <avtMFIXFileFormat.h>

#include <string>

#include <vtkMFIXReader.h>
#include <vtkExecutive.h>
#include <vtkCellData.h>
#include <vtkDataArray.h>
#include <vtkDemandDrivenPipeline.h>
#include <vtkFloatArray.h>
#include <vtkInformation.h>
#include <vtkInformationVector.h>
#include <vtkPointData.h>
#include <vtkRectilinearGrid.h>
#include <vtkStreamingDemandDrivenPipeline.h>
#include <vtkUnstructuredGrid.h>

#include <avtDatabaseMetaData.h>
#include <DBOptionsAttributes.h>
#include <visit-config.h> // for WORDS_BIGENDIAN

#include <InvalidVariableException.h>


// ****************************************************************************
//  Method: avtMFIX constructor
//
//  Programmer: bdotson -- generated by xml2avt
//  Creation:   Fri May 26 08:59:22 PDT 2006
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 22 17:11:35 PDT 2008
//    Reworked to use vtkMFIXReader for bulk of work, and to have read options
//    so user can specify endianness of the data. 
//
// ****************************************************************************

avtMFIXFileFormat::avtMFIXFileFormat(const char *filename, 
    DBOptionsAttributes *rdopts)
    : avtMTMDFileFormat(filename)
{
    strcpy(this->RestartFileName, filename);
    readInData = false; 
    readInformation = false; 
    reader = NULL;
    fileBigEndian = true;
    for (int i = 0; rdopts != 0 && i < rdopts->GetNumberOfOptions(); ++i)
    {
        if (rdopts->GetName(i) == "Big Endian")
        {
            fileBigEndian = rdopts->GetBool("Big Endian");
        }
    }
}


// ****************************************************************************
//  Method: avtMFIX destructor
//
//  Programmer: Kathleen Bonnell 
//  Creation:   October 22, 2008 
//
//  Modifications:
//
// ****************************************************************************

avtMFIXFileFormat::~avtMFIXFileFormat()
{
    if (reader != NULL)
    {
        reader->Delete();
        reader = NULL;
    }
}


// ****************************************************************************
//  Method: avtEMSTDFileFormat::GetNTimesteps
//
//  Purpose:
//      Tells the rest of the code how many timesteps there are in this file.
//
//  Programmer: bdotson -- generated by xml2avt
//  Creation:   Fri May 26 08:59:22 PDT 2006
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 22 17:11:35 PDT 2008
//    Reworked to use vtkMFIXReader for bulk of work.
//
// ****************************************************************************

int
avtMFIXFileFormat::GetNTimesteps(void)
{
    if (!readInformation)
        ReadInformation();

    return reader->GetNumberOfTimeSteps();
}


// ****************************************************************************
//  Method: avtMFIXFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: bdotson -- generated by xml2avt
//  Creation:   Fri May 26 08:59:22 PDT 2006
//
// ****************************************************************************

void
avtMFIXFileFormat::FreeUpResources()
{
}

// ****************************************************************************
// Method: avtMFIXFileFormat::GetTimes
//
// Purpose: 
//   Returns the times.
//
// Arguments:
//   t : The vector that will contain the times.
//
// Programmer: Brad Whitlock
// Creation:   Fri Mar 31 10:48:03 PDT 2006
//
// Modifications:
//    Kathleen Bonnell, Wed Oct 22 17:11:35 PDT 2008
//    Reworked to use vtkMFIXReader for bulk of work.
//
// ****************************************************************************

void
avtMFIXFileFormat::GetTimes(doubleVector &t)
{
    if (!readInformation)
        ReadInformation();

    t = times;
}


// ****************************************************************************
//  Method: avtMFIXFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: bdotson -- generated by xml2avt
//  Creation:   Fri May 26 08:59:22 PDT 2006
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 22 17:11:35 PDT 2008
//    Reworked to use vtkMFIXReader for bulk of work.
//
// ****************************************************************************

void
avtMFIXFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md, 
                                            int timeState)
{
    if (!readInformation)
        ReadInformation();

    avtMeshMetaData *mesh = new avtMeshMetaData;
    mesh->name = "Mesh";
    mesh->meshType = AVT_UNSTRUCTURED_MESH;
    mesh->topologicalDimension = 3;
    mesh->spatialDimension = 3;
    mesh->numBlocks = 4;
    mesh->blockNames.push_back("Fluid");
    mesh->blockNames.push_back("Inlet");
    mesh->blockNames.push_back("Outlet");
    mesh->blockNames.push_back("Walls");
    mesh->hasSpatialExtents = false;
    md->Add(mesh);

    for (int i=0; i < reader->GetNumberOfCellArrays(); ++i)
    {
        const char *name = reader->GetCellArrayName(i);
        if (reader->GetCellArrayNumComponents(i) == 1)
        {
            AddScalarVarToMetaData(md, name, "Mesh", AVT_ZONECENT);
        }
        else
        {
            AddVectorVarToMetaData(md, name, "Mesh", AVT_ZONECENT);
        }
    }
}


// ****************************************************************************
//  Method: avtMFIXFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      domain      The index of the domain.  If there are NDomains, this
//                  value is guaranteed to be between 0 and NDomains-1,
//                  regardless of block origin.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: bdotson -- generated by xml2avt
//  Creation:   Fri May 26 08:59:22 PDT 2006
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 22 17:11:35 PDT 2008
//    Reworked to use vtkMFIXReader for bulk of work.
//
//    Kathleen Bonnell, Wed Jun 24 07:45:11 PDT 2009
//    Set the timestate in the reader.
//
// ****************************************************************************

vtkDataSet *
avtMFIXFileFormat::GetMesh(int timestate, int domain, const char *meshname)
{
    if (!readInData)
        ReadInData();

    reader->SetTimeStep(timestate);
    reader->Update();

    vtkUnstructuredGrid *dataset = vtkUnstructuredGrid::New();
    vtkDataObject *dob = reader->GetOutputDataObject(domain);

    dataset->DeepCopy(vtkUnstructuredGrid::SafeDownCast(dob));

    return dataset;
}


// ****************************************************************************
//  Method: avtMFIXFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      domain     The index of the domain.  If there are NDomains, this
//                 value is guaranteed to be between 0 and NDomains-1,
//                 regardless of block origin.
//      varname    The name of the variable requested.
//
//  Programmer: bdotson -- generated by xml2avt
//  Creation:   Fri May 26 08:59:22 PDT 2006
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 22 17:11:35 PDT 2008
//    Reworked to use vtkMFIXReader for bulk of work.
//
//    Kathleen Bonnell, Wed Jun 24 07:45:11 PDT 2009
//    Set the timestate in the reader.
//
// ****************************************************************************

vtkDataArray *
avtMFIXFileFormat::GetVar(int timestate, int domain, const char *varname)
{
    if (!readInData)
        ReadInData();

    reader->SetTimeStep(timestate);
    reader->Update();

    vtkDataArray *v = NULL;
    if (domain == 0)
    {
        v = reader->GetOutput()->GetPointData()->GetArray(varname);
        if (v == NULL)
            v = reader->GetOutput()->GetCellData()->GetArray(varname);
        if (v == NULL)
            EXCEPTION1(InvalidVariableException, varname);

        v->Register(NULL);
    }
    else 
    {
        v = vtkFloatArray::New();
        vtkDataSet *ds = vtkDataSet::SafeDownCast(
                              reader->GetOutputDataObject(domain));
        int nc = ds->GetNumberOfCells();
        for (int i = 0; i< nc; i++)
        {
          v->InsertTuple1(i, 0);
        }
    }
    return v;
}


// ****************************************************************************
//  Method: avtMFIXFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      domain     The index of the domain.  If there are NDomains, this
//                 value is guaranteed to be between 0 and NDomains-1,
//                 regardless of block origin.
//      varname    The name of the variable requested.
//
//  Programmer: bdotson -- generated by xml2avt
//  Creation:   Fri May 26 08:59:22 PDT 2006
//
//  Modifications:
//    Kathleen Bonnell, Wed Oct 22 17:11:35 PDT 2008
//    Reworked to use vtkMFIXReader for bulk of work.
//
//    Kathleen Bonnell, Wed Jun 24 07:45:11 PDT 2009
//    Set the timestate in the reader.
//
// ****************************************************************************

vtkDataArray *
avtMFIXFileFormat::GetVectorVar(int timestate, int domain,const char *varname)
{
    if (!readInData)
        ReadInData();

    reader->SetTimeStep(timestate);
    reader->Update();

    vtkDataArray *v = NULL;
    if (domain == 0)
    {
        v = reader->GetOutput()->GetPointData()->GetArray(varname);
        if (v == NULL)
            v = reader->GetOutput()->GetCellData()->GetArray(varname);
        if (v == NULL)
            EXCEPTION1(InvalidVariableException, varname);

        v->Register(NULL);
    }
    else 
    {
        v = vtkFloatArray::New();
        v->SetNumberOfComponents(3);
        vtkDataSet *ds = vtkDataSet::SafeDownCast(
                         reader->GetOutputDataObject(domain));
        int nc = ds->GetNumberOfCells();
        for (int i = 0; i < nc; i++)
        {
            v->InsertComponent(i, 0, 0);
            v->InsertComponent(i, 1, 0);
            v->InsertComponent(i, 2, 0);
        }
    }
    return v;
}


// ****************************************************************************
//  Method: avtMFIXFileFormat::SetUpReader
//
//  Purpose:  Sets up the vtk reader.
//
//  Programmer:  Kathleen Bonnell
//  Creation:    October 22, 2008
//
//  Modifications:
//
// ****************************************************************************

void
avtMFIXFileFormat::SetUpReader()
{
    if (reader == NULL)
    {
        reader = vtkMFIXReader::New();
        reader->SetFileName(this->RestartFileName);
#ifdef WORDS_BIGENDIAN
        if (fileBigEndian)
            reader->SetSwapByteOrder(false);
        else 
            reader->SetSwapByteOrder(true);
#else
        if (fileBigEndian)
            reader->SetSwapByteOrder(true);
        else 
            reader->SetSwapByteOrder(false);
#endif
    }
}

// ****************************************************************************
//  Method: avtMFIXFileFormat::ReadInData
//
//  Purpose:  Has the vtk reader read in all the information
//
//  Programmer:  Kathleen Bonnell
//  Creation:    October 22, 2008
//
//  Modifications:
//
// ****************************************************************************

void
avtMFIXFileFormat::ReadInData()
{
    SetUpReader();
    reader->Update();
    readInData = true;
}


// ****************************************************************************
//  Method: avtMFIXFileFormat::ReadInInformation
//
//  Purpose:  Reads meta data information.
//
//  Programmer:  Kathleen Bonnell
//  Creation:    October 22, 2008
//
//  Modifications:
//
// ****************************************************************************

void 
avtMFIXFileFormat::ReadInformation()
{
    SetUpReader();

    // Tell the reader to read in only the meta data (information) 
    vtkExecutive *e = reader->GetExecutive();
    vtkInformation *request = vtkInformation::New();
    request->Set(vtkDemandDrivenPipeline::REQUEST_INFORMATION());
    reader->ProcessRequest(request, e->GetInputInformation(),
                               e->GetOutputInformation());

    // Get time step information  
    vtkInformation *info = e->GetOutputInformation()->GetInformationObject(0); 
    int len = info->Length(vtkStreamingDemandDrivenPipeline::TIME_STEPS());
    double *t = info->Get(vtkStreamingDemandDrivenPipeline::TIME_STEPS());
    for (int i = 0; i < len; i++)
        times.push_back(t[i]);

    readInformation = true;
}
