/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                           avtITAPS_CFileFormat.C                          //
// ************************************************************************* //

#include "iBase.h"
#include "iMesh.h"

#if 0 // PARALLEL
#include "iMeshP.h"
#include <mpi.h>
#include <avtParallel.h>
#endif

#include <snprintf.h>

#include <avtITAPS_CFileFormat.h>
#include <avtITAPS_CUtility.h>

#include <vtkCellType.h>
#include <vtkCharArray.h>
#include <vtkDoubleArray.h>
#include <vtkFloatArray.h>
#include <vtkIntArray.h>
#include <vtkPoints.h>
#include <vtkUnstructuredGrid.h>

#include <avtCallback.h>
#include <avtDatabaseMetaData.h>
#include <avtVariableCache.h>

#include <DBOptionsAttributes.h>
#include <DebugStream.h>
#include <Expression.h>

#include <InvalidVariableException.h>
#include <InvalidFilesException.h>

#include <map>
#include <string>
#include <vector>

using     std::map;
using     std::string;
using     std::vector;
using namespace avtITAPS_CUtility;

static const int maxCoincidentSets = 6;

// ****************************************************************************
//  Method: avtITAPS_C constructor
//
//  Programmer: miller -- generated by xml2avt
//  Creation:   Wed Mar 7 17:15:33 PST 2007
//
//  Modifications:
//    Mark C. Miller, Tue Jun 26 11:27:48 PDT 2007
//    Modified for C interface to ITAPS
//
//    Mark C. Miller, Tue Apr 21 15:56:05 PDT 2009
//    Removed class storage for vert, edge, face and regn entities. We just
//    read them whenever we need to read a tag defined on them.
// ****************************************************************************

avtITAPS_CFileFormat::avtITAPS_CFileFormat(const char *filename,
    DBOptionsAttributes *rdatts) : avtSTMDFileFormat(&filename, 1)
{
    InitDataTypeNames();
    vmeshFileName = filename;
    geomDim = topoDim = 0;
    haveMixedElementMesh = false;
    for (int i = 0; rdatts != 0 && i < rdatts->GetNumberOfOptions(); ++i)
    {
    }
}

avtITAPS_CFileFormat::~avtITAPS_CFileFormat()
{
    messageCounts.clear();
}


// ****************************************************************************
//  Method: avtITAPS_CFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: miller -- generated by xml2avt
//  Creation:   Wed Mar 7 17:15:33 PST 2007
//
//  Modifications:
//    Mark C. Miller, Tue Jun 26 11:27:48 PDT 2007
//    Modified for C interface to ITAPS
//
//    Mark C. Miller, Thu Mar 26 17:30:15 PDT 2009
//    Added call to destroy the iMesh instance (duh!)
//
//    Mark C. Miller, Tue Apr 21 15:56:05 PDT 2009
//    Removed class storage for vert, edge, face and regn entities. We just
//    read them whenever we need to read a tag defined on them.
// ****************************************************************************

void
avtITAPS_CFileFormat::FreeUpResources(void)
{
    iMesh_dtor(itapsMesh, &itapsError);
}


// ****************************************************************************
//  Method: avtITAPS_CFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: miller -- generated by xml2avt
//  Creation:   Wed Mar 7 17:15:33 PST 2007
//
//  Modifications:
//
//    Mark C. Miller, Thu Mar 22 09:37:55 PDT 2007
//    Added code to detect variable centering and populate md with variable
//    info
//
//    Mark C. Miller, Tue Jun 26 11:27:48 PDT 2007
//    Modified for C interface to ITAPS
//
//    Mark C. Miller, Tue Apr 22 21:45:41 PDT 2008
//    Big changes to support MOAB's subsets
//
//    Mark C. Miller, Tue Jan  6 18:48:22 PST 2009
//    Added check for has_data==0 in loop to find tags (variables) centering.
//
//    Mark C. Miller, Wed Mar 25 22:02:06 PDT 2009
//    Made it so the entity iterator in the loop to find dense tags for
//    variable centering is NOT ended until AFTER the returned entity has
//    been queried.
//
//    Mark C. Miller, Thu Mar 26 17:32:01 PDT 2009
//    Added logic to destroy the iMesh instance if we're on the mdserver.
//
//    Mark C. Miller, Mon Mar 30 17:35:44 PDT 2009
//    Ensure dummyStr is initialized.
//
//    Mark C. Miller, Tue Apr 21 15:56:56 PDT 2009
//    Updated to current iMesh/iBase specification. Changed debug level used
//    for various messages. Added to logic which searches for a suitable
//    'domain' class to check of candidate classes can 'cover' the root in
//    the set theoretic sense. Changed loop that searches for tags and
//    determines centering to add stuff to variable name if the same tag is
//    defined on different classes of entities.
//
//    Mark C. Miller, Thu Apr 23 08:10:02 PDT 2009
//    Added logic to handle Zoltan partition as VisIt's domains.
// ****************************************************************************

void
avtITAPS_CFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md)
{
    int i, j;

    // remove extension from filename
    string tmpFileName = vmeshFileName;
    int q = vmeshFileName.size();
    if (vmeshFileName[q-6] == '.' && vmeshFileName[q-5] == 'v' &&
        vmeshFileName[q-4] == 'm' && vmeshFileName[q-3] == 'e' &&
        vmeshFileName[q-2] == 's' && vmeshFileName[q-1] == 'h')
        tmpFileName = string(vmeshFileName, 0, q-6);
    else if (vmeshFileName[q-6] == '.' && vmeshFileName[q-5] == 'c' &&
             vmeshFileName[q-4] == 'u' && vmeshFileName[q-3] == 'b')
        tmpFileName = string(vmeshFileName, 0, q-4);

    char dummyStr[32] = "";
    iMesh_newMesh(dummyStr, &itapsMesh, &itapsError, 0);
    CheckITAPSError(itapsMesh, iMesh_newMesh, NoL);
    iMesh_getRootSet(itapsMesh, &rootSet, &itapsError);
    CheckITAPSError(itapsMesh, iMesh_getRootSet, NoL);

    // ok, try loading the mesh.
    try
    {
#if 0 //PARALLEL
        cerr << "got here" << endl;
        iMeshP_createPartitionAll(itapsMesh, MPI_COMM_WORLD, &itapsPart, &itapsError);
        CheckITAPSError(itapsMesh, iMeshP_createPartitionAll, NoL);
        iMeshP_loadAll(itapsMesh, itapsPart, rootSet, tmpFileName.c_str(), dummyStr, &itapsError,
            tmpFileName.length(), 0);
        CheckITAPSError(itapsMesh, iMeshP_loadAll, NoL);
        itapsParts = 0;
        int parts_allocated = 0, parts_size = 0;
        iMeshP_getLocalParts(itapsMesh, itapsPart, &itapsParts, &parts_allocated, &parts_size, &itapsError);
        CheckITAPSError(itapsMesh, iMeshP_getLocalParts, NoL);
        debug3 << "parts_size = " << parts_size << endl;
#else
        iMesh_load(itapsMesh, rootSet, tmpFileName.c_str(), dummyStr, &itapsError,
            tmpFileName.length(), 0);
        CheckITAPSError(itapsMesh, iMesh_load, NoL);
#endif

        // determine spatial and topological dimensions of mesh
        int geomDim;
        iMesh_getGeometricDimension(itapsMesh, &geomDim, &itapsError);
        CheckITAPSError(itapsMesh, iMesh_getGeometricDimension, NoL);
        topoDim = -1;
        for (i = 0; i < iBase_ALL_TYPES; i++)
        {
            iMesh_getNumOfType(itapsMesh, rootSet, i, &numOfType[i], &itapsError);
            if (numOfType[i] > 0)
                topoDim = i;
            debug3 << "numOfType["<<i<<"] = " << numOfType[i] << endl;
        }

        //
        // Decide if we've got a 'mixed element' mesh
        //
        if ((numOfType[1] > 0 && numOfType[2] > 0) ||
            (numOfType[1] > 0 && numOfType[3] > 0) ||
            (numOfType[2] > 0 && numOfType[3] > 0))
            haveMixedElementMesh = true;

        //
        // Figure out which top-level class we should treat as the 'domain'
        // class.
        //
        string domainSetClassName = "Unknown";
        if (topoDim == 0)
        {
            domainEntType = iBase_VERTEX;
            domainSetClassName = "Vertex";
        }
        else if (topoDim == 1)
        {
            domainEntType = iBase_EDGE;
            domainSetClassName = "Curve";
        }
        else if (topoDim == 2)
        {
            domainEntType = iBase_FACE;
            domainSetClassName = "Surface";
        }
        else if (topoDim == 3)
        {
            domainEntType = iBase_REGION;
            domainSetClassName = "Volume";
        }

        // This first call is purely to provide opportunity to output stuff
        // to debug logs.
        const bool debugOff = true;
        iBase_EntityHandle junk;
        if (DebugStream::Level4())
            TraverseSetHierarchy(itapsMesh, 0, 0, true, junk, rootSet, !debugOff, 0, 0);

        // Ok, do some real work to get top-level sets
        TraverseSetHierarchy(itapsMesh, 0, 0, true, junk, rootSet, debugOff,
            GetTopLevelSets, &topLevelSets);

        //
        // Find the 'domain' set class and assign its sets for the domains.
        //
        map<string, vector<iBase_EntitySetHandle> >::iterator tlsit;
        if (GetTagStuff(itapsMesh, rootSet, -1, "PARALLEL_PARTITION", 0, 0, 0) == iBase_SUCCESS ||
            GetTagStuff(itapsMesh, rootSet, iBase_VERTEX, "PARALLEL_PARTITION", 0, 0, 0) == iBase_SUCCESS ||
            GetTagStuff(itapsMesh, rootSet, iBase_REGION, "PARALLEL_PARTITION", 0, 0, 0) == iBase_SUCCESS)
        {
            domainSets = topLevelSets["PARALLEL_PARTITION"]; 
            domainSetClassName = "PARALLEL_PARTITION";
        }
        else
        {
            {
                for (tlsit = topLevelSets.begin(); tlsit != topLevelSets.end(); tlsit++)
                {
                    if (tlsit->first == domainSetClassName)
                    {
                        //
                        // Lets at least ensure that the choosen class of entity sets
                        // has enough entities to at least 'cover' the root.
                        //
                        int nverts = 0;
                        int nother = 0;
                        for (unsigned int q = 0; q < tlsit->second.size(); q++)
                        {
                            int nverts_tmp;
                            int nother_tmp;
                            iMesh_getNumOfType(itapsMesh, tlsit->second[q], iBase_VERTEX, &nverts_tmp, &itapsError);
                            iMesh_getNumOfType(itapsMesh, tlsit->second[q], domainEntType, &nother_tmp, &itapsError);
                            nverts += nverts_tmp;
                            nother += nother_tmp;
                        }
                        //
                        // MOAB seems to be 'funny' with vertex entities in subsets and so
                        // a vertex cover is never possible to achieve from MOAB. So, we
                        // currently only test for 'other'
                        //if (nverts >= numOfType[0] && nother >= numOfType[topoDim])
                        if (nother >= numOfType[topoDim])
                        {
                            domainSets = tlsit->second;
                            debug3 << "selected \"" << tlsit->first << "\" as the domain class" << endl;
                            break;
                        }
                        else
                        {
                            debug3 << "skipping \"" << tlsit->first << "\" as candidate domain class "
                                   << "because it cannot possible 'cover' root" << endl;
                        }
                    }
                    debug3 << "skipping \"" << tlsit->first << "\" as candidate domain class" << endl;
                }
            }
        }

        //
        // This test for <=1 catches MOAB cases where there is only one Volume type entity
        // set AND yet that set's verticies don't cover the root (brick_10.0.cub).
        //
        if (domainSets.size() <= 1)
        {
            domainSets.clear();
            domainSets.push_back(rootSet);
            domainSetClassName = "Root";
        }

        //
        // Define the mesh
        //
        avtMeshMetaData *mmd = new avtMeshMetaData("mesh", domainSets.size(),
            0, 0, 0, geomDim, topoDim, AVT_UNSTRUCTURED_MESH);
        mmd->blockTitle = domainSetClassName == "Vertex" ? "Verticies" : domainSetClassName + "s";
        mmd->blockPieceName = domainSetClassName;
        md->Add(mmd);

        //
        // Add enumerated scalar variables for top-level subset classes
        // other than the domains.
        //
        for (tlsit = topLevelSets.begin(); tlsit != topLevelSets.end(); tlsit++)
        {
            // skip the 'domain' class
            if (tlsit->first == domainSetClassName)
                continue;

            string className = tlsit->first == "Vertex" ? "Verticies" : tlsit->first + "s";
            avtScalarMetaData *smd = new avtScalarMetaData(className, "mesh", AVT_NODECENT);
            smd->hideFromGUI = true;
            smd->SetEnumerationType(avtScalarMetaData::ByNChooseR);
            smd->SetEnumNChooseRN(tlsit->second.size());
            smd->SetEnumNChooseRMaxR(maxCoincidentSets);
            smd->SetEnumAlwaysExcludeValue(-1.0);
            smd->SetEnumPartialCellMode(avtScalarMetaData::Dissect);

            for (int k = 0; k < tlsit->second.size(); k++)
            {
                char tmpName[64];
                SNPRINTF(tmpName, sizeof(tmpName), "%s_%03d", tlsit->first.c_str(), k);
                smd->AddEnumNameValue(tmpName, k);
            }

            md->Add(smd);
            avtScalarMetaData::BuildEnumNChooseRMap(tlsit->second.size(), maxCoincidentSets,
                pascalsTriangleMaps[className]);
        }

        //
        // Because its so convenient, add the zonetype expression as a
        // variable on the mesh
        //
        Expression expr;
        expr.SetName("zonetype");
        expr.SetDefinition("zonetype(mesh)");
        expr.SetType(Expression::ScalarMeshVar);
        md->AddExpression(&expr);

        //
        // How to determine tag (field) centering WITHOUT invoking problem
        // sized data work; e.g. an array of entity handles? We pick the
        // first entity in each type class (0d, 1d, 2d, and 3d) and
        // find all the tags defined on those entities and then
        // register them as node or zone centered variables.
        //
        int entTypeClass;
        for (entTypeClass = 0; entTypeClass < 4; entTypeClass++)
        {
            iBase_EntityHandle oneEntity;
            int has_data;
            iMesh_EntityIterator entIt;

            // initialize an entity iterator and get the first entity
            iMesh_initEntIter(itapsMesh, domainSets[0], (iBase_EntityType) entTypeClass,
                iMesh_ALL_TOPOLOGIES, &entIt, &itapsError);
            CheckITAPSError(itapsMesh, iMesh_initEntIter, NoL);
            iMesh_getNextEntIter(itapsMesh, entIt, &oneEntity, &has_data, &itapsError);
            CheckITAPSError(itapsMesh, iMesh_getNextEntIter, NoL);

            if (has_data && itapsError == iBase_SUCCESS)
            {
                // get all the tags defined on this one entity
                iBase_TagHandle *tagsOnOneEntity = 0; int tagsOnOneEntity_allocated = 0;
                int tagsOnOneEntity_size = 0;
                iMesh_getAllTags(itapsMesh, oneEntity, &tagsOnOneEntity,
                    &tagsOnOneEntity_allocated, &tagsOnOneEntity_size, &itapsError);
                CheckITAPSError(itapsMesh, iMesh_getAllTags, (0,tagsOnOneEntity,EoL));

                // make a vector of the found handles and copy it
                // to the saved list of primitive tag handles
                vector<iBase_TagHandle> tmpTagHandles;
                for (int kk = 0; kk < tagsOnOneEntity_size; kk++)
                    tmpTagHandles.push_back(tagsOnOneEntity[kk]);
                primitiveTagHandles[entTypeClass] = tmpTagHandles;
                if (tagsOnOneEntity_allocated)
                    free(tagsOnOneEntity);
            }

            iMesh_endEntIter(itapsMesh, entIt, &itapsError);
            CheckITAPSError(itapsMesh, iMesh_endEntIter, NoL);
        }

        for (entTypeClass = 0; entTypeClass < 4; entTypeClass++)
        {
            avtCentering centering = entTypeClass == 0 ? AVT_NODECENT : AVT_ZONECENT;

            vector<iBase_TagHandle> tagHandles = primitiveTagHandles[entTypeClass];

            debug3 << "Checking " << tagHandles.size() << " tags defined on entity class \""
                   << entTypes[entTypeClass] << "\"" << endl;

            for (int tagIdx = 0; tagIdx < tagHandles.size(); tagIdx++)
            {
                iBase_TagHandle theTag = tagHandles[tagIdx]; 
                string tagName = VisIt_iMesh_getTagName(itapsMesh, theTag);
                CheckITAPSError(itapsMesh, VisIt_iMesh_getTagName, NoL);
                int valSize;
                iMesh_getTagSizeValues(itapsMesh, theTag, &valSize, &itapsError);
                CheckITAPSError(itapsMesh, iMesh_getTagSizeValues, NoL);
                avtVarType var_type = GuessVarTypeFromNumDimsAndComps(geomDim, valSize);

                debug3 << "Checking tag \"" << tagName << "\", size = "
                       << valSize << ", candidate variable type is \""
                       << avtVarTypeToString(var_type) << "\"" << endl;

                // ITAPS can sometimes define same tag on multiple entities. If so,
                // be sure to append the class name to the variable name.
                bool shouldAddEntTypeClassNameToVar = false;
                for (int et = 0; et < iBase_ALL_TYPES; et++)
                {
                    // We only want to check the tag name's existence on other entity types
                    if (et == entTypeClass)
                        continue;

                    vector<iBase_TagHandle> tagHandlesOnEnts = primitiveTagHandles[et];
                    for (int t = 0; t < tagHandlesOnEnts.size(); t++)
                    {
                        string vertTagName = VisIt_iMesh_getTagName(itapsMesh, tagHandlesOnEnts[t]);
                        CheckITAPSError(itapsMesh, VisIt_iMesh_getTagName, NoL);
                        if (tagName == vertTagName)
                        {
                            shouldAddEntTypeClassNameToVar = true;
                            break;
                        }
                    }
                }
                string addVarName = "";
                if (shouldAddEntTypeClassNameToVar)
                    addVarName = "_on_" + string(entTypes[entTypeClass]) + "_entities";

                if (var_type == AVT_SCALAR_VAR)
                    AddScalarVarToMetaData(md, tagName.c_str()+addVarName, "mesh", centering);
                else if (var_type == AVT_VECTOR_VAR)
                    AddVectorVarToMetaData(md, tagName.c_str()+addVarName, "mesh", centering, valSize);
                else if (var_type == AVT_SYMMETRIC_TENSOR_VAR)
                    AddSymmetricTensorVarToMetaData(md, tagName.c_str()+addVarName, "mesh", centering, valSize);
                else if (var_type == AVT_TENSOR_VAR)
                    AddTensorVarToMetaData(md, tagName.c_str()+addVarName, "mesh", centering, valSize);
                else
                {
                    vector<string> memberNames;
                    for (int c = 0; c < valSize; c++)
                    {
                        char tmpName[64];
                        SNPRINTF(tmpName, sizeof(tmpName), "%s_%03d", tagName.c_str(), c);
                        memberNames.push_back(tmpName);
                    }
                    AddArrayVarToMetaData(md, tagName.c_str()+addVarName, memberNames, "mesh", centering);
                }
            }
        }
    }
    catch (iBase_Error TErr)
    {
        char msg[512];
        char desc[256];
        desc[0] = '\0';
        int tmpError = itapsError;
#if !defined(ITAPS_GRUMMP)
        iMesh_getDescription(itapsMesh, desc, &itapsError, sizeof(desc));
#endif
        SNPRINTF(msg, sizeof(msg), "Encountered ITAPS error (%d) \"%s\""
            "\nUnable to open file!", tmpError, desc); 
        if (!avtCallback::IssueWarning(msg))
            cerr << msg << endl;
    }
funcEnd: ;

#ifdef MDSERVER
    // We don't need to keep this thing around on the mdserver, so free it up
    iMesh_dtor(itapsMesh, &itapsError);
#endif
}

// ****************************************************************************
//  Method: avtITAPS_CFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      domain      The index of the domain.  If there are NDomains, this
//                  value is guaranteed to be between 0 and NDomains-1,
//                  regardless of block origin.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: miller -- generated by xml2avt
//  Creation:   Wed Mar 7 17:15:33 PST 2007
//
//  Modifications:
//    Mark C. Miller, Tue Jun 26 11:27:48 PDT 2007
//    Modified for C interface to ITAPS
//
//    Mark C. Miller, Tue Apr 22 21:45:41 PDT 2008
//    Big changes to support MOAB's subsets
//
//    Mark C. Miller, Tue Apr 21 16:00:12 PDT 2009
//    Added logic with junkSet and memset to be real careful to ensure that
//    the set did in fact get set to someting. Updated to current iMesh/iBase
//    interface. Added logic where cells are inserted into the mesh to
//    ensure we're not attempting to insert cells that we don't think VTK
//    will think are valid.
//
//    Mark C. Miller, Wed May  6 13:52:16 PDT 2009
//    Changed 'Invalid' in error message to 'Unsupported'.
// ****************************************************************************

vtkDataSet *
avtITAPS_CFileFormat::GetMesh(int domain, const char *meshname)
{
    int i, j;

    debug3 << "avtITAPS_FMDBFileFormat::GetMesh, domain=" << domain << endl;

    try
    {
        //
        // Get the entity set handle for this domain.
        // We use the memset stuff to set the handle to something
        // with low probably of being validly returned from an implementation
        // so that we can check if indeed the logic here sets it. 
        //
        iBase_EntitySetHandle theSet, junkSet;
        memset(&theSet, 0xFF, sizeof(iBase_EntitySetHandle));
        memset(&junkSet, 0xFF, sizeof(iBase_EntitySetHandle));
        int setCount = 0;
        int entType = -1;
#ifdef ITAPS_FMDB
        theSet = rootSet;
        entType = iBase_REGION;
#else
        theSet = domainSets[domain];
        entType = domainEntType;
#endif

        if (memcmp(&junkSet, &theSet, sizeof(iBase_EntitySetHandle)) == 0 || entType == -1)
        {
            char msg[256];
            SNPRINTF(msg, sizeof(msg), "Unable to find entity set handle for domain %d", domain);
            EXCEPTION1(ImproperUseException, msg);
        }

        // This call to iMesh_getAdjEntIndices is being used to obtain all the
        // vertex entities for the elements comprising this domain as well as
        // the list of vertices that comprise each element.
        iBase_EntityHandle *entHdls = 0;
        int entHdls_allocated = 0, entHdls_size;
        iBase_EntityHandle *adjHdls = 0;
        int adjHdls_allocated = 0, adjHdls_size;
        int *adjInds = 0; int adjInds_allocated = 0, adjInds_size;
        int *offs = 0; int offs_allocated = 0, offs_size;
        iMesh_getAdjEntIndices(
            itapsMesh,            // iMesh_Instance instance,
            theSet,               // iBase_EntitySetHandle entity_set_handle,
            entType,              // int entity_type_requestor,
            iMesh_ALL_TOPOLOGIES, // int entity_topology_requestor,
            iBase_VERTEX,         // int entity_type_requested,

            // The returned entities matching type/topo requestor
            // For this particular call, it will be the edge, face and/or
            // volume elements that exist in the set.
            &entHdls,            // iBase_EntityHandle** entity_handles,
            &entHdls_allocated,  // int* entity_handles_allocated,
            &entHdls_size,       // int* entity_handles_size,

            // The UNIQUE set of entities ADJACENT to those in entHdls 
            // For this particular call, this will be the set of UNIQUE
            // vertices shared by all the different entities returned 
            // above.
            &adjHdls,            // iBase_EntityHandle** adj_entity_handles,
            &adjHdls_allocated,  // int* adj_entity_handles_allocated,
            &adjHdls_size,       // int* adj_entity_handles_size,

            // The CONCATENATED listing of indices into adjHdls of
            // entites adjacent to entHdls[i] for all i.
            &adjInds,            // int** adj_entity_indices,
            &adjInds_allocated,  // int* adj_entity_indices_allocated,
            &adjInds_size,       // int* adj_entity_indices_size,

            // The starting index into adjIndcs of the list of adjacent
            // entities for entHdles[i] for all i.
            &offs,                // int** offset,
            &offs_allocated,      // int* offset_allocated,
            &offs_size,           // int* offset_size,

            &itapsError);         //int *err)
        CheckITAPSError(itapsMesh, iMesh_getAdjEntIndices, (0,entHdls,adjHdls,adjInds,offs,EoL));

        if (entHdls_size <= 0 || adjHdls_size <= 0)
        {
            ITAPSErrorCleanupHelper(0,entHdls,adjHdls,adjInds,offs,EoL);
            debug3 << "Mesh is empty" << endl;
            return 0;
        }

        //
        // Get the coordinates for the vertex elements.
        //
        double *coords = 0; int coords_allocated = 0, coords_size;
        iMesh_getVtxArrCoords(itapsMesh, adjHdls, adjHdls_size, iBase_INTERLEAVED,
            &coords, &coords_allocated, &coords_size, &itapsError);
        CheckITAPSError(itapsMesh, iMesh_getVtxArrCoords, (0,coords,EoL));

        //
        // If its a 1D or 2D mesh, the coords could be 3-tuples where
        // the 'extra' values are always zero or they could be reduced
        // dimensioned tuples. The ITAPS interface doesn't specify.
        //
        int tupleSize = coords_size / adjHdls_size;

        //
        // Populate the coordinates.  Put in 3D points with z=0 if the mesh is 2D.
        //
        vtkPoints *points  = vtkPoints::New();
        points->SetNumberOfPoints(adjHdls_size);
        float *pts = (float *) points->GetVoidPointer(0);
        for (i = 0; i < adjHdls_size; i++)
        {
            for (j = 0; j < tupleSize; j++)
                pts[i*3+j] = (float) coords[i*tupleSize+j];
            for (j = tupleSize; j < 3; j++)
                pts[i*3+j] = 0.0;
        }
        if (coords_allocated)
            free(coords);

        vtkUnstructuredGrid *ugrid = vtkUnstructuredGrid::New(); 
        ugrid->SetPoints(points);
        points->Delete();

        for (int i = 0; i < entHdls_size; i++)
        {
            int topoType;
            iMesh_getEntTopo(itapsMesh, entHdls[i], &topoType, &itapsError);
            int vtkZoneType = ITAPSEntityTopologyToVTKZoneType(topoType);
            if (vtkZoneType == -1)
            {
                debug3 << "Unsupported VTK zone type for iMesh entity " << i
                       << " and topology type \"" << entTopologies[topoType]
                       << "\"" << endl;
                continue;
            }
            vtkIdType vertIds[8];
            int jj = 0;
            bool valid = true;
            for (int idx = offs[i];
                 idx < ((i+1) < offs_size ? offs[i+1] : offs_size); idx++)
            {
                if (jj >= sizeof(vertIds) / sizeof(vertIds[0]))
                    valid = false;
                if (valid)
                    vertIds[jj++] = adjInds[idx];
                else
                   jj++;
            }
            if (!valid)
            {
                debug3 << "Unsupported vert count " << jj
                       << " for iMesh entity " << i << endl;
                continue;
            }
            ugrid->InsertNextCell(vtkZoneType, jj, vertIds);
        }

        if (entHdls_allocated)
            free(entHdls);
        if (adjHdls_allocated)
            free(adjHdls);
        if (adjInds_allocated)
            free(adjInds);
        if (offs_allocated)
            free(offs);

        return ugrid;

    }
    catch (iBase_Error TErr)
    {
        char msg[512];
        char desc[256];
        desc[0] = '\0';
        int tmpError = itapsError;
#if !defined(ITAPS_GRUMMP)
        iMesh_getDescription(itapsMesh, desc, &itapsError, sizeof(desc));
#endif
        SNPRINTF(msg, sizeof(msg), "Encountered ITAPS error (%d) \"%s\""
            "\nUnable to open file!", tmpError, desc); 
        if (!avtCallback::IssueWarning(msg))
            cerr << msg << endl;
        return 0;
    }
funcEnd: ;
}


// ****************************************************************************
//  Method: avtITAPS_CFileFormat::GetNodalSubsetVar
//
//  Purpose: Return a node-centered variable indicating subset membership of
//  each node in the associated domain (chunk of mesh).
//
//  I would have liked to use iMesh_getEntities and then iMesh_isEntContained.
//  However, I ran into problems where getEntities for entType of iBase_VERTEX
//  did NOT return as many entities as getAllVtxCoords (which returns the
//  coordinate of vertex entities) for same domain set.
//
//  Programmer: Mark C. Miller 
//  Creation:   Tue Apr 22 14:09:11 PDT 2008
//
// ****************************************************************************
vtkDataArray *
avtITAPS_CFileFormat::GetNodalSubsetVar(int domain, const char *varname,
    const vector<iBase_EntitySetHandle> &theSets)
{
    debug3 << "GetNodalSubsetVar: dom=" << domain << ", var=\"" << varname
           << "\", theSets.size()=" << theSets.size() << endl;
    //
    // Look up the mesh in the cache.
    //
    string meshName = metadata->MeshForVar(varname);
    const avtMeshMetaData *mmd = metadata->GetMesh(meshName);
    vtkDataSet *ds = (vtkDataSet *) cache->GetVTKObject(meshName.c_str(),
                                            avtVariableCache::DATASET_NAME,
                                            timestep, domain, "_all");
    if (ds == 0)
    {
        char msg[256];
        SNPRINTF(msg, sizeof(msg), "Cannot find cached mesh \"%s\" for domain %d to "
            "paint subset variable \"%s\"", meshName.c_str(), domain, varname);
        EXCEPTION1(ImproperUseException, msg);
    }

    vtkUnstructuredGrid *ugrid = vtkUnstructuredGrid::SafeDownCast(ds);
    vtkFloatArray *result = 0;
    try {

        int i,j,k;

        //
        // Build a map from coordinate values to node indices.
        //
        result = vtkFloatArray::New();
        result->SetNumberOfTuples(ugrid->GetPoints()->GetNumberOfPoints());
        float *p = (float *) result->GetVoidPointer(0);
        float *pts = (float *) ugrid->GetPoints()->GetVoidPointer(0);
        map<float, map<float, map<float, int> > > coordToIndexMap;
        for (i = 0; i < ugrid->GetPoints()->GetNumberOfPoints(); i++)
        {
            coordToIndexMap[pts[3*i+0]][pts[3*i+1]][pts[3*i+2]] = i;
            p[i] = -1;
        }

        for (j = 0; j < theSets.size(); j++)
        {
            //
            // Examine this set and any of its subsets, looking for primitive,
            // entities. Find the majority type of entity and the set it occurs
            // in and use that as the set to query for vertices. In most cases
            // this will be the entry in theSets[j]. But, not all.
            //
            int ent_type = -1;
            int max_num_ents = 0;
            iBase_EntitySetHandle qSet;
            vector<iBase_EntitySetHandle> setStack;
            setStack.push_back(theSets[j]);
            while (setStack.size())
            {
                iBase_EntitySetHandle aset = setStack.back();
                setStack.pop_back();

                for (k = 0; k < iBase_ALL_TYPES; k++)
                {
                    int num_ents = 0;
                    iMesh_getNumOfType(itapsMesh, aset, k, &num_ents, &itapsError);
                    debug3 << num_ents << " " << entTypes[k] << endl;
                    if (num_ents > max_num_ents)
                    {
                        max_num_ents = num_ents;
                        ent_type = k;
                        qSet = aset;
                    }
                }

                iBase_EntitySetHandle *esHdls = 0; int esHdls_allocated = 0, esHdls_size;
                iMesh_getEntSets(itapsMesh, aset, 1,
                    &esHdls, &esHdls_allocated, &esHdls_size, &itapsError);
                for (k = 0; k < esHdls_size; k++)
                    setStack.push_back(esHdls[k]);
                free(esHdls);
            }
            debug3 << "Querying adjEntIndices using entity_type of \""
                   << entTypes[ent_type] << "\"" << endl;

            double *coords2 = 0; int coords2_allocated = 0, coords2_size;
            iBase_EntityHandle *verts = 0; int verts_allocated = 0, verts_size;
            iBase_EntityHandle *adjHdls = 0; int adjHdls_allocated = 0, adjHdls_size;
            int *adjInds = 0; int adjInds_allocated = 0, adjInds_size;
            int *off = 0, off_allocated = 0, off_size;
            iMesh_getAdjEntIndices(itapsMesh, qSet, ent_type,
                iMesh_ALL_TOPOLOGIES, iBase_VERTEX,
                &adjHdls, &adjHdls_allocated, &adjHdls_size,
                &verts, &verts_allocated, &verts_size,
                &adjInds, &adjInds_allocated, &adjInds_size,
                &off, &off_allocated, &off_size, &itapsError);
            CheckITAPSError(itapsMesh, iMesh_getAdjEntities, (0,verts,adjHdls,adjInds,off,EoL));
            free(off);
            free(adjHdls);
            free(adjInds);
            debug3 << "For set index " << j << " got " << verts_size << " verts." << endl;
            if (verts_size == 0)
            {
                free(verts);
                continue;
            }
            iMesh_getVtxArrCoords(itapsMesh, verts, verts_size, iBase_INTERLEAVED,
                                  &coords2, &coords2_allocated, &coords2_size, &itapsError);
            CheckITAPSError(itapsMesh, iMesh_getVtxArrCoords, (0,coords2,EoL));
            free(verts);

            int tupleSize = coords2_size / verts_size;
            for (int k = 0; k < verts_size; k++)
            {
                int l;
                float thePoint[3];
                for (l = 0; l < tupleSize; l++)
                    thePoint[l] = (float) coords2[k*tupleSize+l];
                for (l = tupleSize; l < 3; l++)
                    thePoint[l] = 0.0;

                int entIndex = -1;
                map<float, map<float, map<float, int> > >::const_iterator itx =
                    coordToIndexMap.find(thePoint[0]);
                if (itx != coordToIndexMap.end())
                {
                    map<float, map<float, int> >::const_iterator ity =
                        itx->second.find(thePoint[1]);
                    if (ity != itx->second.end())
                    {
                        map<float, int>::const_iterator itz =
                            ity->second.find(thePoint[2]);
                        if (itz != ity->second.end())
                        {
                            entIndex = itz->second;
                        }
                    }
                }

                if (entIndex == -1)
                {
                    debug3 << "skipping vert " << k << " because it is not in map" << endl;
                    continue;
                }

                if (p[entIndex] == -1) // haven't assigned this vertex to any set yet
                    p[entIndex] = j;
                else                   // its already been assigned, so add to it
                {
                    double curval = p[entIndex];
                    avtScalarMetaData::UpdateValByInsertingDigit(&curval,
                        theSets.size(), maxCoincidentSets, pascalsTriangleMaps[varname], j);
                    p[entIndex] = float(curval);
                }
            }
            free(coords2);
        }
    }
    catch (iBase_Error TErr)
    {
        char msg[512];
        char desc[256];
        desc[0] = '\0';
        int tmpError = itapsError;
#if !defined(ITAPS_GRUMMP)
        iMesh_getDescription(itapsMesh, desc, &itapsError, sizeof(desc));
#endif
        SNPRINTF(msg, sizeof(msg), "Encountered ITAPS error (%d) \"%s\""
            "\nUnable to open file!", tmpError, desc); 
        if (!avtCallback::IssueWarning(msg))
            cerr << msg << endl;
        return 0;
    }

funcEnd:
    return result;
}

// ****************************************************************************
//  Method: avtITAPS_CFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      domain     The index of the domain.  If there are NDomains, this
//                 value is guaranteed to be between 0 and NDomains-1,
//                 regardless of block origin.
//      varname    The name of the variable requested.
//
//  Programmer: miller -- generated by xml2avt
//  Creation:   Wed Mar 7 17:15:33 PST 2007
//
//  Modifications:
//    Mark C. Miller, Tue Jun 26 11:27:48 PDT 2007
//    Modified for C interface to ITAPS
//
//    Mark C. Miller, Tue Apr 22 21:45:41 PDT 2008
//    Big changes to support MOAB's subsets
//
//    Mark C. Miller, Tue Apr 21 16:02:11 PDT 2009
//    Removed logic attempting avoid repeated calls to getEntities by having
//    them cached in this object. Added logic to deal with variable names
//    that have had 'stuff' added to them because the are same name on
//    different classes of entities.
// ****************************************************************************

vtkDataArray *
avtITAPS_CFileFormat::GetVar(int domain, const char *varname)
{
    vtkDataArray *result = 0;

    //
    // Check if this is a request for one of the 'special' subsetting variables.
    //
    map<string, vector<iBase_EntitySetHandle> >::iterator tlsit;
    for (tlsit = topLevelSets.begin(); tlsit != topLevelSets.end(); tlsit++)
    {
        string className = tlsit->first == "Vertex" ? "Verticies" : tlsit->first + "s";
        if (className == varname)
            return GetNodalSubsetVar(domain, varname, tlsit->second);
    }

    //
    // Scan through known primitive tags looking for one with the given name
    //
    int entType;
    iBase_TagHandle tagToGet = 0;
    // We make 2 passes here to search of the correct tag and entity type. The
    // first pass matches the varname against the tag+addVarName names. The second
    // pass just matches varnames to tagnames.
    for (int pass = 0; pass < 2; pass++)
    {
        for (entType = 0; entType < 4; entType++)
        {
            vector<iBase_TagHandle> tagHandles = primitiveTagHandles[entType];
            for (int tagIdx = 0; tagIdx < tagHandles.size(); tagIdx++)
            {
                iBase_TagHandle theTag = tagHandles[tagIdx]; 
                string tagName = VisIt_iMesh_getTagName(itapsMesh, theTag);
                CheckITAPSError(itapsMesh, VisIt_iMesh_getTagName, NoL);
                int nmsz = tagName.length();
                if (pass == 0 && string(tagName, 0, nmsz)+"_on_"+string(entTypes[entType])+"_entities" == varname)
                {
                    tagToGet = theTag;
                    goto tagFound;
                }
                else if (pass == 1 && string(tagName, 0, nmsz) == string(varname, 0, nmsz))
                {
                    tagToGet = theTag;
                    goto tagFound;
                }
            }
        }
    }
    EXCEPTION1(InvalidVariableException, varname);

tagFound:
    try
    {
        //
        // Get the array of entities if we haven't already
        //
        IMESH_ADEF(iBase_EntityHandle, varEnts);
        iMesh_getEntities(itapsMesh, domainSets[domain], entType,
            iMesh_ALL_TOPOLOGIES, IMESH_AARG(varEnts), &itapsError);
        CheckITAPSError(itapsMesh, iMesh_getEntities, (0,varEnts,EoL));

        //
        // Now, get the tag data and put it in an appropriate vtk data array 
        //
        int tagSizeValues;
        iMesh_getTagSizeValues(itapsMesh, tagToGet, &tagSizeValues, &itapsError);
        CheckITAPSError(itapsMesh, iMesh_getTagSizeValues, NoL);
        int tagTypeId;
        iMesh_getTagType(itapsMesh, tagToGet, &tagTypeId, &itapsError);
        CheckITAPSError(itapsMesh, iMesh_getTagType, NoL);
        int arraySize;
        switch (tagTypeId)
        {
            case iBase_INTEGER:
            {
                int *intArray; int intArray_allocated = 0;
                iMesh_getIntArrData(itapsMesh, varEnts, varEnts_size, tagToGet,
                    &intArray, &intArray_allocated, &arraySize, &itapsError); 
                CheckITAPSError(itapsMesh, iMesh_getIntArrData, (0,intArray,EoL)); 
                if (arraySize != varEnts_size * tagSizeValues)
                {
                    char tmpMsg[256];
                    SNPRINTF(tmpMsg, sizeof(tmpMsg), "getIntArrData() returned %d values "
                        " but VisIt expected %d * %d", arraySize, varEnts_size, tagSizeValues);
                    EXCEPTION1(InvalidVariableException, tmpMsg);
                }
                vtkIntArray *ia = vtkIntArray::New();
                ia->SetNumberOfComponents(tagSizeValues);
                ia->SetNumberOfTuples(varEnts_size);
                for (int i = 0; i < arraySize; i++)
                    ia->SetValue(i, intArray[i]);
                result = ia;
                if (intArray_allocated)
                    free(intArray);
                break;
            }
            case iBase_DOUBLE:
            {
                double *dblArray; int dblArray_allocated = 0;
                iMesh_getDblArrData(itapsMesh, varEnts, varEnts_size, tagToGet,
                    &dblArray, &dblArray_allocated, &arraySize, &itapsError); 
                CheckITAPSError(itapsMesh, iMesh_getDblArrData, (0,dblArray,EoL)); 
                if (arraySize != varEnts_size * tagSizeValues)
                {
                    char tmpMsg[256];
                    SNPRINTF(tmpMsg, sizeof(tmpMsg), "getDblArrData() returned %d values "
                        " but VisIt expected %d * %d", arraySize, varEnts_size, tagSizeValues);
                    EXCEPTION1(InvalidVariableException, tmpMsg);
                }
                vtkDoubleArray *da = vtkDoubleArray::New();
                da->SetNumberOfComponents(tagSizeValues);
                da->SetNumberOfTuples(varEnts_size);
                for (int i = 0; i < arraySize; i++)
                    da->SetValue(i, dblArray[i]);
                result = da;
                if (dblArray_allocated)
                    free(dblArray);
                break;
            }
            case iBase_ENTITY_HANDLE:
            case iBase_BYTES:
            {
                char tmpMsg[256];
                SNPRINTF(tmpMsg, sizeof(tmpMsg), "Unable to handle data type of \"%s\"",
                    itapsDataTypeNames[tagTypeId]);
                EXCEPTION1(InvalidVariableException, tmpMsg);
            }
        }

        IMESH_AFREE(varEnts);
    }
    catch (iBase_Error TErr)
    {
        char msg[512];
        char desc[256];
        desc[0] = '\0';
        int tmpError = itapsError;
#if !defined(ITAPS_GRUMMP)
        iMesh_getDescription(itapsMesh, desc, &itapsError, sizeof(desc));
#endif
        SNPRINTF(msg, sizeof(msg), "Encountered ITAPS error (%d) \"%s\""
            "\nUnable to open file!", tmpError, desc); 
        if (!avtCallback::IssueWarning(msg))
            cerr << msg << endl;
        return 0;
    }

funcEnd:
    return result;
}


// ****************************************************************************
//  Method: avtITAPS_CFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      domain     The index of the domain.  If there are NDomains, this
//                 value is guaranteed to be between 0 and NDomains-1,
//                 regardless of block origin.
//      varname    The name of the variable requested.
//
//  Programmer: miller -- generated by xml2avt
//  Creation:   Wed Mar 7 17:15:33 PST 2007
//
// ****************************************************************************

vtkDataArray *
avtITAPS_CFileFormat::GetVectorVar(int domain, const char *varname)
{
    return GetVar(domain, varname);
}
