/*****************************************************************************
*
* Copyright (c) 2000 - 2007, The Regents of the University of California
* Produced at the Lawrence Livermore National Laboratory
* All rights reserved.
*
* This file is part of VisIt. For details, see http://www.llnl.gov/visit/. The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or materials provided with the distribution.
*  - Neither the name of the UC/LLNL nor  the names of its contributors may be
*    used to  endorse or  promote products derived from  this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED.  IN  NO  EVENT  SHALL  THE  REGENTS  OF  THE  UNIVERSITY OF
* CALIFORNIA, THE U.S.  DEPARTMENT  OF  ENERGY OR CONTRIBUTORS BE  LIABLE  FOR
* ANY  DIRECT,  INDIRECT,  INCIDENTAL,  SPECIAL,  EXEMPLARY,  OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                            avtHDF_UCFileFormat.C                          //
// ************************************************************************* //

#include <avtHDF_UCFileFormat.h>

#include <string>
#include <vector>
#include <algorithm> 

#include <vtkCellTypes.h>
#include <vtkFloatArray.h>
#include <vtkRectilinearGrid.h>
#include <vtkUnstructuredGrid.h>

#include <avtDatabaseMetaData.h>
#include <avtDataRangeSelection.h>
#include <avtHistogramSpecification.h>
#include <avtIdentifierSelection.h>

#include <DBOptionsAttributes.h>
#include <Expression.h>

#include <DebugStream.h>
#include <InvalidVariableException.h>
#include <InvalidTimeStepException.h>
#include <UnexpectedValueException.h>

#include <float.h>
#include <Utility.h>

#include <TimingsManager.h>

using     std::string;
using     std::vector;

string plugin = "HDF_UC:: ";


// ****************************************************************************
//  Method: avtHDF_UCFileFormat constructor
//
//  Programmer: prabhat -- generated by xml2avt
//  Creation:   Wed Dec 19 09:24:48 PDT 2007
//
// ****************************************************************************

avtHDF_UCFileFormat::avtHDF_UCFileFormat(const char *filename, DBOptionsAttributes *readOpts)
    : avtMTSDFileFormat(&filename, 1)
{
    is2D = false;
    coordType = 0; // Cartesian
    if (readOpts != NULL)
        fbIsAllowed = readOpts->GetBool("Use FastBit");
    else
        fbIsAllowed = true;

    debug4<< "fbIsAllowed = " << fbIsAllowed << endl<< endl;
    
    processAllTimesteps = false;
    readerTimestep = -1;
    findAllTimesteps(filename);
    
    histoCache = new histogramCache();
}


// ****************************************************************************
//  Method: avtHDF_UCFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: prabhat -- generated by xml2avt
//  Creation:   Wed Dec 19 09:24:48 PDT 2007
//
// ****************************************************************************

void
avtHDF_UCFileFormat::FreeUpResources(void)
{
    reader.closeFile();
}


static void 
CheckFileCallback(void *args, const std::string &fname, bool isDir,
                                       bool perms, long filesize)
{
  avtHDF_UCFileFormat *ff = (avtHDF_UCFileFormat *) args;
  ff->AddFileInThisDirectory(fname);
}

void
avtHDF_UCFileFormat::findAllTimesteps(const char *filename)
{
  // Following instructions from 
  // http://visitusers.org/index.php?title=Collab:VisItWiki_filesystem_searches
  char slashChar = '/';
#if defined(_WIN32)
  slashChar = '\\';
#endif
  
  int last_slash = -1;
  
  string fileOpened = string(filename);

  while (fileOpened.find(slashChar, last_slash+1) != string::npos)
    {
      last_slash = fileOpened.find(slashChar, last_slash+1);
    }
  
  string dir = fileOpened.substr(0, last_slash);
  ReadAndProcessDirectory(dir, CheckFileCallback, this);

  for (int i=0; i<fileNames.size(); i++)
    debug4 << "File " << i << " is " << fileNames[i] << endl;
  
  processAllTimesteps = true;
}


void
avtHDF_UCFileFormat::AddFileInThisDirectory(const std::string &filenameWithDir)
{
  char slashChar = '/';
#if defined(_WIN32)
  slashChar = '\\';
#endif
  
  int last_slash = -1;
  while (filenameWithDir.find(slashChar, last_slash+1) != string::npos)
    {
      last_slash = filenameWithDir.find(slashChar, last_slash+1);
    }
  
  // Filename is filenameWithDir without the directory qualification.
  std::string filename = filenameWithDir.substr(last_slash+1);
  
  // now that we have the filename, we need to find 2 patterns
  // test_electrons_ 
  // and h5part
  //  if ((filename.find("test_electrons_") != string::npos) && 
  //      (filename.find(".h5part") != string::npos))
  if (filename.find(".h5part") != string::npos)
    fileNames.push_back(filenameWithDir);
 
  sort(fileNames.begin(), fileNames.end());

}

int
avtHDF_UCFileFormat::GetNTimesteps(void)
{
  if (!processAllTimesteps) {
    debug4 << "WARNING: GetNTimesteps called before all timesteps were processed " << endl;
    EXCEPTION1(InvalidVariableException, "All Timestpes have to be processed");
  }

  debug4 << plugin << "returning " << fileNames.size() << " timesteps.. " << endl;
  return fileNames.size();
}

void 
avtHDF_UCFileFormat::ActivateTimestep(int ts) 
{
  currentTimestep = ts;
  //  debug4 << plugin << "ActivateTimestep setting to " << currentTimestep << endl;
}


// This function is needed in case the new timestep is different from currentTimestep
// If so, we need to close the existing reader and start a new one
void
avtHDF_UCFileFormat::updateReader(int newTimestep, bool runquery)
{
  //  debug4<<plugin<<":updateReader newtime="<<newTimestep<< ", readerTime="<<readerTimestep<<", runQuery="<<runquery<<endl;
  if (newTimestep == readerTimestep)
    return;
  
  reader.closeFile();
  bool h5part = fileNames[newTimestep].find(".h5part") != string::npos;
  
  reader.openFile(fileNames[newTimestep], h5part);
  readerTimestep = newTimestep;

  //  debug4 << plugin << " updateReader switching to new timestep " << readerTimestep << endl;

  if (querySpecified && runquery) 
    runQuery();
}

// queryString and queryResults are implied...
void
avtHDF_UCFileFormat::runQuery()
{
  //  debug4<<"Query string is " << queryString << endl;
  // read the relevant indices at this point; get the actual data in GetVar() and GetMesh()
  queryResults.clear();
  reader.executeQuery((char*)queryString.c_str(), 0, queryResults);
  debug4<<"Execute Query resulted in " << queryResults.size() << " hits.." << endl;
}




// ****************************************************************************
//  Method: avtHDF_UCFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: prabhat -- generated by xml2avt
//  Creation:   Wed Dec 19 09:24:48 PDT 2007
//
// ****************************************************************************

void
avtHDF_UCFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md)
{
    int  i;

    bool h5part = fileNames[0].find(".h5part") != string::npos ;    
    reader.openFile(fileNames[0], h5part);
    currentTimestep = 0;

    std::vector<std::string> vnames;
    reader.getVariableNames(vnames);

    avtMeshMetaData *pmesh = new avtMeshMetaData;

    is2D = true;
    unsigned int cartesian = 0;
    unsigned int cylindrical = 0;
    unsigned int spherical = 0;

    for (i = 0 ; i < vnames.size() ; i++)
    {
        if (vnames[i] == "x" || vnames[i] == "y" || vnames[i] == "z" )
          ++cartesian;

        if (vnames[i] == "r" || vnames[i] == "phi" || vnames[i] == "z" )
          ++cylindrical;

        if (vnames[i] == "r" || vnames[i] == "phi" || vnames[i] == "theta" )
          ++spherical;

        if (vnames[i] == "z" || vnames[i] == "theta")
            is2D = false;
    }

    if( spherical == 3)           { coordType = 2; }
    else if( cylindrical >= 2)    { coordType = 1; }
    else /* if( cartesian >= 2)*/ { coordType = 0; }

    debug4 << (is2D ? "2d Mesh  " :  "3d Mesh  ")
         << (coordType==0 ? "Cartesian" :
             (coordType==1 ? "Cylindrical" :
              (coordType==2 ? "Spherical" : "Unknown") ) )
         << endl;

    pmesh->name = "particles";
    pmesh->originalName = "particles";
    pmesh->meshType = AVT_POINT_MESH;
    pmesh->topologicalDimension = 0;
    pmesh->spatialDimension = (is2D ? 2 : 3);
    debug4 << "Spatial dimension: " << pmesh->spatialDimension << endl;
    pmesh->hasSpatialExtents = false;
    md->Add(pmesh);

    for (i=0; i < vnames.size(); i++)
    {
      /*
        if (vnames[i] == "x")
        continue;
        if (vnames[i] == "y")
        continue;
        if (vnames[i] == "z")
        continue;
      */
      AddScalarVarToMetaData(md, vnames[i], "particles", AVT_NODECENT);
    }
}


// ****************************************************************************
//  Method: avtHDF_UCFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: prabhat -- generated by xml2avt
//  Creation:   Wed Dec 19 09:24:48 PDT 2007
//
// ****************************************************************************

vtkDataSet *
avtHDF_UCFileFormat::GetMesh(int ts, const char *meshname)
{
  char *coord_names[3][3] = { { "x", "y", "z" },
                              { "r", "phi", "z" },
                              { "r", "phi", "theta" } };

  int t1 = visitTimer->StartTimer();
  
  vtkUnstructuredGrid *dataset = vtkUnstructuredGrid::New();
  vtkPoints *vtkpoints = vtkPoints::New();
  
  float *pts;

  unsigned long int numParticles = 0;

  // For queries the number of particles is already known so set up the
  // need vtk variables. Otherwise for reading all do it after the
  // first point variable is read in.
  if (readAllData )
    numParticles = 0;
  else {
    numParticles = queryResults.size();

    vtkpoints->SetNumberOfPoints((vtkIdType) numParticles);

    pts = (float *) vtkpoints->GetVoidPointer(0);

    printOffsets();

    debug4<<"GetMesh:: " << numParticles << " particles... " << endl;
  }

  //  debug4<<"avtHDF_UCFileFormat::GetMesh()" << endl;

  updateReader(ts, true);

  for(unsigned int i=0; i<3; ++i )
  {
    if (i == 2 && is2D) {
      for (unsigned long j=0; j<numParticles; j++) // 2D data
        pts[3*j+i] = 0.;
      
      continue;
    }

//      debug4<<"GetMesh:: readAlldata" << endl;
    vector<int64_t> dims;
    BaseFileInterface::DataType type;

    reader.getVariableInformation(coord_names[coordType][i], 0, dims, &type);
    
    if (dims.size() != 1)
      EXCEPTION1(InvalidVariableException, meshname);
    
    // Set up the need vtk variables. This step was already done for
    // the queries outside ofthe loop.
    if (readAllData ) {
      if( i == 0 ) {
        numParticles = dims[0];

        vtkpoints->SetNumberOfPoints((vtkIdType) numParticles);

        pts = (float *) vtkpoints->GetVoidPointer(0);

        debug4<<"GetMesh:: " << numParticles << " particles... " << endl;
      }
      else if (dims[0] != numParticles) {
        debug4<<coord_names[coordType][i]
              <<" dimensions != numParticles"<<endl;
        EXCEPTION1(InvalidVariableException, coord_names[coordType][i]);
      }
    }

    if (type==BaseFileInterface::H5_Double) {
      double *data = new double[numParticles];
      
      if (readAllData)
        reader.getData(coord_names[coordType][i], 0, data);
      else
        reader.getPointData(coord_names[coordType][i], 0, data, queryResults);
      
      for (unsigned long int j=0; j<numParticles; j++)
        pts[3*j+i] = data[j];
      
      delete[] data;
    }
    else if (type==BaseFileInterface::H5_Float) {
      float *data = new float[numParticles];
      
      if (readAllData)
        reader.getData(coord_names[coordType][i], 0, data);     
      else
        reader.getPointData(coord_names[coordType][i], 0, data, queryResults);

      for (unsigned long int j=0; j<numParticles; j++)
        pts[3*j+i] = data[j];

      delete[] data;
    }
    else {
      debug4 << coord_names[coordType][i]
             <<" is neither float nor double"<<endl;
      EXCEPTION1(InvalidVariableException, coord_names[coordType][i]);
    }

  }
  
  //printMesh(numParticles, pts, "after converting to pts array");
  
  dataset->Allocate(numParticles*2);
  for (unsigned long i=0; i < numParticles ; i++)
  {
    vtkIdType onevertex = (vtkIdType) i;
    dataset->InsertNextCell(VTK_VERTEX, 1, &onevertex);
  }

  dataset->SetPoints(vtkpoints);
  vtkpoints->Delete();
  
  visitTimer->StopTimer(t1, "HDF_UC::GetMesh ");
  return dataset;
}


// ****************************************************************************
//  Method: avtHDF_UCFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      varname    The name of the variable requested.
//
//  Programmer: prabhat -- generated by xml2avt
//  Creation:   Wed Dec 19 09:24:48 PDT 2007
//
// ****************************************************************************

vtkDataArray *
avtHDF_UCFileFormat::GetVar(int ts, const char *varname)
{
    int t1 = visitTimer->StartTimer();
    BaseFileInterface::DataType type;
    vector<int64_t> dims;
    double *VD = NULL;
    float *VF = NULL;
    int numParticles = 0;
    
    //    debug4<<"avtHDF_UCFileFormat::GetVar()" << endl;
    
    updateReader(ts, true);
    
    if (readAllData)
      {
        //        debug4<<"GetVar:: readAlldata" << endl;
        reader.getVariableInformation(varname, 0, dims, &type);
        if (dims.size() != 1)
          EXCEPTION1(InvalidVariableException, varname);
        numParticles = dims[0];
        debug4<<"GetVar() reading all " << numParticles << " particles...." << endl;
        
        if (type==BaseFileInterface::H5_Double) {
          VD = new double[numParticles];
          reader.getData(varname, 0, VD);
        }
        else if (type==BaseFileInterface::H5_Float) {
          VF = new float[numParticles];
          reader.getData(varname, 0, VF);
        }
      }
    else
      {
        // we are going to read a subset of the data based on the query that was previously executed
        // in RegisterDataSelections()
        numParticles = queryResults.size();
        debug4<<"GetVar() returning subset of " << numParticles << " particles...." << endl;
        
        reader.getVariableInformation(varname, 0, dims, &type);
        if (type==BaseFileInterface::H5_Double) {
          VD = new double[numParticles];
          reader.getPointData(varname, 0, VD, queryResults);
        }
        else if (type==BaseFileInterface::H5_Float) {
          VF = new float[numParticles];
          reader.getPointData(varname, 0, VF, queryResults);
        }
        
      }
    
    vtkFloatArray *rv = vtkFloatArray::New();
    rv->SetNumberOfTuples(numParticles);
    
    if (type==BaseFileInterface::H5_Double) {
      for (int i = 0 ; i < numParticles ; i++)
        rv->SetTuple1(i, VD[i]);
      
      delete[]VD;
    }
    else if (type==BaseFileInterface::H5_Float) {
      for (int i = 0 ; i < numParticles ; i++)
        rv->SetTuple1(i, VF[i]);      
      
      delete[]VF;
    }
    
    visitTimer->StopTimer(t1, "HDF_UC::GetVar ");
    return rv;
}


// ****************************************************************************
//  Method: avtHDF_UCFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      varname    The name of the variable requested.
//
//  Programmer: prabhat -- generated by xml2avt
//  Creation:   Wed Dec 19 09:24:48 PDT 2007
//
// ****************************************************************************

vtkDataArray *
avtHDF_UCFileFormat::GetVectorVar(int ts, const char *varname)
{
    return NULL;
}


// ****************************************************************************
//  Method: avtHDF_UCFileFormat::GetAuxiliaryData
//
//  Purpose:
//      Gets the auxiliary data from a HDF_UC file.
//
//  Arguments:
//      var        The variable of interest.
//      type       The type of auxiliary data.
//      <unnamed>  The arguments for that -- not used for any Silo types.
//
//  Returns:    The auxiliary data.  Throws an exception if this is not a
//              supported data type.
//
//  Programmer: Prabhat
//  Creation:   February 26, 2008
//
//  Modifications:
//
//    Hank Childs, Thu Mar  6 08:59:57 PST 2008
//    Add skeleton for creating an avtIdentifierSelection.
//
//    Hank Childs, Thu Jan  7 16:46:10 PST 2010
//    Handle the case where an expression is sent in more gracefully.
//
// ****************************************************************************

void *
avtHDF_UCFileFormat::GetAuxiliaryData(const char *var, int ts, const char *type, 
                                      void *s, DestructorFunction &df)
{
  int t1 = visitTimer->StartTimer();
  
  //  debug4 << plugin << "inside GetAuxiliaryData " << endl<<endl;
  updateReader(ts, false);
  
    if (strcmp(type, AUXILIARY_DATA_HISTOGRAM) == 0)
    {
      debug4 << "HDF_UC TRYING TO GET HISTOGRAM!!" << endl;
      avtHistogramSpecification *spec = (avtHistogramSpecification *) s;
      TRY
      {
          ConstructHistogram(spec);
      }
      CATCHALL
      {
          debug1 << "\nException thrown when constructing a histogram." << endl;
          debug1 << "This is normal when an expression is passed in.\n" << endl;
      }
      ENDTRY
      
      // We don't return the spec ... it was passed in as an input, which we
      // then populated.
      visitTimer->StopTimer(t1, "HDFV_UC::GetAuxilitaryData ");
      return NULL;
    }

    if (strcmp(type, AUXILIARY_DATA_IDENTIFIERS) == 0)
    {
      debug4 << "HDF_UC TRYING TO GET AUXILIARY DATA FOR IDENTIFIERS " << endl<<endl<<endl;
      std::vector<avtDataSelection *> *ds = (std::vector<avtDataSelection *> *) s;
      std::vector<avtDataSelection *> drs;
      for (int i = 0 ; i < ds->size() ; i++)
        {
          if ((strcmp((*ds)[i]->GetType(), "Data Range Selection") == 0)||((strcmp((*ds)[i]->GetType(), "Identifier Data Selection") == 0)))
            drs.push_back((*ds)[i]);
          else
            return NULL;
        }
      
      avtIdentifierSelection *ids = ConstructIdentifiersFromDataRangeSelection(drs);
      df = avtIdentifierSelection::Destruct;
      return (void *) ids;
        visitTimer->StopTimer(t1, "HDFV_UC::GetAuxilitaryData ");
    }
    visitTimer->StopTimer(t1, "HDFV_UC::GetAuxilitaryData ");
    return NULL;
}


// ****************************************************************************
//  Method: avtHDF_UCFileFormat::ConstructHistogram
//
//  Purpose:
//      Given a specification, fill in the histogram data through FastBit calls
//
//  Arguments:
//      spec       The input specification
//
//  Returns:    Specification with count/bounds data filled in. Need to determine 
//              appropriate exceptions to throw
//
//  Programmer: Prabhat
//  Creation:   February 26, 2008
//
// ****************************************************************************

void 
avtHDF_UCFileFormat::ConstructHistogram(avtHistogramSpecification *spec)
{
  int t1 = visitTimer->StartTimer();
  string func = "ConstructHistogram:: ";

  debug4 << plugin << func << endl<<endl;

  if (NULL==spec) 
    EXCEPTION1(InvalidVariableException, "NULL Histogram Specification");
  
  //Compute the hisogram
  int             timestep = spec->GetTimestep();
  bool      regularBinning = spec->IsRegularBinning();
  vector<string> variables = spec->GetVariables();
  vector<int>      numBins = spec->GetNumberOfBins();
  bool     boundsSpecified = spec->BoundsSpecified();
  string         condition = spec->GetCondition();   
  
  int boundsSize                     = spec->GetBounds().size();
  VISIT_LONG_LONG *        counts    = spec->GetCounts();

  updateReader(timestep, false);
  
  if (timestep!=0) {
    debug4 << plugin << func << " WARNING: code needs to be tested.. "<< endl;
    //EXCEPTION1(InvalidVariableException, "Timestep has to be 0 for now");
  }

  /*if (!regularBinning) {
    debug4 << plugin << func << " Need to implement adaptive binning " << endl;
    EXCEPTION1(InvalidVariableException, "regularBinning");
  }*/

  for (int i=0; i<numBins.size(); i++) 
    if (regularBinning) {
      if (numBins[i]<1) {
        debug4 << plugin << func << " Need to specify a valid #bins for histogram " << endl;
        EXCEPTION1(InvalidVariableException, "numBins");
      }
    }
  
  if (variables.empty()) {
    debug4 << plugin << func << " Histogram Spec needs to specify variables to plot " << endl;
    EXCEPTION1(InvalidVariableException, "variables");
  }
  
  for (int i=0; i<variables.size(); i++){
    bool found = reader.checkForVariable(variables[i]);
    if (!found) {
      debug4 << plugin << func << " Histogram Spec lists variable not present in plugin " << endl;
      EXCEPTION1(InvalidVariableException, variables[i]);
    }
  }
  
  if( boundsSpecified ){
    if (variables.size()!= boundsSize) {
      debug4 << plugin << func << " Histogram Spec variable list does not match bounds list size " << endl;
      EXCEPTION1(InvalidVariableException, "Variable bounds list mismatch");
    }
  }
  
    //Try to retrieve the histogram from cache
  bool isCached = false;
  isCached = histoCache->getCached(spec);
  if( isCached ){
      boundsSpecified = spec->BoundsSpecified();  
      debug4 << plugin << func << "Histogram retrieved from cache "<<endl;
  }  
  
  
  //Define the begin and end 
  vector<double> begins;
  vector<double> ends;
  begins.resize(variables.size());
  ends.resize(variables.size());
  //if bounds are specified then copy the extends into the histogram
  if( boundsSpecified ){
      for (int i=0; i<variables.size(); i++) {
            begins[i] =  spec->GetBounds()[i][0];
            ends[i]   =  spec->GetBounds()[i][spec->GetBounds()[i].size()-1];
            debug4<<"boundsSpecified to be "<< begins[i] <<", "<<ends[i]<<std::endl;
      }
  }
  //if bounds are not specified than ask the reader for the extents 
  else{
    debug4<< plugin << func << "Detected that bounds are not set.. setting them to.."<<std::endl;
    for(int i=0; i<variables.size() ; i++){

      
      vector<int64_t> dims;
      BaseFileInterface::DataType type;
      reader.getVariableInformation(variables[i], 0, dims, &type);

      if (type==BaseFileInterface::H5_Float) {
        float start, end;
        std::string varname = variables[i];
        reader.getDataMinMax(varname,
                             0,
                             type,
                             (void*)&start, 
                             (void*)&end);
        begins[i] = start;
        ends[i] = end;
      }
      else if (type==BaseFileInterface::H5_Double) {
        double start, end;
        std::string varname = variables[i];
        reader.getDataMinMax(varname,
                             0,
                             type,
                             (void*)&start,
                             (void*)&end);
        begins[i] = start;
        ends[i] = end;
      }
      else {
        debug4<<"Warning:: data type min/max not yet supported"<<endl;
      }
      
      
      debug4<< "\t HDF-FQ/FastBit calculated " << variables[i] << " bounds to be.. " 
               << begins[i] << ", "<< ends[i] << endl<<endl;
    }
  }
  
  bool cacheCurrentHisto = false;
  
  // We need to check the dimensionality of the histogram to determine what FastBit call
  // to make. We'll assume that the bounds contain junk for now and that num_bins is what will
  // drive the histogram creation.
  if( !isCached )
  if (variables.size()==1) {
    
    vector<uint32_t> count;
    reader.get1DHistogram((int64_t)0, 
                          condition.c_str(),
                          variables[0].c_str(),
                          numBins[0],
                          spec->GetBounds()[0],
                          count);
    spec->SetBoundsSpecified(); //Make sure that the bounds specified flag is defined properly
    //TODO Possibly replace in order to avoid copying of the histograms
    spec->SetCounts( count );   //Copy the counts into the specification
    cacheCurrentHisto = true;
  }
  else if (variables.size()==2) {
    
    vector<uint32_t> count;
    double begin1 = begins[0];
    double end1 = ends[0];
    
    double begin2 = begins[1];
    double end2 = ends[1];
    
    debug4 << "BEGIN1 = " << begin1 << ", END1 " << end1 << ", begin2 " << begin2 << ", end2 = " << end2 << endl;
    debug4 << "numbins1 = " << numBins[0] << ", numBins2 = " << numBins[1] << endl;
  
    if( regularBinning ){ //if regular binning should be used
            reader.get2DHistogram((int64_t)0,
                          condition.c_str(),
                          variables[0].c_str(),
                          begin1, end1, numBins[0], 
                          variables[1].c_str(),
                          begin2, end2, numBins[1],
                          spec->GetBounds()[0] , spec->GetBounds()[1] ,
                          count);
    }else{ //if adaptive binning should be used
          if( condition.empty() ){   //do we have a condition   
                  debug4<<"********************************HDF_UC*************** No condition"<<endl;
                  reader.get2DAdaptiveHistogram(
                              (int64_t)0,
                              variables[0].c_str(),
                              variables[1].c_str(),
                              numBins[0],
                              numBins[1],
                              spec->GetBounds()[0] , spec->GetBounds()[1] ,
                              count);
          }else{ //without a condition
                  debug4<<"********************************HDF_UC*************** Condition"<<endl;               
                  reader.get2DAdaptiveHistogram(
                              (int64_t)0,
                              condition.c_str(),
                              variables[0].c_str(),
                              variables[1].c_str(),
                              numBins[0],
                              numBins[1],
                              spec->GetBounds()[0] , spec->GetBounds()[1] ,
                              count );
          }
    }
    //debug4<<"**********************HDF_UC*************"<<spec->GetBounds().size()<<" "<<spec->GetBounds()[0].size()<<" "<<spec->GetBounds()[1].size()<<endl;     
    
    spec->SetBoundsSpecified();
    spec->SetCounts( count );   //Copy the counts into the
                                //specification

    //debug4<<"**********************HDF_UC*************"<<spec->GetNumberOfBins()[0]<<" "<<spec->GetNumberOfBins()[1]<<endl;      
 
    
    //    debug4<<"Computed histogram successfully and counts are set!"<<endl<<endl;
    cacheCurrentHisto = true;
  }
  
  else { // generic histogram, will need to take slices..
    debug4 << plugin << func << " Need to implement generic histogram slices.. " << endl;
    EXCEPTION1(InvalidVariableException,"variables");
  }
  
  //Add the current histogram to the cache
  if( cacheCurrentHisto )
        histoCache->addToCache(spec);
  
  visitTimer->StopTimer(t1, "HDF-UC::ConstructHistogram ");
}


// ****************************************************************************
//  Method: avtHDF_UCFileFormat::RegisterDataSelections
//
//  Purpose:
//      This is the mechanism that allows the database to communicate to this
//      format what data selections are requested.  This particular file format
//      reader is interested in data range selections, which are produced
//      by the Threshold operator.
//
//  Arguments:
//      sels          the data selections
//      selsApplied   a Boolean list stating whether the file format was able
//                    to apply the selections.
//
//  Programmer: Prabhat
//  Creation:   December 19, 2007
//
// ****************************************************************************

void
avtHDF_UCFileFormat::RegisterDataSelections(
                                const std::vector<avtDataSelection_p> &sels, 
                                std::vector<bool> *selectionsApplied) 
{
  debug4 << plugin << "RegisterDataSelection.." << endl;

  updateReader(currentTimestep, false); 

  this->selList = sels;
  this->selsApplied = selectionsApplied;
  
  userMadeSelection = false;
  querySpecified = false;
  queryString = "";
  
  //Get all available variable names
  std::vector<std::string> varNames; 
  reader.getVariableNames(varNames);
  
  if (fbIsAllowed) {
    debug4 << "selList.size() = " << selList.size() << endl;
    bool firstValidSelection=true; //needed for correct formating of the query string
    for (int i = 0; i < selList.size(); ++i) {
      
      if (std::string(selList[i]->GetType()) == std::string("Data Range Selection")) {
        //Check if the according variable name is available
        avtDataRangeSelection *dr = (avtDataRangeSelection *) *(selList[i]);
        std::string var = dr->GetVariable();       
        bool available = false;    
        for(unsigned int j=0; j<varNames.size() && !available ; ++j ){
              if( varNames[j].compare(dr->GetVariable()) == 0 ){
                  available = true;
              }
        }
        (*(this->selsApplied))[i] = available;
        //If the variable is available then update the query string accordingly
        if( available ){    
              userMadeSelection = true;
              double min = dr->GetMin();
              double max = dr->GetMax();
              std::string min_condition = "("+stringify(min)+"<"+var           +")";
              std::string max_condition = "("+var           +"<"+stringify(max)+")";
              
              if (firstValidSelection)
                    queryString = min_condition + " && " + max_condition;
              else
                    queryString = queryString + "&&" + min_condition + "&&" + max_condition;
              firstValidSelection = false;
              querySpecified = true;
        }
      }
      
      else if (std::string(selList[i]->GetType()) == std::string("Identifier Data Selection")) {
        
        debug4 <<plugin << "Register Data Selection found identifier type" << endl;
        debug4 << "\t for selList[ " << i << "]" << endl;
        
        (*(this->selsApplied))[i] = true;
        userMadeSelection = true;
        avtIdentifierSelection *ids = (avtIdentifierSelection *) *(selList[i]);
        
        const std::vector<double> Identifiers = ids->GetIdentifiers();
        debug4 <<"\t Returned Identifiers list size is " << Identifiers.size() << endl;
        
        string id_string;
        int len = get_string_from_identifiers(Identifiers, id_string);
        
        if (len>0) {

          if (i==0) 
            queryString = id_string;
          else 
            queryString = queryString + "&&" + id_string;
          
          querySpecified = true;
        }
      }
        
    } // over all selList
  }

  if (querySpecified) {
    debug4 << "query Specified " << queryString << endl;
    runQuery();
  } else {
    debug4 << "query NOT specified " << endl;
  }


  if (fbIsAllowed && userMadeSelection)
    readAllData = false;
  else
    readAllData = true;

  //debug4<<fbIsAllowed<<"   "<<userMadeSelection<<" "<<endl;
  //debug4 << "queryString at eof RegisterDataSelection is " << queryString << endl;
}


std::string inline avtHDF_UCFileFormat::stringify(double x)
{
  std::ostringstream o;
  if (!(o << setprecision(32) << x))
    EXCEPTION1(InvalidVariableException, "string conversion");
  //    throw BadConversion("stringify(double)");
  return o.str();
}


// ****************************************************************************
//  Method: avtHDF_UCFileFormat::ConstructIdentifiersFromDataRangeSelection
//
//  Purpose:
//      Constructs the identifiers that fall within a range selection.
//
//  Programmer: Prabhat (Hank added the stub)
//  Creation:   March 6, 2008
//
// ****************************************************************************

avtIdentifierSelection *
avtHDF_UCFileFormat::ConstructIdentifiersFromDataRangeSelection(
                                     std::vector<avtDataSelection *> &drs)
{
  std::string func = "ConstructIdentifiersFromDataRangeSelection";
  debug4<< "Started in " << plugin << func << endl;
  
  avtIdentifierSelection *rv = new avtIdentifierSelection();
  std::vector<double> ids;
  std::string query = "";
  querySpecified = false;
  
  // Go over all the selections
  for (int i=0; i< drs.size(); i++) {

    if (std::string(drs[i]->GetType())==std::string("Data Range Selection")) { // dta Range is valid..
      // create the query string
      avtDataRangeSelection *dr = (avtDataRangeSelection*)drs[i];

      std::string var = dr->GetVariable();
      double min = dr->GetMin();
      double max = dr->GetMax();
      std::string min_condition = "("+stringify(min)+"<"+var           +")";
      std::string max_condition = "("+var           +"<"+stringify(max)+")";
      
      if (i==0)
        query = min_condition + " && " + max_condition;
      else
        query = query + "&&" + min_condition + "&&" + max_condition;
      
      querySpecified = true;
    }
    
    else if (std::string(drs[i]->GetType())==std::string("Identifier Data Selection")) { // identifier selection..

      debug4 << plugin << " Named selection found............."  << endl;
  
      avtIdentifierSelection *id = (avtIdentifierSelection*)drs[i];
      const vector<double>& ids = id->GetIdentifiers();
      
      std::string id_string;
      int len = get_string_from_identifiers(ids, id_string);
      
      if (len>0) {
        if (i==0) 
          query = id_string;
        else 
        query = query + "&&" + id_string;
        
        querySpecified = true;
      }
    }
  }
  debug4 << plugin << func << "query string is " << query << endl;
  debug4 << endl << endl;

  if (querySpecified) {
    
    std::vector<hsize_t> qResults;
    reader.executeQuery((char*)query.c_str(), 0, qResults);
    debug4 << "\t Execution of query string["<<query.size()<<" resulted in " << qResults.size() << " entries.." << endl;
   
    BaseFileInterface::DataType type;
    vector<int64_t> dims;
    double *VD = NULL;
    float *VF = NULL;
    
    reader.getVariableInformation("id", 0, dims, &type); // TODO- id is hardcoded here
    
    if (type==BaseFileInterface::H5_Double) {
      VD = new double[qResults.size()];
      reader.getPointData("id", 0, VD, qResults); 
    }
    else if (type==BaseFileInterface::H5_Float) {
      VF = new float[qResults.size()];
      reader.getPointData("id", 0, VF, qResults); 
    }
    
    if (type==BaseFileInterface::H5_Double) {
      for (int j=0; j<qResults.size(); j++) 
        ids.push_back(VD[j]);
      
      delete [] VD;
    }
    
    else if (type==BaseFileInterface::H5_Float) {
      for (int j=0; j<qResults.size(); j++) 
        ids.push_back(VF[j]);

      delete [] VF;
    }    

    qResults.clear();
  }
  
  rv->SetIdentifiers(ids);

  return rv;
}


int  avtHDF_UCFileFormat::get_string_from_identifiers(const vector<double>& Identifiers, string& id_string) {
  
  id_string = "";
  debug4 << "get_string_from_identifiers id length = " << Identifiers.size() << endl;
  if (Identifiers.size()>0) {
    
    id_string = "( id in ( "; // Prabhat- TODO id is hardcoded here..
    
    for (int j=0; j<Identifiers.size()-1; j++)
      id_string = id_string + stringify(Identifiers[j]) + ", ";
    id_string = id_string + stringify(Identifiers[Identifiers.size()-1]) + " ))";
  }

  debug4 << "id_string is "<< id_string << endl;
  return Identifiers.size();

}

void  avtHDF_UCFileFormat::printOffsets() {
  
  debug4<<"Printing queryResults, size = "<<queryResults.size()<<endl;
  for (int i=0; i<queryResults.size();i++)
    debug4<<queryResults[i]<<", ";
  debug4<<endl;

}

void  avtHDF_UCFileFormat::printMesh(int numParticles, float*ptr, string msg) {

  debug4<<msg<<std::endl;
  debug4<<"\nPrinting mesh with "<< numParticles<<" points************"<<endl;

  for (int i=0; i<numParticles; i++) {
    debug4<<"("<<ptr[3*i+0]<<", "<<ptr[3*i+1]<<", "<<ptr[3*i+2]<<") "<<std::endl;
  }

  debug4<<"****************************"<<std::endl;
}
