/*****************************************************************************
*
* Copyright (c) 2000 - 2010, Lawrence Livermore National Security, LLC
* Produced at the Lawrence Livermore National Laboratory
* LLNL-CODE-400124
* All rights reserved.
*
* This file is  part of VisIt. For  details, see https://visit.llnl.gov/.  The
* full copyright notice is contained in the file COPYRIGHT located at the root
* of the VisIt distribution or at http://www.llnl.gov/visit/copyright.html.
*
* Redistribution  and  use  in  source  and  binary  forms,  with  or  without
* modification, are permitted provided that the following conditions are met:
*
*  - Redistributions of  source code must  retain the above  copyright notice,
*    this list of conditions and the disclaimer below.
*  - Redistributions in binary form must reproduce the above copyright notice,
*    this  list of  conditions  and  the  disclaimer (as noted below)  in  the
*    documentation and/or other materials provided with the distribution.
*  - Neither the name of  the LLNS/LLNL nor the names of  its contributors may
*    be used to endorse or promote products derived from this software without
*    specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT  HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR  IMPLIED WARRANTIES, INCLUDING,  BUT NOT  LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND  FITNESS FOR A PARTICULAR  PURPOSE
* ARE  DISCLAIMED. IN  NO EVENT  SHALL LAWRENCE  LIVERMORE NATIONAL  SECURITY,
* LLC, THE  U.S.  DEPARTMENT OF  ENERGY  OR  CONTRIBUTORS BE  LIABLE  FOR  ANY
* DIRECT,  INDIRECT,   INCIDENTAL,   SPECIAL,   EXEMPLARY,  OR   CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT  LIMITED TO, PROCUREMENT OF  SUBSTITUTE GOODS OR
* SERVICES; LOSS OF  USE, DATA, OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER
* CAUSED  AND  ON  ANY  THEORY  OF  LIABILITY,  WHETHER  IN  CONTRACT,  STRICT
* LIABILITY, OR TORT  (INCLUDING NEGLIGENCE OR OTHERWISE)  ARISING IN ANY  WAY
* OUT OF THE  USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
* DAMAGE.
*
*****************************************************************************/

// ************************************************************************* //
//                            avtH5PartFileFormat.C                           //
// ************************************************************************* //

#include <visit-config.h>
#include <avtH5PartFileFormat.h>

// VTK
#include <vtkCellTypes.h>
#include <vtkFloatArray.h>
#include <vtkDoubleArray.h>
#include <vtkRectilinearGrid.h>
#include <vtkStructuredGrid.h>
#include <vtkUnstructuredGrid.h>

// VisIt
#include <avtDatabaseMetaData.h>

#include <DBOptionsAttributes.h>
#include <Expression.h>

#ifdef HAVE_LIBFASTBIT
#include <avtDataRangeSelection.h>
#include <avtHistogramSpecification.h>
#include <avtIdentifierSelection.h>
#endif

#include <InvalidVariableException.h>
#include <InvalidFilesException.h>
#include <InvalidDBTypeException.h>
#include <BadIndexException.h>
#include <DebugStream.h>

#ifdef PARALLEL
#include <avtParallel.h>
#endif

// ****************************************************************************
//  Method: avtH5PartFileFormat constructor
//
//  Programmer: ghweber -- generated by xml2avt
//  Creation:   Tue eb 9 13:44:50 PST 2010
//
//   Modifications:
//    Gunther H. Weber, Thu Apr  1 18:17:11 PDT 2010
//    Enable domain decomposition per default.
//
// ****************************************************************************

avtH5PartFileFormat::avtH5PartFileFormat(const char *filename, DBOptionsAttributes *readOpts)
    : avtMTSDFileFormat(&filename, 1)
{
    // Defaults
    useFastBitIndex = true;
    disableDomainDecomposition = false;

    // Check options
    if (readOpts != NULL)
        for (int i = 0; i < readOpts->GetNumberOfOptions(); ++i)
            if (readOpts->GetName(i) == "Use FastBit index")
                useFastBitIndex = readOpts->GetBool("Use FastBit index");
            else if (readOpts->GetName(i) == "Disable domain decomposition")
                disableDomainDecomposition = readOpts->GetBool("Disable domain decomposition");

 #ifdef HAVE_LIBFASTBIT
    dataSelectionActive = false;
    querySpecified = false;
    queryString = "";
#endif

    // Here we only open the file broiefly to ensure that it is really an
    // H5Part file
    file = H5PartOpenFile(filename, H5PART_READ);
    if (!file) EXCEPTION1(InvalidFilesException, filename);

    // This function was once supported by H5Part and will be again in the new
    // release. Until it is widely available, comment it out.
    if (H5PartFileIsValid(file) != H5PART_SUCCESS)
    {
        EXCEPTION1(InvalidDBTypeException, "File is not a valid H5Part file.");
    }

    // Activate first time step
    activeTimeStep = 0;
    H5PartSetStep(file, activeTimeStep);

    // Iterate over particle var names
    h5part_int64_t nPointVars = H5PartGetNumDatasets(file);
    if (nPointVars < 0)
    {
        EXCEPTION1(InvalidFilesException,
                "Could not read number of particle variables.");
    }

    // Get information about particle variables
    for (int i=0; i<nPointVars; ++i)
    {
        // Get information. We are interested in name and type
        char varName[maxVarNameLen];
        h5part_int64_t type;
        h5part_int64_t nElem;

        if (H5PartGetDatasetInfo(file, i, varName, maxVarNameLen, &type, &nElem)
                != H5PART_SUCCESS)
        {
                EXCEPTION1(InvalidFilesException,
                        "Could not read particle data set information.");
        }

        // Store information about type in map so that we can access it in GetVar()
        if (particleVarNameToTypeMap.find(varName) != particleVarNameToTypeMap.end())
        {
            EXCEPTION1(InvalidFilesException,
                    "Particle data has two variables with the same name.");
        }
        else 
            particleVarNameToTypeMap[varName] = type;
    }

    particleNSpatialDims = 2;
    unsigned char cartesian = 0;
    unsigned char cylindrical = 0;
    unsigned char spherical = 0;

    // If there is particle data in the file (check needed since a file could contain
    // only block data).
    if (particleVarNameToTypeMap.size())
    {
        // Determine what coordinate system is used and whether the file is 2D or 3D
        for (VarNameToInt64Map_t::const_iterator it = particleVarNameToTypeMap.begin();
                it != particleVarNameToTypeMap.end(); ++it)
        {
            std::string currVarName = it->first; 

            if (currVarName == "x" || currVarName == "y" || currVarName == "z" )
                ++cartesian;

            if (currVarName == "r" || currVarName == "phi" || currVarName == "z" )
                ++cylindrical;

            if (currVarName == "r" || currVarName == "phi" || currVarName == "theta" )
                ++spherical;

            if (currVarName == "z" || currVarName == "theta")
                particleNSpatialDims = 3;;
        }

        if( spherical == 3)           { coordType = sphericalCoordSystem; }
        else if( cylindrical >= 2)    { coordType = cylindricalCoordSystem; }
        else /* if( cartesian >= 2)*/ { coordType = cartesianCoordSystem; }

        debug1 << "File contains a " << particleNSpatialDims << "D mesh ";
        debug1 << "using a " << (coordType == cartesianCoordSystem ? "Cartesian" :
                (coordType == cylindricalCoordSystem ? "Cylindrical" :
                 (coordType == sphericalCoordSystem ? "Spherical" : "Unknown") ) ) <<
            " coordinate system." << std::endl;
    }
        
    // Get information about field variables
    // FIXME: Currently we assume that all fields live on the same mesh. This may not
    // be true.
    h5part_int64_t nFieldVars = H5BlockGetNumFields(file);
    if (nFieldVars < 0)
    {
        EXCEPTION1(InvalidFilesException, "Could not read number of field variables.");
    }
    else if (nFieldVars > 0)
    {
        for (h5part_int64_t idx=0; idx < nFieldVars; ++idx)
        {
            char varName[maxVarNameLen];
            h5part_int64_t gridRank;
            h5part_int64_t gridDims[3];
            h5part_int64_t fieldDims;
            h5part_int64_t type;

            if (H5BlockGetFieldInfo (file, idx, varName, maxVarNameLen,
                        &gridRank, gridDims, &fieldDims, &type) != H5PART_SUCCESS ) 
            {
                EXCEPTION1(InvalidFilesException, "Could not read field information.");
            }
            
            if (fieldDims == 1)
            {
                if (fieldScalarVarNameToTypeMap.find(varName) != fieldScalarVarNameToTypeMap.end())
                {
                    EXCEPTION1(InvalidFilesException,
                            "Field data has two scalar variables with the same name.");
                }
                else 
                {
                    fieldScalarVarNameToTypeMap[varName] = type;
                }
            }
            else
            {
                if (fieldVectorVarNameToTypeMap.find(varName) != fieldVectorVarNameToTypeMap.end())
                {
                    EXCEPTION1(InvalidFilesException,
                            "Field data has two vector variables with the same name.");
                }
                else 
                {
                    fieldVectorVarNameToTypeMap[varName] = type;
                    fieldVectorVarNameToFieldRankMap[varName] = fieldDims;
                }
            }
        }
    }

    // FIXME: We still need to check whether there there are duplicate variable names in field,
    // scalar and vector variables and possibly modify the name so that VisIt can distinguish 
    // between them.

    H5PartCloseFile(file);
    file  = 0; // Mark file as closed
}

// ****************************************************************************
//  Method: avtH5PartFileFormat destructor
//
//  Programmer: ghweber
//  Creation:   Tue eb 9 13:44:50 PST 2010
//
// ****************************************************************************

avtH5PartFileFormat::~avtH5PartFileFormat()
{
}
 

#ifdef HAVE_LIBFASTBIT

// ****************************************************************************
//  Method: avtH5PartFileFormat::GetAuxiliaryData
//
//  Purpose:
//      Gets the auxiliary data, e.g., histograms, from a H5Part file.
//
//  Arguments:
//      var        The variable of interest.
//      type       The type of auxiliary data.
//      <unnamed>  The arguments for that -- not used for any Silo types.
//
//  Returns:    The auxiliary data.  Throws an exception if this is not a
//              supported data type.
//
//  Programmer: Prabhat
//  Creation:   February 26, 2008
//
//  Modifications:
//
//    Hank Childs, Thu Mar  6 08:59:57 PST 2008
//    Add skeleton for creating an avtIdentifierSelection.
//
//    Hank Childs, Thu Jan  7 16:46:10 PST 2010
//    Handle the case where an expression is sent in more gracefully.
// 
//    Gunther H. Weber, Mon Feb 15 20:14:18 PST 2010
//    Merged into new H5Part database plugin
//
// ****************************************************************************

void *
avtH5PartFileFormat::GetAuxiliaryData(const char *var, int ts, const char *type, 
                                      void *s, DestructorFunction &df)
{
    if (strcmp(type, AUXILIARY_DATA_HISTOGRAM) == 0)
    {
        debug5 << "H5Part trying to get histogram!" << std::endl;

        avtHistogramSpecification *spec = (avtHistogramSpecification *) s;
        TRY
        {
            ConstructHistogram(spec);
        }
        CATCHALL
        {
            debug1 << "\nException thrown when constructing a histogram." << endl;
            debug1 << "This is normal when an expression is passed in.\n" << endl;
        }
        ENDTRY

        // We don't return the spec ... it was passed in as an input, which we
        // then populated.
        return NULL;
    }

    if (strcmp(type, AUXILIARY_DATA_IDENTIFIERS) == 0)
    {
        debug5 << "H5Part trying to get auxiliary data for identifiers " << std::endl;

        std::vector<avtDataSelection *> *ds = (std::vector<avtDataSelection *> *) s;
        std::vector<avtDataSelection *> drs;
        for (int i = 0; i < ds->size(); ++i)
        {
            if ((strcmp((*ds)[i]->GetType(), "Data Range Selection") == 0) ||
                ((strcmp((*ds)[i]->GetType(), "Identifier Data Selection") == 0)))
                drs.push_back((*ds)[i]);
            else
                return NULL;
        }

        avtIdentifierSelection *ids = ConstructIdentifiersFromDataRangeSelection(drs);
        df = avtIdentifierSelection::Destruct;

        return (void *) ids;
    }
    return NULL;
}

#endif


// ****************************************************************************
//  Method: avtEMSTDFileFormat::GetNTimesteps
//
//  Purpose:
//      Tells the rest of the code how many timesteps there are in this file.
//
//  Programmer: ghweber -- generated by xml2avt
//  Creation:   Tue Feb 9 13:44:50 PST 2010
//
// ****************************************************************************

int
avtH5PartFileFormat::GetNTimesteps(void)
{
    if (!file)
    {
        file = H5PartOpenFile(filenames[0], H5PART_READ);
#ifdef HAVE_LIBFASTBIT
        reader.openFile(filenames[0], true);
#endif
    } 
    if (!file) EXCEPTION1(InvalidFilesException, "Could not open file.");
    H5PartSetStep(file, activeTimeStep);

    return H5PartGetNumSteps(file);
}


// ****************************************************************************
//  Method: avtH5PartFileFormat::FreeUpResources
//
//  Purpose:
//      When VisIt is done focusing on a particular timestep, it asks that
//      timestep to free up any resources (memory, file descriptors) that
//      it has associated with it.  This method is the mechanism for doing
//      that.
//
//  Programmer: ghweber -- generated by xml2avt
//  Creation:   Tue Feb 9 13:44:50 PST 2010
//
// ****************************************************************************

void
avtH5PartFileFormat::FreeUpResources(void)
{
    if (file)
    {
        H5PartCloseFile(file);
#ifdef HAVE_LIBFASTBIT
        reader.closeFile();
#endif
        file = 0;
    }
}

#ifdef HAVE_LIBFASTBIT

// ****************************************************************************
//  Method: avtH5PartFileFormat::RegisterDataSelections
//
//  Purpose:
//      This is the mechanism that allows the database to communicate to this
//      format what data selections are requested.  This particular database
//      plugin is interested in data range selections, which are produced
//      by the Threshold operator.
//
//  Arguments:
//      sels          the data selections
//      selsApplied   a Boolean list stating whether the file format was able
//                    to apply the selections.
//
//  Programmer: Prabhat
//  Creation:   December 19, 2007
//
//  Modifications:
//    Gunther H. Weber, Thu Feb 11 17:02:13 PST 2010
//    Ported/moved from HDF_UC plugin to H5Part plugin
//    
//
// ****************************************************************************

void
avtH5PartFileFormat::RegisterDataSelections(
                                const std::vector<avtDataSelection_p> &selList, 
                                std::vector<bool> *selsApplied) 
{
    debug5 << "Entering avtH5PartFileFormat::RegisterDataSelection()" << endl;

    dataSelectionActive = false;
    querySpecified = false;
    queryString = "";

    if (useFastBitIndex)
    {
        bool firstValidSelection = true; //needed for correct formating of the query string
        for (int i = 0; i < selList.size(); ++i)
        {
            if (std::string(selList[i]->GetType()) == std::string("Data Range Selection"))
            {
                // Check if the according variable name is available
                avtDataRangeSelection *dr = (avtDataRangeSelection *) *(selList[i]);
                std::string varName = dr->GetVariable();       

                bool available = particleVarNameToTypeMap.find(varName) != particleVarNameToTypeMap.end();
                (*selsApplied)[i] = available;
                debug5 << "H5Part plugin " << (available ? "can" : "*cannot*") << " limit read data based on variable " << varName << std::endl;

                //If the variable is available then update the query string accordingly
                if(available)
                {    
                    dataSelectionActive = true;

                    double min = dr->GetMin();
                    double max = dr->GetMax();

                    std::string min_condition = "("+stringify(min)+"<"+varName           +")";
                    std::string max_condition = "("+varName           +"<"+stringify(max)+")";

                    if (firstValidSelection)
                        queryString = min_condition + " && " + max_condition;
                    else
                        queryString = queryString + "&&" + min_condition + "&&" + max_condition;

                    firstValidSelection = false;
                    querySpecified = true;
                }
            }
            else if (std::string(selList[i]->GetType()) == std::string("Identifier Data Selection"))
            {
                debug5 << "RegisterDataSelection() found identifier type";
                debug5 << " for selList[ " << i << "]" << std::endl;

                (*selsApplied)[i] = true;
                dataSelectionActive = true;
                avtIdentifierSelection *ids = (avtIdentifierSelection *) *(selList[i]);

                const std::vector<double> identifiers = ids->GetIdentifiers();
                debug5 <<"Returned identifiers list size is " << identifiers.size() << endl;

                std::string id_string;
                int len = get_string_from_identifiers(identifiers, id_string);

                if (len>0)
                {

                    if (firstValidSelection)
                        queryString = id_string;
                    else 
                        queryString = queryString + "&&" + id_string;

                    firstValidSelection = false;
                    querySpecified = true;
                }
            }
        } // over all selList

        if (querySpecified)
        {
            debug5 << "Query specified: " << queryString << std::endl;
            queryResults.clear();
            if (!file)
            {
                file = H5PartOpenFile(filenames[0], H5PART_READ);
                reader.openFile(filenames[0], true);
            } 
            if (!file) EXCEPTION1(InvalidFilesException, "Could not open file.");
            H5PartSetStep(file, activeTimeStep);

            reader.executeQuery((char*)queryString.c_str(), activeTimeStep, queryResults);
        }
        else
        {
            debug4 << "NO query specified " << std::endl;
        }
    }
}

#endif


// ****************************************************************************
//  Method: avtH5PartFileFormat::PopulateDatabaseMetaData
//
//  Purpose:
//      This database meta-data object is like a table of contents for the
//      file.  By populating it, you are telling the rest of VisIt what
//      information it can request from you.
//
//  Programmer: ghweber -- generated by xml2avt
//  Creation:   Tue Feb 9 13:44:50 PST 2010
//
// ****************************************************************************

void
avtH5PartFileFormat::PopulateDatabaseMetaData(avtDatabaseMetaData *md, int timeState)
{
    if (particleVarNameToTypeMap.size())
    {
        // AddMeshToMetaData(md, meshname, mt, extents, nblocks, block_origin,
        //                   spatial_dimension, topological_dimension);
        std::string particleMeshName = "particles";
        AddMeshToMetaData(md, particleMeshName, AVT_POINT_MESH, 0 /* extents not known */,
                1 /* single domain */, 0, particleNSpatialDims, 1);
        for (VarNameToInt64Map_t::const_iterator it = particleVarNameToTypeMap.begin();
                it != particleVarNameToTypeMap.end(); ++it)
        {
            // AddScalarVarToMetaData(md, varname, mesh_for_this_var, cent);
            AddScalarVarToMetaData(md, it->first, particleMeshName, AVT_NODECENT);
        }
    }

    // FIXME: Currently we assume that all fields live on the same mesh. This may not
    // be true.
    if (fieldScalarVarNameToTypeMap.size() || fieldVectorVarNameToTypeMap.size())
    {
        std::string fieldMeshName = "fields"; 

        // AddMeshToMetaData(md, meshname, mt, extents, nblocks, block_origin,
        //                   spatial_dimension, topological_dimension);
        AddMeshToMetaData(md, fieldMeshName, AVT_RECTILINEAR_MESH, 0, 1, 0, 3, 3);
        // FIXME: Assume three dimensions for grid data
        // FIXME: Need to determine dimensions of mesh and create a new mesh
        // if dimensions for variables differ

        for (VarNameToInt64Map_t::const_iterator it = fieldScalarVarNameToTypeMap.begin();
                it != fieldScalarVarNameToTypeMap.end(); ++it)
        {
            // AddScalarVarToMetaData(md, varname, mesh_for_this_var, cent);
            AddScalarVarToMetaData(md, it->first, fieldMeshName, AVT_NODECENT);
        }

         for (VarNameToInt64Map_t::const_iterator it = fieldVectorVarNameToFieldRankMap.begin();
                it != fieldVectorVarNameToFieldRankMap.end(); ++it)
        {
            // AddVectorVarToMetaData(md, varname, mesh_for_this_var, cent,vector_dim);
            AddVectorVarToMetaData(md, it->first, fieldMeshName, AVT_NODECENT, it->second);
        }
    }

    if (!disableDomainDecomposition)
        md->SetFormatCanDoDomainDecomposition(true);

    //
    // Here's the way to add expressions:
    //Expression momentum_expr;
    //momentum_expr.SetName("momentum");
    //momentum_expr.SetDefinition("{u, v}");
    //momentum_expr.SetType(Expression::VectorMeshVar);
    //md->AddExpression(&momentum_expr);
    //Expression KineticEnergy_expr;
    //KineticEnergy_expr.SetName("KineticEnergy");
    //KineticEnergy_expr.SetDefinition("0.5*(momentum*momentum)/(rho*rho)");
    //KineticEnergy_expr.SetType(Expression::ScalarMeshVar);
    //md->AddExpression(&KineticEnergy_expr);
    //
}

// ****************************************************************************
//  Method: avtH5PartFileFormat::SelectParticlesToRead
//
//  Purpose:
//      Restrict the "view" in H5Part to those particles that we are
//      interested in: serial, no query -> all particles; serial, query ->
//      query result particles; parallel -> a subset of particles based on
//      query and domain decomposition settings.
//
//  Arguments:
//
//  Programmer: ghweber
//  Creation:   Tue Feb 9 13:44:50 PST 2010
//
//  Modifications:
//    Gunther H. Weber, Thu Apr  1 18:09:09 PDT 2010
//    Fixed computation of index range. Indices passed to H5PartSetView are
//    inclusive!
//
// ****************************************************************************

void
avtH5PartFileFormat::SelectParticlesToRead()
{
    // Make sure we are seeing the entire file
    H5PartSetView(file, -1, -1);

#ifdef PARALLEL
#ifdef HAVE_LIBFASTBIT
    if (useFastBitIndex && dataSelectionActive)
    {
        // There is a query/selection. Read only those particles.
        if (disableDomainDecomposition)
        {
            // Serial version and parallel if domain decomposition disabled
            h5part_int64_t *indices = new h5part_int64_t[queryResults.size()];
            std::copy(queryResults.begin(), queryResults.end(), indices);
            H5PartSetViewIndices(file, indices, queryResults.size());
            delete[] indices;
        }
        else
        {
            h5part_int64_t minParticlesPerProc = 100;
            h5part_int64_t particlesPerProc = std::min(minParticlesPerProc,
                    h5part_int64_t(queryResults.size() / PAR_Size()));
            h5part_int64_t myStart = std::max( h5part_int64_t(PAR_Rank() * particlesPerProc),
                    h5part_int64_t(queryResults.size()));
            h5part_int64_t myEnd = std::max(h5part_int64_t((PAR_Rank()+1) * particlesPerProc),
                    h5part_int64_t(queryResults.size()));
            if (myStart < queryResults.size())
            {
                h5part_int64_t *indices = new h5part_int64_t[myEnd - myStart];
                for (h5part_int64_t i = myStart; i < myEnd; ++i) indices[i-myStart] = queryResults[i];
                H5PartSetViewIndices(file, indices, myEnd - myStart);
                delete[] indices;
            }
            else
            {
                h5part_int64_t dummy[] = { 0 };
                // Empty selection since processor does not have any particles
                H5PartSetViewIndices(file, dummy, 0);
                debug5 << "Insufficient number of particles. ";
                debug5 << "Proc. " << PAR_Rank() << " will not read any data." << std::endl;
            }
        }
    }
    else
    {
#endif
        debug5 << "No selection active or FastBit disabled." << std::endl;
        // No selection, decompose entire data set
        if (!disableDomainDecomposition)
        {
            // Determine what portion of the file we need to read
            int nProcs = PAR_Size();
            int procNo = PAR_Rank();

            h5part_int64_t totalNPoints= H5PartGetNumParticles(file);
            h5part_int64_t idStart = (totalNPoints/nProcs)*procNo;
            h5part_int64_t idEnd;
            if (procNo < nProcs-1) 
                idEnd   = (totalNPoints/nProcs)*(procNo+1)-1;
            else 
                idEnd   = totalNPoints-1;

            // Restrict the view to the particles we want to read on this processor
            H5PartSetView(file, idStart, idEnd);

            debug5 << "SelectParticlesToRead(): Processor " << procNo;
            debug5 << " reads particles from " << idStart << " to " << idEnd << std::endl;
        }
        else
        {
            debug5 << "SelectParticlesToRead(): Domain decomposition disabled." << std::endl;
        }
#ifdef HAVE_LIBFASTBIT
    }
#endif
#else
#ifdef HAVE_LIBFASTBIT
    if (useFastBitIndex && dataSelectionActive)
    {
            h5part_int64_t *indices = new h5part_int64_t[queryResults.size()];
            std::copy(queryResults.begin(), queryResults.end(), indices);
            H5PartSetViewIndices(file, indices, queryResults.size());
            delete[] indices;
    }
    else
    {
        debug5 << "No selection active or FastBit disabled." << std::endl;
    }
#endif
#endif
}



// ****************************************************************************
//  Method: avtH5PartFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: ghweber -- generated by xml2avt
//  Creation:   Tue Feb 9 13:44:50 PST 2010
//
//  Modifications:
//
// ****************************************************************************

vtkDataSet *
avtH5PartFileFormat::GetMesh(int timestate, const char *meshname)
{
    if (strcmp(meshname, "particles") == 0)
    {
        return GetParticleMesh(timestate); // There is only one particle mesh
    }
    else
    {
        return GetFieldMesh(timestate, meshname); // In the future there may be
        // more than one mesh for field data
    }
}

// ****************************************************************************
//  Method: avtH5PartFileFormat::GetMesh
//
//  Purpose:
//      Gets the mesh associated with this file.  The mesh is returned as a
//      derived type of vtkDataSet (ie vtkRectilinearGrid, vtkStructuredGrid,
//      vtkUnstructuredGrid, etc).
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      meshname    The name of the mesh of interest.  This can be ignored if
//                  there is only one mesh.
//
//  Programmer: ghweber -- generated by xml2avt
//  Creation:   Tue Feb 9 13:44:50 PST 2010
//
//  Modifications:
//
// ****************************************************************************

vtkDataSet *
avtH5PartFileFormat::GetParticleMesh(int timestate)
{
    // Switch to appropriate time step (opens file if necessary)
    ActivateTimestep(timestate);

    // Select particles to actually read
    SelectParticlesToRead();

    // Read data
    h5part_int64_t nPoints = H5PartGetNumParticles(file);
    // If we do domain decomposition or particle selection, this will only
    // return the number of particles in the current view, i.e., the number of
    // particles we actually read

    debug5 << "GetParticleMesh(): Reading " << nPoints << " points." << std::endl;;

    if (nPoints ==  0)
        return 0;

    const char *coordNames[3][3] = { { "x", "y", "z" }, {"r", "phi", "z" } , { "r", "phi", "theta" } };
    h5part_int64_t coordValType = particleVarNameToTypeMap[coordNames[coordType][0]]; 
    if (coordValType != particleVarNameToTypeMap[coordNames[coordType][1]] ||
        (particleNSpatialDims > 2) && coordValType != particleVarNameToTypeMap[coordNames[coordType][2]]) 
    {
        EXCEPTION1(InvalidFilesException, "Coordinate data sets do not have the same value type (double, float, ...).");
    }

    void* coordsRead[3] = { 0, 0, 0 };
    for (int i=0; i < particleNSpatialDims; ++i)
    {
        h5part_int64_t status = H5PART_SUCCESS; 

        if (coordValType == H5PART_INT64)
        {
            coordsRead[i] = malloc(sizeof(h5part_int64_t) * nPoints);
            status = H5PartReadDataInt64(file, coordNames[coordType][i],
                    static_cast<h5part_int64_t*>(coordsRead[i]));
        }
        else if (coordValType == H5PART_FLOAT64)
        {
            coordsRead[i] = malloc(sizeof(h5part_float64_t) * nPoints);
            status = H5PartReadDataFloat64(file, coordNames[coordType][i],
                    static_cast<h5part_float64_t*>(coordsRead[i]));
 
        }
        else if (coordValType == H5PART_FLOAT32)
        {
            coordsRead[i] = malloc(sizeof(h5part_float32_t) * nPoints);
            status = H5PartReadDataFloat32(file, coordNames[coordType][i],
                    static_cast<h5part_float32_t*>(coordsRead[i]));
        }
        else
        {
            EXCEPTION1(InvalidFilesException, "Unsupported value type for coordinates.");
        }

        if (status != H5PART_SUCCESS)
          EXCEPTION1(InvalidFilesException, "Could not read coordinates");
    }

    // Reset view
    H5PartSetView(file, -1, -1);

    // Construct VTK data set
    vtkUnstructuredGrid *dataset = vtkUnstructuredGrid::New();
    vtkPoints *vtkpoints = vtkPoints::New();
    // FIXME: Once VisIt supports dobule data, we need to ensure that
    // native precsission is preserved
    vtkpoints->SetNumberOfPoints((vtkIdType) nPoints);

    float *pts = (float *) vtkpoints->GetVoidPointer(0);

    if (coordValType == H5PART_INT64)
    {
        for (h5part_int64_t ptNo = 0; ptNo < nPoints; ptNo++)
        {
            pts[3*ptNo] =
                float(static_cast<h5part_int64_t*>(coordsRead[0])[ptNo]);
            pts[3*ptNo+1] =
                float(static_cast<h5part_int64_t*>(coordsRead[1])[ptNo]);
            if (particleNSpatialDims > 2)
                pts[3*ptNo+2] =
                    float(static_cast<h5part_int64_t*>(coordsRead[2])[ptNo]);
            else
                pts[3*ptNo+2] = 0.0;
        }
    }
    else if (coordValType == H5PART_FLOAT64)
    {
        for (h5part_int64_t ptNo = 0; ptNo < nPoints; ptNo++)
        {
            pts[3*ptNo] =
                float(static_cast<h5part_float64_t*>(coordsRead[0])[ptNo]);
            pts[3*ptNo+1] =
                float(static_cast<h5part_float64_t*>(coordsRead[1])[ptNo]);
            if (particleNSpatialDims > 2)
                pts[3*ptNo+2] =
                    float(static_cast<h5part_float64_t*>(coordsRead[2])[ptNo]);
            else
                pts[3*ptNo+2] = 0.0;
        }
    }
    else if (coordValType == H5PART_FLOAT32)
    {
        for (h5part_int64_t ptNo = 0; ptNo < nPoints; ptNo++)
        {
            pts[3*ptNo] =
                float(static_cast<h5part_float32_t*>(coordsRead[0])[ptNo]);
            pts[3*ptNo+1] =
                float(static_cast<h5part_float32_t*>(coordsRead[1])[ptNo]);
            if (particleNSpatialDims > 2)
                pts[3*ptNo+2] =
                    float(static_cast<h5part_float32_t*>(coordsRead[2])[ptNo]);
            else
                pts[3*ptNo+2] = 0.0;
        }
    }
    else
    {
        EXCEPTION1(InvalidFilesException, "Unsupported value type for coordinates.");
    }

    for (int i=0; i <particleNSpatialDims; ++i) free(coordsRead[i]);

    dataset->Allocate(nPoints);
    for (h5part_int64_t i=0; i<nPoints; ++i)
    {
        vtkIdType onevertex = (vtkIdType) i;
        dataset->InsertNextCell(VTK_VERTEX, 1, &onevertex);
    }
    dataset->SetPoints(vtkpoints);
    vtkpoints->Delete();

    return dataset;
}

// ****************************************************************************
//  Method: avtH5PartFileFormat::GetFieldMesh
//
//  Purpose:
//      Gets the mesh associated with fields of this file
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      domain      The index of the domain.  If there are NDomains, this
//                  value is guaranteed to be between 0 and NDomains-1,
//                  regardless of block origin.
//
//  Programmer: kurts
//  Creation:   Tue Aug 28 17:35:50 PDT 2007
//
//  Modifications:
//    Gunther H. Weber, Fri Apr 17 13:55:07 PDT 2009
//    Read block origin from file.
//
//    Gunther H. Weber, Fri Jul 17 11:11:36 PDT 2009
//    Clean up. Read block origin via dedicated API call.
//
//    Gunther H. Weber, Wed Feb 10 11:47:32 PST 2010
//    Conversion from MD plugin to SD plugin supporting domain decomposition
//    and clean-up.
//
//    Gunther H. Weber, Thu Apr  1 18:14:00 PDT 2010
//    Adapted to GetSubBlock paramter change.
//
// ****************************************************************************

vtkDataSet *
avtH5PartFileFormat::GetFieldMesh(int timestate, const char *meshname)
{
    // Switch to appropriate time step (opens file if necessary)
    ActivateTimestep(timestate);

    // FIXME: assume all fields have the same dimensions
    // get information from first field
    char fieldName[maxVarNameLen];
    h5part_int64_t gridRank;
    h5part_int64_t gridDims[3];
    h5part_int64_t fieldDims;
    h5part_int64_t type;
    if (H5BlockGetFieldInfo (file, 0, fieldName, maxVarNameLen,
            &gridRank, gridDims, &fieldDims, &type) != H5PART_SUCCESS)
        EXCEPTION1(InvalidFilesException, "Could not read field information");

    // Determine which data portion we need to construct a mesh for
    h5part_int64_t subBlockDims[6];
    GetSubBlock(gridDims, subBlockDims);

    double xOrigin = 0;
    double yOrigin = 0;
    double zOrigin = 0;

    if (H5Block3dGetFieldOrigin(file, fieldName,
                &xOrigin, &yOrigin, &zOrigin) != H5PART_SUCCESS)
    {
        EXCEPTION1(InvalidFilesException, "Could not read field origin.");
    }

    debug5 << "xOrigin: " << xOrigin << "\n";
    debug5 << "yOrigin: " << yOrigin << "\n";
    debug5 << "zOrigin: " << zOrigin << "\n";

    double xSpacing = 0;
    double ySpacing = 0;
    double zSpacing = 0;
    
    if (H5Block3dGetFieldSpacing (file, fieldName,
                &xSpacing, &ySpacing, &zSpacing) != H5PART_SUCCESS)
    {
        EXCEPTION1(InvalidFilesException, "Could not read field spacing.");
    }

    debug5 << "xSpacing: " << xSpacing << "\n";
    debug5 << "ySpacing: " << ySpacing << "\n";
    debug5 << "zSpacing: " << zSpacing << "\n";

    vtkFloatArray *coords[3] = {0, 0, 0};
    // set x coordinates
    coords[0] = vtkFloatArray::New();
    coords[0]->SetNumberOfTuples(subBlockDims[1] - subBlockDims[0] + 1);
    float *xarray = (float *) coords[0]->GetVoidPointer(0);
    for (int i=subBlockDims[0]; i <= subBlockDims[1]; i++)
    {
        xarray[i-subBlockDims[0]] = xOrigin + i * xSpacing;
    }

    // set y coordinates
    coords[1] = vtkFloatArray::New();
    coords[1]->SetNumberOfTuples(subBlockDims[3] - subBlockDims[2] + 1);
    float *yarray = (float *) coords[1]->GetVoidPointer(0);
    for (int i=subBlockDims[2]; i <= subBlockDims[3]; i++) {
        yarray[i-subBlockDims[2]] = yOrigin + i * ySpacing;
    }

    // set z coordinates
    coords[2] = vtkFloatArray::New();
    coords[2]->SetNumberOfTuples(subBlockDims[5] - subBlockDims[4] + 1);
    float *zarray = (float *) coords[2]->GetVoidPointer(0);
    for (int i=subBlockDims[4]; i <= subBlockDims[5]; i++) {
        zarray[i-subBlockDims[4]] = zOrigin + i * zSpacing;
    }

    // create vtkRectilinearGrid objects + set dims and coords
    vtkRectilinearGrid *rgrid = vtkRectilinearGrid::New();
    rgrid->SetDimensions(
            coords[0]->GetNumberOfTuples(),
            coords[1]->GetNumberOfTuples(),
            coords[2]->GetNumberOfTuples());

    rgrid->SetXCoordinates(coords[0]);
    coords[0]->Delete();
    rgrid->SetYCoordinates(coords[1]);
    coords[1]->Delete();
    rgrid->SetZCoordinates(coords[2]);
    coords[2]->Delete();

    return rgrid;
}

// ****************************************************************************
//  Method: avtH5PartFileFormat::GetVar
//
//  Purpose:
//      Gets a scalar variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: ghweber -- generated by xml2avt
//  Creation:   Tue Feb 9 13:44:50 PST 2010
//
// ****************************************************************************

vtkDataArray *
avtH5PartFileFormat::GetVar(int timestate, const char *varname)
{
    VarNameToInt64Map_t::const_iterator it = particleVarNameToTypeMap.find(varname);
    if (it == particleVarNameToTypeMap.end())
        return GetFieldVar(timestate, varname);

    // Switch to appropriate time step (opens file if necessary)
    ActivateTimestep(timestate);

    // Select particles to actually read
    SelectParticlesToRead();

    // Read data
    h5part_int64_t nPoints = H5PartGetNumParticles(file);

    if (nPoints == 0)
        return 0;

    vtkDataArray *scalars = 0;
    if (it->second == H5PART_FLOAT64)
    {
        scalars = vtkDoubleArray::New();
        scalars->SetNumberOfTuples(nPoints);
        if (H5PartReadDataFloat64(file, varname,
                    static_cast<h5part_float64_t*>(scalars->GetVoidPointer(0)))
                != H5PART_SUCCESS)
        {
            EXCEPTION1(InvalidVariableException, varname);
        }
    }
    else if (it->second == H5PART_FLOAT32)
    {
        scalars = vtkFloatArray::New();
        scalars->SetNumberOfTuples(nPoints);
        if (H5PartReadDataFloat32(file, varname,
                    static_cast<h5part_float32_t*>(scalars->GetVoidPointer(0)))
                != H5PART_SUCCESS)
        {
            EXCEPTION1(InvalidVariableException, varname);
        }
    }
    else if (it->second == H5PART_INT64)
    {
        // Unlike FLOAT32 and FLOAT64 which should map to float and
        // double we are not sure what int type is used to achieve 64 bits.
        // To be safe, read into separate array and copy.
        h5part_int64_t *idvar = new h5part_int64_t[nPoints];
        if (H5PartReadDataInt64(file, varname, idvar) != H5PART_SUCCESS)
            EXCEPTION1(InvalidVariableException, varname);
        scalars = vtkFloatArray::New(); // FIXME: Can we use an integer array here?
        scalars->SetNumberOfTuples(nPoints);
        float *ptr = static_cast<float*>(scalars->GetVoidPointer(0));
        for (h5part_int64_t i=0; i<nPoints; ++i)
        {
            ptr[i] = float(idvar[i]);
        }
        delete[] idvar;
    }
    else
    {
        EXCEPTION1(InvalidVariableException, varname);
    }

    // Reset view
    H5PartSetView(file, -1, -1);

    return scalars;
}


// ****************************************************************************
//  Method: avtH5PartFileFormat::GetFieldVar
//
//  Purpose:
//      Get a scalar variable associated with the field mesh of this file
//
//  Arguments:
//      timestate   The index of the timestate.  If GetNTimesteps returned
//                  'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: kurts
//  Creation:   Tue Aug 28 17:35:50 PDT 2007
//
//  Modifications:
//    Gunther H. Weber, Fri Jul 17 11:13:39 PDT 2009
//    Clean-up.
//
//    Gunther H. Weber, Wed Feb 10 11:47:32 PST 2010
//    Conversion from MD plugin to SD plugin supporting domain decomposition.
//
//    Gunther H. Weber, Thu Apr  1 18:14:00 PDT 2010
//    Adapted to GetSubBlock paramter change.
//
// ****************************************************************************

vtkDataArray *
avtH5PartFileFormat::GetFieldVar(int timestate, const char* varname)
{
    // Check if we know about variable and get type
    VarNameToInt64Map_t::const_iterator it = fieldScalarVarNameToTypeMap.find(varname);
    if (it == fieldScalarVarNameToTypeMap.end())
        EXCEPTION1(InvalidVariableException, varname);

   // Activate correct time step (opens file if necessary)
    ActivateTimestep(timestate);

    char fieldName[maxVarNameLen];
    h5part_int64_t gridRank;
    h5part_int64_t gridDims[3];
    h5part_int64_t fieldDims;
    h5part_int64_t type;

    // FIXME: This assumes that all blocks have the same mesh
    if (H5BlockGetFieldInfo (file, 0, fieldName, maxVarNameLen,
            &gridRank, gridDims, &fieldDims, &type) != H5PART_SUCCESS)
        EXCEPTION1(InvalidVariableException, "Could not read field information.");

    h5part_int64_t subBlockDims[6];
    GetSubBlock(gridDims, subBlockDims);

    // set field layout
    if(H5BlockDefine3DFieldLayout(file, subBlockDims[0], subBlockDims[1], 
            subBlockDims[2], subBlockDims[3], 
            subBlockDims[4], subBlockDims[5]) != H5PART_SUCCESS)
        EXCEPTION1(InvalidVariableException, "Could not set field layout");

    // read data
    int nValues = (subBlockDims[1] - subBlockDims[0] + 1) *
        (subBlockDims[3] - subBlockDims[2] + 1) *
        (subBlockDims[5] - subBlockDims[4] + 1);

    vtkDataArray *scalars;
    if (it->second == H5PART_FLOAT64)
    {
        scalars = vtkDoubleArray::New();
        scalars->SetNumberOfTuples(nValues);
        if (H5Block3dReadScalarFieldFloat64(file, varname,
                    static_cast<h5part_float64_t*>(scalars->GetVoidPointer(0)))
                != H5PART_SUCCESS)
        {
            EXCEPTION1(InvalidVariableException, varname);
        }
    }
    else if (it->second == H5PART_FLOAT32)
    {
        scalars = vtkFloatArray::New();
        scalars->SetNumberOfTuples(nValues);
        if (H5Block3dReadScalarFieldFloat32(file, varname,
                    static_cast<h5part_float32_t*>(scalars->GetVoidPointer(0)))
                != H5PART_SUCCESS)
        {
            EXCEPTION1(InvalidVariableException, varname);
        }
    }
    else if (it->second == H5PART_INT64)
    {
        // Unlike FLOAT32 and FLOAT64 which should map to float and
        // double we are not sure what int type is used to achieve 64 bits.
        // To be safe, read into separate array and copy.
        h5part_int64_t *idvar = new h5part_int64_t[nValues];
        if (H5Block3dReadScalarFieldInt64(file, varname, idvar) != H5PART_SUCCESS)
        {
            EXCEPTION1(InvalidVariableException, varname);
        }
        scalars = vtkFloatArray::New(); // FIXME: Can we use an integer array here?
        scalars->SetNumberOfTuples(nValues);
        float *ptr = static_cast<float*>(scalars->GetVoidPointer(0));
        for (h5part_int64_t i=0; i<nValues; ++i)
        {
            ptr[i] = float(idvar[i]);
        }
        delete[] idvar;
    }

    return scalars;
}

// ****************************************************************************
//  Method: avtH5PartFileFormat::GetVectorVar
//
//  Purpose:
//      Gets a vector variable associated with this file.  Although VTK has
//      support for many different types, the best bet is vtkFloatArray, since
//      that is supported everywhere through VisIt.
//
//  Arguments:
//      timestate  The index of the timestate.  If GetNTimesteps returned
//                 'N' time steps, this is guaranteed to be between 0 and N-1.
//      varname    The name of the variable requested.
//
//  Programmer: ghweber -- generated by xml2avt
//  Creation:   Tue Feb 9 13:44:50 PST 2010
//
//  Modifications:
//    Kurt Stockinger, Tue Aug 28 17:35:50 PDT 2007
//    Added support for field data
//
//    Gunther H. Weber, Fri Jul 17 11:13:39 PDT 2009
//    Clean-up.
//
//    Gunther H. Weber, Thu Apr  1 18:14:00 PDT 2010
//    Adapted to GetSubBlock paramter change.
//
// ****************************************************************************
vtkDataArray *
avtH5PartFileFormat::GetVectorVar(int timestate, const char *varname)
{
    // Check if we know about variable and get type
    VarNameToInt64Map_t::const_iterator it = fieldVectorVarNameToTypeMap.find(varname);
    if (it == fieldVectorVarNameToTypeMap.end())
        EXCEPTION1(InvalidVariableException, varname);

   // Activate correct time step (opens file if necessary)
    ActivateTimestep(timestate);

    char fieldName[maxVarNameLen];
    h5part_int64_t gridRank;
    h5part_int64_t gridDims[3];
    h5part_int64_t fieldDims;
    h5part_int64_t type;

    // FIXME: This assumes that all blocks have the same size
    if (H5BlockGetFieldInfo (file, 0, fieldName, maxVarNameLen,
            &gridRank, gridDims, &fieldDims, &type) != H5PART_SUCCESS)
        EXCEPTION1(InvalidVariableException, "Could not read field information.");

    h5part_int64_t subBlockDims[6];
    GetSubBlock(gridDims, subBlockDims);

    // set field layout
    if (H5BlockDefine3DFieldLayout (file, 
            subBlockDims[0], subBlockDims[1], 
            subBlockDims[2], subBlockDims[3], 
            subBlockDims[4], subBlockDims[5]) != H5PART_SUCCESS)
        EXCEPTION1(InvalidVariableException, "Could not set field layout");

    // Read data
    int nValues = (subBlockDims[1] - subBlockDims[0] + 1) *
        (subBlockDims[3] - subBlockDims[2] + 1) *
        (subBlockDims[5] - subBlockDims[4] + 1);

    void* vecsRead[3] = { 0, 0, 0 };
    h5part_int64_t status = H5PART_SUCCESS; 
    if (it->second == H5PART_INT64)
    {
        for (int i=0; i<3; ++i)
            vecsRead[i] = malloc(sizeof(h5part_int64_t) * nValues);

        status = H5Block3dRead3dVectorFieldInt64(file, varname,  
                static_cast<h5part_int64_t*>(vecsRead[0]),
                static_cast<h5part_int64_t*>(vecsRead[1]),
                static_cast<h5part_int64_t*>(vecsRead[2]));
    }
    else if (it->second == H5PART_FLOAT64)
    {
        for (int i=0; i<3; ++i)
            vecsRead[i] = malloc(sizeof(h5part_float64_t) * nValues);

        status = H5Block3dRead3dVectorFieldFloat64(file, varname,  
                static_cast<h5part_float64_t*>(vecsRead[0]),
                static_cast<h5part_float64_t*>(vecsRead[1]),
                static_cast<h5part_float64_t*>(vecsRead[2]));
 

    }
    else if (it->second == H5PART_FLOAT32)
    {
        for (int i=0; i<3; ++i)
            vecsRead[i] = malloc(sizeof(h5part_float32_t) * nValues);

        status = H5Block3dRead3dVectorFieldFloat32(file, varname,  
                static_cast<h5part_float32_t*>(vecsRead[0]),
                static_cast<h5part_float32_t*>(vecsRead[1]),
                static_cast<h5part_float32_t*>(vecsRead[2]));
    }
    else
    {
        EXCEPTION1(InvalidFilesException, "Unsupported value type for coordinates.");
    }

 
    // Store data in VTK object
    vtkDataArray *array;
    if (it->second == H5PART_FLOAT64)
    {
        array = vtkDoubleArray::New();
    }
    else
    {
        vtkFloatArray *array = vtkFloatArray::New();
        // FIXME: Possibly check for integer and create an integer artay
    }

    // Set array size
    array->SetNumberOfComponents(3);
    array->SetNumberOfTuples(nValues);

    // Copy data
    if (it->second == H5PART_INT64)
    {
        // FIXME: Possibly check for integer and create an integer artay
        float *vecs = static_cast<float*>(array->GetVoidPointer(0));
        for (h5part_int64_t valNo = 0; valNo < nValues; valNo++)
        {
            vecs[3*valNo] =
                float(static_cast<h5part_int64_t*>(vecsRead[0])[valNo]);
            vecs[3*valNo+1] =
                float(static_cast<h5part_int64_t*>(vecsRead[1])[valNo]);
            vecs[3*valNo+2] =
                float(static_cast<h5part_int64_t*>(vecsRead[2])[valNo]);
        }
    }
    else if (it->second == H5PART_FLOAT64)
    {
        double *vecs = static_cast<double*>(array->GetVoidPointer(0));
        for (h5part_int64_t valNo = 0; valNo < nValues; valNo++)
        {
            vecs[3*valNo] =
                double(static_cast<h5part_float64_t*>(vecsRead[0])[valNo]);
            vecs[3*valNo+1] =
                double(static_cast<h5part_float64_t*>(vecsRead[1])[valNo]);
            vecs[3*valNo+2] =
                double(static_cast<h5part_float64_t*>(vecsRead[2])[valNo]);
        }
    }
    else if (it->second == H5PART_FLOAT32)
    {
        float *vecs = static_cast<float*>(array->GetVoidPointer(0));
        for (h5part_int64_t valNo = 0; valNo < nValues; valNo++)
        {
            vecs[3*valNo] =
                float(static_cast<h5part_float32_t*>(vecsRead[0])[valNo]);
            vecs[3*valNo+1] =
                float(static_cast<h5part_float32_t*>(vecsRead[1])[valNo]);
            vecs[3*valNo+2] =
                float(static_cast<h5part_float32_t*>(vecsRead[2])[valNo]);
        }
    }
    else
    {
        EXCEPTION1(InvalidFilesException, "Unsupported value type for coordinates.");
    }

    for (int i=0; i <3; ++i) free(vecsRead[i]);

    return array;
}

// ****************************************************************************
//  Method: avtH5PartFileFormat::ActivateTimestep
//
//  Purpose:
//      Select the current time step
//
//  Arguments:
//      ts      The new current time step
//
//  Output:
//
//  Programmer: ghweber
//  Creation:   Thu Feb 11 15:38:49 PST 2010
//
// ****************************************************************************

void
avtH5PartFileFormat::ActivateTimestep(int ts)
{
    // Open file if necessary
    if (!file)
    {
        file = H5PartOpenFile(filenames[0], H5PART_READ);
#ifdef HAVE_LIBFASTBIT
        reader.openFile(filenames[0], true);
#endif
        // Force the check ts != activeTimeStep to be true thus ensuring that
        // H5PartSetStep is called for the newly opened file.
        activeTimeStep = -1;
    } 
    if (!file) EXCEPTION1(InvalidFilesException, "Could not open file.");
 
    debug5 << "H5Part: Activating time step " << ts << std::endl;
    if (ts != activeTimeStep)
    {
        activeTimeStep = ts;

        H5PartSetStep(file, ts);

#ifdef HAVE_LIBFASTBIT
        if (querySpecified)
        {
            // Re-run query
            debug5 << "Re-running query in new time step." << std::endl;
            queryResults.clear();
            reader.executeQuery((char*)queryString.c_str(), activeTimeStep, queryResults);
        }
#endif
    }
}

// ****************************************************************************
//  Method: avtH5PartFileFormat::GetSubBlock
//
//  Purpose:
//      Compute extents of a sub bliock for a given processor id
//
//  Arguments:
//      gridDims      Number of nodes in x-/y-/z-direction
//
//  Output:
//      subBlockDims  Range of samples that this processor should read. Order
//                    i_min, i_max, j_min, j_max, k_min, k_max. Indices are
//                    inclusive, i.e., processor should also read samples i_min
//                    and i_max as well as all samples with indices in between.
//
//  Programmer: kurts and ghweber
//  Creation:   Tue Aug 28 17:35:50 PDT 2007
//
//  Modifications:
//    Gunther H. Weber, Thu Apr  1 15:43:05 PDT 2010
//    Use PAR_Rank() instead of block number as parameter. Fix problem that
//    cuased data to be only read partially if domain decomposition was disabled.
//
// ****************************************************************************

void avtH5PartFileFormat::GetSubBlock(h5part_int64_t gridDims[3], h5part_int64_t subBlockDims[6])
{
#ifdef PARALLEL
    if (disableDomainDecomposition)
    {
        debug1 << "Domain decomposition disabled. Returning entire block." << std::endl;
#endif
        subBlockDims[2*0+0] = 0;
        subBlockDims[2*0+1] = gridDims[0] - 1;

        subBlockDims[2*1+0] = 0;
        subBlockDims[2*1+1] = gridDims[1] - 1;

        subBlockDims[2*2+0] = 0;
        subBlockDims[2*2+1] = gridDims[2] - 1;

        debug1 << "avtH5PartFileFormat::GetSubBlock() returning block comprising entire data set:"
            << subBlockDims[0] << " " << subBlockDims[1] << " "
            << subBlockDims[2] << " " << subBlockDims[3] << " "
            << subBlockDims[4] << " " << subBlockDims[5] << std::endl;
#ifdef PARALLEL
    }
    else
    {
        int partitionAxis =0;
        int axis1 = 1;
        int axis2 = 2;

        if (gridDims[1] > gridDims[0] && gridDims[1] > gridDims[2])
        {
            partitionAxis = 1;
            axis1 = 0;
            axis2 = 2;
        }
        else if (gridDims[2] > gridDims[0] && gridDims[2] > gridDims[1])
        {
            axis1 = 0;
            axis2 = 1;
            partitionAxis = 2;
        }
        debug1 << "Grid dims are "
            << gridDims[0] << " " << gridDims[1] << " " << gridDims[2]
            << " partition axis is " << partitionAxis
            << " other axes are " << axis1 << " " << axis2 << std::endl;

        subBlockDims[2*axis1+0] = 0;
        subBlockDims[2*axis1+1] = gridDims[axis1] - 1;

        subBlockDims[2*axis2+0] = 0;
        subBlockDims[2*axis2+1] = gridDims[axis2] - 1;

        int numPartsForPartition = std::min(h5part_int64_t(PAR_Size()), gridDims[partitionAxis]);
        float slicesPerProcessor = float(gridDims[partitionAxis]) / float(numPartsForPartition);
        debug1 << "Slices per processor: " << slicesPerProcessor << std::endl;

        subBlockDims[2*partitionAxis+0] = int(PAR_Rank()*slicesPerProcessor);
        subBlockDims[2*partitionAxis+1] = int((PAR_Rank()+1)*slicesPerProcessor);
        if (PAR_Rank() == numPartsForPartition - 1)
        {
            // Make sure that we include the final slice and do not include anything beyond
            subBlockDims[2*partitionAxis+1] = gridDims[partitionAxis] - 1; 
        }
        debug1 << "Processor " << PAR_Rank() << " has block "
            << subBlockDims[0] << " " << subBlockDims[1] << " "
            << subBlockDims[2] << " " << subBlockDims[3] << " "
            << subBlockDims[4] << " " << subBlockDims[5] << std::endl;
    }
#endif
}

#ifdef HAVE_LIBFASTBIT

std::string inline avtH5PartFileFormat::stringify(double x)
{
  std::ostringstream o;
  if (!(o << setprecision(32) << x))
    EXCEPTION1(InvalidVariableException, "string conversion");
  //    throw BadConversion("stringify(double)");
  return o.str();
}


// ****************************************************************************
//  Method: avtH5PartFileFormat::ConstructHistogram
//
//  Purpose:
//      Given a specification, fill in the histogram data through FastBit calls
//
//  Arguments:
//      spec       The input specification
//
//  Returns:    Specification with count/bounds data filled in. Need to determine 
//              appropriate exceptions to throw
//
//  Programmer: Prabhat
//  Creation:   February 26, 2008
//
//  Modifications:
//
//    Gunther H. Weber, Mon Feb 15 20:20:56 PST 2010
//    Merged into new H5Part database plugin
//
// ****************************************************************************

void 
avtH5PartFileFormat::ConstructHistogram(avtHistogramSpecification *spec)
{
    if (!file)
    {
        file = H5PartOpenFile(filenames[0], H5PART_READ);
        reader.openFile(filenames[0], true);
    } 
    if (!file) EXCEPTION1(InvalidFilesException, "Could not open file.");
    H5PartSetStep(file, activeTimeStep);
 
    std::string method = "avtH5PartFileFormat::ConstructHistogram(): ";
    if (NULL==spec) 
        EXCEPTION1(InvalidVariableException, "NULL Histogram Specification");

    //Compute the hisogram
    int             timestep = spec->GetTimestep();
    bool      regularBinning = spec->IsRegularBinning();
    vector<string> variables = spec->GetVariables();
    vector<int>      numBins = spec->GetNumberOfBins();
    bool     boundsSpecified = spec->BoundsSpecified();
    string         condition = spec->GetCondition();   

    int           boundsSize = spec->GetBounds().size();
    VISIT_LONG_LONG * counts = spec->GetCounts();

    for (int i=0; i<numBins.size(); i++) 
    {
        if (regularBinning)
        {
            if (numBins[i]<1)
            {
                debug5 << method <<  "Need to specify a valid #bins for histogram!" << std::endl;
                EXCEPTION1(InvalidVariableException, "numBins");
            }
        }
    }

    if (variables.empty())
    {
        debug5 << method << "Histogram Spec needs to specify variables to plot!" << std::endl;
        EXCEPTION1(InvalidVariableException, "variables");
    }

    for (int i=0; i<variables.size(); i++)
    {
        bool found = reader.checkForVariable(variables[i]);
        if (!found)
        {
            debug5 << method << "Histogram Spec lists variable not present in file!" << std::endl;
            EXCEPTION1(InvalidVariableException, variables[i]);
        }
    }

    if (boundsSpecified)
    {
        if (variables.size() != boundsSize)
        {
            debug5 << method << " Histogram Spec variable list does not match bounds list size " << std::endl;
            EXCEPTION1(InvalidVariableException, "Variable bounds list mismatch");
        }
    }

    // Try to retrieve the histogram from cache
    bool isCached = false;
    isCached = histoCache.getCached(spec);
    if (isCached)
    {
        // FIXME: Does it really make sense to override boundsSpecified?
        boundsSpecified = spec->BoundsSpecified();  
        debug5 << method << "Histogram retrieved from cache!" << std::endl;
    }  

    // Define the begin and end 
    vector<double> begins;
    vector<double> ends;
    begins.resize(variables.size());
    ends.resize(variables.size());

    // If bounds are specified then copy the extends into the histogram
    if (boundsSpecified)
    {
        for (int i=0; i<variables.size(); i++)
        {
            begins[i] = spec->GetBounds()[i][0];
            ends[i]   = spec->GetBounds()[i][spec->GetBounds()[i].size()-1];
            debug5 << "boundsSpecified to be "<< begins[i] <<", "<<ends[i]<<std::endl;
        }
    }
    //if bounds are not specified than ask the reader for the extents 
    else
    {
        debug5 << method << "Detected that bounds are not set.. setting them to.." << std::endl;
        for(int i=0; i<variables.size() ; i++)
        {
            vector<int64_t> dims;
            BaseFileInterface::DataType type;
            reader.getVariableInformation(variables[i], timestep, dims, &type);

            if (type==BaseFileInterface::H5_Float)
            {
                float start, end;
                std::string varname = variables[i];
                reader.getDataMinMax(varname,
                        timestep,
                        type,
                        (void*)&start, 
                        (void*)&end);
                begins[i] = start;
                ends[i] = end;
            }
            else if (type==BaseFileInterface::H5_Double)
            {
                double start, end;
                std::string varname = variables[i];
                reader.getDataMinMax(varname,
                        timestep,
                        type,
                        (void*)&start,
                        (void*)&end);
                begins[i] = start;
                ends[i] = end;
            }
            else
            {
                debug5 << method << "Warning:: data type min/max not yet supported!" << std::endl;
            }

            debug5 << "HDF-FQ/FastBit calculated " << variables[i] << " bounds to be.. " 
                << begins[i] << ", "<< ends[i] << endl<<endl;
        }
    }

    bool cacheCurrentHisto = false;

    // We need to check the dimensionality of the histogram to determine what FastBit call
    // to make. We'll assume that the bounds contain junk for now and that num_bins is what will
    // drive the histogram creation.
    if (!isCached)
    {
        if (variables.size()==1)
        {
            vector<uint32_t> count;
            reader.get1DHistogram((int64_t)timestep, 
                    condition.c_str(),
                    variables[0].c_str(),
                    numBins[0],
                    spec->GetBounds()[0],
                    count);
            spec->SetBoundsSpecified(); //Make sure that the bounds specified flag is defined properly
            //TODO Possibly replace in order to avoid copying of the histograms
            spec->SetCounts( count );   //Copy the counts into the specification
            cacheCurrentHisto = true;
        }
        else if (variables.size()==2)
        {
            vector<uint32_t> count;
            double begin1 = begins[0];
            double end1 = ends[0];

            double begin2 = begins[1];
            double end2 = ends[1];

            debug5 << "BEGIN1 = " << begin1 << ", END1 " << end1 << ", begin2 " << begin2 << ", end2 = " << end2 << std::endl;
            debug5 << "numbins1 = " << numBins[0] << ", numBins2 = " << numBins[1] << std::endl;

            if (regularBinning)
            { //if regular binning should be used
                reader.get2DHistogram((int64_t)timestep,
                        condition.c_str(),
                        variables[0].c_str(),
                        begin1, end1, numBins[0], 
                        variables[1].c_str(),
                        begin2, end2, numBins[1],
                        spec->GetBounds()[0] , spec->GetBounds()[1] ,
                        count);
            }
            else
            { //if adaptive binning should be used
                if (condition.empty())
                {   //do we have a condition   
                    debug5<<"********************************H5Part*************** No condition"<<endl;
                    reader.get2DAdaptiveHistogram(
                            (int64_t)timestep,
                            variables[0].c_str(),
                            variables[1].c_str(),
                            numBins[0],
                            numBins[1],
                            spec->GetBounds()[0] , spec->GetBounds()[1] ,
                            count);
                }
                else
                { //without a condition
                    debug5<<"********************************H5Part*************** Condition"<<endl;               
                    reader.get2DAdaptiveHistogram(
                            (int64_t)timestep,
                            condition.c_str(),
                            variables[0].c_str(),
                            variables[1].c_str(),
                            numBins[0],
                            numBins[1],
                            spec->GetBounds()[0] , spec->GetBounds()[1] ,
                            count );
                }
            }

            spec->SetBoundsSpecified();
            spec->SetCounts( count );   //Copy the counts into the
            //specification

            //    debug4<<"Computed histogram successfully and counts are set!"<<endl<<endl;
            cacheCurrentHisto = true;
        }
        else
        { // generic histogram, will need to take slices..
            debug5 << method << " Need to implement generic histogram slices.. " << std::endl;
            EXCEPTION1(InvalidVariableException,"variables");
        }
    }

    //Add the current histogram to the cache
    if( cacheCurrentHisto )
        histoCache.addToCache(spec);
}

// ****************************************************************************
//  Method: avtH5PartFileFormat::ConstructIdentifiersFromDataRangeSelection
//
//  Purpose:
//      Constructs the identifiers that fall within a range selection.
//
//  Programmer: Prabhat (Hank added the stub)
//  Creation:   March 6, 2008
//
//  Modifications:
//
//    Gunther H. Weber, Mon Feb 15 20:48:08 PST 2010
//    Ported/moved to new H5Part plugin.
//
// ****************************************************************************

avtIdentifierSelection *
avtH5PartFileFormat::ConstructIdentifiersFromDataRangeSelection(
                                     std::vector<avtDataSelection *> &drs)
{
    std::string method = "avtH5PartFileFormat::ConstructIdentifiersFromDataRangeSelection(): ";

    avtIdentifierSelection *rv = new avtIdentifierSelection();
    std::vector<double> ids;
    std::string query = "";
    querySpecified = false;

    debug5 << method << "Creating query string from selections." << std::endl;
    // Go over all the selections
    for (int i=0; i< drs.size(); i++)
    {
        if (std::string(drs[i]->GetType())==std::string("Data Range Selection"))
        { // dta Range is valid..
            // create the query string
            avtDataRangeSelection *dr = (avtDataRangeSelection*)drs[i];

            std::string var = dr->GetVariable();
            double min = dr->GetMin();
            double max = dr->GetMax();
            std::string min_condition = "("+stringify(min)+"<"+var           +")";
            std::string max_condition = "("+var           +"<"+stringify(max)+")";

            if (i==0)
                query = min_condition + " && " + max_condition;
            else
                query = query + "&&" + min_condition + "&&" + max_condition;

            querySpecified = true;
        }
        else if (std::string(drs[i]->GetType())==std::string("Identifier Data Selection"))
        { // identifier selection..

            debug5 << method << " Named selection found............."  << endl;

            avtIdentifierSelection *id = (avtIdentifierSelection*)drs[i];
            const vector<double>& ids = id->GetIdentifiers();

            std::string id_string;
            int len = get_string_from_identifiers(ids, id_string);

            if (len>0)
            {
                if (i==0) 
                    query = id_string;
                else 
                    query = query + "&&" + id_string;

                querySpecified = true;
            }
        }
    }
    debug5 << method << "query string is " << query << std::endl;

    if (querySpecified)
    {
        if (!file)
        {
            file = H5PartOpenFile(filenames[0], H5PART_READ);
#ifdef HAVE_LIBFASTBIT
            reader.openFile(filenames[0], true);
#endif
        } 
        if (!file) EXCEPTION1(InvalidFilesException, "Could not open file.");
        H5PartSetStep(file, activeTimeStep);

        std::vector<hsize_t> qResults;
        reader.executeQuery((char*)query.c_str(), activeTimeStep, qResults);
        debug5 << method << "Execution of query string["<<query.size()<<" resulted in " << qResults.size() << " entries.." << std::endl;

        BaseFileInterface::DataType type;
        vector<int64_t> dims;
        double *VD = NULL;
        float *VF = NULL;

        reader.getVariableInformation("id", activeTimeStep, dims, &type); // FIXME- id is hardcoded here

        if (type==BaseFileInterface::H5_Double)
        {
            VD = new double[qResults.size()];
            reader.getPointData("id", activeTimeStep, VD, qResults); 
        }
        else if (type==BaseFileInterface::H5_Float)
        {
            VF = new float[qResults.size()];
            reader.getPointData("id", activeTimeStep, VF, qResults); 
        }

        if (type==BaseFileInterface::H5_Double)
        {
            for (int j=0; j<qResults.size(); j++) 
                ids.push_back(VD[j]);

            delete [] VD;
        }
        else if (type==BaseFileInterface::H5_Float)
        {
            for (int j=0; j<qResults.size(); j++) 
                ids.push_back(VF[j]);

            delete [] VF;
        }    

        qResults.clear();
    }

    rv->SetIdentifiers(ids);

    return rv;
}

// ****************************************************************************
//  Method: avtH5PartFileFormat::ConstructIdentifiersFromDataRangeSelection
//
//  Purpose:
//      Generate a string from a list of identifiers
//
//  Programmer: Prabhat
//  Creation:   March 6, 2008
//
// ****************************************************************************

int  avtH5PartFileFormat::get_string_from_identifiers(const vector<double>& identifiers, string& id_string)
{
    id_string = "";
    debug5 << "get_string_from_identifiers id length = " << identifiers.size() << std::endl;

    if (identifiers.size()>0)
    {

        id_string = "( id in ( "; // Prabhat- FIXME id is hardcoded here..

        for (int j=0; j<identifiers.size()-1; j++)
            id_string = id_string + stringify(identifiers[j]) + ", ";
        id_string = id_string + stringify(identifiers[identifiers.size()-1]) + " ))";
    }

    debug5 << "id_string is "<< id_string << std::endl;
    return identifiers.size();
}

#endif

