#include <VariableMenuPopulator.h>

#include <Init.h>
#include <avtDatabaseMetaData.h>
#include <avtSIL.h>
#include <PlotPluginInfo.h>
#include <Expression.h>
#include <ExpressionList.h>
#include <QvisVariablePopupMenu.h>
#include <qobject.h>

using std::string;
using std::map;

// ****************************************************************************
// Method: VariableMenuPopulator::VariableMenuPopulator
//
// Purpose: 
//   Constructor for the VariableMenuPopulator class.
//
// Programmer: Brad Whitlock
// Creation:   Mon Mar 17 14:58:14 PST 2003
//
// Modifications:
//   Brad Whitlock, Tue Sep 2 09:56:54 PDT 2003
//   I added materialVars.
//
//   Brad Whitlock, Tue Feb 24 15:50:54 PST 2004
//   I added cachedDBName and cachedExpressionList.
//
// ****************************************************************************

VariableMenuPopulator::VariableMenuPopulator() :
    cachedDBName(), cachedExpressionList(),
    meshVars(), scalarVars(), materialVars(), vectorVars(), subsetVars(),
    speciesVars(), curveVars(), tensorVars(), symmTensorVars()
{
}

// ****************************************************************************
// Method: VariableMenuPopulator::VariableMenuPopulator
//
// Purpose: 
//   Destructor for the VariableMenuPopulator class.
//
// Programmer: Brad Whitlock
// Creation:   Mon Mar 17 14:58:14 PST 2003
//
// Modifications:
//   
// ****************************************************************************

VariableMenuPopulator::~VariableMenuPopulator()
{
}

// ****************************************************************************
// Method: VariableMenuPopulator::ClearDatabaseName
//
// Purpose: 
//   Clears out the database name that is used to determine whether the
//   list of variables needs to be regenerated.
//
// Programmer: Brad Whitlock
// Creation:   Thu Feb 26 08:43:58 PDT 2004
//
// Modifications:
//   
// ****************************************************************************

void
VariableMenuPopulator::ClearDatabaseName()
{
    cachedDBName = "";
}

// ****************************************************************************
// Method: VariableMenuPopulator::PopulateVariableLists
//
// Purpose: 
//   Populates internal variable lists using the metadata, sil, and expression
//   list.
//
// Arguments:
//   md       : A pointer to metadata for the file we're interested in.
//   sil      : A pointer to SIL for the file we're interested in.
//   exprList : A pointer to the list of expression definitions.
//
// Returns:    
//
// Note:  I adapted this code from code that I originally wrote for 
//        QvisPlotManagerWidget.
//
// Programmer: Brad Whitlock
// Creation:   Mon Mar 17 14:19:06 PST 2003
//
// Modifications:
//   Brad Whitlock, Tue May 20 12:49:43 PDT 2003
//   Made it work with an updated version of Expression::ExprType..
//
//   Hank Childs, Fri Aug  1 10:44:45 PDT 2003
//   Add support for curves.
//
//   Kathleen Bonnell, Fri Aug 22 18:03:55 PDT 2003
//   Create 'compound' subset vars (e.g. 'materials(mesh1)') only when
//   necessary.
//
//   Brad Whitlock, Tue Sep 2 09:55:50 PDT 2003
//   I added the materialVars map, which only contains materials, so the
//   VAR_CATEGORY_MATERIALS flag works and does not populate the variable
//   list with all of the subset variables.
//
//   Hank Childs, Tue Sep 23 22:05:54 PDT 2003
//   Add support for tensors.
//
//   Brad Whitlock, Thu Oct 23 16:27:38 PST 2003
//   I made the code take into account whether material and subset variables
//   are valid variables using information from the metadata. I also changed
//   how expressions are added to the list so that expressions from the
//   expression list and expressions from metadata are added separately.
//
//   Brad Whitlock, Tue Feb 24 15:49:58 PST 2004
//   I added code to check the name of the database that we're using against
//   the name of the database that we've used before.
//
//   Mark C. Miller, Wed Apr 14 10:51:23 PDT 2004
//   I added code to include the catch-all mesh name in the list of meshes
//   when the plugin has activiated this feature
//
//   Jeremy Meredith, Mon Jun 21 15:13:20 PDT 2004
//   SUBSET variables now no-longer include MATERIAL variables.  The
//   Filled Boundary and Boundary plots have been changed to MATERIAL, and
//   the Subset plot is the only one that will do other kinds of subsets like
//   domains -- at least in the GUI.
//
//   Brad Whitlock, Thu Aug 5 14:42:13 PST 2004
//   I made it use VariableList instead of StringBoolMap.
//
//   Jeremy Meredith, Tue Aug 24 16:18:19 PDT 2004
//   Force an update if it is metadata from a simulation.
//
//   Brad Whitlock, Fri Feb 18 11:39:20 PDT 2005
//   I made it use only the user-defined expressions and the expressions
//   from the metadata so we don't accidentally get menus that contain
//   database expressions from databases other than the one that we're
//   using here.
//
// ****************************************************************************

bool
VariableMenuPopulator::PopulateVariableLists(const std::string &dbName,
    const avtDatabaseMetaData *md, const avtSIL *sil,
    const ExpressionList *exprList)
{
    if(md == 0 || sil == 0 || exprList == 0)
        return false;

    // The expression list can sometimes contain database expressions for
    // a database that is not the database specified by dbName. To combat
    // this problem, we only use the user-defined expressions from the
    // passed in exprList and we supplement it with any database expressions
    // that are contained in the metadata md.
    ExpressionList newExpressionList;
    GetRelevantExpressions(newExpressionList, md, *exprList);

    //
    // If the database name is the same and the expression list is the
    // same then return false, indicating that no updates are required.
    // If this is a simulation, then the variable list might change at
    // any time, so treat that as equivalent to MustRepopulateOnStateChange.
    //
    bool expressionsSame = newExpressionList == cachedExpressionList;
    bool variableMetaData = md->GetMustRepopulateOnStateChange() ||
                            md->GetIsSimulation();
    if(dbName == cachedDBName && expressionsSame && !variableMetaData)
        return false;

    //
    // Save the database name and the expression list so we can check them
    // again later so we can get out of this routine early if they are the
    // same.
    //
    cachedDBName = dbName;
    if(!expressionsSame)
        cachedExpressionList = newExpressionList;

    // Clear out the variable lists and set their sorting method..
    meshVars.Clear();        meshVars.SetSorted(md->GetMustAlphabetizeVariables());
    scalarVars.Clear();      scalarVars.SetSorted(md->GetMustAlphabetizeVariables());
    vectorVars.Clear();      vectorVars.SetSorted(md->GetMustAlphabetizeVariables());
    materialVars.Clear();    materialVars.SetSorted(md->GetMustAlphabetizeVariables());
    subsetVars.Clear();      subsetVars.SetSorted(md->GetMustAlphabetizeVariables());
    speciesVars.Clear();     speciesVars.SetSorted(md->GetMustAlphabetizeVariables());
    curveVars.Clear();       curveVars.SetSorted(md->GetMustAlphabetizeVariables());
    tensorVars.Clear();      tensorVars.SetSorted(md->GetMustAlphabetizeVariables());
    symmTensorVars.Clear();  symmTensorVars.SetSorted(md->GetMustAlphabetizeVariables());

    // Do stuff with the metadata
    int i;
    for (i = 0; i < md->GetNumMeshes(); ++i)
    {
        const avtMeshMetaData *mmd = md->GetMesh(i);
        meshVars.AddVariable(mmd->name, mmd->validVariable);
    }
    if (md->GetUseCatchAllMesh())
        meshVars.AddVariable(Init::CatchAllMeshName, true);
    for (i = 0; i < md->GetNumScalars(); ++i)
    {
        const avtScalarMetaData *smd = md->GetScalar(i);
        scalarVars.AddVariable(smd->name, smd->validVariable);
    }
    for (i = 0; i < md->GetNumVectors(); ++i)
    {
        const avtVectorMetaData *vmd = md->GetVector(i);
        vectorVars.AddVariable(vmd->name, vmd->validVariable);
    }
    for (i = 0; i < md->GetNumSpecies(); ++i)
    {
        const avtSpeciesMetaData *smd = md->GetSpecies(i);
        speciesVars.AddVariable(smd->name, smd->validVariable);
    }
    for (i = 0; i < md->GetNumCurves(); ++i)
    {
        const avtCurveMetaData *cmd = md->GetCurve(i);
        curveVars.AddVariable(cmd->name, cmd->validVariable);
    }
    for (i = 0; i < md->GetNumTensors(); ++i)
    {
        const avtTensorMetaData *tmd = md->GetTensor(i);
        tensorVars.AddVariable(tmd->name, tmd->validVariable);
    }
    for (i = 0; i < md->GetNumSymmTensors(); ++i)
    {
        const avtSymmetricTensorMetaData *tmd = md->GetSymmTensor(i);
        symmTensorVars.AddVariable(tmd->name, tmd->validVariable);
    }

    // Do stuff with the sil
    const intVector &topSets = sil->GetWholes();

    // There are currently 9 role types, lets count their occurrance
    // in the top sets.
    int roleCount[9] = { 0, 0, 0, 0, 0, 0, 0, 0, 0};

    for(i = 0; i < topSets.size(); ++i)
    {
        int tsIndex = topSets[i];
        const intVector &maps = sil->GetSILSet(tsIndex)->GetMapsOut();
        for (int j = 0; j < maps.size(); ++j)
        {
            int idx = maps[j];
            int role = (int)sil->GetSILCollection(idx)->GetRole();
            if (role > 0 && role < 8)
                roleCount[role]++;
        }
    }

    for(i = 0; i < topSets.size(); ++i)
    {
        int tsIndex = topSets[i];
        const intVector &maps = sil->GetSILSet(tsIndex)->GetMapsOut();
        string setName("(" + sil->GetSILSet(tsIndex)->GetName() + ")");
        for(int j = 0; j < maps.size(); ++j)
        {
            int     idx = maps[j];

            //
            // DISABLE SPECIES FOR SUBSET PLOTS.  THIS IS A HACK AND SHOULD
            // BE REMOVED WHEN EVERYTHING WORKS DOWNSTREAM.
            //
            if (sil->GetSILCollection(idx)->GetRole() == SIL_SPECIES)
            {
                continue;
            }

            string  varName = sil->GetSILCollection(idx)->GetCategory();
            int     role = sil->GetSILCollection(idx)->GetRole();
            bool    validVariable = true;

            //
            // If we're looking at a material, test to see if the material
            // is a valid variable
            //
            if(role == SIL_MATERIAL)
            {
                for(int m = 0; m < md->GetNumMaterials(); ++m)
                {
                    const avtMaterialMetaData *mmd = md->GetMaterial(m);
                    if(mmd->name == varName)
                    {
                        validVariable = mmd->validVariable;
                        break;
                    }
                }
            }

            //
            // Only add the set name if necessary.
            //
            if (roleCount[role] > 1)
                varName += setName;

            //
            // Also add the varName to the material variable map so we can
            // have plots, etc that just use materials instead of any type
            // of subset variable.
            //
            if (role == SIL_MATERIAL)
                materialVars.AddVariable(varName, validVariable);
            else
                subsetVars.AddVariable(varName, validVariable);
        }
    }

    //
    // Add the expressions from the cached expression list.
    //
    int nexp = cachedExpressionList.GetNumExpressions();
    for(i = 0; i < nexp; ++i)
    {
        const Expression &expr = cachedExpressionList[i];
        if(!expr.GetHidden())
            AddExpression(expr);
    }

    return true;
}

// ****************************************************************************
// Method: VariableMenuPopulator::AddExpression
//
// Purpose: 
//   This method adds an expression to the appropriate variable list.
//
// Arguments:
//   expr : The expression to add to a varialbe list.
//
// Programmer: Brad Whitlock
// Creation:   Fri Oct 24 15:39:47 PST 2003
//
// Modifications:
//   Brad Whitlock, Thu Aug 5 14:35:26 PST 2004
//   Made it use VariableList.
//
// ****************************************************************************

void
VariableMenuPopulator::AddExpression(const Expression &expr)
{
    // Figure out which list this expression should be added to.
    VariableList *m = 0;
    switch (expr.GetType())
    {
    case Expression::ScalarMeshVar:
        m = &scalarVars;
        break;
    case Expression::VectorMeshVar:
        m = &vectorVars;
        break;
    case Expression::Mesh:
        m = &meshVars;
        break;
    case Expression::Material:
        m = &materialVars;
        break;
    case Expression::Species:
        m = &speciesVars;
        break;
    case Expression::TensorMeshVar:
        m = &tensorVars;
        break;
    case Expression::SymmetricTensorMeshVar:
        m = &symmTensorVars;
        break;
    default:
        break;
    }

    // Check if the name is already in the list
    if(m)
    {
        if(!m->Contains(expr.GetName()))
            m->AddVariable(expr.GetName(), true);
        else
            m->AddVariable(expr.GetName() + " (expression)", true);
    }
}

// ****************************************************************************
// Method: VariableMenuPopulator::GetRelevantExpressions
//
// Purpose: 
//   Gets the list of user-defined expressions and expressions that come
//   from the metadata.
//
// Arguments:
//   newExpressionList : The new expression list.
//   md                : The metadata that we're searching for expressions.
//   exprList          : The expression list to use for the user-defined
//                       expressions.
//
// Programmer: Brad Whitlock
// Creation:   Fri Feb 18 11:36:26 PDT 2005
//
// Modifications:
//   
// ****************************************************************************

void
VariableMenuPopulator::GetRelevantExpressions(ExpressionList &newExpressionList,
    const avtDatabaseMetaData *md, const ExpressionList &exprList)
{
    int i;

    // Get the user-defined expressions.
    for(i = 0; i < exprList.GetNumExpressions(); ++i)
    {
        const Expression &e = exprList[i];
        if(!e.GetHidden() && !e.GetFromDB())
            newExpressionList.AddExpression(e);
    }

    // Get the expressions from the metadata.
    for(i = 0; i < md->GetNumberOfExpressions(); ++i)
    {
        const Expression *e = md->GetExpression(i);
        if(e != 0 && !e->GetHidden())
            newExpressionList.AddExpression(*e);
    }
}

// ****************************************************************************
// Method: VariableMenuPopulator::UpdateSingleVariableMenu
//
// Purpose: 
//   Updates the a variable menu with a list of variables that is a combination
//   of the variables in the variable lists.
//
// Arguments:
//   menu      : A pointer to the variable menu that will be updated.
//   varTypes  : A flag that contains various variable types orred together.
//   receiver  : The QObject whose slot function will be called.
//   slot      : The slot function that will be called on the receiver.
//
// Returns:    The number of variables in the list that was added to the menu.
//
// Note:       I moved this code from QvisPlotManagerWidget.
//
// Programmer: Brad Whitlock
// Creation:   Mon Mar 17 14:55:17 PST 2003
//
// Modifications:
//   Hank Childs, Fri Aug  1 10:44:45 PDT 2003
//   Added support for curves.
//
//   Brad Whitlock, Tue Sep 2 09:45:45 PDT 2003
//   I made it possible for materials to be treated differently than subset
//   variables. The materialVars map is a subset of subsetVars.
//
//   Hank Childs, Tue Sep 23 22:09:33 PDT 2003
//   Added support for tensors.
//
//   Brad Whitlock, Thu Aug 5 14:30:13 PST 2004
//   I made it use VariableList instead of StringBoolMap.
//
//   Brad Whitlock, Wed Dec 8 14:07:55 PST 2004
//   I changed how the slot functions for the menu are hooked up.
//
// ****************************************************************************

int
VariableMenuPopulator::UpdateSingleVariableMenu(QvisVariablePopupMenu *menu,
    int varTypes, QObject *receiver, const char *slot)
{
    int numVarTypes = 0;
    int retval = 0;

    // See if we need to create a variable list that has more than one
    // category of variable in it.
    if(varTypes & VAR_CATEGORY_MESH)
       ++numVarTypes;
    if(varTypes & VAR_CATEGORY_SCALAR)
       ++numVarTypes;
    if(varTypes & VAR_CATEGORY_MATERIAL)
       ++numVarTypes;
    if(varTypes & VAR_CATEGORY_VECTOR)
       ++numVarTypes;
    if(varTypes & VAR_CATEGORY_SUBSET)
       ++numVarTypes;
    if(varTypes & VAR_CATEGORY_SPECIES)
       ++numVarTypes;
    if(varTypes & VAR_CATEGORY_CURVE)
       ++numVarTypes;
    if(varTypes & VAR_CATEGORY_TENSOR)
       ++numVarTypes;
    if(varTypes & VAR_CATEGORY_SYMMETRIC_TENSOR)
       ++numVarTypes;

    if(numVarTypes > 1)
    {
        VariableList vars;
        vars.SetSorted(meshVars.GetSorted());

        // Construct a variable list that contains variables from multiple
        // categories then sort the list.
        if(varTypes & VAR_CATEGORY_MESH)
            AddVars(vars, meshVars);
        if(varTypes & VAR_CATEGORY_SCALAR)
            AddVars(vars, scalarVars);
        if(varTypes & VAR_CATEGORY_MATERIAL)
            AddVars(vars, materialVars);
        if(varTypes & VAR_CATEGORY_VECTOR)
            AddVars(vars, vectorVars);
        if(varTypes & VAR_CATEGORY_SUBSET)
            AddVars(vars, subsetVars);
        if(varTypes & VAR_CATEGORY_SPECIES)
            AddVars(vars, speciesVars);
        if(varTypes & VAR_CATEGORY_CURVE)
            AddVars(vars, curveVars);
        if(varTypes & VAR_CATEGORY_TENSOR)
            AddVars(vars, tensorVars);
        if(varTypes & VAR_CATEGORY_SYMMETRIC_TENSOR)
            AddVars(vars, symmTensorVars);
       
        // Update the menu with the composite variable list.
        UpdateSingleMenu(menu, vars, receiver, slot);
        retval = vars.Size();
    }
    else
    {
        // The menu only contains one type of variable so update the menu
        // based on the variable type.
        switch(varTypes)
        {
        case VAR_CATEGORY_MESH:
            UpdateSingleMenu(menu, meshVars, receiver, slot);
            retval = meshVars.Size();
            break;
        case VAR_CATEGORY_SCALAR:
            UpdateSingleMenu(menu, scalarVars, receiver, slot);
            retval = scalarVars.Size();
            break;
        case VAR_CATEGORY_VECTOR:
            UpdateSingleMenu(menu, vectorVars, receiver, slot);
            retval = vectorVars.Size();
            break;
        case VAR_CATEGORY_MATERIAL:
            UpdateSingleMenu(menu, materialVars, receiver, slot);
            retval = materialVars.Size();
            break;
        case VAR_CATEGORY_SUBSET:
            UpdateSingleMenu(menu, subsetVars, receiver, slot);
            retval = subsetVars.Size();
            break;
        case VAR_CATEGORY_SPECIES:
            UpdateSingleMenu(menu, speciesVars, receiver, slot);
            retval = speciesVars.Size();
            break;
        case VAR_CATEGORY_CURVE:
            UpdateSingleMenu(menu, curveVars, receiver, slot);
            retval = curveVars.Size();
            break;
        case VAR_CATEGORY_TENSOR:
            UpdateSingleMenu(menu, tensorVars, receiver, slot);
            retval = curveVars.Size();
            break;
        case VAR_CATEGORY_SYMMETRIC_TENSOR:
            UpdateSingleMenu(menu, symmTensorVars, receiver, slot);
            retval = curveVars.Size();
            break;
        }
    }

    return retval;
}

// ****************************************************************************
// Method: VariableMenuPopulator::ItemEnabled
//
// Purpose: 
//   Returns whether a plot with the given variable type signature should be
//   enabled given the current variables.
//
// Arguments:
//   varType : The types of variables that the plot handles.
//
// Returns:    true if the plot is enabled, false otherwise.
//
// Programmer: Brad Whitlock
// Creation:   Tue Mar 18 08:31:34 PDT 2003
//
// Modifications:
//   
//   Hank Childs, Fri Aug  1 10:44:45 PDT 2003
//   Add support for curves.
//
//   Hank Childs, Tue Sep 23 22:09:33 PDT 2003
//   Added support for tensors.
//
//   Brad Whitlock, Thu Aug 5 15:32:50 PST 2004
//   Made it use VariableList.
//
// ****************************************************************************

bool
VariableMenuPopulator::ItemEnabled(int varType) const
{
    bool retval = false;

    if(varType & VAR_CATEGORY_MESH)
       retval |= (meshVars.Size() > 0);
    if(varType & VAR_CATEGORY_SCALAR)
       retval |= (scalarVars.Size() > 0);
    if(varType & VAR_CATEGORY_MATERIAL)
       retval |= (subsetVars.Size() > 0);
    if(varType & VAR_CATEGORY_VECTOR)
       retval |= (vectorVars.Size() > 0);
    if(varType & VAR_CATEGORY_SUBSET)
       retval |= (subsetVars.Size() > 0);
    if(varType & VAR_CATEGORY_SPECIES)
       retval |= (speciesVars.Size() > 0);
    if(varType & VAR_CATEGORY_CURVE)
       retval |= (curveVars.Size() > 0);
    if(varType & VAR_CATEGORY_TENSOR)
       retval |= (tensorVars.Size() > 0);
    if(varType & VAR_CATEGORY_SYMMETRIC_TENSOR)
       retval |= (symmTensorVars.Size() > 0);

    return retval;
}

// ****************************************************************************
// Method: VariableMenuPopulator::UpdateSingleMenu
//
// Purpose: 
//   Updates a variable list so it contains the correct variables.
//
// Arguments:
//   menu     : The menu that we want to update.
//   vars     : A map of strings and bools representing the variable list.
//   receiver : The QObject that will handle signals emitted by the menu.
//   slot     : The slot function that will be called on the receiver when
//              signals are emitted by the menu.
//
// Programmer: Brad Whitlock
// Creation:   Mon Mar 17 14:31:20 PST 2003
//
// Modifications:
//   Brad Whitlock, Thu Aug 5 14:27:22 PST 2004
//   I made it use VariableList.
//
//   Brad Whitlock, Fri Dec 3 13:26:06 PST 2004
//   I removed the code to clear the menu and changed the slot hookup code
//   so it is more general.
//
// ****************************************************************************

void
VariableMenuPopulator::UpdateSingleMenu(QvisVariablePopupMenu *menu,
    VariableList &vars, QObject *receiver, const char *slot)
{
    if (menu == 0)
        return;

    // Add each variable to the variable menu.
    std::map <std::string, QvisVariablePopupMenu *> popups;
    int j, varCount = menu->count();
    std::string var;
    bool        validVar;
    vars.InitTraversal();
    while(vars.GetNextVariable(var, validVar))
    {
        // Split the variable's path into a vector of strings.
        stringVector pathvar;
        Split(var, pathvar);

        // Add the submenus.
        QvisVariablePopupMenu *parent = menu;
        string path;
        for (j = 0; j < pathvar.size() - 1; ++j)
        {
            // Create the current path.
            path += (pathvar[j] + "/");

            // See if the current path is in the map. If it is then
            // do nothing. If the path is not in the map then we
            // add a new popup menu.
            std::map<std::string, QvisVariablePopupMenu *>::const_iterator p =
                popups.find(path);
            if(p == popups.end())
            {
                QvisVariablePopupMenu *newPopup =
                    new QvisVariablePopupMenu(menu->getPlotType(), parent,
                                              path.c_str());
                newPopup->setVarPath(path.c_str());
                if (receiver != 0 && slot != 0)
                {
                    QObject::connect(newPopup, SIGNAL(activated(int, const QString &)),
                                     receiver, slot);
                }

                popups[path] = newPopup;
                parent->insertItem(pathvar[j].c_str(), newPopup, -1, parent->count());
                parent = newPopup;
            }
            else
                parent = p->second;
        }

        // Add the variable.
        int id = parent->insertItem(pathvar[j].c_str(), varCount++, parent->count());
        parent->setItemEnabled(id, validVar);
    }
}

// ****************************************************************************
// Method: VariableMenuPopulator::Split
//
// Purpose: 
//   Splits a variable name but it ignores slashes that are enclosed in
//   parenthesis so Subset variables in subdirectories are split correctly.
//
// Arguments:
//   varName : The path to split.
//   pieces  : The pieces of the path.
//
// Programmer: Brad Whitlock
// Creation:   Tue Aug 20 17:23:19 PST 2002
//
// Modifications:
//   
// ****************************************************************************

void
VariableMenuPopulator::Split(const std::string &varName, stringVector &pieces) const
{
    std::string word;
    int         parenthesis = 0;

    // Iterate through the characters in the word splitting
    for(int i = 0; i < varName.size(); ++i)
    {
        char c = varName[i];
        if(c == '(')
        {
            ++parenthesis;
            word += c;
        }
        else if(c == ')')
        {
            --parenthesis;
            word += c;
        }
        else if(c == '/')
        {
            if(parenthesis > 0)
                word += c;
            else
            {
                pieces.push_back(word);
                word = "";
            }
        }
        else
            word += c;
    }

    if(word.size() > 0)
        pieces.push_back(word);
}

// ****************************************************************************
// Method: VariableMenuPopulator::AddVars
//
// Purpose: 
//   Internal helper method that adds the contents of one map to another map.
//
// Arguments:
//   to   : The destination map.
//   from : The source map.
//
// Programmer: Brad Whitlock
// Creation:   Tue Mar 18 08:26:22 PDT 2003
//
// Modifications:
//   Brad Whitlock, Thu Aug 5 14:23:48 PST 2004
//   Changed to VariableList instead of StringBoolMap.
//
// ****************************************************************************

void
VariableMenuPopulator::AddVars(VariableMenuPopulator::VariableList &to,
    VariableMenuPopulator::VariableList &from)
{
    std::string var;
    bool        validVar;

    from.InitTraversal();
    while(from.GetNextVariable(var, validVar))
        to.AddVariable(var, validVar);
}

//
// VariableMenuPopulator::VariableList
//

// ****************************************************************************
// Method: VariableMenuPopulator::VariableList::VariableList
//
// Purpose: 
//   Constructor for the VariableList class.
//
// Note:       This class is an interface on top of different containers that
//             allow us to create sorted or unsorted variable lists. The
//             containers used depend on whether we're sorting.
//
// Programmer: Brad Whitlock
// Creation:   Thu Aug 5 14:45:02 PST 2004
//
// Modifications:
//   
// ****************************************************************************

VariableMenuPopulator::VariableList::VariableList() : sortedVariables(),
    sortedVariablesIterator(), unsortedVariableNames(), unsortedVariableValid()
{
    sorted = true;
    unsortedVariableIndex = -1;
}

// ****************************************************************************
// Method: VariableMenuPopulator::VariableList::~VariableList
//
// Purpose: 
//   Destructor for the VariableList class.
//
// Programmer: Brad Whitlock
// Creation:   Thu Aug 5 14:46:44 PST 2004
//
// Modifications:
//   
// ****************************************************************************

VariableMenuPopulator::VariableList::~VariableList()
{
}

// ****************************************************************************
// Method: VariableMenuPopulator::VariableList::AddVariable
//
// Purpose: 
//   Adds a variable to the appropriate container.
//
// Arguments:
//   var      : The name of the variable.
//   validVar : Whether the variable is valid.
//
// Programmer: Brad Whitlock
// Creation:   Thu Aug 5 14:47:34 PST 2004
//
// Modifications:
//   
// ****************************************************************************

void
VariableMenuPopulator::VariableList::AddVariable(const std::string &var, bool validVar)
{
    if(sorted)
        sortedVariables[var] = validVar;
    else
    {
        unsortedVariableNames.push_back(var);
        unsortedVariableValid.push_back(validVar);
    }
}

// ****************************************************************************
// Method: VariableMenuPopulator::VariableList::Clear
//
// Purpose: 
//   Clears the variable list.
//
// Programmer: Brad Whitlock
// Creation:   Thu Aug 5 15:28:48 PST 2004
//
// Modifications:
//   
// ****************************************************************************

void
VariableMenuPopulator::VariableList::Clear()
{
    sortedVariables.clear();
    unsortedVariableNames.clear();
    unsortedVariableValid.clear();
}

// ****************************************************************************
// Method: VariableMenuPopulator::VariableList::Contains
//
// Purpose: 
//   Returns whether the Variable list contains a variable.
//
// Arguments:
//   var : The variable to check for.
//
// Returns:    True if the list contains the variable; false otherwise.
//
// Programmer: Brad Whitlock
// Creation:   Thu Aug 5 15:29:04 PST 2004
//
// Modifications:
//   
// ****************************************************************************

bool
VariableMenuPopulator::VariableList::Contains(const std::string &var) const
{
    if(sorted)
        return (sortedVariables.find(var) != sortedVariables.end());
    else
    {
        for(int i = 0; i < unsortedVariableNames.size(); ++i)
            if(unsortedVariableNames[i] == var)
                return true;
    }

    return false;
}

// ****************************************************************************
// Method: VariableMenuPopulator::VariableList::InitTraversal
//
// Purpose: 
//   Initializes the variable list for traversal.
//
// Programmer: Brad Whitlock
// Creation:   Thu Aug 5 15:30:06 PST 2004
//
// Modifications:
//   
// ****************************************************************************

void
VariableMenuPopulator::VariableList::InitTraversal()
{
    unsortedVariableIndex = 0;
    sortedVariablesIterator = sortedVariables.begin();
}

// ****************************************************************************
// Method: VariableMenuPopulator::VariableList::GetNextVariable
//
// Purpose: 
//   Returns the current variable.
//
// Arguments:
//   var      : The name of the variable.
//   validVar : Whether the variable is valid.
//
// Returns:    True if a variable was returned; false otherwise.
//
// Programmer: Brad Whitlock
// Creation:   Thu Aug 5 15:30:24 PST 2004
//
// Modifications:
//   
// ****************************************************************************

bool 
VariableMenuPopulator::VariableList::GetNextVariable(std::string &var, bool &validVar)
{
    bool retval;

    if(sorted)
    {
        retval = (sortedVariablesIterator != sortedVariables.end());
        if(retval)
        {
            var = sortedVariablesIterator->first;
            validVar = sortedVariablesIterator->second;
            ++sortedVariablesIterator;
        }
    }
    else
    {
        retval = (unsortedVariableIndex < unsortedVariableNames.size());
        if(retval)
        {
            var = unsortedVariableNames[unsortedVariableIndex];
            validVar = unsortedVariableValid[unsortedVariableIndex];
            ++unsortedVariableIndex;
        }
    }

    return retval;
}

// ****************************************************************************
// Method: VariableMenuPopulator::VariableList::Size
//
// Purpose: 
//   Returns the number of variables in the container.
//
// Returns:    The number of variables.
//
// Programmer: Brad Whitlock
// Creation:   Thu Aug 5 15:33:52 PST 2004
//
// Modifications:
//   
// ****************************************************************************

int
VariableMenuPopulator::VariableList::Size() const
{
    return sorted ? sortedVariables.size() : unsortedVariableNames.size();
}
