// ************************************************************************* //
//                           avtHistogramFilter.C                            //
// ************************************************************************* //

#include <avtHistogramFilter.h>

#include <float.h>

#include <vtkCellData.h>
#include <vtkPointData.h>
#include <vtkPointDataToCellData.h>
#include <vtkPolyData.h>

#include <avtDataAttributes.h>
#include <avtExtents.h>
#include <avtParallel.h>

#include <ImproperUseException.h>


// ****************************************************************************
//  Method: avtHistogramFilter constructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Thu Jun 26 09:04:54 PDT 2003
//
// ****************************************************************************

avtHistogramFilter::avtHistogramFilter()
{
    bins = NULL;
}


// ****************************************************************************
//  Method: avtHistogramFilter destructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Thu Jun 26 09:04:54 PDT 2003
//
// ****************************************************************************

avtHistogramFilter::~avtHistogramFilter()
{
    if (bins != NULL)
        delete [] bins;
}


// ****************************************************************************
//  Method: avtHistogramFilter::SetAttributes
//
//  Purpose:
//      Sets the attributes for this filter.
//
//  Programmer: Hank Childs
//  Creation:   June 26, 2003
//
// ****************************************************************************

void
avtHistogramFilter::SetAttributes(const HistogramAttributes &h_atts)
{
    atts = h_atts;
}


// ****************************************************************************
//  Method: avtHistogramFilter::PreExecute
//
//  Purpose:
//      Initializes for our actual execute.  Sets up the bins, etc.
//
//  Programmer: Hank Childs
//  Creation:   June 26, 2003
//
// ****************************************************************************

void
avtHistogramFilter::PreExecute(void)
{
    avtStreamer::PreExecute();

    InputSetActiveVariable(pipelineVariable);

    bool extentsSpecified = atts.GetSpecifyRange();
    double minmax[2];
    if (!extentsSpecified)
    {
        GetDataExtents(minmax, pipelineVariable);
    }

    if (extentsSpecified)
        workingMin = atts.GetMin();
    else
        workingMin = minmax[0];

    if (extentsSpecified)
        workingMax = atts.GetMax();
    else
        workingMax = minmax[1];

    workingNumBins = atts.GetNumBins();
    if (workingNumBins <= 1)
        EXCEPTION0(ImproperUseException);

    if (bins != NULL)
        delete [] bins;
    bins = new float[workingNumBins];
    for (int i = 0 ; i < workingNumBins ; i++)
        bins[i] = 0.;
}


// ****************************************************************************
//  Method: avtHistogramFilter::PostExecute
//
//  Purpose:
//      Sets up the final curve based on the bins.
//
//  Programmer: Hank Childs
//  Creation:   June 26, 2003
//
//  Modifications:
//
//    Hank Childs, Sat Oct 18 11:14:06 PDT 2003
//    Make sure the limits of the histogram plot always have y=0.
//
//    Hank Childs, Sat Oct 18 11:45:13 PDT 2003
//    Make sure that the bins are of a uniform size.
//
//    Brad Whitlock, Thu Jul 22 17:02:11 PST 2004
//    Changed the xlabel and xunits in the output data attributes so the
//    histogram plot would have the units along the axes.
//
//    Kathleen Bonnell, Thu Jan  6 10:34:57 PST 2005 
//    Remove TRY-CATCH block in favor of testing for ValidActiveVariable.
//
// ****************************************************************************

void
avtHistogramFilter::PostExecute(void)
{
    avtStreamer::PreExecute();

    int  i;

    float *newBins = new float[workingNumBins];
    SumFloatArrayAcrossAllProcessors(bins, newBins, workingNumBins);
    for (i = 0 ; i < workingNumBins ; i++)
        bins[i] = newBins[i];

    if (PAR_Rank() != 0)
    {
        SetOutputDataTree(new avtDataTree());
        return;
    }

    vtkPolyData *output = vtkPolyData::New();

    if (atts.GetOutputType() == HistogramAttributes::Curve)
    {
        vtkPoints *pts = vtkPoints::New();
        pts->SetNumberOfPoints(workingNumBins);
        float jump = (workingMax - workingMin) / (workingNumBins);
        for (i = 0 ; i < workingNumBins ; i++)
        {
            float pt[3];
            pt[0] = workingMin + (i+0.5)*jump;
            pt[1] = bins[i];
            pt[2] = 0.;
            pts->SetPoint(i, pt);
        }
    
        output->SetPoints(pts);
        pts->Delete();
    
        int nSegments = workingNumBins-1;
        output->Allocate(nSegments);
        vtkIdType line[2];
        for (int i = 0 ; i < nSegments ; i++)
        {
            line[0] = i;
            line[1] = i+1;
            output->InsertNextCell(VTK_LINE, 2, line);
        }
    }
    else
    {
        vtkPoints *pts = vtkPoints::New();
        pts->SetNumberOfPoints(4*workingNumBins);
        float jump = (workingMax - workingMin) / (workingNumBins);

        int ptIndex = 0;
        float pt[3] = { 0., 0., 0. };

        for (i = 0 ; i < workingNumBins ; i++)
        {
            //
            // Do the last two points for bin i.
            //
            pt[0] = jump*i + workingMin;
            pt[1] = 0.;
            pts->SetPoint(ptIndex++, pt);

            pt[1] = bins[i];
            pts->SetPoint(ptIndex++, pt);
 
            //
            // Now do the first two points for bin i+1.
            //
            pt[0] = jump*(i+1) + workingMin;
            pt[1] = 0.;
            pts->SetPoint(ptIndex++, pt);

            pt[1] = bins[i];
            pts->SetPoint(ptIndex++, pt);
        }
    
        output->SetPoints(pts);
        pts->Delete();
    
        int ncells = 5*workingNumBins;
        output->Allocate(ncells);

        //
        // Add the line segments for each bin.
        //
        vtkIdType line[2];
        for (i = 0 ; i < workingNumBins ; i++)
        {
            line[0] = 4*i;
            line[1] = 4*i+1;
            output->InsertNextCell(VTK_LINE, 2, line);
            line[0] = 4*i+2;
            line[1] = 4*i+3;
            output->InsertNextCell(VTK_LINE, 2, line);
            line[0] = 4*i;
            line[1] = 4*i+2;
            output->InsertNextCell(VTK_LINE, 2, line);
            line[0] = 4*i+1;
            line[1] = 4*i+3;
            output->InsertNextCell(VTK_LINE, 2, line);
        }

        //
        // Now add the polygons in the middle.
        //
        vtkIdType quad[2];
        for (int i = 0 ; i < workingNumBins ; i++)
        {
            quad[0] = 4*i;
            quad[1] = 4*i+2;
            quad[2] = 4*i+3;
            quad[3] = 4*i+1;
            output->InsertNextCell(VTK_QUAD, 4, quad);
        }
    }

    SetOutputDataTree(new avtDataTree(output, -1));
    output->Delete();

    //
    // Set up the extents of the output.
    //
    float low = +FLT_MAX;
    float hi  = -FLT_MAX;
    for (i = 0 ; i < workingNumBins ; i++)
    {
        low = (low < bins[i] ? low : bins[i]);
        hi  = (hi  > bins[i] ? hi  : bins[i]);
    }
    double extents[6];
    extents[0] = workingMin;
    extents[1] = workingMax;
    extents[2] = 0.;  // We always start from 0.  lo is misleading.
    extents[3] = hi;
    extents[4] = extents[5] = 0.;

    // Try to do something reasonable if we have constant data.
    if (extents[0] == extents[1])
    {
        extents[0] = workingMin - hi/2.;
        extents[1] = workingMax + hi/2.;
    }

    avtDataAttributes &outAtts = GetOutput()->GetInfo().GetAttributes();
    outAtts.GetTrueSpatialExtents()->Set(extents);
    outAtts.GetCumulativeTrueSpatialExtents()->Set(extents);
    outAtts.GetEffectiveSpatialExtents()->Set(extents);
    outAtts.GetCurrentSpatialExtents()->Set(extents);
    outAtts.GetCumulativeCurrentSpatialExtents()->Set(extents);

    //
    // Set the X-axis's units to match the variable units.
    // 
    if (GetInput()->GetInfo().GetAttributes().ValidActiveVariable())
    {
        if(GetInput()->GetInfo().GetAttributes().GetVariableUnits() != "")
        {
            outAtts.SetXLabel(string("Variable ") + pipelineVariable);
            outAtts.SetXUnits(GetInput()->GetInfo().GetAttributes().GetVariableUnits());
        }
        else
            outAtts.SetXUnits(pipelineVariable);
    }

    string yunits = "";
    if (GetInput()->GetInfo().GetAttributes().GetTopologicalDimension() == 3)
    {
        yunits = "Volume";
    }
    else
    {
        if (atts.GetTwoDAmount() == HistogramAttributes::Area)
            yunits = "Area";
        else
            yunits = "Revolved Volume";

    }
    outAtts.SetYUnits(yunits);
}


// ****************************************************************************
//  Method: avtHistogramFilter::ExecuteData
//
//  Purpose:
//      Does the actual VTK code to modify the dataset.
//
//  Arguments:
//      inDS      The input dataset.
//      <unused>  The domain number.
//      <unused>  The label.
//
//  Returns:      The output dataset.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Thu Jun 26 09:04:54 PDT 2003
//
//  Modifications:
//
//    Hank Childs, Sat Oct 18 11:41:13 PDT 2003
//    Make the bins be uniform in size.
//
// ****************************************************************************

vtkDataSet *
avtHistogramFilter::ExecuteData(vtkDataSet *inDS, int, std::string)
{
    //
    // Get the "_amounts".  This is the area or volume that each cell
    // takes up.
    //
    vtkDataArray *amount_arr = inDS->GetCellData()->GetArray("_amounts");
    if (amount_arr == NULL)
        EXCEPTION0(ImproperUseException);

    //
    // Get the variable that we are binning by.
    //
    const char *var = pipelineVariable;
    vtkDataArray *bin_arr = NULL;
    bool ownBinArr = false;
    if (inDS->GetPointData()->GetArray(var) != NULL)
    {
        //
        // The input is point-centered, but we would prefer zone-centered.
        //
        vtkDataSet *new_in_ds = (vtkDataSet *) inDS->NewInstance();
        new_in_ds->CopyStructure(inDS);
        new_in_ds->GetPointData()->AddArray(
                                          inDS->GetPointData()->GetArray(var));
        vtkPointDataToCellData *pd2cd = vtkPointDataToCellData::New();
        pd2cd->SetInput(new_in_ds);
        pd2cd->Update();

        bin_arr = pd2cd->GetOutput()->GetCellData()->GetArray(var);
        bin_arr->Register(NULL);
        ownBinArr = true;

        new_in_ds->Delete();
        pd2cd->Delete();
    }
    else
    {
        bin_arr = inDS->GetCellData()->GetArray(var);
    }
    if (bin_arr == NULL)
        EXCEPTION0(ImproperUseException);

    //
    // Now we will walk through each value and sort them into bins.
    //
    int nvals = bin_arr->GetNumberOfTuples();
    float binStep = (workingMax - workingMin) / (workingNumBins);
    for (int i = 0 ; i < nvals ; i++)
    {
        float val = bin_arr->GetTuple1(i);
        if (val < workingMin || val > workingMax)
            continue;
        int index = (int)((val - workingMin) / binStep);
        if (index >= workingNumBins)
            index = workingNumBins-1;
        float amount = amount_arr->GetTuple1(i);
        bins[index] += amount;
    }

    if (ownBinArr)
        bin_arr->Delete();

    return NULL;
}


// ****************************************************************************
//  Method: avtHistogramFilter::RefashionDataObjectInfo
//
//  Purpose:
//      Allows the filter to change its output's data object information, which
//      is a description of the data object.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Thu Jun 26 09:04:54 PDT 2003
//
//  Modifications:
//    Kathleen Bonnell, Thu Aug 12 08:44:36 PDT 2004
//    Allow this plot to be queryable.
//
// ****************************************************************************

void
avtHistogramFilter::RefashionDataObjectInfo(void)
{
    avtDataAttributes &outAtts     = GetOutput()->GetInfo().GetAttributes();
    avtDataValidity   &outValidity = GetOutput()->GetInfo().GetValidity();
 
    outAtts.SetTopologicalDimension(1);
    outAtts.SetSpatialDimension(2);
    outValidity.InvalidateZones();
    outValidity.SetNormalsAreInappropriate(true);
    outValidity.InvalidateSpatialMetaData();
    outValidity.SetPointsWereTransformed(true);
}


// ****************************************************************************
//  Method: avtHistogramFilter::PerformRestriction
//
//  Purpose:
//      Tells the input that we don't want ghost zones.  They cost time to
//      calculate (in some cases) and skew results towards the domain
//      boundaries.
//
//  Programmer: Hank Childs
//  Creation:   October 18, 2003
//
//  Modifications:
//
//    Hank Childs, Wed Aug 11 09:01:27 PDT 2004
//    Allow the input to have ghost zones, so that we can play better with 
//    other filters.  The histogram filter will now remove ghost zones before
//    executing.
//
// ****************************************************************************

avtPipelineSpecification_p
avtHistogramFilter::PerformRestriction(avtPipelineSpecification_p spec)
{
    avtPipelineSpecification_p newspec = new avtPipelineSpecification(spec);

    newspec->NoDynamicLoadBalancing();

    return newspec;
}


