// ************************************************************************* //
//  File: avtThresholdFilter.C
// ************************************************************************* //

#include <avtThresholdFilter.h>

#include <vtkCellData.h>
#include <vtkPointData.h>
#include <vtkPolyData.h>
#include <vtkThreshold.h>
#include <vtkUnstructuredGrid.h>

#include <avtIntervalTree.h>
#include <avtMetaData.h>

#include <DebugStream.h>
#include <ImproperUseException.h>
#include <NoDefaultVariableException.h>


// ****************************************************************************
//  Method:  avtThresholdFilter::Create
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Oct 23 16:38:18 PST 2001
//
// ****************************************************************************

avtFilter *
avtThresholdFilter::Create()
{
    return new avtThresholdFilter();
}


// ****************************************************************************
//  Method:      avtThresholdFilter::SetAtts
//
//  Purpose:
//      Sets the state of the filter based on the attribute object.
//
//  Arguments:
//      a        The attributes to use.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Oct 23 16:38:18 PST 2001
//
//  Modifications:
//
//    Hank Childs, Thu Oct 25 09:17:54 PDT 2001
//    Register the active variable.
//
// ****************************************************************************

void
avtThresholdFilter::SetAtts(const AttributeGroup *a)
{
    atts = *(const ThresholdAttributes*)a;

    // We need to specify that we want a secondary variable as soon as 
    // possible.
    if (strcmp(atts.GetVariable().c_str(), "default") != 0)
    {
        SetActiveVariable(atts.GetVariable().c_str());
    }
}


// ****************************************************************************
//  Method: avtThresholdFilter::Equivalent
//
//  Purpose:
//      Returns true if creating a new avtThresholdFilter with the given
//      parameters would result in an equivalent avtThresholdFilter.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Oct 23 16:38:18 PST 2001
//
// ****************************************************************************

bool
avtThresholdFilter::Equivalent(const AttributeGroup *a)
{
    return (atts == *(ThresholdAttributes*)a);
}


// ****************************************************************************
//  Method: avtThresholdFilter::ExecuteData
//
//  Purpose:
//      Sends the specified input and output through the Threshold filter.
//
//  Arguments:
//      in_ds      The input dataset.
//      <unused>   The domain number.
//      <unused>   The label.
//
//  Returns:       The output dataset.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Tue Oct 23 16:38:18 PST 2001
//
//  Modifications:
//
//    Hank Childs, Thu Oct 25 09:17:54 PDT 2001
//    Removed references to active variables.
//
//    Hank Childs, Tue Oct 30 09:29:13 PST 2001
//    Account for cases when we are not switching active variables.
//
//    Kathleen Bonnell, Fri Feb  8 11:03:49 PST 2002
//    vtkScalars has been deprecated in VTK 4.0, use vtkDataArray instead.
//
//    Hank Childs, Tue Sep 10 18:29:14 PDT 2002
//    Make memory management be self-contained in this routine.
//
//    Hank Childs, Sat Jun 21 09:25:34 PDT 2003
//    If we have poly data input, then we should have poly data output.
//
//    Hank Childs, Fri May  7 08:33:17 PDT 2004
//    If the variable is not a scalar, then issue a warning.
//
//    Hank Childs, Thu Jul 29 09:43:58 PDT 2004
//    Do not convert output to poly-data, since the base class will now
//    take care of this (when appropriate).
//
// ****************************************************************************

vtkDataSet *
avtThresholdFilter::ExecuteData(vtkDataSet *in_ds, int, std::string)
{
    string v1 = atts.GetVariable();
    string thres_var;
    if (v1 == "default")
        thres_var = pipelineVariable;
    else
        thres_var = v1;
    vtkDataArray *arr = in_ds->GetPointData()->GetArray(thres_var.c_str());
    if (arr == NULL)
        arr = in_ds->GetCellData()->GetArray(thres_var.c_str());

    if (arr == NULL)
    {
        char str[1024];
        sprintf(str, "The threshold operator was not applied because the "
                "variable \"%s\" could not be located.", thres_var.c_str());
        EXCEPTION1(VisItException, str);
    }
    if (arr->GetNumberOfComponents() != 1)
    {
        char str[1024];
        sprintf(str, "The threshold operator was not applied because the "
                "variable \"%s\" is not a scalar.", thres_var.c_str());
        EXCEPTION1(VisItException, str);
    }

    vtkThreshold *threshold = vtkThreshold::New();

    //
    // Set up the threshold filter from the attributes.
    //
    threshold->SetInput(in_ds);
    threshold->SetAttributeModeToDefault();
    if (atts.GetAmount() == ThresholdAttributes::Some)
    {
        threshold->AllScalarsOff();
    }
    else if (atts.GetAmount() == ThresholdAttributes::All)
    {
        threshold->AllScalarsOn();
    }
    else
    {
        debug1 << "No good option for threshold attributes for what amount to "
               << "restrict by." << endl;
        threshold->AllScalarsOff();
    }
    threshold->ThresholdBetween(atts.GetLbound(), atts.GetUbound());

    bool usePointData = false;
    if (switchVariables)
    {
        usePointData = activeVariableIsPointData;
    }
    else
    {
        vtkDataArray *s = in_ds->GetPointData()->GetScalars();
        usePointData = (s != NULL ? true : false);
    }
    if (usePointData)
    {
        threshold->SetAttributeModeToUsePointData();
    }
    else
    {
        threshold->SetAttributeModeToUseCellData();
    }

    vtkDataSet *out_ds = threshold->GetOutput();
    out_ds->Update();

    if (out_ds->GetNumberOfCells() <= 0)
    {
        out_ds = NULL;
    }

    ManageMemory(out_ds);
    threshold->Delete();

    return out_ds;
}


// ****************************************************************************
//  Method: avtThresholdFilter::RefashionDataObjectInfo
//
//  Purpose:
//      Indicates the zones no longer correspond to the original problem.
//
//  Programmer: Hank Childs
//  Creation:   October 23, 2001
//
// ****************************************************************************

void
avtThresholdFilter::RefashionDataObjectInfo(void)
{
    GetOutput()->GetInfo().GetValidity().InvalidateZones();
}


// ****************************************************************************
//  Method: avtThresholdFilter::PreExecute
//
//  Purpose:
//      Determine if there is a "default" variable to work with.
//
//  Programmer: Hank Childs
//  Creation:   August 28, 2002
//
//  Modifications:
//
//    Hank Childs, Thu Aug 29 08:12:22 PDT 2002
//    If the variable was not the 'default' variable, the test was sometimes
//    wrong.
//
// ****************************************************************************

void
avtThresholdFilter::PreExecute(void)
{
    avtPluginStreamer::PreExecute();

    if (atts.GetVariable() == "default" &&
        GetInput()->GetInfo().GetAttributes().GetVariableName() == "<unknown>")
    {
        //
        // Somehow the variable we asked for didn't make it down far enough.
        // This often happens when we are doing a plot that doesn't have a
        // variable (say a material plot) and then we apply the threshold
        // operator.
        //
        EXCEPTION1(NoDefaultVariableException, "Threshold");
    }
}


// ****************************************************************************
//  Method: avtThresholdFilter::PerformRestriction
//
//  Purpose:
//      Restrict the data processed by looking at the data extents.
//
//  Programmer: Hank Childs
//  Creation:   May 14, 2003
//
//  Modifications:
//
//    Mark C. Miller, Mon Oct 18 13:02:37 PDT 2004
//    Added code to pass variable name in call to GetDataExtents
//
// ****************************************************************************

avtPipelineSpecification_p
avtThresholdFilter::PerformRestriction(avtPipelineSpecification_p in_spec)
{
    string thres_var = atts.GetVariable();;
    if (thres_var == "default")
        thres_var = in_spec->GetDataSpecification()->GetVariable();

    avtIntervalTree *it = GetMetaData()->GetDataExtents(thres_var.c_str());
    if (it == NULL)
    {
        return in_spec;
    }

    avtPipelineSpecification_p spec = new avtPipelineSpecification(in_spec);
    float min = atts.GetLbound();
    float max = atts.GetUbound();
    vector<int> dl;
    it->GetDomainsListFromRange(&min, &max, dl);
    spec->GetDataSpecification()->GetRestriction()->RestrictDomains(dl);

    return spec;
}


