// ************************************************************************* //
//  File: avtRevolveFilter.C
// ************************************************************************* //

#include <avtRevolveFilter.h>

#include <math.h>
#include <float.h> // for FLT_MAX

#include <vtkCell.h>
#include <vtkCellData.h>
#include <vtkIdList.h>
#include <vtkMatrix4x4.h>
#include <vtkPointData.h>
#include <vtkUnstructuredGrid.h>

#include <avtExtents.h>

#include <BadVectorException.h>
#include <InvalidCellTypeException.h>
#include <InvalidDimensionsException.h>


static void GetRotationMatrix(double angle, double axis[3], vtkMatrix4x4 *mat);


// ****************************************************************************
//  Method: avtRevolveFilter constructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Dec 11 11:31:52 PDT 2002
//
// ****************************************************************************

avtRevolveFilter::avtRevolveFilter()
{
}


// ****************************************************************************
//  Method: avtRevolveFilter destructor
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Dec 11 11:31:52 PDT 2002
//
//  Modifications:
//
// ****************************************************************************

avtRevolveFilter::~avtRevolveFilter()
{
}


// ****************************************************************************
//  Method:  avtRevolveFilter::Create
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Dec 11 11:31:52 PDT 2002
//
// ****************************************************************************

avtFilter *
avtRevolveFilter::Create()
{
    return new avtRevolveFilter();
}


// ****************************************************************************
//  Method:      avtRevolveFilter::SetAtts
//
//  Purpose:
//      Sets the state of the filter based on the attribute object.
//
//  Arguments:
//      a        The attributes to use.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Dec 11 11:31:52 PDT 2002
//
//  Modifications:
//    Kathleen Bonnell, Wed May 21 11:10:58 PDT 2003   
//    Test for bad Rotation Axis (0, 0, 0), throw exception when encountered.
//
// ****************************************************************************

void
avtRevolveFilter::SetAtts(const AttributeGroup *a)
{
    atts = *(const RevolveAttributes*)a;
    const double *axis = atts.GetAxis();
    if (axis[0] == 0. && axis[1] == 0. && axis[2] == 0.)
    {
        EXCEPTION1(BadVectorException, "Axis of Revolution");
        return;
    }
}


// ****************************************************************************
//  Method: avtRevolveFilter::Equivalent
//
//  Purpose:
//      Returns true if creating a new avtRevolveFilter with the given
//      parameters would result in an equivalent avtRevolveFilter.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Dec 11 11:31:52 PDT 2002
//
// ****************************************************************************

bool
avtRevolveFilter::Equivalent(const AttributeGroup *a)
{
    return (atts == *(RevolveAttributes*)a);
}


// ****************************************************************************
//  Method: avtRevolveFilter::ExecuteData
//
//  Purpose:
//      Sends the specified input and output through the Revolve filter.
//
//  Arguments:
//      in_ds      The input dataset.
//      <unused>   The domain number.
//      <unused>   The label.
//
//  Returns:       The output dataset.
//
//  Programmer: childs -- generated by xml2info
//  Creation:   Wed Dec 11 11:31:52 PDT 2002
//
// ****************************************************************************

vtkDataSet *
avtRevolveFilter::ExecuteData(vtkDataSet *in_ds, int, std::string)
{
    int   i, j;

    //
    // Get information about how to revolve.
    //
    int    nsteps      = atts.GetSteps();
    double start_angle = atts.GetStartAngle();
    double stop_angle  = atts.GetStopAngle();
    if (start_angle == stop_angle)
    {
        return in_ds;
    }
    if (start_angle > stop_angle)
    {
        start_angle = atts.GetStopAngle();
        stop_angle  = atts.GetStartAngle();
    }
    double axis[3];
    axis[0] = atts.GetAxis()[0];
    axis[1] = atts.GetAxis()[1];
    axis[2] = atts.GetAxis()[2];

    //
    // Set up our VTK structures.
    //
    vtkMatrix4x4 *mat = vtkMatrix4x4::New();
    vtkUnstructuredGrid *ugrid = vtkUnstructuredGrid::New();
    vtkPoints *pts = vtkPoints::New();
    int npts = in_ds->GetNumberOfPoints();
    int ncells = in_ds->GetNumberOfCells();
    int n_out_pts = npts*nsteps;
    pts->SetNumberOfPoints(n_out_pts);
    ugrid->SetPoints(pts);

    //
    // Create the points for each timestep.
    //
    int niter = (fabs(stop_angle-start_angle-360.) < 0.001 ? nsteps-1 : nsteps);
    float *ptr = (float *) pts->GetVoidPointer(0);
    for (i = 0 ; i < niter ; i++)
    {
        double angle = ((stop_angle-start_angle)*i)/(nsteps-1) + start_angle;
        GetRotationMatrix(angle, axis, mat);
        for (j = 0 ; j < npts ; j++)
        {
            float pt[4];
            in_ds->GetPoint(j, pt);
            pt[3] = 1.;
            float outpt[4];
            mat->MultiplyPoint(pt, outpt);
            ptr[0] = outpt[0];
            ptr[1] = outpt[1];
            ptr[2] = outpt[2];
            ptr += 3;
        }
    }

    //
    // Now set up the connectivity.  The output will consist of revolved
    // quads (-> hexes) and revolved triangles (-> wedges).  No special care is
    // given to the case where an edge of a cell lies directly on the axis of
    // revolution (ie: you get a degenerate hex, not a wedge).
    //
    int n_out_cells = ncells*(nsteps-1);
    ugrid->Allocate(8*n_out_cells);
    bool overlap_ends = (fabs(stop_angle-start_angle-360.) < 0.001);
    for (i = 0 ; i < ncells ; i++)
    {
         vtkCell *cell = in_ds->GetCell(i);
         int c = cell->GetCellType();
         if (c != VTK_QUAD && c != VTK_TRIANGLE && c != VTK_PIXEL)
         {
             EXCEPTION1(InvalidCellTypeException, "anything but quads and"
                                                  " tris.");
         }
         vtkIdList *list = cell->GetPointIds();

         if (c == VTK_TRIANGLE)
         {
             int pt0 = list->GetId(0);
             int pt1 = list->GetId(1);
             int pt2 = list->GetId(2);
             for (j = 0 ; j < nsteps-1 ; j++)
             {
                 vtkIdType wedge[6];
                 wedge[0] = npts*j + pt0;
                 wedge[1] = npts*j + pt1;
                 wedge[2] = npts*j + pt2;
                 wedge[3] = npts*(j+1) + pt0;
                 wedge[4] = npts*(j+1) + pt1;
                 wedge[5] = npts*(j+1) + pt2;
                 if (j == nsteps-2 && overlap_ends)
                 {
                     wedge[3] = pt0;
                     wedge[4] = pt1;
                     wedge[5] = pt2;
                 }
                 ugrid->InsertNextCell(VTK_WEDGE, 6, wedge);
             }
         }
         else
         {
             int pt0 = list->GetId(0);
             int pt1 = list->GetId(1);
             int pt2 = list->GetId(2);
             int pt3 = list->GetId(3);
             if (c == VTK_PIXEL)
             {
                 pt2 = list->GetId(3);
                 pt3 = list->GetId(2);
             }
             for (j = 0 ; j < nsteps-1 ; j++)
             {
                 vtkIdType hex[6];
                 hex[0] = npts*j + pt0;
                 hex[1] = npts*j + pt1;
                 hex[2] = npts*j + pt2;
                 hex[3] = npts*j + pt3;
                 hex[4] = npts*(j+1) + pt0;
                 hex[5] = npts*(j+1) + pt1;
                 hex[6] = npts*(j+1) + pt2;
                 hex[7] = npts*(j+1) + pt3;
                 if (j == nsteps-2 && overlap_ends)
                 {
                     hex[4] = pt0;
                     hex[5] = pt1;
                     hex[6] = pt2;
                     hex[7] = pt3;
                 }
                 ugrid->InsertNextCell(VTK_HEXAHEDRON, 8, hex);
             }
         }
    }

    vtkCellData *incd   = in_ds->GetCellData();
    vtkCellData *outcd  = ugrid->GetCellData();
    outcd->CopyAllocate(incd, n_out_cells);
    for (i = 0 ; i < n_out_cells ; i++)
    {
        outcd->CopyData(incd, i/(nsteps-1), i);
    }
    
    vtkPointData *inpd  = in_ds->GetPointData();
    vtkPointData *outpd = ugrid->GetPointData();
    outpd->CopyAllocate(inpd, n_out_pts);
    for (i = 0 ; i < n_out_pts ; i++)
    {
        outpd->CopyData(inpd, i%npts, i);
    }
    
    //
    // Clean up.
    //
    ManageMemory(ugrid);
    ugrid->Delete();
    mat->Delete();
    pts->Delete();

    return ugrid;
}


// ****************************************************************************
//  Method: avtRevolveFilter::RefashionDataObjectInfo
//
//  Purpose:
//      Tells the output that it is now higher in topological dimension.
//
//  Programmer: Hank Childs
//  Creation:   December 11, 2002
//
//  Modifications:
//    Kathleen Bonnell, Mon Apr 14 09:57:39 PDT 2003 
//    Set CanUseTransform to false.
//
// ****************************************************************************

void
avtRevolveFilter::RefashionDataObjectInfo(void)
{
    avtDataAttributes &inAtts      = GetInput()->GetInfo().GetAttributes();
    avtDataAttributes &outAtts     = GetOutput()->GetInfo().GetAttributes();
    avtDataValidity   &outValidity = GetOutput()->GetInfo().GetValidity();

    outAtts.SetTopologicalDimension(inAtts.GetTopologicalDimension()+1);
    if (inAtts.GetSpatialDimension() >= 2)
    {
        outAtts.SetSpatialDimension(3);
    }
    else
    {
        outAtts.SetSpatialDimension(inAtts.GetSpatialDimension()+1);
    }
    outValidity.InvalidateZones();
    outValidity.SetPointsWereTransformed(true);
    outValidity.InvalidateSpatialMetaData();

    //
    // This filter invalidates any transform matrix in the pipeline.
    //
    outAtts.SetCanUseTransform(false);

    //
    // Now revolve the extents.
    //
    double b[6];
    int olddim = inAtts.GetSpatialDimension();
    if (inAtts.GetTrueSpatialExtents()->HasExtents())
    {
        inAtts.GetTrueSpatialExtents()->CopyTo(b);
        RevolveExtents(b, olddim);
        outAtts.GetTrueSpatialExtents()->Set(b);
    }

    if (inAtts.GetCumulativeTrueSpatialExtents()->HasExtents())
    {
        inAtts.GetCumulativeTrueSpatialExtents()->CopyTo(b);
        RevolveExtents(b, olddim);
        outAtts.GetCumulativeTrueSpatialExtents()->Set(b);
    }

    if (inAtts.GetEffectiveSpatialExtents()->HasExtents())
    {
        inAtts.GetEffectiveSpatialExtents()->CopyTo(b);
        RevolveExtents(b, olddim);
        outAtts.GetEffectiveSpatialExtents()->Set(b);
    }

    if (inAtts.GetCurrentSpatialExtents()->HasExtents())
    {
        inAtts.GetCurrentSpatialExtents()->CopyTo(b);
        RevolveExtents(b, olddim);
        outAtts.GetCurrentSpatialExtents()->Set(b);
    }

    if (inAtts.GetCumulativeCurrentSpatialExtents()->HasExtents())
    {
        inAtts.GetCumulativeCurrentSpatialExtents()->CopyTo(b);
        RevolveExtents(b, olddim);
        outAtts.GetCumulativeCurrentSpatialExtents()->Set(b);
    }
}


// ****************************************************************************
//  Method: avtRevolveFilter::RevolveExtents
//
//  Purpose:
//      Determines the extents of a dataset revolved around an axis.
//
//  Programmer: Hank Childs
//  Creation:   December 11, 2002
//
// ****************************************************************************

void
avtRevolveFilter::RevolveExtents(double *dbounds, int spat_dim)
{
    int    nsteps      = atts.GetSteps();
    double start_angle = atts.GetStartAngle();
    double stop_angle  = atts.GetStopAngle();
    double axis[3];
    axis[0] = atts.GetAxis()[0];
    axis[1] = atts.GetAxis()[1];
    axis[2] = atts.GetAxis()[2];

    vtkMatrix4x4 *mat = vtkMatrix4x4::New();
    double new_bounds[6];
    new_bounds[0] = FLT_MAX;
    new_bounds[1] = -FLT_MAX;
    new_bounds[2] = FLT_MAX;
    new_bounds[3] = -FLT_MAX;
    new_bounds[4] = FLT_MAX;
    new_bounds[5] = -FLT_MAX;

    //
    // Revolve the bounding box for each increment in theta.
    //
    for (int i = 0 ; i < nsteps ; i++)
    {
        double angle = ((stop_angle-start_angle)*i)/(nsteps-1) + start_angle;
        GetRotationMatrix(angle, axis, mat);
        int iters = 1;
        for (int k = 0 ; k < spat_dim ; k++)
            iters *= 2;
        for (int j = 0 ; j < iters ; j++)
        {
            double pt[4];
            pt[0] = (spat_dim >= 1 ? (j & 1 ? dbounds[1] : dbounds[0]) : 0.);
            pt[1] = (spat_dim >= 2 ? (j & 2 ? dbounds[3] : dbounds[2]) : 0.);
            pt[2] = (spat_dim >= 3 ? (j & 4 ? dbounds[5] : dbounds[4]) : 0.);
            pt[3] = 1.;

            double outpt[4];
            mat->MultiplyPoint(pt, outpt);
            new_bounds[0] = (outpt[0]<new_bounds[0] ? outpt[0] :new_bounds[0]);
            new_bounds[1] = (outpt[0]>new_bounds[1] ? outpt[0] :new_bounds[1]);
            new_bounds[2] = (outpt[1]<new_bounds[2] ? outpt[1] :new_bounds[2]);
            new_bounds[3] = (outpt[1]>new_bounds[3] ? outpt[1] :new_bounds[3]);
            new_bounds[4] = (outpt[2]<new_bounds[4] ? outpt[2] :new_bounds[4]);
            new_bounds[5] = (outpt[2]>new_bounds[5] ? outpt[2] :new_bounds[5]);
        }
    }

    mat->Delete();
    dbounds[0] = new_bounds[0];
    dbounds[1] = new_bounds[1];
    dbounds[2] = new_bounds[2];
    dbounds[3] = new_bounds[3];
    dbounds[4] = new_bounds[4];
    dbounds[5] = new_bounds[5];
}


// ****************************************************************************
//  Method: avtRevolveFilter::VerifyInput
//
//  Purpose:
//      Verifies that the input does not already have a topological dimension
//      of 3.
//
//  Programmer: Hank Childs
//  Creation:   December 11, 2002
//
// ****************************************************************************

void
avtRevolveFilter::VerifyInput(void)
{
    if  (GetInput()->GetInfo().GetAttributes().GetTopologicalDimension() >= 3)
    {
        EXCEPTION2(InvalidDimensionsException, "Revolve", "<=2D");
    }
}


// ****************************************************************************
//  Function: GetRotationMatrix
//
//  Purpose:
//      Given an angle and an axis, this creates a matrix that will rotate a
//      point around that axis.
//
//  Programmer: Hank Childs
//  Creation:   December 11, 2002
//
// ****************************************************************************

static void
GetRotationMatrix(double angle, double axis[3], vtkMatrix4x4 *mat)
{
    //
    // The game plan is to transform into a coordinate space that we are
    // familiar with, perform the rotation there, and then rotate back.
    //

    //
    // First rotate to the yz plane.  We will do this by rotating by theta
    // around the y-axis, where theta is arctan(axis[0] / axis[2]).
    //
    vtkMatrix4x4 *rot1 = vtkMatrix4x4::New();
    rot1->Identity();
    vtkMatrix4x4 *rot5 = vtkMatrix4x4::New();
    rot5->Identity();
    if (axis[0] != 0.)
    {
        double theta = atan2(axis[0], axis[2]);
        double cos_theta = cos(theta);
        double sin_theta = sin(theta);
        rot1->SetElement(0, 0, cos_theta);
        rot1->SetElement(2, 0, -sin_theta);
        rot1->SetElement(0, 2, sin_theta);
        rot1->SetElement(2, 2, cos_theta);

        //
        // Rot 5 will be a rotation around -theta.
        //
        double cos_minus_theta = cos_theta;
        double sin_minus_theta = -sin_theta;
        rot5->SetElement(0, 0, cos_minus_theta);
        rot5->SetElement(2, 0, -sin_minus_theta);
        rot5->SetElement(0, 2, sin_minus_theta);
        rot5->SetElement(2, 2, cos_minus_theta);
    }

    //
    // Now rotate around the x-axis until we get the to the z-axis.
    //
    vtkMatrix4x4 *rot2 = vtkMatrix4x4::New();
    rot2->Identity();
    vtkMatrix4x4 *rot4 = vtkMatrix4x4::New();
    rot4->Identity();
    if (axis[1] != 0.)
    {
        double theta = atan2(axis[1], sqrt(axis[0]*axis[0] + axis[2]*axis[2]));
        double cos_theta = cos(theta);
        double sin_theta = sin(theta);
        rot2->SetElement(1, 1, cos_theta);
        rot2->SetElement(2, 1, sin_theta);
        rot2->SetElement(1, 2, -sin_theta);
        rot2->SetElement(2, 2, cos_theta);

        //
        // Rot 4 will be a rotation around -theta.
        //
        double cos_minus_theta = cos_theta;
        double sin_minus_theta = -sin_theta;
        rot4->SetElement(1, 1, cos_minus_theta);
        rot4->SetElement(2, 1, sin_minus_theta);
        rot4->SetElement(1, 2, -sin_minus_theta);
        rot4->SetElement(2, 2, cos_minus_theta);
    }

    //
    // Now we can do the easy rotation around the z-axis.
    //
#if defined(_WIN32)
    double angle_rad = (angle / 360. * 2. * 3.14159);
#else
    double angle_rad = (angle / 360. * 2. * M_PI);
#endif
    vtkMatrix4x4 *rot3 = vtkMatrix4x4::New();
    rot3->Identity();
    double cos_angle = cos(angle_rad);
    double sin_angle = sin(angle_rad);
    rot3->SetElement(0, 0, cos_angle);
    rot3->SetElement(1, 0, sin_angle);
    rot3->SetElement(0, 1, -sin_angle);
    rot3->SetElement(1, 1, cos_angle);

    //
    // Now set up our matrix.
    //
    vtkMatrix4x4 *tmp  = vtkMatrix4x4::New();
    vtkMatrix4x4 *tmp2 = vtkMatrix4x4::New();
    vtkMatrix4x4::Multiply4x4(rot5, rot4, tmp);
    vtkMatrix4x4::Multiply4x4(tmp, rot3, tmp2);
    vtkMatrix4x4::Multiply4x4(tmp2, rot2, tmp);
    vtkMatrix4x4::Multiply4x4(tmp, rot1, mat);

    tmp->Delete();
    tmp2->Delete();
    rot1->Delete();
    rot2->Delete();
    rot3->Delete();
    rot4->Delete();
    rot5->Delete();
}


